﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace LayoutEditor.src.CombinerEditor
{
    /// <summary>
    /// Enum for effect combiner communication status.
    /// </summary>
    public enum CombinerCommunicationStatus
    {
        /// <summary>The communication is stopped.</summary>
        Stopped,

        /// <summary>The communication is established.</summary>
        Running,

        /// <summary>The communication is failed.</summary>
        Failed,
    }

    /// <summary>
    /// Interface of Manager class for the communications between EffectCombinerEditor and EffectMaker.
    /// </summary>
    public interface ICombinerCommunicationBridge
    {
        /// <summary>
        /// Event fired when communication status changed.
        /// </summary>
        event EventHandler<EffectCombinerCommunicationStatusEventArgs> StatusChanged;

        /// <summary>
        /// Event fired when shader code changed.
        /// </summary>
        event EventHandler<CombinerShaderChangedEventArgs> ShaderChanged;

        /// <summary>
        /// Get the current communication status.
        /// </summary>
        CombinerCommunicationStatus Status
        {
            get;
        }

        /// <summary>
        /// Get the error while communication.
        /// </summary>
        Exception Error { get; }

        /// <summary>
        /// Initialize.
        /// </summary>
        void Init();

        /// <summary>
        /// Deinitialize.
        /// </summary>
        void Deinit();

        /// <summary>
        /// Set shader source code that is received from connected effect combiner editor.
        /// </summary>
        /// <param name="combinerProjectPath">The combiner editor project file path.</param>
        /// <param name="shaderSourceCode">The generated shader code.</param>
        void SetReceivedShaderSourceCode(string combinerProjectPath, string shaderSourceCode, string workFilePath);

        /// <summary>
        /// 指定したパスのキャッシュを破棄して、もう一度ファイルからシェーダが生成されるようにします。
        /// </summary>
        /// <param name="combinerProjectPath">コンバイナシェーダのファイルパス</param>
        void DiscardCache(string combinerProjectPath);

        /// <summary>
        /// Clear all the shader source code received from connected effect combiner editor.
        /// </summary>
        void ClearReceivedShaderSourceCode();

        /// <summary>
        /// Get the shader source code that is generated from
        /// the given combiner editor project.
        /// </summary>
        /// <param name="combinerProjectPath">The combiner editor project file path.</param>
        /// <returns>The shader source code if found, or empty string is returned.</returns>
        string GetShaderSourceCode(string combinerProjectPath);
    }

    /// <summary>
    /// Event arguments for effect combiner communication status.
    /// </summary>
    public class EffectCombinerCommunicationStatusEventArgs : EventArgs
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="oldStatus">The old status.</param>
        /// <param name="newStatus">The new status.</param>
        public EffectCombinerCommunicationStatusEventArgs(
            CombinerCommunicationStatus oldStatus,
            CombinerCommunicationStatus newStatus)
        {
            this.OldStatus = oldStatus;
            this.NewStatus = newStatus;
        }

        /// <summary>
        /// Get the old status.
        /// </summary>
        public CombinerCommunicationStatus OldStatus { get; private set; }

        /// <summary>
        /// Get the new status.
        /// </summary>
        public CombinerCommunicationStatus NewStatus { get; private set; }
    }

    /// <summary>
    /// Event arguments for effect combiner ShaderChanged event.
    /// </summary>
    public class CombinerShaderChangedEventArgs : EventArgs
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="combinerProjectPath">The effect combiner project path of the changed shader.</param>
        public CombinerShaderChangedEventArgs(string combinerProjectPath, string combinerWorkFilePath)
        {
            this.ProjectPath = combinerProjectPath;
            this.WorkFilePath = combinerWorkFilePath;
        }

        /// <summary>
        /// Get the effect combiner project path of the changed shader.
        /// </summary>
        public string ProjectPath { get; private set; }

        /// <summary>
        /// Get the effect combiner work file path of the changed shader.
        /// </summary>
        public string WorkFilePath { get; private set; }
    }
}
