﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace LayoutEditor.src.CombinerEditor
{
    public class CombinerCommunication : LayoutEditor.src.CombinerEditor.ICombinerCommunicationBridge
    {
        /// <summary>
        /// A dictionary stores the shader source codes received from connected effect combiner.
        /// </summary>
        private Dictionary<string, string> ReceivedShaderCodes = new Dictionary<string, string>();

        /// <summary>The communication task.</summary>
        private Task runningTask;

        /// <summary>The communications status.</summary>
        private LayoutEditor.src.CombinerEditor.CombinerCommunicationStatus status = LayoutEditor.src.CombinerEditor.CombinerCommunicationStatus.Stopped;

        /// <summary>The communication server.</summary>
        private LayoutEditor.src.CombinerEditor.CommunicationServer<Communication.Contracts.IShaderTransferServiceV2> communicationServer = null;

        /// <summary>
        /// Get the error while communication.
        /// </summary>
        public Exception Error { get; private set; }

        /// <summary>
        /// Get the current communication status.
        /// </summary>
        public LayoutEditor.src.CombinerEditor.CombinerCommunicationStatus Status
        {
            get
            {
                return status;
            }

            private set
            {
                if (status == value)
                {
                    return;
                }

                var oldStatus = status;
                status = value;

                OnStatusChanged(
                    new LayoutEditor.src.CombinerEditor.EffectCombinerCommunicationStatusEventArgs(oldStatus, status));
            }
        }

        public event EventHandler<LayoutEditor.src.CombinerEditor.CombinerShaderChangedEventArgs> ShaderChanged;
        public event EventHandler<LayoutEditor.src.CombinerEditor.EffectCombinerCommunicationStatusEventArgs> StatusChanged;

        public void ClearReceivedShaderSourceCode()
        {
            var combinerProjectPaths = ReceivedShaderCodes.Keys.ToArray();
            ReceivedShaderCodes.Clear();
            if (ShaderChanged != null)
            {
                foreach (string path in combinerProjectPaths)
                {
                    ShaderChanged(
                        null,
                        new CombinerShaderChangedEventArgs(path, null));
                }
            }
        }

        public void Deinit()
        {
            throw new NotImplementedException();
        }

        public void DiscardCache(String combinerProjectPath)
        {
            throw new NotImplementedException();
        }

        public String GetShaderSourceCode(String combinerProjectPath)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Initialize.
        /// </summary>
        public void Init()
        {
            // Create the communication server.
            communicationServer = new LayoutEditor.src.CombinerEditor.CommunicationServer<Communication.Contracts.IShaderTransferServiceV2>();

            runningTask = communicationServer.Start(new CombinerShaderTransferService(this))
                .ContinueWith(t =>
                {
                    if (t.IsFaulted)
                    {
                        Error = t.Exception;
                        Status = LayoutEditor.src.CombinerEditor.CombinerCommunicationStatus.Failed;
                    }
                    else
                    {
                        Error = null;
                        Status = LayoutEditor.src.CombinerEditor.CombinerCommunicationStatus.Running;
                    }
            });
        }

        public void SetReceivedShaderSourceCode(String combinerProjectPath, String shaderSourceCode, String workFilePath)
        {
            ReceivedShaderCodes[combinerProjectPath] = shaderSourceCode;
            if (ShaderChanged != null)
            {
                ShaderChanged(
                    null,
                    new LayoutEditor.src.CombinerEditor.CombinerShaderChangedEventArgs(combinerProjectPath, workFilePath));
            }
        }

        /// <summary>
        /// Helper method for firing communication status changed event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        private void OnStatusChanged(
            LayoutEditor.src.CombinerEditor.EffectCombinerCommunicationStatusEventArgs e)
        {
            var handler = StatusChanged;
            if (handler != null)
            {
                handler(null, e);
            }
        }
    }
}
