﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Runtime.InteropServices;
using System.Threading;
using System.Windows.Forms;
using System.Xml.Serialization;

namespace LayoutEditor
{
    using System.Linq;
    using System.Text;
    using System.Web;
    using LayoutEditor.Forms;
    using LECore;
    using LECore.Manipulator;
    using LECore.Save_Load;
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Util;
    using Structures.SerializableObject;
    using Nintendo.Foundation.IO;

    /// <summary>
    /// アプリケーションを表すクラスです。
    /// </summary>
    public class AppMain
    {
        #region 定数
        /// <summary>
        /// 環境設定ファイル名
        /// </summary>
        const string _SettingFileNameOld = "setteing.xml";
        const string _SettingFileName = "setting.xml";


        #endregion 定数

        #region フィールド
        static Font _guiFont = null;
        // ＧＵＩフォント（太字）
        static Font _guiFontBold = null;
        // アプリケーション調整値
        static AppConfig _config = null;
        // ヘルプ
        static HelpProvider _hppLayoutEditor = new HelpProvider();
        static HelpProvider _hppViewer = new HelpProvider();
        // アプリケーション設定(永続化されます)
        static AppSetting _theAppSetting = null;

        static CommandLineOptions _commandLineOptions;

        // エラーメッセージ表示ダイアログ
        static internal AppMessageReporter _appMessageReporter;

        static LayoutEditor.Controls.SubsceneThumbnailCreator _thumbnailCreator;
        #endregion


        #region プロパティ

        /// <summary>
        /// フォントデータ
        /// </summary>
        public static Font GuiFont
        {
            get { return _guiFont; }
        }

        /// <summary>
        /// ＧＵＩフォント（太字）。
        /// </summary>
        public static Font GuiFontBold
        {
            get { return _guiFontBold; }
        }

        /// <summary>
        /// コンフィグデータ。
        /// </summary>
        public static AppConfig Config
        {
            get { return _config; }
        }

        /// <summary>
        /// ヘルププロバイダ
        /// </summary>
        public static HelpProvider LayoutEditorHelp
        {
            get { return _hppLayoutEditor; }
        }

        /// <summary>
        /// ビューア用ヘルププロバイダ
        /// </summary>
        public static HelpProvider ViewerHelp
        {
            get { return _hppViewer; }
        }
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AppMain() { }

        /// <summary>
        /// XXXマネージャのような、シングルトンクラスを初期化します。
        /// </summary>
        static void InitSingletonManagers_(CommandLineOptions commandLineOptions)
        {
            // デバッグコンソール作成
            DbgConsole.Initialize();

            // 文字列リソースマネージャ
            StringResMgr.Initialize();

            // メッセージ報告クラスを初期化
            Func<AppSetting> func = () => { return _theAppSetting; };
            _appMessageReporter = new AppMessageReporter(func);
            _appMessageReporter.DialogDisabled = commandLineOptions.SaveScreenCapture || commandLineOptions.BakeInfinity;

            // ログ出力クラスを初期化します。
            LECore.LayoutEditorCore.MsgReporter.OnReportMsg += _appMessageReporter.ReportMsgHandler;
            LECore.LayoutEditorCore.MsgReporter.OnReportLog += MsgReporter_OnReportLog;
            LECore.LayoutEditorCore.MsgReporter.OnShowDialog += MsgReporter_OnShowDialog;
            LECore.LayoutEditorCore.MsgReporter.OnBeginPacking += _appMessageReporter.BeginPacking;
            LECore.LayoutEditorCore.MsgReporter.OnEndPacking += _appMessageReporter.EndPacking;

            LECore.LayoutEditorCore.BeforeImport += LayoutEditorCore_BeforeImportExport_;

            LECore.LayoutEditorCore.BeforeExport += LayoutEditorCore_BeforeExport_;
            LECore.LayoutEditorCore.AfterExport += LayoutEditorCore_AfterExport_;
            LECore.LayoutEditorCore.ExportThumbnail += LayoutEditorCore_ExportThumbnail_;
            LECore.LayoutEditorCore.AddToRecentDocs += LayoutEditorCore_AddToRecentDocs;

            // ビューア実行用の初期化
            LECore.ViewerExecuter.Initialize();
        }

        /// <summary>
        /// 最近使ったファイルに登録
        /// </summary>
        private static void LayoutEditorCore_AddToRecentDocs(object sender, LayoutEditorCore.AddToRecentDocsEventArgs e)
        {
            try
            {
                var path = Path.GetFullPath(e.filePath);

                // レイアウトファイルのみ登録する
                if (string.Compare(Path.GetExtension(path), AppConstants.LayoutFileExt, true) == 0)
                {
                    _theAppSetting.RecentlyUsedFiles =
                        Enumerable.Repeat(path, 1)
                        .Concat(_theAppSetting.RecentlyUsedFiles.Where(x => string.Compare(x, path, true) != 0 && File.Exists(x)))
                        .Take(AppSetting.MaxRecentlyUsedFiles)
                        .ToArray();
                }
            }
            catch
            {
                Debug.Assert(false);
            }
        }

        static void LayoutEditorCore_BeforeExport_(object sender, LayoutEditorCore.BeforeExportEventArgs args)
        {
            LayoutEditorCore_BeforeImportExport_(args.FilePath);
            LECore.LayoutEditorCore.PrepareExport(args);
        }

        static bool LayoutEditorCore_AfterExport_(string filePath)
        {
            return true;
        }

        /// <summary>
        /// サムネイルの出力
        /// </summary>
        private static void LayoutEditorCore_ExportThumbnail_(object sender, LayoutEditorCore.ExportThumbnailEventArgs e)
        {
            var layoutPath = string.IsNullOrEmpty(e.SubSceneOriginalPath) ? e.SubScenePath : e.SubSceneOriginalPath;
            try
            {
                if (!_theAppSetting.ProjectSettings.SaveThumbnailImage)
                {
                    e.Result = false;
                    return;
                }

                if (_thumbnailCreator == null)
                {
                    _thumbnailCreator = new Controls.SubsceneThumbnailCreator(AppSetting.RendererType.D3D, _theAppSetting);

                }
                else
                {
                    _thumbnailCreator.SetAppSetting(_theAppSetting);
                }

                RectangleF rect = Controls.SubsceneThumbnailCreator.GetThubmnailBound(e.SubScene);
                var size = new Size(
                    Math.Max(Math.Min((int)Math.Ceiling(rect.Width), _theAppSetting.ProjectSettings.ThumbnailMaximumWidth), 1),
                    Math.Max(Math.Min((int)Math.Ceiling(rect.Width), _theAppSetting.ProjectSettings.ThumbnailMaximumHeight), 1)
                    );

                _thumbnailCreator.Size = size;
                var bitmap = _thumbnailCreator.GetCaptureImage(e.SubScene);
                if (bitmap == null)
                {
                    if (_thumbnailCreator.IsDeviceLost)
                    {
                        // 例外がおきるようになるので、作り直し
                        _thumbnailCreator.Dispose();
                        _thumbnailCreator = new Controls.SubsceneThumbnailCreator(AppSetting.RendererType.D3D, _theAppSetting);
                        _thumbnailCreator.Size = size;
                        bitmap = _thumbnailCreator.GetCaptureImage(e.SubScene);
                    }
                    else
                    {
                        e.Error = StringResMgr.Get("SYSTEM_UPDATE_THUMBNAIL_FILE_ERROR", layoutPath) + Environment.NewLine;
                        return;
                    }
                }

                using (bitmap)
                {
                    var directory = $"{Path.GetDirectoryName(layoutPath)}\\{AppConstants.ThumbnailDirectory}";
                    var path = $"{directory}\\{Path.GetFileName(layoutPath)}{AppConstants.ThumbnailExt}";
                    if (!Directory.Exists(directory))
                    {
                        Directory.CreateDirectory(directory);
                    }
                    bitmap.Save(path, System.Drawing.Imaging.ImageFormat.Png);
                }

                e.Result = true;
            }
            catch (Exception exception)
            {
                e.Error = StringResMgr.Get("SYSTEM_UPDATE_THUMBNAIL_FILE_ERROR", layoutPath) + Environment.NewLine + exception.Message + Environment.NewLine;
#if DEBUG
                e.Error += exception.ToString() + Environment.NewLine;
#endif
            }
        }

        /// <summary>
        /// ファイル入出力イベントハンドラ
        /// </summary>
        static bool LayoutEditorCore_BeforeImportExport_(string filePath)
        {
            MsgReporter_OnReportLog(null,  new MessageArgs() {title = "AppMain", msg = "BeforeImportExportHandler is called."});

            return true;
        }

        /// <summary>
        /// メッセージボックス表示ハンドラ
        /// </summary>
        static DialogResult MsgReporter_OnShowDialog(string title, string msg, MessageBoxButtons messageBoxButtons)
        {
            return MessageBox.Show(null, msg, title, messageBoxButtons);
        }

        /// <summary>
        /// アプリケーションのログ出力処理。
        /// </summary>
        static void MsgReporter_OnReportLog(object sender, MessageArgs args)
        {
            if (_commandLineOptions.Verbose)
            {
                Console.Out.WriteLine(string.Format("[{0}] ... {1}", args.title, args.msg));
            }
        }

        /// <summary>
        /// 多重起動の処理
        /// </summary>
        /// <param name="option">コマンドラインオプション</param>
        static void HandleMultipleLaunching_(Process layoutEditorProcess, CommandLineOptions option)
        {
            string optionString = CommandLineOptions.EncodeForWindowMessage(option);
            if (!string.IsNullOrEmpty(optionString))
            {
                // 入力ファイル名が指定されていれば...
                // Rlytファイルのダブルクリック起動など、
                // 入力ファイル名が指定されて起動した場合のみ、
                // 多重起動が禁止される点に注意してください。
                // すでに起動している、プロセスに文字列データを送信します。
                LECore.Win32.COPYDATASTRUCT cds = new LECore.Win32.COPYDATASTRUCT();
                cds.lpData = Marshal.StringToHGlobalAuto(optionString);
                cds.cbData = (optionString.Length + 1) * 2;

                // WM_COPYDATAでデータを送信する。
                // lparamにobject型を利用すると、実行に失敗するので注意。
                // (多分呼び出しの際に、マーシャリングが行われないため。）
                IntPtr result = LECore.Win32.User32.SendMessage(layoutEditorProcess.MainWindowHandle, LECore.Win32.WM.WM_COPYDATA, IntPtr.Zero, ref cds);
                Debug.Assert(result == IntPtr.Zero);
            }

            // フォアグランドウインドウに設定します。
            LECore.Win32.User32.SetForegroundWindow(layoutEditorProcess.MainWindowHandle);
        }

        /// <summary>
        /// cygwin の仕様変更によって、1.7.10 以降の bash からツールが実行された場合、tmp と TMP など大文字・小文字違いの同名
        /// 環境変数が定義され、XmlSerializer内部で例外が発生してしまう問題があります。
        /// 対策として、小文字の環境変数を削除します。
        /// </summary>
        static void RemoveMultipleEnvironmentVariableWorkaround_(string envName)
        {
            // 環境変数を取得する。
            string envVal = Environment.GetEnvironmentVariable(envName);
            if (string.IsNullOrEmpty(envVal))
            {
                return;
            }

            // 手始めに一つ削除して、その後再度取得してみる。
            Environment.SetEnvironmentVariable(envName, null);
            string envVal2 = Environment.GetEnvironmentVariable(envName);
            if (!string.IsNullOrEmpty(envVal2))
            {
                // 2つ目が取得できた場合は、多重定義されている状況。
                // POSIX スタイルの環境変数を削除するようにする。
                if (envVal2.StartsWith("/"))
                {
                    Environment.SetEnvironmentVariable(envName, envVal);
                }
                return;
            }
            else
            {
                // 多重定義がなかったので、削除した環境変数を元に元しておく。
                Environment.SetEnvironmentVariable(envName, envVal);
            }
        }

        /// <summary>
        /// アプリケーションのメイン エントリ ポイントです。
        /// </summary>
        [STAThread]
        static void Main(string[] args)
        {
            //--------------------------------------------
            // cygwin 挙動変更による環境変数多重定義の回避コード
            RemoveMultipleEnvironmentVariableWorkaround_("tmp");
            RemoveMultipleEnvironmentVariableWorkaround_("temp");

            //--------------------------------------------
            // コマンドラインを解釈します。
            _commandLineOptions = ParseArgs(args);

            // ファイルに対する処理を行わない場合は、他のプロセスがあれば処理をすぐに委譲する
            if (!_commandLineOptions.SaveScreenCapture &&
                !_commandLineOptions.BakeInfinity &&
                string.IsNullOrEmpty(_commandLineOptions.OutputFilePath) &&
                !_commandLineOptions.UpdatePartsLayout &&
                !_commandLineOptions.Multiprocess)
            {
                if (SearchAndDelegateToOtherProcess())
                {
                    // 他のプロセスが見つかったので終了
                    return;
                }
            }

            Application.EnableVisualStyles();

            // ------------------ ＵＩ言語初期化
            if (Directory.Exists(Application.StartupPath + "\\en"))
            {
                Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);
            }

            //--------------------------------------------
            // アプリケーション・コンフィグを初期化します。
            _config = new AppConfig();

            // アプリケーションで使用するフォントを初期化します。
            _guiFont = new Font("Tahoma", 8.25F, FontStyle.Regular, GraphicsUnit.Point, 0);
            _guiFontBold = new Font(_guiFont, FontStyle.Bold);

            // ログ出力ハンドラに登録。
            List<string> initialOperationErrors = new List<string>();
            LEReportMsgHandler logHandler = new LEReportMsgHandler((sender, arg) => initialOperationErrors.Add(string.Format("{0} - {1}", arg.title, arg.msg)));
            LECore.LayoutEditorCore.MsgReporter.OnReportLog += logHandler;

            // マネージャなど、シングルトンクラスを初期化します。
            InitSingletonManagers_(_commandLineOptions);

            try
            {
                // 環境が正しく設定されてい無い場合は警告
                if (!Directory.Exists(AppConstants.NwToolsRootPath))
                {
                    initialOperationErrors.Add("WARNING : " + LECoreStringResMgr.Get("SYSTEM_NW4F_ROOT_WARNING"));
                }

                //--------------------------------------------
                // アプリケーション設定の読み込み
                // 読み込めなければ規定値を使う。
                _theAppSetting = InitializeAppSettings_();

                // 引数からの設定
                _theAppSetting.DebugConverter = _commandLineOptions.DebugConverter || !Environment.Is64BitProcess;

                //--------------------------------------------
                // プラットフォームの設定
                var platformOwner = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
                if (!string.IsNullOrEmpty(_theAppSetting.UiPlatformName))
                {
                    var index = platformOwner.PlatformPreferences.FindIndex(x => x.UiPlatformName == _theAppSetting.UiPlatformName);
                    if (index != -1)
                    {
                        platformOwner.PlatformPreference = platformOwner.PlatformPreferences[index];
                    }
                }
                _theAppSetting.UiPlatformName = platformOwner.PlatformPreference.UiPlatformName;

                //--------------------------------------------
                // 部品ルートの引数指定
                if (!string.IsNullOrEmpty(_commandLineOptions.PartsRootPath))
                {
                    _theAppSetting.PartsRootPath = _commandLineOptions.PartsRootPath;
                }

                //--------------------------------------------
                // 部品ルートを設定します。
                SetPartsRoot_(_theAppSetting.PartsRootPath);

                //--------------------------------------------
                // プロジェクト設定をロードします。
                ProjectSettings.TryReadProjectSettingsFromPartsRoot(_theAppSetting.PartsRootPath, _theAppSetting.ProjectSettings);

                LayoutEditorCore.SetTextureCacheSizeForScalableFont(_theAppSetting.ProjectSettings.TextureCacheSizeForScalableFont);
                LayoutEditorCore.SetForceConvertBffntFromSameNameFfnt(_theAppSetting.ProjectSettings.AutoFilteredFontGenerationFromSameNameFfntEnabled);
                LayoutEditorCore.SetFloatColorWriteFlag(_theAppSetting.ProjectSettings.FloatColorWriteEnabled);
                LayoutEditorCore.AstcTextureEnabled = _theAppSetting.ProjectSettings.AstcTextureEnabled;
                LayoutEditorCore.DefaultTextureFormat = _theAppSetting.ProjectSettings.DefaultTextureFormat;
                LayoutEditorCore.ExtendedTagEnabled = _theAppSetting.ProjectSettings.ExtendedTagEnabled;
                LayoutEditorCore.MovieTextureEnabled = _theAppSetting.ProjectSettings.MovieTextureEnabled;
                LayoutEditorCore.MovieTextureOutputDirectory = ToAbsolutePath(_theAppSetting.PartsRootPath, _theAppSetting.ProjectSettings.MovieTextureOutputDirectory);
                LayoutEditorCore.KeepingSizeScaleEnabled = _theAppSetting.ProjectSettings.KeepingSizeScaleEnabled;
                LayoutEditorCore.DefaultInitialPaneSize = _theAppSetting.ProjectSettings.DefaultInitialPaneSize;

                TileSettingParams.TileOptimize = _theAppSetting.ProjectSettings.TileOptimize;
                TileSettingParams.TileSizeThreshold = _theAppSetting.ProjectSettings.TileSizeThreshold;

                FileIOUserScriptHandler.Init(
                    _theAppSetting.ProjectSettings.PreSaveCommandPath,
                    _theAppSetting.ProjectSettings.PostSaveCommandPath,
                    _theAppSetting.ProjectSettings.PreDeleteCommandPath,
                    _theAppSetting.ProjectSettings.PostDeleteCommandPath,
                    _theAppSetting.ProjectSettings.PreLoadCommandPath,
                    _theAppSetting.ProjectSettings.PostCloseCommandPath,
                    _theAppSetting.ProjectSettings.UserCommandTimeOutMilliSec,
                    LECore.LayoutEditorCore.MsgReporter);

                AutoFilteredFontUtil.IsGpuEncodingEnalbed = _theAppSetting.ProjectSettings.IsGpuEncodingEnabled;

                //--------------------------------------------
                // 部品レイアウトの一括同期
                if (_commandLineOptions.UpdatePartsLayout)
                {
                    var param = new FileUpdateHelper.UpdateParam();

                    if (string.IsNullOrEmpty(_commandLineOptions.InputFilePath))
                    {
                        initialOperationErrors.Add(StringResMgr.Get("UPDATEPARTS_NOINPUTFILEPATH"));
                    }
                    else
                    {
                        param.Path = _commandLineOptions.InputFilePath;
                        param.Owner = null;
                        param.DialogTitle = StringResMgr.Get("SYSTEM_UPDATE_PARTS_TITLE");
                        param.DoShowConfirmDialog = false;
                        param.ConfirmMessage = StringResMgr.Get("SYSTEM_UPDATE_PARTS_CONFIRM");
                        param.DoUpdatePartsLayout = true;
                        param.DoUpdateAnimation = true;
                        param.DoSaveThumbnail = _theAppSetting.ProjectSettings.SaveThumbnailImage;

                        UpdateAllFiles(param);
                    }

                    // プロセスを終了
                    return;
                }

                //--------------------------------------------
                {
                    var combiner = LayoutEditor.src.CombinerEditor.CombinerCommunicationManager.CommunicationBridge;
                    if (combiner != null)
                    {
                        combiner.Init();
                    }

                    // CombinerEditorService を初期化
                    if (_theAppSetting.ProjectSettings.IsCombinerUserShaderEnabled)
                    {
                        string combinerEditorDirectry = Path.GetDirectoryName(_theAppSetting.ProjectSettings.UserCombinerShaderSettings.CombinerEditorPath);
                        if (!Path.IsPathRooted(_theAppSetting.ProjectSettings.UserCombinerShaderSettings.CombinerEditorPath))
                        {
                            combinerEditorDirectry = SceneHelper.GetAbsolutePathFromPartsRootBasedPath(LayoutEditorCore.Scene, combinerEditorDirectry);
                        }

                        string path = Path.Combine(combinerEditorDirectry,　"CoreModules", "CombinerEditor.Service.Impl.dll");
                        bool resInitialize = CombinerEditor.Service.CombinerEditorService.Initialize(path, "CombinerEditor.Service.CombinerEditorServiceImpl");
                        if (resInitialize == false)
                        {
                            LayoutEditorCore.MsgReporter.ReportError(
                                StringResMgr.Get("COMBINEREDITORSERVICE"),
                                StringResMgr.Get("COMBINEREDITORSERVICE_INITIALIZE_FAILED", path)
                            );
                        }
                    }

                }

                //--------------------------------------------
                // ファイル読み込み指定があれば読み込みます。
                LayoutDocument loadResult = null;
                ISubScene subScene = null;
                if (!string.IsNullOrEmpty(_commandLineOptions.InputFilePath))
                {
                    if (!File.Exists(_commandLineOptions.InputFilePath))
                    {
                        string message = string.Format("WARNING : Can't find the input argument file. [{0}]", _commandLineOptions.InputFilePath);
                        initialOperationErrors.Add(message);
                    }

                    loadResult = LayoutEditorCore.LoadLayoutFile(_commandLineOptions.InputFilePath, LayoutEditorCore.LoadOption.TryToOpenRlan);
                    subScene = loadResult != null ? loadResult.ISubScene : null;
                }

                //--------------------------------------------
                // 現在フレームの指定
                if (subScene != null && _commandLineOptions.CurrentFrameEnabled)
                {
                    GlobalTime.Inst.Time = _commandLineOptions.CurrentFrame;
                }

                //--------------------------------------------
                // キャプチャ保存
                if (subScene != null && _commandLineOptions.SaveScreenCapture)
                {
                    if (string.IsNullOrEmpty(_commandLineOptions.InputFilePath))
                    {
                        initialOperationErrors.Add(StringResMgr.Get("CAPTURE_NOINPUTFILEPATH"));
                    }
                    else
                    {
                        string oldPath = _theAppSetting.CaptureDestinationPath;
                        try
                        {

                            if (!string.IsNullOrEmpty(_commandLineOptions.CaptureFilePath) &&
                                Directory.Exists(Path.GetDirectoryName(_commandLineOptions.CaptureFilePath)))
                            {
                                _theAppSetting.CaptureDestinationPath = _commandLineOptions.CaptureFilePath;
                            }

                            if (_commandLineOptions.CurrentFrameEnabled)
                            {
                                _theAppSetting.CaptureFrame = _commandLineOptions.CurrentFrame;
                            }

                            _theAppSetting.CaptureSize = _commandLineOptions.CaptureSize;

                            var setting = _theAppSetting.GetPreviewSettings();
                            var finishedLocker = new object();
                            var finished = false;
                            setting.Finished += (s, e) =>
                            {
                                lock (finishedLocker)
                                {
                                    finished = true;
                                    Monitor.Pulse(finishedLocker);
                                }
                            };

                            // キャプチャ
                            ViewerExecuter.RunPreviewer(
                                setting,
                                subScene,
                                Path.GetFileNameWithoutExtension(_commandLineOptions.InputFilePath),
                                null,
                                string.Empty,
                                ViewerPreviewParam.TransferDestination.NoSpecify,
                                ViewerPreviewParam.PreviewMode.PCViewerScreenCapture);

                            // 処理の終了を待つ
                            if (setting.PushedToViewerThread)
                            {
                                lock (finishedLocker)
                                {
                                    while (!finished)
                                    {
                                        Monitor.Wait(finishedLocker);
                                    }
                                }
                            }
                        }
                        finally
                        {
                            _theAppSetting.CaptureDestinationPath = oldPath;
                        }
                    }
                }

                //--------------------------------------------
                // ファイル書き出し指定があれば書き出します。
                if (subScene != null && !string.IsNullOrEmpty(_commandLineOptions.OutputFilePath))
                {
                    ExportOption option = new ExportOption() {
                        BakeAnimationInfinityCurve = _commandLineOptions.BakeInfinity,
                        UseBaseValue = true,
                        Frame = GlobalTime.Inst.Time
                    };

                    LayoutEditorCore.ExportToFileAll(subScene, _commandLineOptions.OutputFilePath, option);
                }

                //--------------------------------------------
                // ウインドウを開くか？
                if (!_commandLineOptions.SaveScreenCapture && !_commandLineOptions.BakeInfinity)
                {
                    if (!_commandLineOptions.Multiprocess)
                    {
                        // 多重起動の確認
                        if (SearchAndDelegateToOtherProcess())
                        {
                            // 他のプロセスが見つかったので終了
                            return;
                        }
                    }

                    //------------- ビューマネージャの初期化
                    AppForm theAppForm = new AppForm(_theAppSetting, _appMessageReporter);
                    theAppForm.Closed += Event_AppForm_Closed;

                    // ログ出力ハンドラに登録。
                    LECore.LayoutEditorCore.MsgReporter.OnReportLog += new LEReportMsgHandler(
                        delegate (object sender, MessageArgs arg)
                        {
                            theAppForm.WriteMessage(string.Format("{0} - {1}", arg.title, arg.msg));
                        });

                    // 初期処理のエラーをステータスに表示します。
                    {
                        // ハンドラは不要になるので外す
                        LECore.LayoutEditorCore.MsgReporter.OnReportLog -= logHandler;

                        if (initialOperationErrors.Count > 0)
                        {
                            initialOperationErrors.ForEach((emsg) => theAppForm.WriteMessage(emsg));
                        }
                    }

                    // Host/IOのdllをあらかじめ読み込んでおきます。
                    string dllFilePath = Environment.ExpandEnvironmentVariables("%KMC_CTRHOSTIO%\\Bin\\devhio.dll");
                    int hostioDllModule = LECore.Win32.Kernel32.LoadLibrary(dllFilePath);

                    //------------- DirectXが利用できるかチェックします。
                    bool bD3DValid;
                    try
                    {
                        // ためしに、レンダラを作成して、
                        // 詳細なハード性能のチェックを行います。
                        Type type = Type.GetType("LayoutEditor.DynamicBinding.D3DRenderer");
                        System.Reflection.ConstructorInfo constructorInfo = type.GetConstructor(Type.EmptyTypes);

                        IRenderer testRenderer = constructorInfo.Invoke(null) as IRenderer;
                        bD3DValid = testRenderer.Initialize(theAppForm);
                        testRenderer.Dispose();
                    }
                    catch
                    {
                        bD3DValid = false;
                    }

                    if (!bD3DValid)
                    {
                        // 初期化に失敗した場合は、毎回、警告を表示します。
                        string nl = System.Environment.NewLine;
                        MessageBox.Show(
                            StringResMgr.Get("SYSTEM_DIRECTX_INIT_EROOR"),
                            StringResMgr.Get("SYSTEM_LAYOUTEDITOR"));

                        _theAppSetting.SetAvailableRrendererType(AppSetting.RendererType.GDI);
                        _theAppSetting.RendererKind = AppSetting.RendererType.GDI;
                    }
                    else
                    {
                        _theAppSetting.SetAvailableRrendererType(AppSetting.RendererType.D3D | AppSetting.RendererType.GDI);
                        // 利用可能な場合は、D3Dモードへの切り替えを促します。
                        if (_theAppSetting.RendererKind != AppSetting.RendererType.D3D)
                        {
                            DialogResult res = MessageBox.Show(
                                StringResMgr.Get("SYSTEM_DIRECTX_RECOMMEND_CAHNGE_TO_D3D"),
                                StringResMgr.Get("SYSTEM_LAYOUTEDITOR"),
                                MessageBoxButtons.YesNo);
                            if (res == DialogResult.Yes)
                            {
                                _theAppSetting.RendererKind = AppSetting.RendererType.D3D;
                            }
                        }

                        DbgConsole.WriteLine("D3D-Initilizing is done.");
                    }


                    // シーンが読み込まれていれば、ウインドウを開きます。
                    if (subScene != null)
                    {
                        theAppForm.AddNewLayoutWindow(loadResult);

                        // ビューアへの転送を行います。
                        if (_commandLineOptions.SendViewer)
                        {
                            theAppForm.SendViewer();
                        }
                    }

                    // ビューアへのコマンド通知転送を行います。
                    if (_commandLineOptions.IsViewerCommandEnabled)
                    {
                        theAppForm.SendViewerCommand(_commandLineOptions.ViewerCommand);
                    }

                    //---------------------------------
                    // 強制的にシーン更新イベントを起こします。
                    // シーン更新イベントをフックして初期化を行うサブウインドウ
                    // を正しく初期化するために必要です。
                    SceneManipulator sceneMnp = new SceneManipulator();
                    sceneMnp.BindTarget(LECore.LayoutEditorCore.Scene);
                    sceneMnp.CurrentISubScene = null;
                    sceneMnp.CurrentISubScene = subScene;

                    //------------- ヘルプの初期化
                    InitializeHelp_(theAppForm);


                    // ------------------ メインループをスタート
                    try
                    {
                        DbgConsole.WriteLine("App-Main-Loop Starts.");
                        Application.Run(theAppForm);
                    }
                    catch (Exception e)
                    {
                        writeErrorLog(e);

#if DEBUG
                        // デバッグビルド時のみ、スタックトレースをstrに吐き出してブレーク
                        StringBuilder sb = new StringBuilder();
                        StackTrace stackTrace = new StackTrace(e, true);

                        foreach (StackFrame stackFrame in stackTrace.GetFrames())
                        {
                            string line = string.Format("{0} ({1}): {2}", stackFrame.GetFileName(), stackFrame.GetFileLineNumber(), stackFrame.GetMethod());
                            sb.AppendLine(line);
                        }

                        string str = sb.ToString();
                        Debugger.Break();
#endif
                    }

                    // Host/IOのdllを解放しておきます。
                    if (hostioDllModule != 0)
                    {
                        LECore.Win32.Kernel32.FreeLibrary(hostioDllModule);
                    }
                }
            }
            finally
            {
                // 一時ディレクトリ削除
                LECore.ViewerExecuter.Cleanup();

                // 古い LayoutEditor が生成していたディレクトリを削除
                AppConstants.DeleteDeprecatedDirectoriesAll();

                // デバッグコンソール終了
                DbgConsole.Terminate();
            }
        }

        /// <summary>
        ///　多重起動の確認して処理を委譲する
        /// </summary>
        /// <returns></returns>
        static bool SearchAndDelegateToOtherProcess()
        {
            //--------------------------------------------

            Process currentProcess = Process.GetCurrentProcess();
            foreach (System.Diagnostics.Process p in Process.GetProcessesByName(currentProcess.ProcessName))
            {
                // MDI アプリケーションでは、サブウインドウが最大化されているときに MainWindowTitle が 変化するので StartsWith で判定します。
                if (p.Id != currentProcess.Id && p.MainWindowTitle.StartsWith(AppConstants.MainWindowTitle))
                {
                    // アプリケーションの多重起動を中断します。
                    HandleMultipleLaunching_(p, _commandLineOptions);
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 相対パスを絶対パスに変換する
        /// </summary>
        static string ToAbsolutePath(string baseDirectory, string relativePath)
        {
            if (baseDirectory == null || relativePath == null)
            {
                return null;
            }
            System.IO.FileInfo fileInfo = new FileInfo(System.IO.Path.Combine(baseDirectory, relativePath));
            return fileInfo.FullName;
        }

        /// <summary>
        /// コマンドライン引数ををパースします。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>パース結果を返します。</returns>
        static CommandLineOptions ParseArgs(string[] args)
        {
            // パース設定
            StringWriter helpWriter = new StringWriter();
            StringWriter errorWriter = new StringWriter();

            var settings = new CommandLineParserSettings()
            {
                ApplicationName = StringResMgr.Get("SYSTEM_LAYOUTEDITOR"),
                ApplicationDescription = StringResMgr.Get("COMMANDLINE_DESC_APPLICATION"),
                HelpWriter = helpWriter.Write,
                ErrorWriter = errorWriter.Write
            };

            // パーサーを作成
            bool resParse = false;
            CommandLineOptions option = null;
            var parser = new CommandLineParser(settings);

            // パース処理を実行
            try {
                //// ParseArgs()の挙動
                //// パース成功     : params = input, return = true , throw = false
                //// パースエラー   : params = null , return = false, throw = true
                //// ヘルプ表示     : params = null , return = false, throw = false
                //// バージョン表示 : params = null , return = false, throw = false
                resParse = parser.ParseArgs(args, out option);

                // ファイルパスの'/'を'\'に置き換えます
                if (option.InputFilePath != null)
                {
                    option.InputFilePath = option.InputFilePath.Replace('/', '\\');
                }
            }
            catch (Exception)
            {
                // TODO: エラー情報をコンソールに表示
                string errorText = errorWriter.ToString();
                DbgConsole.Write(errorText);

                Debug.Assert(false);

                return new CommandLineOptions();
            }

            // ヘルプ情報またはバージョン情報を表示
            if (resParse == false)
            {
                // TODO: ヘルプ情報またはバージョン情報をコンソールに表示
                string helpText = helpWriter.ToString();
                DbgConsole.Write(helpText);

                return new CommandLineOptions();
            }

            return option;
        }

        /// <summary>
        /// アプリケーション設定の読み込み
        /// </summary>
        static AppSetting InitializeAppSettings_()
        {
            // 読み込めなければ規定値を使う。
            AppSetting appSetting = LoadViewSetting_();
            return (appSetting == null) ? new AppSetting() : appSetting;
        }

        /// <summary>
        /// ヘルプの初期化
        /// </summary>
        static void InitializeHelp_(AppForm theAppForm)
        {
            // TODO:
        }

        #region アプリケーション設定読み込み関連
        /// <summary>
        /// 実行パスを取得します。
        /// </summary>
        /// <returns></returns>
        static string GetExecutableDirectry_()
        {
            return Path.GetDirectoryName(Application.ExecutablePath) + "\\";
        }

        /// <summary>
        /// ユーザーのアプリケーションデータのパスを取得
        /// </summary>
        public static string LocalAppDataDirectory
        {
            get
            {
                var folder = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData), "Nintendo", "LayoutEditor");
                return folder;
            }
        }

        /// <summary>
        /// ビュー設定を保存します。
        /// </summary>
        static void SaveViewSetting_(AppForm appForm)
        {
            _theAppSetting.PartsRootPath = LayoutEditorCore.Scene.PartsRootPath;
            _theAppSetting.ViewManagerSetting = appForm.SaveViewSetting();



            try
            {
                var directory = Path.Combine(LocalAppDataDirectory, Application.ProductVersion);
                if (!Directory.Exists(directory))
                {
                    Directory.CreateDirectory(directory);
                }

                XmlSerializer serializer = XmlSerializerCache.GetXmlSerializer(typeof(AppSetting));
                using (FileStream fs = new FileStream(Path.Combine(directory,_SettingFileName), FileMode.Create))
                {
                    serializer.Serialize(fs, _theAppSetting);
                }
            }
            catch (Exception e)
            {
                LECore.DbgConsole.WriteLine(e.ToString());
            }
        }

        /// <summary>
        /// ビュー設定を保存します。
        /// </summary>
        public static void SaveUserViewSetting_(AppForm appForm)
        {
            _theAppSetting.PartsRootPath = LayoutEditorCore.Scene.PartsRootPath;
            _theAppSetting.ViewManagerUserSetting = appForm.SaveUserViewSetting();

            try
            {
                var directory = Path.Combine(LocalAppDataDirectory, Application.ProductVersion);
                if (!Directory.Exists(directory))
                {
                    Directory.CreateDirectory(directory);
                }

                XmlSerializer serializer = XmlSerializerCache.GetXmlSerializer(typeof(AppSetting));
                using (FileStream fs = new FileStream(Path.Combine(directory, _SettingFileName), FileMode.Create))
                {
                    serializer.Serialize(fs, _theAppSetting);
                }
            }
            catch (Exception e)
            {
                LECore.DbgConsole.WriteLine(e.ToString());
            }
        }

        public static int[] ParseVersion(string version)
        {
            return version.Split('.').Select(x =>
            {
                int v;
                if (!int.TryParse(x, out v))
                {
                    v = -1;
                }
                return v;
            }).ToArray();
        }

        /// <summary>
        /// 0.0.0... 形式のバージョンを比較する
        /// </summary>
        internal class VersionComparer : IComparer<string>
        {
            public int Compare(string x, string y)
            {
                var ix = ParseVersion(Path.GetFileName(x));
                var iy = ParseVersion(Path.GetFileName(y));
                var ixcount = ix.Count();
                var iycount = iy.Count();
                var mincount = Math.Min(ixcount, iycount);

                // ドットで区切られた各数を比較して大小が有れば比較終了
                for (var i = 0; i < mincount; i++)
                {
                    if (ix[i] < iy[i])
                    {
                        return -1;
                    }
                    else if (ix[i] > iy[i])
                    {
                        return 1;
                    }
                }

                //ドットで区切られた数の多さで比較
                // ここに来るのは 1.1.1 と 1.1.1.1の比較の場合
                if (ixcount < iycount)
                {
                    return -1;
                }
                else if (ixcount > iycount)
                {
                    return 1;
                }
                return 0; // 1.1.1 と 1.1.01 のときにここに来る
            }
        }

        /// <summary>
        /// アプリケーション設定をロードします。
        /// </summary>
        /// <param name="appForm"></param>
        public static AppSetting LoadViewSetting_()
        {
            try
            {
                // 新しい設定ファイルが存在しなければ、過去の名前で読み込みます。
                string settingFilePath = Path.Combine(LocalAppDataDirectory, Application.ProductVersion,_SettingFileName);
                if (!File.Exists(settingFilePath))
                {
                    var dirs = Directory.GetDirectories(LocalAppDataDirectory, "*.*.*.*").OrderByDescending(x => x, new VersionComparer())
                        .ToList();
                    // バージョン番号のないパスは最後に探す
                    dirs.Add(LocalAppDataDirectory);
                    settingFilePath = dirs.Select(x => Path.Combine(x, _SettingFileName)).FirstOrDefault(File.Exists);

                    if (settingFilePath == null)
                    {
                        // 古いファイル名のものを探す
                        var oldPath = Path.Combine(LocalAppDataDirectory, _SettingFileNameOld);
                        if (File.Exists(oldPath))
                        {
                            settingFilePath = oldPath;
                        }
                    }
                }

                if (settingFilePath != null && File.Exists(settingFilePath))
                {
                    XmlSerializer serializer = XmlSerializerCache.GetXmlSerializer(typeof(AppSetting));
                    using (FileStream fs = new FileStream(settingFilePath, FileMode.Open, FileAccess.Read))
                    {
                        AppSetting setting = XmlSerializerCache.Deserialize(serializer, fs) as AppSetting;
                        return setting;
                    }
                }
            }
            catch
            {
                Debug.Assert(false);
                // TODO：警告表示
            }
            return null;
        }

        static void SetPartsRoot_(string partRootPath)
        {
            if (!Directory.Exists(partRootPath))
            {
                LayoutEditorCore.MsgReporter.ReportError(StringResMgr.Get("PARTS_ROOT_TITLE"), StringResMgr.Get("PARTS_PALLET_INVALID_PARTSROOT"));
            }

            SceneManipulator mnp = new SceneManipulator();
            mnp.BindTarget(LayoutEditorCore.Scene);
            mnp.RefreshPartsSubScenes(partRootPath);
        }

        /// <summary>
        /// 閉じるハンドラ
        /// </summary>
        static private void Event_AppForm_Closed(object sender, EventArgs e)
        {
            // ビュー設定を保存します。
            SaveViewSetting_(sender as AppForm);
        }
        #endregion アプリケーション設定読み込み関連


        #region ファイル一括更新
        /// <summary>
        /// ファイルまたはフォルダを一括更新します。
        /// </summary>
        /// <param name="param">更新パラメータ</param>
        public static void UpdateAllFiles(FileUpdateHelper.UpdateParam param)
        {
            MessageReportDlg reportDlg = null;
            Utility.WaitCursor wc = null;

            param.ShowConfirmMessage = (string msg) =>
            {
                DialogResult dialogResult = MessageBox.Show(param.Owner, msg, param.DialogTitle, MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
                return dialogResult == DialogResult.OK;
            };

            param.BeginProcess = () =>
            {
                wc = new Utility.WaitCursor();
                reportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Cancel, false);

                reportDlg.Title = param.DialogTitle;
                reportDlg.Message = string.Empty;
                reportDlg.Show(param.Owner);
            };

            param.ReportMessage = (string msg) =>
            {
                reportDlg.Message += msg;
            };

            param.IsCancelled = () =>
            {
                return reportDlg.DialogResult == DialogResult.Cancel;
            };

            param.EndProcess = () =>
            {
                wc.Dispose();
                wc = null;

                // もう一回モーダルで開きなおしてユーザに結果を通知。
                MessageReportDlg reportDlg2 = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                reportDlg2.Title = reportDlg.Title;
                reportDlg2.Message = reportDlg.Message;
                reportDlg.Close();
                reportDlg = null;
                reportDlg2.ShowDialog(param.Owner);
            };

            param.ReportError = (string msg) =>
            {
                MessageReportDlg reportDlg2 = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);

                reportDlg2.Title = param.DialogTitle;
                reportDlg2.Message = msg;
                reportDlg2.ShowDialog(param.Owner);
            };

            // 一括更新を実行
            FileUpdateHelper.UpdateAll(param);

            _thumbnailCreator?.ClearBitmapTextureMap();

            if (reportDlg != null)
            {
                reportDlg.Close();
                reportDlg = null;
            }

            if (wc != null)
            {
                wc.Dispose();
                wc = null;
            }
        }

        #endregion ファイル一括更新

        #region サムネール一括更新
        public static void UpdateThumbnail(string directoryPath, IWin32Window owner)
        {
            string[] filePaths;
            if (Directory.Exists(directoryPath))
            {
                // フォルダ以下のflytファイルを列挙
                try
                {
                    filePaths = Directory.GetFiles(directoryPath, AppConstants.LayoutFileExtPattern, SearchOption.AllDirectories);
                }
                catch (Exception e)
                {

                    MessageReportDlg reportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                    reportDlg.Title = StringResMgr.Get("SYSTEM_UPDATE_THUMBNAIL");
                    reportDlg.Message = StringResMgr.Get("SYSTEM_UPDATE_THUMBNAIL_FILE_EXCEPTION") + Environment.NewLine + e.Message;
                    reportDlg.ShowDialog(owner);
                    return;
                }

                var message = new StringBuilder();
                int updatedCount = 0;
                using (var wc = new Utility.WaitCursor())
                {
                    var subScenesBeforeExportThumbnail = LayoutEditorCore.Scene.ISubSceneSet;
                    var loadedSubScenes = new List<ISubScene>();
                    foreach (string filePath in filePaths)
                    {
                        try
                        {
                            var loadedSubScene = LayoutEditorCore.CreateSubSceneWithoutEventNotifications(filePath, LayoutEditorCore.LoadOption.None);

                            if (loadedSubScene == null)
                            {
                                message.AppendLine(StringResMgr.Get("SYSTEM_UPDATE_THUMBNAIL_FILEOPEN", filePath));
                                continue;
                            }

                            Debug.Assert(!subScenesBeforeExportThumbnail.Contains(loadedSubScene));
                            loadedSubScenes.Add(loadedSubScene);

                            var args = new LayoutEditorCore.ExportThumbnailEventArgs(loadedSubScene, filePath, string.Empty);
                            LayoutEditorCore_ExportThumbnail_(null, args);
                            if (args.Result)
                            {
                                updatedCount++;

                                // メモリが増大しすぎないようにときどきリソースを開放する
                                if (loadedSubScenes.Count == 20)
                                {
                                    Debug.Assert(loadedSubScenes.Intersect(LayoutEditorCore.Scene.PartsSubScenes.Select(x => x.SubScene)).Count() == 0);

                                    SceneManipulator sceneMnp = new SceneManipulator();
                                    sceneMnp.BindTarget(LayoutEditorCore.Scene);
                                    foreach (ISubScene subScene in loadedSubScenes)
                                    {
                                        sceneMnp.RemoveSubScene(subScene);
                                    }

                                    loadedSubScenes.Clear();
                                }
                            }
                            else
                            {
                                if (!string.IsNullOrEmpty(args.Error))
                                {
                                    message.Append(args.Error);
                                }
                            }
                        }
                        catch (Exception e)
                        {
                            message.AppendLine(StringResMgr.Get("SYSTEM_UPDATE_THUMBNAIL_FILE_ERROR", filePath));
                            message.AppendLine(e.ToString());
                        }
                    }

                    {
                        SceneManipulator sceneMnp = new SceneManipulator();
                        sceneMnp.BindTarget(LayoutEditorCore.Scene);

                        foreach (ISubScene subScene in LayoutEditorCore.Scene.ISubSceneSet.Except(subScenesBeforeExportThumbnail))
                        {
                            sceneMnp.RemoveSubScene(subScene);
                        }

                        // LayoutEditorCore.Scene.PartsSubScenes に削除されたインスタンスへの参照が残っている可能性があるので初期化が必要
                        sceneMnp.RefreshPartsSubScenes(LayoutEditorCore.Scene.PartsRootPath);

                        _thumbnailCreator?.ClearBitmapTextureMap();
                    }
                }

                {
                    MessageReportDlg reportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                    reportDlg.Title = StringResMgr.Get("SYSTEM_UPDATE_THUMBNAIL");
                    reportDlg.Message = StringResMgr.Get("SYSTEM_UPDATE_THUMBNAIL_RESULT", updatedCount) + Environment.NewLine + message.ToString();
                    reportDlg.ShowDialog(owner);
                }
            }
        }
        #endregion サムネール一括更新



        #region ヘルプ関連付け
        /// <summary>
        /// 関連付けをします。
        /// </summary>
        static public void BindHelp(Control control, string keyWord)
        {
            _hppLayoutEditor.SetHelpKeyword(control, keyWord);
            _hppLayoutEditor.SetHelpNavigator(control, HelpNavigator.TableOfContents);
            //_hppLayoutEditor.SetHelpNavigator( control, HelpNavigator.Topic );
            //control.HandleDestroyed += AppMain.UnbindHelp;
        }

        /// <summary>
        /// Viewer用の関連付けをします。
        /// </summary>
        static public void BindViewerHelp(Control control, string keyWord)
        {
            _hppViewer.SetHelpKeyword(control, keyWord);
            _hppViewer.SetHelpNavigator(control, HelpNavigator.TableOfContents);
            //_hppViewer.SetHelpNavigator( control, HelpNavigator.Topic );
            //control.HandleDestroyed += AppMain.UnbindHelp;
        }

        /// <summary>
        /// 関連付けを解除します。
        /// 関連付けを行ったコントロールの破棄イベントハンドラで呼び出されます。
        /// </summary>
        static public void UnbindHelp(object sender, EventArgs args)
        {
            Control control = sender as Control;
            Debug.Assert(control != null);
            _hppLayoutEditor.SetHelpKeyword(control, null);
            _hppViewer.SetHelpKeyword(control, null);
        }

        #endregion ヘルプ関連付け


        #region ErrorLog
        /// <summary>
        /// エラーログの書き出し。
        /// </summary>
        public static void writeErrorLog(Exception error)
        {
            string errorLogFilePath = Application.StartupPath + "\\error.log";
            try
            {

                StreamWriter writer = new StreamWriter(errorLogFilePath, false);
                writer.WriteLine(error.Message);
                writer.WriteLine(error.StackTrace);
                writer.Close();


                // 社内のユーザーか判定し、メール送信をたずねます。
                string fullMachineName = System.Net.Dns.GetHostEntry("localhost").HostName;
                bool sendMail = false;
                if(fullMachineName.EndsWith("ncl.nintendo.co.jp"))
                {
                    sendMail = TrySendRepoertMail_(error);
                }

                // 送信されなかった場合は、ログの場所をユーザに通知
                if (!sendMail)
                {
                    string errorMessage = StringResMgr.Get("MSG_CRITICAL_ERROR");

                    errorMessage += "\n" + errorLogFilePath + StringResMgr.Get("MSG_CRITICAL_ERROR_LOG");
                    errorMessage += "\n" + error.Message;
                    MsgBox.Error(errorMessage);
                }
            }
            catch (Exception)
            {
                // エラーログの書き出し失敗
                MsgBox.Error(StringResMgr.Get("MSG_CRITICAL_ERROR_LOG_FAILED") +
                    "\n" + errorLogFilePath);
            }
        }

        /// <summary>
        /// レポートメールの送信試行
        /// </summary>
        static bool TrySendRepoertMail_(Exception error)
        {
            MessageReportDlg msgDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.OkCancel, false);
            msgDlg.OKButtonText = "メール送信へ";
            msgDlg.Title = "LayoutEditor の致命的なエラー";
            msgDlg.Message =
                "LayoutEditor の不具合でご不便をおかして申し訳ありません。" + Environment.NewLine +
                "よろしければ担当者にメール送信いただき不具合調査にご協力ください。" + Environment.NewLine +
                "【キャンセル】で報告をスキップします。" + Environment.NewLine +
                Environment.NewLine +
                " ・メールの内容は、レイアウトエディタに関するものだけで、ゲーム固有の情報は一切含まれないようになっています" + Environment.NewLine +
                " ・このメッセージは、任天堂社内で LayoutEditor を利用いただいている方だけに表示されています。" + Environment.NewLine;

            string repertBody =
                "-----" + Environment.NewLine +
                LECore.AppConstants.GetVersionString() + Environment.NewLine +
                "-----" + Environment.NewLine +
                error.Message + Environment.NewLine +
                "-----" + Environment.NewLine +
                error.StackTrace;

            msgDlg.Message = msgDlg.Message + Environment.NewLine + repertBody;

            repertBody =
                "# そのまま「送信」としてくださって構いません。" + Environment.NewLine +
                "# お名前などは省略してください。" + Environment.NewLine + repertBody;

            if (msgDlg.ShowDialog() == DialogResult.OK)
            {
                repertBody = repertBody.Replace("場所", "_");
                string encoded = HttpUtility.UrlPathEncode(repertBody);
                Process.Start(string.Format("mailto:kitani_toshikazu@nintendo.co.jp?subject=LayoutEditor 不具合レポート&body={0}", encoded));
                return true;
            }

            return false;
        }
        #endregion

        //-----------------------------------------------------
        // コンソールアタッチ(ネイティブ関数呼び出し)
        // 現在は利用されていません。
        // アプリケーション呼び出しをリダイレクトすることで、
        // 同等の効果が得られることがわかったためです。
        #region
        [DllImport("kernel32.dll")]
        private static extern bool AttachConsole(uint dwProcessId);
        [DllImport("kernel32.dll")]
        private static extern bool FreeConsole();
        #endregion
    }




    #region enum LanguageMode
    /// <summary>
    /// 言語モード。
    /// </summary>
    public enum LanguageMode
    {
        /// <summary>日本語</summary>
        Japanese = 0,
        /// <summary>英語</summary>
        English,
    }

    #endregion

}
