﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Xml.Serialization;

namespace LayoutEditor
{
    using Forms.ToolWindows;
    using Forms.ToolWindows.common;
    using LayoutEditor.Forms;
    using LayoutEditor.Plugin;
    using LayoutEditor.Utility;
    using LECore;
    using LECore.Manipulator;
    using LECore.Save_Load;
    using LECore.Structures;
    using Structures.SerializableObject;
    using System.Text;
    using LayoutWindow = Forms.ToolWindows.LayoutWindow.LayoutWindow;

    /// <summary>
    /// 外部から の ProcessCmdKey メッセージを受け取るクラス。
    /// </summary>
    public interface ILECmdKeyAcceptor
    {
        int SuspendProcessCmdKey { get; set; }
        bool DoProcessCmdKey(ref Message msg, Keys keyData);
    }

    /// <summary>
    /// アプリケーションフォーム
    /// </summary>
    public class AppForm :
        System.Windows.Forms.Form,
        ILECmdKeyAcceptor,
        IViewManagerMessageSender
    {
        #region デザイナ変数

        private IContainer components = null;


        private MenuStrip _mspAppForm;
        private ToolStripMenuItem _tmiFile;
        private ToolStripMenuItem _tmiFileNewDocument;
        private ToolStripMenuItem _tmiFileOpen;
        private ToolStripMenuItem _tmiFileQuitApp;
        private ToolStripMenuItem _tmiEdit;
        private ToolStripMenuItem _tmiHelp;
        private ToolStripMenuItem _tmiEditToolSetting;
        private ToolStripMenuItem _tmiHelpShowHelp;
        private ToolStripMenuItem _tmiFileRecentlyUsedFile;
        private ToolStripMenuItem _tmiFileQuitApp2;
        private ToolStripMenuItem _tmiHelpVersion;
        private ToolStripSeparator toolStripSeparator1;
        private ToolStripSeparator toolStripSeparator2;
        private ToolStripMenuItem _tmiViewSubWindows;
        private ToolStripMenuItem _tmiViewSubWindows0;
        private ToolStripMenuItem _tmiViewSubWindows1;
        private ToolStripMenuItem _tmiViewSubWindows2;
        private ToolStripMenuItem _tmiViewSubWindows3;
        private ToolStripMenuItem _tmiViewSubWindows4;
        private ToolStripMenuItem _tmiViewSubWindows5;
        private ToolStripMenuItem _tmiViewSubWindows6;
        private ToolStripMenuItem _tmiViewSubWindows7;
        private ToolStripMenuItem _tmiViewSubWindows8;
        private ToolStripMenuItem _tmiViewSubWindows9;
        private ToolStripMenuItem _tmiViewSubWindows10;
        private ToolStripMenuItem _tmiViewSubWindows11;
        private ToolStripMenuItem _tmiViewSubWindows12;
        private ToolStripSeparator toolStripSeparator3;
        private LayoutEditor.Controls.UI.UIToolStrip _tspMain;
        private ToolStripMenuItem _tmiViewHideToolabar;
        private ToolStripSeparator toolStripSeparator4;
        private StatusStrip statusStrip1;
        private ToolStripMenuItem _tmiFileReExportAll;
        private ToolStripSeparator toolStripSeparator5;
        private ToolStripMenuItem _tmiHelpFontConverter;
        private ToolStripMenuItem _tmiSaveWorkSpace;
        private ToolStripStatusLabel _lblLog;
        private ToolStripMenuItem _tmiFileUpdatePartsAll;
        private ToolStripMenuItem _tmiHelpShowQuickStart;
        private ToolStripMenuItem _tmiHelpShowTutorial;
        private ToolStripSeparator toolStripSeparator6;
        private ToolStripSeparator toolStripSeparator7;
        private ToolTip warningToolTip;
        private ToolStripMenuItem _tmiHelpShowNews;
        private ToolStripMenuItem _tmiHelpShowTroubleshooting;
        internal WeifenLuo.WinFormsUI.Docking.DockPanel DockPanel;
        private ToolStripMenuItem _tmiLoadWorkspace;
        private ToolStripMenuItem _tmiLoadDefaultWorkspace;
        private ToolStripSeparator toolStripSeparator8;
        private ToolStripSeparator toolStripSeparator9;
        private ToolStripMenuItem _tmiFixDocking;
        private ToolStripMenuItem _tmiMdi;
        private ToolStripMenuItem _tmiFileUpdateThumbnailAll;
        private ToolStripButton _loadWorkspaceButton;

        #endregion デザイナ変数

        #region Windows フォーム デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(AppForm));
            this._mspAppForm = new System.Windows.Forms.MenuStrip();
            this._tmiFile = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiFileNewDocument = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiFileOpen = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator1 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiFileRecentlyUsedFile = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator2 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiFileReExportAll = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiFileUpdatePartsAll = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiFileUpdateThumbnailAll = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator5 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiFileQuitApp = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiFileQuitApp2 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiEdit = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSaveWorkSpace = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiLoadWorkspace = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiLoadDefaultWorkspace = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator8 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiEditToolSetting = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows0 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows1 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows2 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows3 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows4 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows5 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows6 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows7 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows8 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows9 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows10 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows11 = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiViewSubWindows12 = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator9 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiFixDocking = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiMdi = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator3 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiViewHideToolabar = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator4 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiHelp = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiHelpShowHelp = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiHelpShowNews = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator6 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiHelpShowQuickStart = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiHelpShowTutorial = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiHelpShowTroubleshooting = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiHelpFontConverter = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator7 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiHelpVersion = new System.Windows.Forms.ToolStripMenuItem();
            this.statusStrip1 = new System.Windows.Forms.StatusStrip();
            this._lblLog = new System.Windows.Forms.ToolStripStatusLabel();
            this.warningToolTip = new System.Windows.Forms.ToolTip(this.components);
            this.DockPanel = new WeifenLuo.WinFormsUI.Docking.DockPanel();
            this._tspMain = new LayoutEditor.Controls.UI.UIToolStrip();
            this._mspAppForm.SuspendLayout();
            this.statusStrip1.SuspendLayout();
            this.SuspendLayout();
            //
            // _mspAppForm
            //
            this._mspAppForm.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._tmiFile,
            this._tmiEdit,
            this._tmiViewSubWindows,
            this._tmiHelp});
            resources.ApplyResources(this._mspAppForm, "_mspAppForm");
            this._mspAppForm.Name = "_mspAppForm";
            //
            // _tmiFile
            //
            this._tmiFile.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._tmiFileNewDocument,
            this._tmiFileOpen,
            this.toolStripSeparator1,
            this._tmiFileRecentlyUsedFile,
            this.toolStripSeparator2,
            this._tmiFileReExportAll,
            this._tmiFileUpdatePartsAll,
            this._tmiFileUpdateThumbnailAll,
            this.toolStripSeparator5,
            this._tmiFileQuitApp,
            this._tmiFileQuitApp2});
            this._tmiFile.MergeAction = System.Windows.Forms.MergeAction.MatchOnly;
            this._tmiFile.MergeIndex = 0;
            this._tmiFile.Name = "_tmiFile";
            resources.ApplyResources(this._tmiFile, "_tmiFile");
            this._tmiFile.DropDownOpening += new System.EventHandler(this._tmiFile_DropDownOpening);
            //
            // _tmiFileNewDocument
            //
            this._tmiFileNewDocument.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this._tmiFileNewDocument.MergeIndex = 0;
            this._tmiFileNewDocument.Name = "_tmiFileNewDocument";
            resources.ApplyResources(this._tmiFileNewDocument, "_tmiFileNewDocument");
            this._tmiFileNewDocument.Click += new System.EventHandler(this.Event_MniNewDocumentLayout_Click);
            //
            // _tmiFileOpen
            //
            this._tmiFileOpen.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this._tmiFileOpen.MergeIndex = 2;
            this._tmiFileOpen.Name = "_tmiFileOpen";
            resources.ApplyResources(this._tmiFileOpen, "_tmiFileOpen");
            this._tmiFileOpen.Click += new System.EventHandler(this.Event_MniFileOpen_Click);
            //
            // toolStripSeparator1
            //
            this.toolStripSeparator1.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this.toolStripSeparator1.MergeIndex = 4;
            this.toolStripSeparator1.Name = "toolStripSeparator1";
            resources.ApplyResources(this.toolStripSeparator1, "toolStripSeparator1");
            //
            // _tmiFileRecentlyUsedFile
            //
            this._tmiFileRecentlyUsedFile.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this._tmiFileRecentlyUsedFile.MergeIndex = 4;
            this._tmiFileRecentlyUsedFile.Name = "_tmiFileRecentlyUsedFile";
            resources.ApplyResources(this._tmiFileRecentlyUsedFile, "_tmiFileRecentlyUsedFile");
            this._tmiFileRecentlyUsedFile.DropDownOpening += new System.EventHandler(this.Event_MniRecentlyUsedFiles_Popup);
            //
            // toolStripSeparator2
            //
            this.toolStripSeparator2.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this.toolStripSeparator2.MergeIndex = 4;
            this.toolStripSeparator2.Name = "toolStripSeparator2";
            resources.ApplyResources(this.toolStripSeparator2, "toolStripSeparator2");
            //
            // _tmiFileReExportAll
            //
            this._tmiFileReExportAll.Name = "_tmiFileReExportAll";
            resources.ApplyResources(this._tmiFileReExportAll, "_tmiFileReExportAll");
            this._tmiFileReExportAll.Click += new System.EventHandler(this.Event_tmiFileReExportAll_Click);
            //
            // _tmiFileUpdatePartsAll
            //
            this._tmiFileUpdatePartsAll.Name = "_tmiFileUpdatePartsAll";
            resources.ApplyResources(this._tmiFileUpdatePartsAll, "_tmiFileUpdatePartsAll");
            this._tmiFileUpdatePartsAll.Click += new System.EventHandler(this.Event_tmiFileUpdatePartsAll_Click);
            //
            // _tmiFileUpdateThumbnailAll
            //
            this._tmiFileUpdateThumbnailAll.Name = "_tmiFileUpdateThumbnailAll";
            resources.ApplyResources(this._tmiFileUpdateThumbnailAll, "_tmiFileUpdateThumbnailAll");
            this._tmiFileUpdateThumbnailAll.Click += new System.EventHandler(this._tmiFileUpdateThumbnailAll_Click);
            //
            // toolStripSeparator5
            //
            this.toolStripSeparator5.Name = "toolStripSeparator5";
            resources.ApplyResources(this.toolStripSeparator5, "toolStripSeparator5");
            //
            // _tmiFileQuitApp
            //
            this._tmiFileQuitApp.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this._tmiFileQuitApp.MergeIndex = 6;
            this._tmiFileQuitApp.Name = "_tmiFileQuitApp";
            resources.ApplyResources(this._tmiFileQuitApp, "_tmiFileQuitApp");
            this._tmiFileQuitApp.Click += new System.EventHandler(this.Event_MniExitApp_Click);
            //
            // _tmiFileQuitApp2
            //
            this._tmiFileQuitApp2.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this._tmiFileQuitApp2.MergeIndex = 6;
            this._tmiFileQuitApp2.Name = "_tmiFileQuitApp2";
            resources.ApplyResources(this._tmiFileQuitApp2, "_tmiFileQuitApp2");
            this._tmiFileQuitApp2.Click += new System.EventHandler(this.Event_MniExitApp_Click);
            //
            // _tmiEdit
            //
            this._tmiEdit.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._tmiSaveWorkSpace,
            this._tmiLoadWorkspace,
            this._tmiLoadDefaultWorkspace,
            this.toolStripSeparator8,
            this._tmiEditToolSetting});
            this._tmiEdit.MergeAction = System.Windows.Forms.MergeAction.MatchOnly;
            this._tmiEdit.MergeIndex = 1;
            this._tmiEdit.Name = "_tmiEdit";
            resources.ApplyResources(this._tmiEdit, "_tmiEdit");
            //
            // _tmiSaveWorkSpace
            //
            this._tmiSaveWorkSpace.Name = "_tmiSaveWorkSpace";
            resources.ApplyResources(this._tmiSaveWorkSpace, "_tmiSaveWorkSpace");
            this._tmiSaveWorkSpace.Click += new System.EventHandler(this.Event_SaveWorkspace_Click);
            //
            // _tmiLoadWorkspace
            //
            this._tmiLoadWorkspace.Name = "_tmiLoadWorkspace";
            resources.ApplyResources(this._tmiLoadWorkspace, "_tmiLoadWorkspace");
            this._tmiLoadWorkspace.Click += new System.EventHandler(this.Event_LoadWorkspace_Click);
            //
            // _tmiLoadDefaultWorkspace
            //
            this._tmiLoadDefaultWorkspace.Name = "_tmiLoadDefaultWorkspace";
            resources.ApplyResources(this._tmiLoadDefaultWorkspace, "_tmiLoadDefaultWorkspace");
            this._tmiLoadDefaultWorkspace.Click += new System.EventHandler(this._tmiLoadDefaultWorkspace_Click);
            //
            // toolStripSeparator8
            //
            this.toolStripSeparator8.Name = "toolStripSeparator8";
            resources.ApplyResources(this.toolStripSeparator8, "toolStripSeparator8");
            //
            // _tmiEditToolSetting
            //
            this._tmiEditToolSetting.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this._tmiEditToolSetting.MergeIndex = 10;
            this._tmiEditToolSetting.Name = "_tmiEditToolSetting";
            resources.ApplyResources(this._tmiEditToolSetting, "_tmiEditToolSetting");
            this._tmiEditToolSetting.Click += new System.EventHandler(this.Event_MniEnviromentSetting_Click);
            //
            // _tmiViewSubWindows
            //
            this._tmiViewSubWindows.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._tmiViewSubWindows0,
            this._tmiViewSubWindows1,
            this._tmiViewSubWindows2,
            this._tmiViewSubWindows3,
            this._tmiViewSubWindows4,
            this._tmiViewSubWindows5,
            this._tmiViewSubWindows6,
            this._tmiViewSubWindows7,
            this._tmiViewSubWindows8,
            this._tmiViewSubWindows9,
            this._tmiViewSubWindows10,
            this._tmiViewSubWindows11,
            this._tmiViewSubWindows12,
            this.toolStripSeparator9,
            this._tmiFixDocking,
            this._tmiMdi,
            this.toolStripSeparator3,
            this._tmiViewHideToolabar,
            this.toolStripSeparator4});
            this._tmiViewSubWindows.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this._tmiViewSubWindows.MergeIndex = 0;
            this._tmiViewSubWindows.Name = "_tmiViewSubWindows";
            resources.ApplyResources(this._tmiViewSubWindows, "_tmiViewSubWindows");
            this._tmiViewSubWindows.DropDownOpening += new System.EventHandler(this.Event_MniSubWindows_Popup);
            //
            // _tmiViewSubWindows0
            //
            this._tmiViewSubWindows0.Name = "_tmiViewSubWindows0";
            resources.ApplyResources(this._tmiViewSubWindows0, "_tmiViewSubWindows0");
            this._tmiViewSubWindows0.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows1
            //
            this._tmiViewSubWindows1.Name = "_tmiViewSubWindows1";
            resources.ApplyResources(this._tmiViewSubWindows1, "_tmiViewSubWindows1");
            this._tmiViewSubWindows1.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows2
            //
            this._tmiViewSubWindows2.Name = "_tmiViewSubWindows2";
            resources.ApplyResources(this._tmiViewSubWindows2, "_tmiViewSubWindows2");
            this._tmiViewSubWindows2.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows3
            //
            this._tmiViewSubWindows3.Name = "_tmiViewSubWindows3";
            resources.ApplyResources(this._tmiViewSubWindows3, "_tmiViewSubWindows3");
            this._tmiViewSubWindows3.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows4
            //
            this._tmiViewSubWindows4.Name = "_tmiViewSubWindows4";
            resources.ApplyResources(this._tmiViewSubWindows4, "_tmiViewSubWindows4");
            this._tmiViewSubWindows4.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows5
            //
            this._tmiViewSubWindows5.Name = "_tmiViewSubWindows5";
            resources.ApplyResources(this._tmiViewSubWindows5, "_tmiViewSubWindows5");
            this._tmiViewSubWindows5.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows6
            //
            this._tmiViewSubWindows6.Name = "_tmiViewSubWindows6";
            resources.ApplyResources(this._tmiViewSubWindows6, "_tmiViewSubWindows6");
            this._tmiViewSubWindows6.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows7
            //
            this._tmiViewSubWindows7.Name = "_tmiViewSubWindows7";
            resources.ApplyResources(this._tmiViewSubWindows7, "_tmiViewSubWindows7");
            this._tmiViewSubWindows7.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows8
            //
            this._tmiViewSubWindows8.Name = "_tmiViewSubWindows8";
            resources.ApplyResources(this._tmiViewSubWindows8, "_tmiViewSubWindows8");
            this._tmiViewSubWindows8.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows9
            //
            this._tmiViewSubWindows9.Name = "_tmiViewSubWindows9";
            resources.ApplyResources(this._tmiViewSubWindows9, "_tmiViewSubWindows9");
            this._tmiViewSubWindows9.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows10
            //
            this._tmiViewSubWindows10.Name = "_tmiViewSubWindows10";
            resources.ApplyResources(this._tmiViewSubWindows10, "_tmiViewSubWindows10");
            this._tmiViewSubWindows10.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows11
            //
            this._tmiViewSubWindows11.Name = "_tmiViewSubWindows11";
            resources.ApplyResources(this._tmiViewSubWindows11, "_tmiViewSubWindows11");
            this._tmiViewSubWindows11.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // _tmiViewSubWindows12
            //
            this._tmiViewSubWindows12.Name = "_tmiViewSubWindows12";
            resources.ApplyResources(this._tmiViewSubWindows12, "_tmiViewSubWindows12");
            this._tmiViewSubWindows12.Click += new System.EventHandler(this.Event_SetViewVisibleFlag_);
            //
            // toolStripSeparator9
            //
            this.toolStripSeparator9.Name = "toolStripSeparator9";
            resources.ApplyResources(this.toolStripSeparator9, "toolStripSeparator9");
            //
            // _tmiFixDocking
            //
            this._tmiFixDocking.Name = "_tmiFixDocking";
            resources.ApplyResources(this._tmiFixDocking, "_tmiFixDocking");
            this._tmiFixDocking.Click += new System.EventHandler(this._tmiFixDocking_Click);
            //
            // _tmiMdi
            //
            this._tmiMdi.Name = "_tmiMdi";
            resources.ApplyResources(this._tmiMdi, "_tmiMdi");
            this._tmiMdi.Click += new System.EventHandler(this._tmiMdi_Click);
            //
            // toolStripSeparator3
            //
            this.toolStripSeparator3.Name = "toolStripSeparator3";
            resources.ApplyResources(this.toolStripSeparator3, "toolStripSeparator3");
            //
            // _tmiViewHideToolabar
            //
            this._tmiViewHideToolabar.Name = "_tmiViewHideToolabar";
            resources.ApplyResources(this._tmiViewHideToolabar, "_tmiViewHideToolabar");
            this._tmiViewHideToolabar.Click += new System.EventHandler(this.Event_TmiViewHideToolbar_Click);
            //
            // toolStripSeparator4
            //
            this.toolStripSeparator4.Name = "toolStripSeparator4";
            resources.ApplyResources(this.toolStripSeparator4, "toolStripSeparator4");
            //
            // _tmiHelp
            //
            this._tmiHelp.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._tmiHelpShowHelp,
            this._tmiHelpShowNews,
            this.toolStripSeparator6,
            this._tmiHelpShowQuickStart,
            this._tmiHelpShowTutorial,
            this._tmiHelpShowTroubleshooting,
            this._tmiHelpFontConverter,
            this.toolStripSeparator7,
            this._tmiHelpVersion});
            this._tmiHelp.MergeAction = System.Windows.Forms.MergeAction.MatchOnly;
            this._tmiHelp.MergeIndex = 7;
            this._tmiHelp.Name = "_tmiHelp";
            resources.ApplyResources(this._tmiHelp, "_tmiHelp");
            //
            // _tmiHelpShowHelp
            //
            this._tmiHelpShowHelp.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this._tmiHelpShowHelp.MergeIndex = 0;
            this._tmiHelpShowHelp.Name = "_tmiHelpShowHelp";
            resources.ApplyResources(this._tmiHelpShowHelp, "_tmiHelpShowHelp");
            this._tmiHelpShowHelp.Click += new System.EventHandler(this.Event_tmiShowHelp_Click);
            //
            // _tmiHelpShowNews
            //
            this._tmiHelpShowNews.Name = "_tmiHelpShowNews";
            resources.ApplyResources(this._tmiHelpShowNews, "_tmiHelpShowNews");
            this._tmiHelpShowNews.Click += new System.EventHandler(this.Event_tmiShowNews_Click);
            //
            // toolStripSeparator6
            //
            this.toolStripSeparator6.Name = "toolStripSeparator6";
            resources.ApplyResources(this.toolStripSeparator6, "toolStripSeparator6");
            //
            // _tmiHelpShowQuickStart
            //
            this._tmiHelpShowQuickStart.Name = "_tmiHelpShowQuickStart";
            resources.ApplyResources(this._tmiHelpShowQuickStart, "_tmiHelpShowQuickStart");
            this._tmiHelpShowQuickStart.Click += new System.EventHandler(this.Event_tmiHelpShowQuickStart_Click);
            //
            // _tmiHelpShowTutorial
            //
            this._tmiHelpShowTutorial.Name = "_tmiHelpShowTutorial";
            resources.ApplyResources(this._tmiHelpShowTutorial, "_tmiHelpShowTutorial");
            this._tmiHelpShowTutorial.Click += new System.EventHandler(this.Event_tmiHelpShowTutorial_Click);
            //
            // _tmiHelpShowTroubleshooting
            //
            this._tmiHelpShowTroubleshooting.Name = "_tmiHelpShowTroubleshooting";
            resources.ApplyResources(this._tmiHelpShowTroubleshooting, "_tmiHelpShowTroubleshooting");
            this._tmiHelpShowTroubleshooting.Click += new System.EventHandler(this.Event_tmiHelpShowTroubleshooting_Click);
            //
            // _tmiHelpFontConverter
            //
            this._tmiHelpFontConverter.Name = "_tmiHelpFontConverter";
            resources.ApplyResources(this._tmiHelpFontConverter, "_tmiHelpFontConverter");
            this._tmiHelpFontConverter.Click += new System.EventHandler(this.Event_tmiHelpFontConverter_Click);
            //
            // toolStripSeparator7
            //
            this.toolStripSeparator7.Name = "toolStripSeparator7";
            resources.ApplyResources(this.toolStripSeparator7, "toolStripSeparator7");
            //
            // _tmiHelpVersion
            //
            this._tmiHelpVersion.MergeAction = System.Windows.Forms.MergeAction.Insert;
            this._tmiHelpVersion.MergeIndex = 0;
            this._tmiHelpVersion.Name = "_tmiHelpVersion";
            resources.ApplyResources(this._tmiHelpVersion, "_tmiHelpVersion");
            this._tmiHelpVersion.Click += new System.EventHandler(this.Event_MniVersion_Click);
            //
            // statusStrip1
            //
            this.statusStrip1.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._lblLog});
            this.statusStrip1.LayoutStyle = System.Windows.Forms.ToolStripLayoutStyle.Flow;
            resources.ApplyResources(this.statusStrip1, "statusStrip1");
            this.statusStrip1.Name = "statusStrip1";
            this.statusStrip1.ShowItemToolTips = true;
            //
            // _lblLog
            //
            this._lblLog.Name = "_lblLog";
            resources.ApplyResources(this._lblLog, "_lblLog");
            this._lblLog.MouseEnter += new System.EventHandler(this.Event_lblLog_MouseEnter);
            //
            // warningToolTip
            //
            this.warningToolTip.ToolTipIcon = System.Windows.Forms.ToolTipIcon.Warning;
            //
            // DockPanel
            //
            resources.ApplyResources(this.DockPanel, "DockPanel");
            this.DockPanel.DockRightPortion = 0.3D;
            this.DockPanel.Name = "DockPanel";
            this.DockPanel.MouseDown += new System.Windows.Forms.MouseEventHandler(this.Event_AppForm_MouseDown);
            //
            // _tspMain
            //
            this._tspMain.ClickThrough = false;
            this._tspMain.GripStyle = System.Windows.Forms.ToolStripGripStyle.Hidden;
            resources.ApplyResources(this._tspMain, "_tspMain");
            this._tspMain.Name = "_tspMain";
            //
            // AppForm
            //
            this.AllowDrop = true;
            resources.ApplyResources(this, "$this");
            this.Controls.Add(this.DockPanel);
            this.Controls.Add(this._tspMain);
            this.Controls.Add(this.statusStrip1);
            this.Controls.Add(this._mspAppForm);
            this.Cursor = System.Windows.Forms.Cursors.Default;
            this.IsMdiContainer = true;
            this.MainMenuStrip = this._mspAppForm;
            this.Name = "AppForm";
            this.Deactivate += new System.EventHandler(this.AppForm_Deactivate);
            this.FormClosed += new System.Windows.Forms.FormClosedEventHandler(this.AppForm_Closed);
            this.Load += new System.EventHandler(this.AppForm_Load);
            this.Shown += new System.EventHandler(this.AppForm_Shown);
            this.DragDrop += new System.Windows.Forms.DragEventHandler(this.Event_AppForm_DragDrop);
            this.DragEnter += new System.Windows.Forms.DragEventHandler(this.Event_AppForm_DragEnter);
            this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.Event_AppForm_MouseDown);
            this._mspAppForm.ResumeLayout(false);
            this._mspAppForm.PerformLayout();
            this.statusStrip1.ResumeLayout(false);
            this.statusStrip1.PerformLayout();
            this.ResumeLayout(false);
            this.PerformLayout();

        }
        #endregion


        #region フィールド
        /// <summary>
        /// ビューマネージャ実体
        /// </summary>
        readonly ViewManager _viewManager = null;
        readonly SceneManipulator _sceneMnp = new SceneManipulator();
        readonly PaneSearchWindow _paneSearchDlg = new PaneSearchWindow();
        static Icon _appIcon = null;

        private Image _InfoImage;
        private Image _WarningImage;

        private Queue<string> _statusLineQueue = new Queue<string>(100);

        private ToolStripComboBox _cmbWorkspace;
        private ToolStripComboBox _cmbPlatform;
        private ToolStripComboBox _cmbTarget;
        private ToolStripButton _btnAutoReload;
        private ToolStripButton _btnSendViewerAnimation;

        private AppSetting appSetting;

        public event EventHandler PlatformChanged;

        #endregion

        #region プロパティ

        public event Action<ViewManagerMessage> SendMessageToViewManager;

        /// <summary>
        /// アプリケーションアイコン
        /// </summary>
        static public Icon AppIcon
        {
            get
            {
                if (_appIcon == null)
                {
                    _appIcon = LayoutEditorCore.PlatformDetail.GetIcon();
                }

                return _appIcon;
            }
        }

        /// <summary>
        /// Load が既に呼ばれたか？
        /// </summary>
        public bool IsLoaded { get; set; } = false;

        #endregion

        /// <summary>
        /// Mdiクライアントサイズを取得します。
        /// </summary>
        public Size GetMdiCliantSize()
        {
            Control mdiCilent = FindMdiCliant_(this);
            return mdiCilent != null ? mdiCilent.Size : Size.Empty;
        }

        /// <summary>
        /// MdiClient コントロールを探します。
        /// </summary>
        Control FindMdiCliant_(Form form)
        {
            foreach (Control ctrl in this.Controls)
            {
                if (ctrl is MdiClient)
                {
                    return ctrl;
                }
            }
            // 発見できなかった
            return null;
        }

        /// <summary>
        /// 最近利用したファイルメニューを初期化します。
        /// </summary>
        void InitializeRecentlyUsedFiles_()
        {
            // 最近使ったファイル名を取得します。
            // 以前のバージョンで RecentlyUsedFiles にアニメーションファイルが含まれる不具合があったので念のために拡張子も確認
            string[] filePathSet = appSetting.RecentlyUsedFiles
                .Where(x => File.Exists(x) && string.Compare(Path.GetExtension(x), AppConstants.LayoutFileExt, true) == 0)
                .Take(AppSetting.MaxRecentlyUsedFiles).ToArray();

            // 初回時は、アイテムを作成します。
            // ノードを一定数つくりおきして非表示にするのは、
            // マージ時に、アイテムの動的な追加、削除が正しく行われない
            // 仕様に対処するためです。
            if (_tmiFileRecentlyUsedFile.DropDownItems.Count == 0)
            {
                for (int i = 0; i < AppSetting.MaxRecentlyUsedFiles; i++)
                {
                    _tmiFileRecentlyUsedFile.DropDownItems.Add(new ToolStripMenuItem());
                    _tmiFileRecentlyUsedFile.DropDownItems[i].Click += new EventHandler(Event_MniRecentlyUsedFile_Click);
                }
            }

            // 有効な項目を、メニュー項目として登録します。
            for (int i = 0; i < AppSetting.MaxRecentlyUsedFiles; i++)
            {
                ToolStripMenuItem mniFile = _tmiFileRecentlyUsedFile.DropDownItems[i] as ToolStripMenuItem;

                if (i < filePathSet.Length)
                {
                    string filePath = filePathSet[i];
                    string text = String.Format("{0}    ({1})", Path.GetFileNameWithoutExtension(filePath), filePath);
                    mniFile.Text = text;
                    mniFile.Tag = filePath;

                    mniFile.Visible = true;
                }
                else
                {
                    mniFile.Visible = false;
                }
            }
        }

        /// <summary>
        /// ステータスバーに文字列を出力します。
        /// </summary>
        public void WriteMessage(string title)
        {
            _lblLog.Text = title;
            OnStatusStrChangedUpdate_();

            // 表示を更新します。
            statusStrip1.Refresh();
        }

        /// <summary>
        /// ステータスバーに文字列を出力します。
        /// </summary>
        public void WriteWarning(string title)
        {
            WriteMessage(title);
        }

        /// <summary>
        /// ビューアへの転送を行います。
        /// </summary>
        public void SendViewer()
        {
            SendMessageToViewManager(new ViewManagerMessage(ViewManagerMessageKind.SendViewer));
        }

        /// <summary>
        /// ビューアーへのコマンド通知を行います。
        /// </summary>
        public void SendViewerCommand(string command)
        {
            Debug.Assert(this.appSetting != null);
            ViewerExecuter.SendCommandToViewer(command, this.appSetting.GetPreviewSettings());
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AppForm(AppSetting setting, AppMessageReporter messageReporter)
        {
            Debug.Assert(setting != null);

            appSetting = setting;

            //
            // Windows フォーム デザイナ サポートに必要です。
            //
            InitializeComponent();

            // DockPanel の設定
            DockPanel.Theme = new WeifenLuo.WinFormsUI.Docking.VS2015LightTheme();
            DockPanel.Theme.Extender.FloatWindowFactory = new LayoutEditor.Controls.CustomizedFloatWindowFactory();
            DockPanel.ActiveDocumentChanged += DockPanel_ActiveDocumentChanged;
            DockPanel.ActiveContentChanged += DockPanel_ActiveContentChanged;

            this.Text = AppConstants.MainWindowTitle;

            // アイコンの設定
            this.Icon = AppIcon;

            // 位置、サイズの設定はOnLoadのタイミングで行います。

            // DLLプラグインをロード
            PluginManager.Instance.LoadPlugins();

            // NX用テクスチャDLLのロード
            LayoutEditorCore.PlatformDetail.LoadNXDll();

            // viewManagerの初期化
            _viewManager = new ViewManager(this, setting, messageReporter);

            // ビュー設定を読み込み後の初期化処理を行います。
            _viewManager.InitializeAfterLoadSetting();

            // レイアウトエディタコアのシーンを操作するクラスを初期化します。
            _sceneMnp.BindTarget(LayoutEditorCore.Scene);

            // MDIクライアント領域のコントロールを取得し、マウスダウンハンドラを設定します。
            Control mdiCiant = FindMdiCliant_(this);
            Debug.Assert(mdiCiant != null);
            mdiCiant.MouseDown += new MouseEventHandler(this.Event_AppForm_MouseDown);

            // ツールバーを初期化します。
            InitToolbar_();

            // プラットフォーム関係のコンボボックスを初期化します。
            InitializeComboBoxes(setting);

            // ツールウインドウメニューを初期化するために、明示的に呼び出します。
            // ショートカット等を有効にします。
            Event_MniSubWindows_Popup(_tmiViewSubWindows, null);

            // 最近使ったファイル名を取得し、メニュー項目を初期化します。
            InitializeRecentlyUsedFiles_();

            messageReporter.InitializeReportDlgOwner(this);

            // 不正なサムネイル発生時ハンドラの定義
            TextureThumbnailHelper.OnInvalidThumbnailAction += ReportInvalidThumbnail_;

            _InfoImage = ImageResMgr.GetManifestResourcePng("Info_16x16.png");
            _WarningImage = ImageResMgr.GetManifestResourcePng("Warning_16.png");

            // 配置ボタンを更新
            CheckEnableLoadWorkspace();

            // ツールバーの表示を設定
            this._tmiViewHideToolabar.Checked = appSetting.ViewManagerSetting.HideToolbar;
            this._tspMain.Visible = !appSetting.ViewManagerSetting.HideToolbar;

            // ドッキングの固定を設定
            _tmiFixDocking.Checked = !appSetting.ViewManagerSetting.AllowEndUserDocking;
            DockPanel.AllowEndUserDocking = appSetting.ViewManagerSetting.AllowEndUserDocking;

            // ウインドウスタイルの設定
            // ドキュメントが読み込まれる前に行う必要がある
            _tmiMdi.Checked = appSetting.ViewManagerSetting.Mdi;
            DockPanel.DocumentStyle = appSetting.ViewManagerSetting.Mdi ?
                WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi :
                WeifenLuo.WinFormsUI.Docking.DocumentStyle.DockingMdi;

            // メッセージフィルタの設定
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter(_mspAppForm);
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter(_tspMain);

            appSetting.Modified += AppSetting_Modified;

            LECore.LayoutEditorCore.MsgReporter.ShowMessageDialogAsync += MsgReporter_ShowMessageDialogAsync;

            _tmiViewSubWindows.DropDownOpened += new EventHandler(Event_tmiViewSubWindows_DropDownOpened);
        }


        private void DockPanel_ActiveContentChanged(object sender, EventArgs e)
        {
            // アクティブなプロパティウインドウの編集対象の情報を更新する
            var propertyWindow =
                DockPanel.ActiveContent as Forms.ToolWindows.PropertyEditWindow.PropertyWindow;

            if (propertyWindow != null)
            {
                propertyWindow.UpdateCurrentPane();
            }
            else if (LECore.LayoutEditorCore.Scene.CurrentISubScene != null)
            {
                LECore.LayoutEditorCore.Scene.CurrentISubScene.CurrentPane = null;
            }
        }

        private void DockPanel_ActiveDocumentChanged(object sender, EventArgs e)
        {
            // MDI 表示時以外のときに ActiveDocument が Activated されないことがある
            if (DockPanel.DocumentStyle != WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi)
            {
                // ActivateMdiChild を呼ぶと例外になることがあるので呼ばない。

                var layoutWindow = DockPanel.ActiveDocument as LayoutWindow;
                layoutWindow?.SetCurrentISubScene();
            }
        }

        private void MsgReporter_ShowMessageDialogAsync(string title, string msg)
        {
            if (IsHandleCreated)
            {
                BeginInvoke((Action)(() => {
                    AppMain._appMessageReporter.ShowDialog_(
                        new MessageArgs() {
                            title = title,
                            msg = msg,
                        });
                }));
            }
            else
            {
                MessageBox.Show(msg, title);
            }
        }

        private void AppSetting_Modified(object sender, EventArgs e)
        {
            UpdateViewerToolStripItems();
        }

        private void UpdateViewerToolStripItems()
        {
            _btnAutoReload.Image = ((Bitmap[])_btnAutoReload.Tag)[AutoUpdateViewer ? 1 : 0];
            _btnAutoReload.ToolTipText = AutoUpdateViewer ? StringResMgr.Get("VIEWER_AUTO_RELOAD_ON") : StringResMgr.Get("VIEWER_AUTO_RELOAD_OFF");
            _btnAutoReload.Invalidate();
            _btnSendViewerAnimation.Image = ((Bitmap[])_btnSendViewerAnimation.Tag)[SendViewerAnimation ? 1 : 0];
            _btnSendViewerAnimation.ToolTipText = SendViewerAnimation ? StringResMgr.Get("VIEWER_SEND_ANIMATION_ON") : StringResMgr.Get("VIEWER_SEND_ANIMATION_OFF");
        }

        /// <summary>
        ///
        /// </summary>
        private void ReportInvalidThumbnail_(string msg)
        {
            msg = LECore.LECoreStringResMgr.Get("LECORE_TUMBNAIL_ERROR") + Environment.NewLine
                + Environment.NewLine
                + msg;

            MessageBox.Show(this, msg, LECore.LECoreStringResMgr.Get("LAYOUT_WARNING_DLG_TITLE"), MessageBoxButtons.OK, MessageBoxIcon.Warning);
        }

        /// <summary>
        ///
        /// </summary>
        public void AddLayoutWindowMenuItem(LayoutWindow lw)
        {
            foreach (ToolStripItem item in _tmiViewSubWindows.DropDownItems)
            {
                if (object.ReferenceEquals(item.Tag, lw))
                {
                    return;
                }
            }

            var newItem = new ToolStripMenuItem();
            newItem.Tag = lw;
            newItem.Click += new EventHandler((sender, e) => ((sender as ToolStripItem).Tag as LayoutWindow).Activate());
            _tmiViewSubWindows.DropDownItems.Add(newItem);
        }

        /// <summary>
        ///
        /// </summary>
        public void RemoveUnusedLayoutWindowMenuItem()
        {
            // 開いているレイアウトウインドウノード文字列を更新
            int currentLayoutCount = 0;
            List<ToolStripItem> removedItems = new List<ToolStripItem>();

            foreach (ToolStripItem item in _tmiViewSubWindows.DropDownItems)
            {
                LayoutWindow lw = item.Tag as LayoutWindow;
                if (lw == null)
                {
                    continue;
                }

                // まだ存在しているか確認する。
                var currentLw = _viewManager.FindLayoutWindowBySubScene(lw.CurrentSubScene);
                if (currentLw == null)
                {
                    removedItems.Add(item);
                }
                else
                {
                    item.Text = string.Format("{0} : {1}", currentLayoutCount, lw.Text);
                    currentLayoutCount++;
                }
            }

            foreach (var removedItem in removedItems)
            {
                removedItem.Tag = null;
                _tmiViewSubWindows.DropDownItems.Remove(removedItem);
            }
        }

        private Bitmap GetToolbarIcon_(string name)
        {
            return ImageResMgr.GetManifestResourceBitmap(string.Format("toolbar.{0}.bmp", name));
        }

        /// <summary>
        /// ツールバーを初期化します。
        /// </summary>
        private void InitToolbar_()
        {
            _tspMain.ClickThrough = true;

            // ツールウインドウ用アイコン
            Bitmap[] icons = new Bitmap[]{
                GetToolbarIcon_("Curve"),
                GetToolbarIcon_("Texture"),
                GetToolbarIcon_("Group"),
                GetToolbarIcon_("Hier"),
                GetToolbarIcon_("Paste"),
                GetToolbarIcon_("AnimTag"),
                GetToolbarIcon_("Search"),
                null,
                null,
                GetToolbarIcon_("Color"),
                null,
                GetToolbarIcon_("PartsSettings"),
                GetToolbarIcon_("PartsPallet"),
                null,
#if DEBUG
                null,
#endif
            };

            Debug.Assert(_viewManager.VisibleToolWindowSet.Length == icons.Length);

            int index = 0;
            foreach (ILEToolWindow toolView in _viewManager.VisibleToolWindowSet)
            {
                var icon = index < icons.Length ? icons[index] : null;
                index++;

                // アイコンがあるものだけ登録する。
                if (icon != null)
                {
                    var button = new ToolStripButton(toolView.LEWindowName);
                    button.AutoSize = false;
                    button.Size = new Size(28, 28);
                    button.Margin = new Padding(1, 0, 0, 1);
                    button.Padding = new Padding(0, 0, 0, 0);
                    button.DisplayStyle = ToolStripItemDisplayStyle.Image;
                    button.Click += new EventHandler((sender, e) =>
                    {
                        SetViewVisibleFlag_((sender as ToolStripButton).Text);
                    });

                    button.ImageScaling = ToolStripItemImageScaling.None;

                    button.Image = icon;
                    _tspMain.Items.Add(button);
                }
            }

            // ワークスペース
            _tspMain.Items.Add(new ToolStripSeparator());
            _tspMain.Items.Add(new ToolStripLabel()
            {
                Text = StringResMgr.Get("APPFORM_EDIT_MODE"),
            });
            _cmbWorkspace = new ToolStripComboBox()
            {
                Size = new Size(120, 26),
                DropDownStyle = ComboBoxStyle.DropDownList,
            };
            _tspMain.Items.Add(_cmbWorkspace);

            // プラットフォーム
            _tspMain.Items.Add(new ToolStripSeparator());
            _tspMain.Items.Add(new ToolStripLabel()
            {
                Text = StringResMgr.Get("PLATFORM"),
            });
            _cmbPlatform = new ToolStripComboBox()
            {
                Size = new Size(80, 26),
                DropDownStyle = ComboBoxStyle.DropDownList,
            };
            _tspMain.Items.Add(_cmbPlatform);

            // 自動転送
            _btnAutoReload = new ToolStripButton()
            {
                AutoSize = false,
                ImageScaling = ToolStripItemImageScaling.None,
                Size = new Size(28, 28),
                Margin = new Padding(1, 0, 0, 1),
                Padding = new Padding(0, 0, 0, 0),
                DisplayStyle = ToolStripItemDisplayStyle.Image,
            };
            _btnAutoReload.Tag = new[] {
                LayoutEditor.Properties.Resources.AutoReload_Off,
                LayoutEditor.Properties.Resources.AutoReload_On };
            _btnAutoReload.Click += _btnAutoReload_Click;
            _tspMain.Items.Add(_btnAutoReload);

            // 接続先
            _tspMain.Items.Add(new ToolStripSeparator());
            _cmbTarget = new ToolStripComboBox()
            {
                Size = new Size(80, 26),
                DropDownStyle = ComboBoxStyle.DropDownList,
            };
            _tspMain.Items.Add(_cmbTarget);

            // アニメーションモード
            // 自動転送
            _btnSendViewerAnimation = new ToolStripButton()
            {
                AutoSize = false,
                ImageScaling = ToolStripItemImageScaling.None,
                Size = new Size(28, 28),
                Margin = new Padding(1, 0, 0, 1),
                Padding = new Padding(0, 0, 0, 0),
                DisplayStyle = ToolStripItemDisplayStyle.Image,
            };
            _btnSendViewerAnimation.Tag = new[] {
                LayoutEditor.Properties.Resources.SendViewer_Animation_Off,
                LayoutEditor.Properties.Resources.SendViewer_Animation_On };
            _btnSendViewerAnimation.Click += _btnSendViewerAnimation_Click;
            _tspMain.Items.Add(_btnSendViewerAnimation);


            // ワークスペースの配置用ボタン
            {
                _loadWorkspaceButton = new ToolStripButton("ワークスペースの配置")
                    {
                        AutoSize = false,
                        Size = new Size(28, 28),
                        Alignment = ToolStripItemAlignment.Right,
                        DisplayStyle = ToolStripItemDisplayStyle.Image
                    };
                _loadWorkspaceButton.Click += Event_LoadWorkspace_Click;
                _loadWorkspaceButton.Image = GetToolbarIcon_("WorkSpace");

                _loadWorkspaceButton.ToolTipText = StringResMgr.Get("WORKSPACE_LOADBUTTON_TEXT"); ;

                _loadWorkspaceButton.ImageScaling = ToolStripItemImageScaling.None;
                _tspMain.Items.Add(_loadWorkspaceButton);
            }

            // ボタンプラグインを適用
            {
                var appInfo = new ButtonPluginParam();
                appInfo.AppForm = this;
                appInfo.ViewManager = _viewManager;

                foreach (var plugin in PluginManager.Instance.ButtonPlugins)
                {
                    plugin.Initialize(appInfo);

                    // セパレータなどでもよい
                    IEnumerable<ToolStripItem> buttons = plugin.EnumerateButtons();
                    foreach (ToolStripItem button in buttons)
                    {
                        _tspMain.Items.Add(button);
                    }
                }
            }

            // アイコンを更新
            UpdateViewerToolStripItems();
        }

        private void _btnAutoReload_Click(object sender, EventArgs e)
        {
            AutoUpdateViewer = !AutoUpdateViewer;
        }

        private void _btnSendViewerAnimation_Click(object sender, EventArgs e)
        {
            SendViewerAnimation = !SendViewerAnimation;
            _viewManager.RunPreviewer(SendViewerAnimation ? ViewerPreviewParam.PreviewMode.Animation: ViewerPreviewParam.PreviewMode.Control);
        }

        /// <summary>
        /// LayoutEditor での変更を自動的にビューアに反映させます。
        /// </summary>
        [Browsable(false)]
        public bool AutoUpdateViewer
        {
            get { return appSetting.AutoUpdateViewer; }
            set
            {
                if (appSetting.AutoUpdateViewer != value)
                {
                    appSetting.AutoUpdateViewer = value;
                    appSetting.RaiseModifyEvent();
                }
            }
        }

        /// <summary>
        /// レイアウトビューアでアニメーションを再生するかの設定です。
        /// </summary>
        [Browsable(false)]
        public bool SendViewerAnimation
        {
            get { return appSetting.SendViewerAnimation; }
            set
            {
                if (appSetting.SendViewerAnimation != value)
                {
                    appSetting.SendViewerAnimation = value;
                    appSetting.RaiseModifyEvent();
                }
            }
        }

        private bool SuspendIndexChanged = false;

        private class WorkspaceItem
        {
            public override string ToString()
            {
                return Label ?? ID;
            }

            public string ID { get; set; }
            public string Label { get; set; }
            public string toolTip { get; set; } // 今は使われていない
        }

        private const string DockPanelDefault = "DockPanelDefault";
        private void InitializeWorkspaceComboBoxes(AppSetting setting)
        {
            SuspendIndexChanged = true;
            try
            {
                _cmbWorkspace.Items.Clear();

                _cmbWorkspace.Items.Add(new WorkspaceItem() { ID = ViewManagerSetting.DockPanelModeLayout, Label = StringResMgr.Get("APPFORM_EDIT_MODE_LAYOUT") });
                _cmbWorkspace.Items.Add(new WorkspaceItem() { ID = ViewManagerSetting.DockPanelModeAnimation, Label = StringResMgr.Get("APPFORM_EDIT_MODE_ANIMATION") });
                _cmbWorkspace.Items.Add(new WorkspaceItem() { ID = ViewManagerSetting.DockPanelModelParts, Label = StringResMgr.Get("APPFORM_EDIT_MODE_PARTS") });

                foreach (var item in setting.ViewManagerSetting.DockPanelSettings)
                {
                    if (!_cmbWorkspace.Items.OfType<WorkspaceItem>().Any(x => x.ID == item.id) && !string.IsNullOrEmpty(item.id))
                    {
                        _cmbWorkspace.Items.Add(new WorkspaceItem() { ID = item.id, Label = item.label });
                    }
                }

                bool selected = false;
                for (int i = 0; i < _cmbWorkspace.Items.Count; i++)
                {
                    if (((WorkspaceItem)_cmbWorkspace.Items[i]).ID == setting.ViewManagerSetting.WorkspaceName)
                    {
                        _cmbWorkspace.SelectedIndex = i;
                        selected = true;
                    }
                }

                if (!selected)
                {
                    _cmbWorkspace.SelectedIndex = 0;
                    setting.ViewManagerSetting.WorkspaceName = ((WorkspaceItem)_cmbWorkspace.Items[0]).ID;
                }

                _cmbWorkspace.ToolTipText = ((WorkspaceItem)_cmbWorkspace.SelectedItem).toolTip;
            }
            finally
            {
                SuspendIndexChanged = false;
            }
        }

        /// <summary>
        /// コンボボックスを初期化します。
        /// </summary>
        private void InitializeComboBoxes(AppSetting setting)
        {
            InitializeWorkspaceComboBoxes(setting);

            _cmbWorkspace.SelectedIndexChanged += (s, e) =>
            {
                if (!SuspendIndexChanged)
                {
                    var mode = ((WorkspaceItem)_cmbWorkspace.SelectedItem).ID;
                    if (setting.ViewManagerSetting.WorkspaceName != mode)
                    {
                        _cmbWorkspace.ToolTipText = ((WorkspaceItem)(_cmbWorkspace.SelectedItem)).toolTip;
                        _viewManager.ChangeDockPanelMode(mode);
                    }
                }
            };

            var owner = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
            if (owner != null && owner.PlatformPreferences.Any())
            {
                // プラットフォームのコンボボックスを初期化します。
                foreach (var item in owner.PlatformPreferences)
                {
                    _cmbPlatform.Items.Add(item.UiPlatformName);
                }

                var index = owner.PlatformPreferences.IndexOf(owner.PlatformPreference);
                Debug.Assert(index != -1);

                _cmbPlatform.SelectedIndex = index;
                _cmbPlatform.SelectedIndexChanged += (s, e) =>
                    {
                        owner.PlatformPreference = owner.PlatformPreferences[_cmbPlatform.SelectedIndex];
                        setting.UiPlatformName = owner.PlatformPreference.UiPlatformName;

                        // チェック
                        List<ILEFont> fonts = new List<ILEFont>();
                        foreach (var fm in MdiChildren)
                        {
                            LayoutWindow lw = fm as LayoutWindow;
                            if (lw != null)
                            {
                                var _fonts = (from font in (lw.CurrentSubScene?.ILEFontManager?.ILEFontSet ?? Enumerable.Empty<ILEFont>()).Distinct()
                                              where (LEFontHelper.IsBitmapFontFile(font.FontPath) || LEFontHelper.IsComplexFont(font)) && !LEFontHelper.IsAcceptableBntxPlatform(font, owner.PlatformPreference)
                                              select font).ToArray();

                                if (_fonts.Any())
                                {
                                    foreach (ILEFont ft in _fonts)
                                    {
                                        // 複数レイアウトで同じファイルを使用していると重複列挙されてしまうため
                                        // 同一ファイルは除外する
                                        if (!fonts.Contains(ft))
                                        {
                                            fonts.Add(ft);
                                        }
                                    }
                                }
                            }
                        }

                        if (fonts.Any())
                        {
                            var builder = new StringBuilder();
                            builder.AppendLine(StringResMgr.Get("CHANGE_PLATFORM_ERROR", setting.UiPlatformName));

                            foreach (var item in fonts)
                            {
                                 builder.AppendLine(item.FontPath);
                            }

                            warningToolTip.Show(builder.ToString(), _cmbPlatform.Owner, 5000);
                        }

                        UpdateTargetComboBoxItems();
                        UpdateTargetStatus();

                        if (PlatformChanged != null)
                        {
                            PlatformChanged(this, new EventArgs());
                        }
                    };

                // 接続先を初期化します。
                UpdateTargetComboBoxItems();
                UpdateTargetStatus();

                _viewManager.AppSetting.Modified += (s, e) => UpdateTargetStatus();
                _cmbTarget.SelectedIndexChanged += (s, e) =>
                {
                    var types = TargetViewerTypes(owner.PlatformPreference).ToArray();

                    Debug.Assert(_cmbTarget.SelectedIndex < types.Length);

                    var type = types[_cmbTarget.SelectedIndex];

                    var old = _viewManager.AppSetting.TargetViewerKind;
                    _viewManager.AppSetting.TargetViewerKind = type;

                    if (old != _viewManager.AppSetting.TargetViewerKind)
                    {
                        _viewManager.AppSetting.RaiseModifyEvent();
                    }
                };
            }
            else
            {
                _cmbPlatform.Enabled = false;
            }
        }

        /// <summary>
        /// 接続先の候補を更新します。
        /// </summary>
        private void UpdateTargetComboBoxItems()
        {
            var owner = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
            Debug.Assert(owner != null);

            var items = TargetViewerTypes(owner.PlatformPreference).Select(
                x => x == AppSetting.TargetViewerType.PCViewer ?
                    StringResMgr.Get("APPFORM_TARGET_PC") :
                    StringResMgr.Get("APPFORM_TARGET_AE")
                    ).ToArray();

            if (!_cmbTarget.Items.OfType<string>().SequenceEqual(items))
            {
                _cmbTarget.Items.Clear();
                _cmbTarget.Items.AddRange(items);
            }
        }

        /// <summary>
        /// 接続先の選択を更新します。
        /// </summary>
        private void UpdateTargetStatus()
        {
            AppSetting.TargetViewerType type = _viewManager.AppSetting.TargetViewerKind;
            var owner = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
            if (owner != null)
            {
                _cmbTarget.SelectedIndex = TargetViewerTypes(owner.PlatformPreference).Select((value, index) => new { value, index }).Where(x => x.value == type)
                    .Select(x => x.index).FirstOrDefault();
            }
        }

        /// <summary>
        /// プラットフォームに対する接続先を列挙します。
        /// </summary>
        private static IEnumerable<AppSetting.TargetViewerType> TargetViewerTypes(PlatformPreference preference)
        {
            if (!string.IsNullOrEmpty(preference.PcTransferProtocol))
            {
                yield return AppSetting.TargetViewerType.PCViewer;
            }

            if (!string.IsNullOrEmpty(preference.TargetTransferProtocol))
            {
                yield return AppSetting.TargetViewerType.AEViewer;
            }
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        #region View設定保存

        ///// <summary>
        ///// ビュー設定を保存します。
        ///// </summary>
        public ViewManagerSetting SaveViewSetting()
        {
            var setting = _viewManager.SaveSetting();
            setting.HideToolbar = !this._tspMain.Visible;
            setting.AllowEndUserDocking = DockPanel.AllowEndUserDocking;
            setting.Mdi = DockPanel.DocumentStyle == WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi;

            return setting;
        }
        #endregion View設定保存

        #region 基底View設定保存

        ///// <summary>
        ///// ビュー設定を保存します。
        ///// </summary>
        public ViewManagerSetting SaveUserViewSetting()
        {
            var setting = _viewManager.SaveUserSetting();
            setting.HideToolbar = !this._tspMain.Visible;

            return setting;
        }
        #endregion 基底View設定保存

        #region イベントハンドラ

        /// <summary>
        ///
        /// </summary>
        /// <param name="m"></param>
        protected override void WndProc(ref Message m)
        {
            // 別プロセスからのメッセージ送信。
            if (m.Msg == LECore.Win32.WM.WM_COPYDATA)
            {
                try
                {
                    HandleOtherProccessMessage_(ref m);
                }
                catch
                {
                    // うまく処理できなかった
                    Debug.Assert(false, "WM_COPYDATA メッセージが正しく処理できませんでした。");
                }
            }

            base.WndProc(ref m);
        }

        /// <summary>
        /// ウインドウをアクティブに
        /// </summary>
        private void DoActivate_()
        {
            if (this.WindowState == FormWindowState.Minimized)
            {
                this.WindowState = FormWindowState.Normal;
            }

            this.Activate();
        }

        /// <summary>
        /// Message を CommandLineOptions にデコードします。
        /// </summary>
        CommandLineOptions DecodeFromProccessMessage_(ref Message m)
        {
            var cds = (LECore.Win32.COPYDATASTRUCT)Marshal.PtrToStructure(m.LParam, typeof(LECore.Win32.COPYDATASTRUCT));
            string optionString = Marshal.PtrToStringAuto(cds.lpData);

            return CommandLineOptions.DecodeForWindowMessage(optionString);
        }

        /// <summary>
        /// 別プロセスからのメッセージ送信を処理
        /// </summary>
        private void HandleOtherProccessMessage_(ref Message m)
        {
            CommandLineOptions option = DecodeFromProccessMessage_(ref m);
            if (option.IsViewerCommandEnabled)
            {
                SendViewerCommand(option.ViewerCommand);
            }
            else
            {
                // ViewManagerMessage 送信
                if (!string.IsNullOrEmpty(option.InputFilePath) && !option.CloseIfLoaded)
                {
                    DoActivate_();

                    ViewManagerMessage arg =
                        new ViewManagerMessage(ViewManagerMessageKind.OpenLayout, option.InputFilePath);
                    this.SendMessageToViewManager(arg);
                }
                else if (option.CloseIfLoaded)
                {
                    DoActivate_();

                    ViewManagerMessage arg = string.IsNullOrEmpty(option.InputFilePath) ?
                        new ViewManagerMessage(ViewManagerMessageKind.ForceCloseLayoutAll, string.Empty) :
                        new ViewManagerMessage(ViewManagerMessageKind.ForceCloseLayout, option.InputFilePath);
                    this.SendMessageToViewManager(arg);
                }

                // ビューアへの転送
                if (option.SendViewer)
                {
                    SendViewer();
                }
            }

        }

        /// <summary>
        ///
        /// </summary>
        private void AppForm_Deactivate(object sender, EventArgs e)
        {
            // TODO: 目的の確認
            LayoutWindow lw = ActiveMdiChild as LayoutWindow;
            if (lw != null)
            {
                lw.LayoutWindow_Deactivate();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void AppForm_Load(object sender, EventArgs e)
        {
            // AppFormの設定
            AppSetting appSetting = _viewManager.AppSetting;

            this.DesktopLocation = LEToolFormSetting.ClampWindowPosition(appSetting.ViewManagerSetting.AppFormLocation, appSetting.ViewManagerSetting.AppFormSize);
            if (appSetting.ViewManagerSetting.AppFormSize != Size.Empty)
            {
                this.Size = appSetting.ViewManagerSetting.AppFormSize;
            }
            else
            {
                var size = Screen.PrimaryScreen.Bounds.Size;
                var defaultScreenSize = LECore.Util.MathUtil.ScaleVec(new PointF(size.Width, size.Height), 0.9f);
                this.Size = new Size(LECore.Util.MathUtil.PointFToPoint(defaultScreenSize));
            }

            IsLoaded = true;
        }

        /// <summary>
        ///
        /// </summary>
        void Event_tmiViewSubWindows_DropDownOpened(object sender, EventArgs e)
        {
            RemoveUnusedLayoutWindowMenuItem();
        }

        /// <summary>
        ///
        /// </summary>
        private void AppForm_Closed(object sender, FormClosedEventArgs e)
        {
            ViewerExecuter.DisposeStaticResources();
        }

        /// <summary>
        /// コンテクストメニューを表示します。
        /// </summary>
        void HandleContextMenu_(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            InitializeRecentlyUsedFiles_();

            ToolStripMenuItem[] itemSet = new ToolStripMenuItem[] {
                _tmiFileNewDocument,
                _tmiFileOpen,
                _tmiFileRecentlyUsedFile };

            ContextMenuStrip cm =
                ToolStripMenuItemHelper.MakeContextMenuStripFromToolStripMenuItem(itemSet);

            Control ctrl = sender as Control;
            Debug.Assert(ctrl != null);

            cm.Show(ctrl, new Point(e.X, e.Y));
        }

        /// <summary>
        /// マウスハンドラ：MouseDown
        /// MDIクライアント領域のマウスイベントもうけとっています。
        /// </summary>
        private void Event_AppForm_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                HandleContextMenu_(sender, e);
            }
        }

        #region メニュー項目
        /// <summary>
        /// 最近使ったファイル
        /// </summary>
        void Event_MniRecentlyUsedFile_Click(object sender, EventArgs e)
        {
            ToolStripMenuItem mni = sender as ToolStripMenuItem;
            _viewManager.LoadFileAndRegisterLayoutWindow((string)mni.Tag, LayoutEditorCore.LoadOption.TryToOpenRlan);
        }

        /// <summary>
        /// 最近使ったファイル：ポップアップ
        /// </summary>
        private void Event_MniRecentlyUsedFiles_Popup(object sender, EventArgs e)
        {
            InitializeRecentlyUsedFiles_();
        }

        /// <summary>
        /// ツールダイアログ：ポップアップ
        /// </summary>
        private void Event_MniSubWindows_Popup(object sender, System.EventArgs e)
        {
            _tmiViewHideToolabar.Checked = !_tspMain.Visible;
            _tmiFixDocking.Checked = !DockPanel.AllowEndUserDocking;
            _tmiMdi.Checked = DockPanel.DocumentStyle == WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi;

            ToolStripMenuItem miTools = sender as ToolStripMenuItem;
            ILEToolWindow[] viewSet = _viewManager.VisibleToolWindowSet;

            // ビューセットに変更が発見された場合は、メニュー項目を更新します。
            // Debug.Assert( miTools.MenuItems.Count == viewSet.MatchLength );

            miTools.Enabled = false;

            for (int i = 0; i < miTools.DropDownItems.Count; ++i)
            {
                ToolStripMenuItem menuItem = miTools.DropDownItems[i] as ToolStripMenuItem;

                // セパレータにぶち当たった
                if (menuItem == null)
                {
                    if (i < viewSet.Length)
                    {
                        // まだviewSetにアイテムが残っているうちはメニュー項目を挿入して続行
                        var pluginItem = new ToolStripMenuItem();
                        pluginItem.Click += this.Event_SetViewVisibleFlag_;
                        miTools.DropDownItems.Insert(i, pluginItem);
                        menuItem = pluginItem;
                    }
                    else
                    {
                        // なければbreak
                        break;
                    }
                }

                if (i < viewSet.Length && !viewSet[i].IsNeededToHide)
                {
                    // メニュー項目をviewSetの内容で上書き
                    menuItem.Text = viewSet[i].LEWindowName;
                    menuItem.ShortcutKeys = viewSet[i].CustomShortcut;
                }
                else
                {
                    // 余剰分があったら不可視に
                    menuItem.Visible = false;
                }
            }

            miTools.Enabled = true;
        }

        /// <summary>
        /// ビューの表示フラグを設定します。
        /// </summary>
        private void Event_SetViewVisibleFlag_(object sender, System.EventArgs e)
        {
            ToolStripMenuItem mi = sender as ToolStripMenuItem;
            SetViewVisibleFlag_(mi.Text);
        }

        /// <summary>
        /// ビューの表示フラグを設定します。
        /// </summary>
        private void SetViewVisibleFlag_(string text)
        {
            _viewManager.Show(text);
            this.Focus();
        }

        /// <summary>
        /// ツールバーを隠す。
        /// </summary>
        private void Event_TmiViewHideToolbar_Click(object sender, EventArgs e)
        {
            _tspMain.Visible = !_tspMain.Visible;
        }

        /// <summary>
        /// 新規ドキュメントの追加
        /// </summary>
        private void Event_MniNewDocumentLayout_Click(object sender, System.EventArgs e)
        {
            // サブシーンを追加します。
            var result = LayoutEditorCore.CreateAsNewFile();
            Debug.Assert(result != null);

            LECore.Structures.ISubScene newScene = result.ISubScene;
            newScene.BackGround.ScreenSize = new FVec2(_viewManager.AppSetting.LastSetScreenSize);

            // サブシーンを表示するViewをViewマネージャに登録します。
            LayoutWindow lw = _viewManager.AddNewLayoutWindowAsNewDocument(result);
        }

        /// <summary>
        /// 新しいレイアウトウインドウを追加します。
        /// dataFilePath は 絶対パスで指定する必要があります。
        /// </summary>
        public void AddNewLayoutWindow(LayoutDocument lytDoc)
        {
            Debug.Assert(lytDoc.ISubScene != null);

            _viewManager.AddNewLayoutWindow(lytDoc);
        }

        /// <summary>
        /// 開く：すべて
        /// </summary>
        private void Event_MniFileOpen_Click(object sender, System.EventArgs e)
        {
            string[] inputPathSet;
            if (SaveLoadUIHelper.AskInputFilePath(LECore.Save_Load.ImportExportFileKind.All, out inputPathSet))
            {
                foreach (string inputPath in inputPathSet)
                {
                    _viewManager.LoadFileAndRegisterLayoutWindow(inputPath, LayoutEditorCore.LoadOption.TryToOpenRlan);
                }
            }
        }

        /// <summary>
        /// メニュー項目：終了
        /// </summary>
        private void Event_MniExitApp_Click(object sender, System.EventArgs e)
        {
            this.Close();
        }

        /// <summary>
        /// 環境設定
        /// </summary>
        private void Event_MniEnviromentSetting_Click(object sender, System.EventArgs e)
        {
            Forms.EnviromentSettingDlg dlg = new Forms.EnviromentSettingDlg();
            dlg.AppSetting = _viewManager.AppSetting;

            dlg.Owner = this;

            if (dlg.ShowDialog() == DialogResult.OK)
            {
                dlg.CopyParamaters();

                // 変更された内容をプロジェクトファイルにも反映する。
                ProjectSettings.WriteProjectSettingsFromPartsRoot(LayoutEditorCore.Scene.PartsRootPath, _viewManager.AppSetting.ProjectSettings);

                _viewManager.OnAppEvent(this, new AppEventArgs(AppEventKind.RedrawAllLayout));
            }
        }

        #region ヘルプ関連

        /// <summary>
        /// ヘルプ表示試行
        /// </summary>
        private void TryShowHelp_(string pageId)
        {
            try
            {
                string documentRootPath = Path.Combine(AppConstants.NwToolsRootPath, "../../Documents");

                // プロジェクト設定でドキュメントパスが設定されていたらそちらで上書き
                if (Directory.Exists(_viewManager.AppSetting.PartsRootPath))
                {
                    // NwToolsRootPath は Tools/Graphics
                    string documentPathPrj = _viewManager.AppSetting.ProjectSettings.DocumentPath;

                    // ドキュメントパスが絶対パスでかつディレクトリパスになるように変換
                    if (!string.IsNullOrEmpty(documentPathPrj))
                    {
                        if (!Path.IsPathRooted(documentPathPrj))
                        {
                            documentPathPrj = Path.Combine(_viewManager.AppSetting.PartsRootPath, documentPathPrj);
                        }

                        if (!string.IsNullOrEmpty(Path.GetExtension(documentPathPrj)))
                        {
                            documentPathPrj = Path.GetDirectoryName(documentPathPrj);
                        }

                        documentRootPath = documentPathPrj;
                    }
                }

                // ページIDを含めたページパスを取得
                string pagePath = Path.Combine(documentRootPath, string.Format("Package/contents/Pages/Page_{0}.html", pageId));

                if (File.Exists(pagePath))
                {
                    var fullPath = Path.GetFullPath(pagePath);
                    Process.Start(string.Format("file:///{0}", fullPath.Replace("\\", "/")));
                }
                else
                {
                    _viewManager.ShowMessageBox(
                        this,
                        StringResMgr.Get("MSG_FILEIO_DOCUMENT_FILE_NOT_FOUND"),
                        StringResMgr.Get("SYSTEM_LAYOUTEDITOR"),
                        MessageBoxButtons.OK);
                }
            }
            catch
            {
                // 何もしない
            }
        }

        /// <summary>
        /// ヘルプ
        /// </summary>
        private void Event_tmiShowHelp_Click(object sender, EventArgs e)
        {
            TryShowHelp_("83956239");
        }

        /// <summary>
        /// ニュース
        /// </summary>
        private void Event_tmiShowNews_Click(object sender, EventArgs e)
        {
            TryShowHelp_("113826765");
        }

        /// <summary>
        /// クイックスタート
        /// </summary>
        private void Event_tmiHelpShowQuickStart_Click(object sender, EventArgs e)
        {
            TryShowHelp_("106338839");
        }

        /// <summary>
        /// チュートリアル
        /// </summary>
        private void Event_tmiHelpShowTutorial_Click(object sender, EventArgs e)
        {
            TryShowHelp_("106345483");
        }

        /// <summary>
        /// トラブルシューティング
        /// </summary>
        private void Event_tmiHelpShowTroubleshooting_Click(object sender, EventArgs e)
        {
            TryShowHelp_("83955529");
        }

        /// <summary>
        /// フォントコンバータのヘルプ
        /// </summary>
        private void Event_tmiHelpFontConverter_Click(object sender, EventArgs e)
        {
            TryShowHelp_("83956036");
        }

        /// <summary>
        /// バージョン報告
        /// </summary>
        private void Event_MniVersion_Click(object sender, EventArgs e)
        {
            Forms.AboutDlg aboutDlg = new Forms.AboutDlg();
            aboutDlg.SetAppIconImage(AppForm.AppIcon.ToBitmap());
            aboutDlg.ShowDialog(this);
        }
        #endregion ヘルプ関連

        #endregion メニュー項目

        #endregion イベントハンドラ

        #region ILECmdKeyAccepter メンバ
        public bool DoProcessCmdKey(ref Message msg, Keys keyData)
        {
            // 自らの、メッセージとして解釈するように、
            // ウインドウハンドルを書き換えます。
            var origHandle = msg.HWnd;
            msg.HWnd = this.Handle;

            var ret = this.ProcessCmdKey(ref msg, keyData);

            msg.HWnd = origHandle;
            return ret;
        }

        public int SuspendProcessCmdKey { get; set; } = 0;

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if (SuspendProcessCmdKey > 0)
            {
                return false;
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }
        #endregion ILECmdKeyAccepter メンバ

        #region ドラッグ&ドロップ

        /// <summary>
        /// TODO:共通ルーチン化
        /// </summary>
        bool IsFilePathValid_(string name)
        {
            return File.Exists(name);
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_AppForm_DragEnter(object sender, DragEventArgs e)
        {
            string[] dropFiles = GetValidDropFiles_(e);
            if (dropFiles.Length > 0)
            {
                e.Effect = DragDropEffects.All;
            }
        }

        /// <summary>
        /// 開くのに適切なファイルか判定します。
        /// </summary>
        bool IsFileValidForOpen_(string filePath)
        {
            return SaveLoadUIHelper.IsFileExtValidForDrop(filePath) && _viewManager.FindLayoutWindow(filePath) == null;
        }

        /// <summary>
        /// ドロップされた有効なファイル列を取得します。
        /// </summary>
        private string[] GetValidDropFiles_(DragEventArgs e)
        {
            List<string> validFilePathSet = new List<string>();
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                string[] filePathSet = (string[])e.Data.GetData(DataFormats.FileDrop);
                foreach (string filePath in filePathSet)
                {
                    if (IsFileValidForOpen_(filePath))
                    {
                        validFilePathSet.Add(filePath);
                    }
                }

                return filePathSet;
            }
            else if (e.Data.GetDataPresent(DataFormats.StringFormat))
            {
                string filePath = (string)e.Data.GetData(DataFormats.StringFormat);
                if (IsFileValidForOpen_(filePath))
                {
                    validFilePathSet.Add(filePath);
                }
            }

            return validFilePathSet.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_AppForm_DragDrop(object sender, DragEventArgs e)
        {
            string[] dropFiles = GetValidDropFiles_(e);
            if (dropFiles.Length > 0)
            {
                List<string> loadedFileNameSet = new List<string>();
                foreach (string fileName in dropFiles)
                {
                    string fileNameWithoutExt = Path.GetFileNameWithoutExtension(fileName);
                    if (!loadedFileNameSet.Contains(fileNameWithoutExt))
                    {
                        if (_viewManager.LoadFileAndRegisterLayoutWindow(fileName, LayoutEditorCore.LoadOption.TryToOpenRlan)
                            == ViewManager.LoadResult.Loaded)
                        {
                            loadedFileNameSet.Add(fileNameWithoutExt);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 部品ルート以下を再保存します。
        /// </summary>
        private void Event_tmiFileReExportAll_Click(object sender, EventArgs e)
        {
            FolderSelectOperationImple_(
                _viewManager.AppSetting.LastOutputtedLayoutFolderPath,
                LayoutEditorCore.Scene.PartsRootPath,
                StringResMgr.Get("SYSTEM_OUTPUT_SELECT_FOLDER"),
                (selectedPath) => {
                    var param = new FileUpdateHelper.UpdateParam();

                    param.Path = selectedPath;
                    param.Owner = this;
                    param.DialogTitle = StringResMgr.Get("SYSTEM_REEXPORT_TITLE");
                    param.DoShowConfirmDialog = true;
                    param.ConfirmMessage = StringResMgr.Get("SYSTEM_REEXPORT_CONFIRM");
                    param.DoUpdatePartsLayout = false;
                    param.DoUpdateAnimation = true;
                    param.DoSaveThumbnail = appSetting.ProjectSettings.SaveThumbnailImage;

                    AppMain.UpdateAllFiles(param);

                    _viewManager.AppSetting.LastOutputtedLayoutFolderPath = selectedPath;
                });
        }

        /// <summary>
        /// 指定フォルダ以下のレイアウトファイルを一括同期します。
        /// </summary>
        private void Event_tmiFileUpdatePartsAll_Click(object sender, EventArgs e)
        {
            FolderSelectOperationImple_(
                _viewManager.AppSetting.LastUpdatedLayoutFolderPath,
                LayoutEditorCore.Scene.PartsRootPath,
                StringResMgr.Get("SYSTEM_UPDATE_PARTS_SELECT_FOLDER"),
                (selectedPath) => {
                    var param = new FileUpdateHelper.UpdateParam();

                    param.Path = selectedPath;
                    param.Owner = this;
                    param.DialogTitle = StringResMgr.Get("SYSTEM_UPDATE_PARTS_TITLE");
                    param.DoShowConfirmDialog = true;
                    param.ConfirmMessage = StringResMgr.Get("SYSTEM_UPDATE_PARTS_CONFIRM");
                    param.DoUpdatePartsLayout = true;
                    param.DoUpdateAnimation = true;
                    param.DoSaveThumbnail = appSetting.ProjectSettings.SaveThumbnailImage;

                    AppMain.UpdateAllFiles(param);

                    _viewManager.AppSetting.LastUpdatedLayoutFolderPath = selectedPath;
                });
        }

        private void FolderSelectOperationImple_(string lastSelectedPath, string defaultPath, string caption, Action<string> func)
        {
            string initialPath = string.Empty;

            // 前に選択したフォルダを取得
            if (!string.IsNullOrEmpty(lastSelectedPath))
            {
                if (Directory.Exists(lastSelectedPath))
                {
                    initialPath = lastSelectedPath;
                }
            }

            // デフォルト位置を指定
            if (string.IsNullOrEmpty(initialPath))
            {
                if (string.IsNullOrEmpty(defaultPath) == false)
                {
                    FileInfo fileInfo = new FileInfo(LayoutEditorCore.Scene.PartsRootPath);
                    if (Directory.Exists(fileInfo.FullName))
                    {
                        initialPath = fileInfo.FullName;
                    }
                }

                // 部品ルートが未指定ならばドキュメントフォルダになる
            }

            // フォルダ選択ダイアログを開く
            FolderBrowserUility folderBrowser = new FolderBrowserUility();

            string selectedPath = folderBrowser.SelectFolder(caption, initialPath, this.Handle);

            if (selectedPath == null)
            {
                return;
            }
            else
            {
                // 実行
                func(selectedPath);
            }
        }

        #endregion ドラッグ&ドロップ

        private void AppForm_Shown(object sender, EventArgs e)
        {
            // GLが初期化できるか確認しておく。
            {
                PictureBox bmp = new PictureBox();
                Graphics gfx = Graphics.FromHwnd(bmp.Handle);

                bool initSccessful = LayoutEditorCore.InitializeOpenGLContext(gfx.GetHdc());
                if (!initSccessful)
                {
                    LayoutEditorCore.MsgReporter.ReportError("",LECoreStringResMgr.Get("LECORE_INVALID_OPENGL"));
                }

                LayoutEditorCore.FinalizeOpenGLContext();
            }
        }

        /// <summary>
        /// ステータス文がエラー、警告を含むか
        /// </summary>
        private bool StatusStrConatainsWarningOrError_(string str)
        {
            return str.Contains(AppConstants.WarnigLogPrefix) || str.Contains(AppConstants.ErrorLogPrefix);
        }

        /// <summary>
        /// ステータス文変更時ハンドラ
        /// </summary>
        private void OnStatusStrChangedUpdate_()
        {
            if (!string.IsNullOrEmpty(_lblLog.ToolTipText))
            {
                _lblLog.ToolTipText = string.Empty;
                _statusLineQueue.Clear();
                _lblLog.Image = _InfoImage;
            }

            _statusLineQueue.Enqueue(_lblLog.Text);
            if (_statusLineQueue.Count > 100)
            {
                _statusLineQueue.Dequeue();
            }

            bool containsError = _lblLog.Image == _WarningImage || StatusStrConatainsWarningOrError_(_lblLog.Text);
            _lblLog.Image = containsError ? _WarningImage : _InfoImage;
        }

        /// <summary>
        /// マウスエンター(ユーザに警告を通知するツールチップを作る)
        /// </summary>
        private void Event_lblLog_MouseEnter(object sender, EventArgs e)
        {
            if (string.IsNullOrEmpty(_lblLog.ToolTipText))
            {
                int errorCount = 0;
                string toolTip = StringResMgr.Get("MSG_ERRORLOG") + "\n";
                string[] lines = _statusLineQueue.ToArray();
                for (int i = lines.Length - 1; i >= 0; i--)
                {
                    if (StatusStrConatainsWarningOrError_(lines[i]))
                    {
                        toolTip += lines[i] + "\n";
                        errorCount++;
                    }
                }

                _lblLog.ToolTipText = errorCount == 0 ? _lblLog.Text : toolTip;
            }
        }

        /// <summary>
        /// ワークスペースの配置ボタンを更新
        /// </summary>
        private void CheckEnableLoadWorkspace()
        {
            if (_loadWorkspaceButton == null)
            {
                return;
            }

            var size = new Size(0, 0);
            if (_viewManager.AppSetting.ViewManagerUserSetting.AppFormSize != size)
            {
                _loadWorkspaceButton.Enabled = true;
                _tmiLoadWorkspace.Enabled = true;
            }
            else
            {
                _loadWorkspaceButton.Enabled = false;
                _tmiLoadWorkspace.Enabled = false;
            }
        }

        /// <summary>
        /// ワークスペースの配置
        /// </summary>
        public void Event_LoadWorkspace_Click(object sender, EventArgs e)
        {
            var userSetting = AppMain.LoadViewSetting_().ViewManagerUserSetting;

            var item = userSetting.DockPanelSettings.FirstOrDefault(x => x.id == userSetting.WorkspaceName);

            // 設定の読み込み
            _viewManager.LoadUserSetting_(userSetting);

            // ワークスペースボタンを更新
            CheckEnableLoadWorkspace();

            // ワークスペースのコンボボックスを更新
            InitializeWorkspaceComboBoxes(appSetting);
        }

        /// <summary>
        /// ワークスペースの記憶
        /// </summary>
        private void Event_SaveWorkspace_Click(object sender, EventArgs e)
        {
            AppMain.SaveUserViewSetting_(this);

            CheckEnableLoadWorkspace();
        }

        private void _tmiLoadDefaultWorkspace_Click(object sender, EventArgs e)
        {
            // ウインドウの復元
            _viewManager.LoadDefaultDockPanelSetting();

            // ワークスペースのコンボボックスを更新
            InitializeWorkspaceComboBoxes(appSetting);
        }

        public void UpdateText()
        {
            var activeWindow = _viewManager.GetActiveLayoutWindow_();
            if (activeWindow != null)
            {
                Text = AppConstants.MainWindowTitle + " - " + activeWindow.Text;
            }
            else
            {
                Text = AppConstants.MainWindowTitle;
            }
        }

        private void _tmiFixDocking_Click(object sender, EventArgs e)
        {
            DockPanel.AllowEndUserDocking = !DockPanel.AllowEndUserDocking;
        }

        private void _tmiMdi_Click(object sender, EventArgs e)
        {
            if (_viewManager.IsLayoutWindowExsit)
            {
                _viewManager.ShowMessageBox(
                    this,
                    StringResMgr.Get("CHANGE_MDI"),
                    AppConstants.MainWindowTitle,
                    MessageBoxButtons.OK
                    );
                return;
            }

            DockPanel.DocumentStyle = DockPanel.DocumentStyle == WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi ?
                WeifenLuo.WinFormsUI.Docking.DocumentStyle.DockingMdi:
                WeifenLuo.WinFormsUI.Docking.DocumentStyle.SystemMdi;

        }

        private void _tmiFileUpdateThumbnailAll_Click(object sender, EventArgs e)
        {
            FolderSelectOperationImple_(
                _viewManager.AppSetting.LastUpdatedThumbnailFolderPath,
                LayoutEditorCore.Scene.PartsRootPath,
                StringResMgr.Get("SYSTEM_UPDATE_THUMBNAIL_SELECT_FOLDER"),
                (selectedPath) => {
                    AppMain.UpdateThumbnail(selectedPath, this);

                    _viewManager.AppSetting.LastUpdatedThumbnailFolderPath = selectedPath;
                });
        }

        /// <summary>
        /// File メニューを開くときの処理
        /// </summary>
        private void _tmiFile_DropDownOpening(object sender, EventArgs e)
        {
            _tmiFileUpdateThumbnailAll.Enabled = _tmiFileUpdateThumbnailAll.Visible = appSetting.ProjectSettings.SaveThumbnailImage;
        }
    }
}
