﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Linq;
using System.Reflection;

namespace LECore.Structures
{
    using LECoreInterface;
    using System.Collections.Generic;
    using LECore.Structures.Nsrif.Attributes;
    using System.IO;
    using LECore.Save_Load;
    using System.Xml.Serialization;

    /// <summary>
    ///
    /// </summary>
    internal class PlatformDetailGeneric : IPlatformDetail, IPlatformPreferenceOwner
    {
        static private readonly string _NwRootPath;
        static private readonly string _NwToolsRootPath;
        static public readonly string _NwFontToolsRootPath;
        static private readonly bool _OnNintendoSdk;
        static private PlatformPreference _PlatformPreference;
        static private readonly List<PlatformPreference> _PlatformPreferences;

        static private bool _IsNXDllLoaded = false;
        static private bool _GenericTextureWriterInitialized = false;
        static private NW4F.LayoutBinaryConverter.GenericTextureWriter _GenericTextureWriter;

        static PlatformDetailGeneric()
        {
            _PlatformPreferences = new List<PlatformPreference>();

            // テスト：特定のフォルダ構成が検出された場合は処理を切り替えます。
            string executingFileName = Assembly.GetExecutingAssembly().Location;
            string executingFolder = Path.GetDirectoryName(executingFileName);
            string layoutEditorFolderParentPath = Path.GetDirectoryName(executingFolder);

            bool isNintendoSDKEnviroment =
                layoutEditorFolderParentPath.Contains("Tools\\Graphics") ||
                layoutEditorFolderParentPath.Contains("Tests\\Outputs");

            if (isNintendoSDKEnviroment)
            {
                if (layoutEditorFolderParentPath.EndsWith("Tools\\Graphics"))
                {
                    // パッケージフォルダ構成での動作
                    _NwRootPath = Path.GetDirectoryName(Path.GetDirectoryName(layoutEditorFolderParentPath));
                    _NwToolsRootPath = layoutEditorFolderParentPath;
                }
                else
                {
                    // リポジトリフォルダ構成での動作
                    // 親フォルダをたどって基点を見つけ、Toolフォルダを決定する。
                    string parentDir = layoutEditorFolderParentPath;
                    try
                    {
                        while (!parentDir.EndsWith("Programs") && !parentDir.EndsWith("Tests"))
                        {
                            parentDir = Path.GetDirectoryName(parentDir);
                        }
                        parentDir = Path.GetDirectoryName(parentDir);
                    }
                    catch
                    {
                        // 想定外のフォルダ構成だった。
                        // 適当な値を設定しておく。
                        parentDir = layoutEditorFolderParentPath;
                    }

                    _NwRootPath = parentDir;
                    _NwToolsRootPath = Path.Combine(_NwRootPath, "Tools\\Graphics");
                }

                _NwFontToolsRootPath = _NwToolsRootPath;
                _OnNintendoSdk = true;

                var platformsDirectory = Path.Combine(_NwToolsRootPath, "LayoutEditor\\Platforms");
                try
                {
                    foreach (var file in Directory.EnumerateFiles(platformsDirectory, "*.preset").OrderBy(x => x))
                    {
                        XmlSerializer serializer = XmlSerializerCache.GetXmlSerializer(typeof(PlatformPreference));
                        using (FileStream fs = new FileStream(file, FileMode.Open, FileAccess.Read))
                        {
                            var preference = (PlatformPreference)XmlSerializerCache.Deserialize(serializer, fs);
                            _PlatformPreferences.Add(preference);
                        }
                    }
                    _PlatformPreference = _PlatformPreferences.FirstOrDefault();
                }
                catch
                {
                }
            }
            else
            {
                _NwRootPath = System.Environment.GetEnvironmentVariable("NW4F_ROOT");

                _NwToolsRootPath = string.IsNullOrEmpty(_NwRootPath) ? string.Empty : Path.Combine(_NwRootPath, "Tool");
                _NwFontToolsRootPath = _NwToolsRootPath;
                _OnNintendoSdk = false;
            }

            if (_PlatformPreferences.Count == 0)
            {
                _PlatformPreferences.Add(
                    new PlatformPreference()
                    {
                        PlatformName = "Generic",
                        UiPlatformName = "Win",
                        ArchiverPlatformName = "Win",
                        AcceptableBntxPlatformNames = new List<string>() { "Gen" },
                        PcViewerPath = @"LayoutViewer\Win64\lytviewerGfx.exe",
                        TargetViewerRunnnerPath = @"LayoutViewer\Cafe\run_viewer.bat",
                        PcTransferProtocol = "HTCS",
                        PcPeerType = "Generic-Windows",
                        TargetTransferProtocol = "HIO",
                        TargetConverterPath = @"LayoutTools\LayoutConverter.exe",
                        PCConverterPath = @"LayoutTools\LayoutConverter.exe",
                        ArchiverPath = @"LayoutTools\LayoutArchiver.exe",
                        FontConverterConsolePath = @"FontConverter\FontConverterConsole.exe",
                        IsNW4FMode = false,
                        PcConverterAdditionalArguments = "",
                        ArchiverAdditionalArguments = @"--tile-mode linear --api-type Gl --code-type Source",
                        FontConverterAdditionalArguments = @"-tile-mode linear",
                        BgTextureConverterAdditionalArguments = @"-f srgb_8_8_8_8 --tile-mode linear",
                    });
            }
        }

        /// <summary>
        /// プレビュー前(バイナリコンバート前)の準備
        /// </summary>
        public bool PrepareBeforeBinaryConvert(bool forPCViewer, string tempDirPath)
        {
            // 何もしません
            return true;
        }

        /// <summary>
        /// プレビュー前のフォント変換
        /// </summary>
        public bool PrepareBinaryFontForPreview(bool forPCViewer, string tempDirPath)
        {
            // 何もしません
            return true;
        }

        /// <summary>
        /// PCビューアの画面サイズ制限が有効か
        /// </summary>
        public bool PCViewerScreenSizeLimitEnabled { get { return false; } }

        /// <summary>
        // テクスチャサイズが小さすぎる場合に問題が発生する。(ランタイム側で正しくテクスチャを復元できない問題がある)
        // 出力されたフォントのサイズがかなり小さいことを検出して、シートサイズを指定して再度フォント出力をおこなう。
        // ランタイム側の問題が解消したら、この処理は不要になるので削除する。
        /// </summary>
        public bool SmallSizeFontProblemWorkaroundEnabled { get { return true; } }

        /// <summary>
        /// リニア編集環境が有効かどうか
        /// </summary>
        public bool IsLinearGammaSettingEnabled
        {
            get { return true; }
        }

        /// <summary>
        /// ツールルートパス
        /// </summary>
        public string NwToolsRootPath
        {
            get { return _NwToolsRootPath; }
        }

        /// <summary>
        /// レイアウトツールルートパス
        /// </summary>
        public string LayoutToolsRootPath
        {
            get { return Path.Combine(NwToolsRootPath, _OnNintendoSdk ? "LayoutTools" : "LytTool"); }
        }

        /// <summary>
        /// フォントツールルートパス
        /// </summary>
        public string NwFontToolsRootPath
        {
            get { return _NwFontToolsRootPath; }
        }

        /// <summary>
        /// バイナリコンバータのパス
        /// </summary>
        public string LayoutConverterPath
        {
            get { return Path.Combine(AppConstants.LayoutToolsPath, _OnNintendoSdk ? @"LayoutConverter.exe" : @"LayoutConverter\NW4F_LayoutConverter.exe"); }
        }

        /// <summary>
        /// バイナリコンバータのパス
        /// </summary>
        public string LayoutConverterForPCViewerPath
        {
            get { return LayoutConverterPath; }
        }

        /// <summary>
        /// フォントコンバータのパス
        /// </summary>
        public string FontConverterConsolePath
        {
            get { return Path.Combine(AppConstants.NwToolsRootPath, @"FontConverter\FontConverterConsole.exe"); }
        }

        /// <summary>
        /// PCビューアのパス
        /// </summary>
        public string PCLayoutViewerPath
        {
            get
            {
                var exePath = @"LayoutViewerGfx\win32\lytviewerGfx.exe";
                if (_PlatformPreference != null)
                {
                    exePath = _PlatformPreference.PcViewerPath;
                }

                return Path.Combine(
                    _OnNintendoSdk ? AppConstants.NwToolsRootPath : AppConstants.LayoutToolsPath, exePath);
            }
        }

        /// <summary>
        /// 実機ビューア起動バッチのパス
        /// </summary>
        public string RunCafeViewerBatPath
        {
            get
            {
                var batPath = @"LayoutViewerGfx\Target\run_viewer.bat";
                if (_PlatformPreference != null)
                {
                    batPath = _PlatformPreference.TargetViewerRunnnerPath;
                }

                return Path.Combine(
                    _OnNintendoSdk ? AppConstants.NwToolsRootPath : AppConstants.LayoutToolsPath, batPath);
            }
        }

        /// <summary>
        /// プレビュースターターのパス（NW4F パッチで使われます。それ以外でもアーカイブファイルの出力場所に関係します。）
        /// </summary>
        public string ViewerStarterPath
        {
            get
            {
                var starterPath = @"LayoutViewerGfx\Cafe\host\preview_starter.exe";
                if (_PlatformPreference != null)
                {
                    starterPath = _PlatformPreference.PreviewStarterPath;
                }

                if (!string.IsNullOrEmpty(starterPath))
                {
                    return Path.Combine(
                        _OnNintendoSdk ? AppConstants.NwToolsRootPath : AppConstants.LayoutToolsPath, starterPath);
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        /// <summary>
        /// プレビュー一時ファイルパスを取得します。
        /// </summary>
        public string PreviewTemporaryPath
        {
            get {
                var previewTempPath = @"LayoutEditor\preview_temp";
                if (_PlatformPreference != null && !string.IsNullOrEmpty(_PlatformPreference.PreviewTemporaryPath))
                {
                    previewTempPath = _PlatformPreference.PreviewTemporaryPath;
                }

                return Path.Combine(AppConstants.NwToolsRootPath, previewTempPath);
            }
        }

        /// <summary>
        /// UI に表示するプラットフォーム名を取得します。
        /// </summary>
        public string UIPlatformName
        {
            get
            {
                if　(_PlatformPreference != null)
                {
                    return _PlatformPreference.UiPlatformName;
                }

                return "Win";
            }
        }

        /// <summary>
        /// プラットフォーム名を取得します。
        /// </summary>
        public string PlatformName
        {
            get
            {
                if (_PlatformPreference != null)
                {
                    return _PlatformPreference.PlatformName;
                }

                return "Generic";
            }
        }

        /// <summary>
        /// サンプルデータパスです。
        /// </summary>
        public string NwDefaultSampleRoot
        {
            get
            {
                if(_OnNintendoSdk)
                {
                    return AppConstants.GetNintendoSDKDefaultSamplePath(_NwRootPath);
                }
                else
                {
                    if (!string.IsNullOrEmpty(AppConstants.NwToolsRootPath))
                    {
                        return Path.Combine(AppConstants.NwToolsRootPath, "..\\SampleData\\Layout\\flyt");
                    }
                    else
                    {
                        return string.Empty;
                    }
                }
            }
        }

        /// <summary>
        /// 実機通信に必要な環境変数設定
        /// </summary>
        public IEnumerable<string> GetTargetComEnvList()
        {
            yield break;
        }

        /// <summary>
        /// アイコンを取得します。
        /// </summary>
        public System.Drawing.Icon GetIcon()
        {
            return LEImageResMgr.GetManifestResourceIco("LayoutEditor.ico");
        }

        /// <summary>
        /// プラットフォーム設定を取得します。
        /// </summary>
        public PlatformPreference PlatformPreference
        {
            get { return _PlatformPreference; }
            set { _PlatformPreference = value; }
        }

        public List<PlatformPreference> PlatformPreferences { get { return _PlatformPreferences; } }

        /// <summary>
        /// NX用テクスチャDLLの読み込み状態を取得します。
        /// </summary>
        public bool IsNXDllLoaded
        {
            get { return _IsNXDllLoaded; }
            set { _IsNXDllLoaded = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public float GetBPP(TexImagePixelFmt PixelFmt)
        {
            switch (PixelFmt)
            {
                case TexImagePixelFmt.BC4L:
                case TexImagePixelFmt.BC4A:

                case TexImagePixelFmt.BC1:
                case TexImagePixelFmt.RGB5A1:
                case TexImagePixelFmt.RGB8:
                    return 0.5f;
                case TexImagePixelFmt.L8:
                case TexImagePixelFmt.A8:

                case TexImagePixelFmt.BC2:
                case TexImagePixelFmt.BC3:
                case TexImagePixelFmt.BC7:
                case TexImagePixelFmt.RGBA4:

                case TexImagePixelFmt.BC5:
                    return 1.0f;
                case TexImagePixelFmt.LA4:
                case TexImagePixelFmt.LA8:
                    return 2;
                case TexImagePixelFmt.RGB565:
                case TexImagePixelFmt.RGB565_INDIRECT:
                    return 2;
                case TexImagePixelFmt.RGBA8:
                    return 4;
                // 以下は使われていないフォーマット
                case TexImagePixelFmt.A4:
                case TexImagePixelFmt.L4:
                case TexImagePixelFmt.HILO8:
                case TexImagePixelFmt.ETC1:
                case TexImagePixelFmt.ETC1A4:
                default:
                    return 0;
            }
        }

        /// <summary>
        /// 変換テーブル
        /// </summary>
        private NW4F.LayoutBinaryConverter.TexConv.TexelFormat ConvetTexConvTexelFmt_(TexImagePixelFmt fmt)
        {
            switch (fmt)
            {
                case TexImagePixelFmt.A4: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.A4;
                case TexImagePixelFmt.A8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.A8;
                case TexImagePixelFmt.BC1: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC1;
                case TexImagePixelFmt.BC2: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC2;
                case TexImagePixelFmt.BC3: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC3;
                case TexImagePixelFmt.BC4A: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC4A;
                case TexImagePixelFmt.BC4L: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC4L;
                case TexImagePixelFmt.BC5: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC5;
                case TexImagePixelFmt.BC7: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.BC7;
                case TexImagePixelFmt.L4: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.L4;
                case TexImagePixelFmt.L8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.L8;
                case TexImagePixelFmt.LA4: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.LA4;
                case TexImagePixelFmt.LA8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.LA8;
                case TexImagePixelFmt.RGB565: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.RGB565;
                case TexImagePixelFmt.RGB565_INDIRECT: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.RGB565_INDIRECT;
                case TexImagePixelFmt.RGBA8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.RGBA8;

                case TexImagePixelFmt.ASTC4x4: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC4x4;
                case TexImagePixelFmt.ASTC5x4: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC5x4;
                case TexImagePixelFmt.ASTC5x5: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC5x5;
                case TexImagePixelFmt.ASTC6x5: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC6x5;
                case TexImagePixelFmt.ASTC6x6: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC6x6;
                case TexImagePixelFmt.ASTC8x5: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC8x5;
                case TexImagePixelFmt.ASTC8x6: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC8x6;
                case TexImagePixelFmt.ASTC8x8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC8x8;
                case TexImagePixelFmt.ASTC10x5: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC10x5;
                case TexImagePixelFmt.ASTC10x6: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC10x6;
                case TexImagePixelFmt.ASTC10x8: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC10x8;
                case TexImagePixelFmt.ASTC10x10: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC10x10;
                case TexImagePixelFmt.ASTC12x10: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC12x10;
                case TexImagePixelFmt.ASTC12x12: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.ASTC12x12;

                default: return NW4F.LayoutBinaryConverter.TexConv.TexelFormat.RGB565;
            }
        }

        /// <summary>
        /// UIに表示する文字列を取得します。
        /// </summary>
        public string GetPixcelFormatString(TexImagePixelFmt fmt)
        {
            string result = fmt.ToString();
            switch (fmt)
            {
                case TexImagePixelFmt.RGB565:
                    result += string.Format(" : ({0}_RGB)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.RGB565_INDIRECT:
                    result += string.Format(" : ({0}_UVA)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.RGB8:
                    result += string.Format(" : ({0})", LECoreStringResMgr.Get("TEXTURE_FMT_CONVERTTO_", "BC1"));
                    break;
                case TexImagePixelFmt.RGB5A1:
                    result += string.Format(" : ({0}_RGB5_A1)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.RGBA4:
                    result += string.Format(" : ({0}_RGBA4)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.R10G10B10A2:
                    result += string.Format(" : ({0})", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.BC1:
                    result += " : (RGB+1A)";
                    break;
                case TexImagePixelFmt.BC2:
                    result += " : (RGB+4A)";
                    break;
                case TexImagePixelFmt.BC3:
                case TexImagePixelFmt.BC7:
                    result += " : (RGB+8A)";
                    break;
                case TexImagePixelFmt.BC4A:
                    result += " : (A)";
                    break;
                case TexImagePixelFmt.BC4L:
                    result += " : (I)";
                    break;
                case TexImagePixelFmt.BC5:
                    result += " : (I+A)";
                    break;
                case TexImagePixelFmt.RGBA8:
                    result += string.Format(" : ({0}_RGB+8A)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.L4:
                case TexImagePixelFmt.L8:
                    result += string.Format(" : ({0}_I)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.A4:
                case TexImagePixelFmt.A8:
                    result += string.Format(" : ({0}_A)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.LA4:
                    // nvn がサポートするテクスチャフォーマットに合わせて適宜フォーマットを変換している。
                    // Generic 実装としてはこういったコードをハードコードするのは適切ではないので、将来は設定ファイルからロードするように拡張が必要かもしれない。
                    result += string.Format(" : ({0})", LECoreStringResMgr.Get("TEXTURE_FMT_CONVERTTO_", "BC5"));
                    break;
                case TexImagePixelFmt.LA8:
                    result += string.Format(" : ({0}_I+A)", LECoreStringResMgr.Get("TEXTURE_FMT_NOCMP"));
                    break;
                case TexImagePixelFmt.ASTC4x4:
                case TexImagePixelFmt.ASTC5x4:
                case TexImagePixelFmt.ASTC5x5:
                case TexImagePixelFmt.ASTC6x5:
                case TexImagePixelFmt.ASTC6x6:
                case TexImagePixelFmt.ASTC8x5:
                case TexImagePixelFmt.ASTC8x6:
                case TexImagePixelFmt.ASTC8x8:
                case TexImagePixelFmt.ASTC10x5:
                case TexImagePixelFmt.ASTC10x6:
                case TexImagePixelFmt.ASTC10x8:
                case TexImagePixelFmt.ASTC10x10:
                case TexImagePixelFmt.ASTC12x10:
                case TexImagePixelFmt.ASTC12x12:
                    result += string.Format(" : (RGB+8A) {0}", "BC3 on PCViewer");
                    break;
                default:
                    Debug.Assert(false);
                    break;
            }

            return result;
        }

        private void InitializeTextureWriter()
        {
            if (_GenericTextureWriterInitialized)
            {
                return;
            }

            _GenericTextureWriterInitialized = true;

            try
            {
                _GenericTextureWriter = new NW4F.LayoutBinaryConverter.GenericTextureWriter();
                var grahicsToolPath = Path.Combine(LayoutEditorCore.PlatformDetail.NwToolsRootPath, "GraphicsTools");
                if (_GenericTextureWriter.InitializeEncoder(grahicsToolPath) == false)
                {
                    _GenericTextureWriter = null;
                }
            }
            catch (Exception)
            {
                _GenericTextureWriter = null;
            }
        }

        public void LoadNXDll()
        {
            if (!_GenericTextureWriterInitialized)
            {
                InitializeTextureWriter();
            }

            if (_GenericTextureWriter != null)
            {
                _IsNXDllLoaded = _GenericTextureWriter.InitializeNXApi();
            }
        }

        public int CalcNativeTextureSize(int width, int height, TexImagePixelFmt fmt)
        {
            // DLL に丸ごと切り出します。
            // DLL に切り出されたら、この ifdef は不要になります。
#if NW_LYT_TOOLS_PLATFORM_CAFE
            return NW4F.LayoutBinaryConverter.GX2TextureWriter.GetGX2TextureSize(width, height, ConvetTexConvTexelFmt_(fmt));
#else
            if (!_GenericTextureWriterInitialized)
            {
                InitializeTextureWriter();
            }

            if (_GenericTextureWriter != null)
            {
                return _GenericTextureWriter.GetNativeDataSize(ConvetTexConvTexelFmt_(fmt), width, height);
            }

            return 0;
#endif
        }

        public int CalcNativeTextureSizeNX(int width, int height, TexImagePixelFmt fmt, uint tilingFlags, int sizeThreshold)
        {
            // DLL に丸ごと切り出します。
            // DLL に切り出されたら、この ifdef は不要になります。
#if NW_LYT_TOOLS_PLATFORM_CAFE
#else
            if (!_GenericTextureWriterInitialized)
            {
                InitializeTextureWriter();
            }

            if (_GenericTextureWriter != null)
            {
                return _GenericTextureWriter.GetNativeDataSizeNX(ConvetTexConvTexelFmt_(fmt), width, height, tilingFlags, sizeThreshold);
            }

            return 0;
#endif
        }

        private object IsGpuEncodingAvailableLocker = new object();
        private bool? _IsGpuEncodingAvailable = null;
        public bool IsGpuEncodingAvailable
        {
            get
            {
                lock (IsGpuEncodingAvailableLocker)
                {
                    if (!_IsGpuEncodingAvailable.HasValue)
                    {

                        if (!_GenericTextureWriterInitialized)
                        {
                            InitializeTextureWriter();
                        }

                        if (_GenericTextureWriter != null)
                        {
                            _IsGpuEncodingAvailable = _GenericTextureWriter.IsGpuEncodingAvailable();
                        }
                        else
                        {
                            _IsGpuEncodingAvailable = false;
                        }
                    }

                    return _IsGpuEncodingAvailable.Value;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<TexImagePixelFmt> GetGrayScaleFormatList(bool hasAlpha)
        {
            if (hasAlpha)
            {
                yield return TexImagePixelFmt.BC4A;
                yield return TexImagePixelFmt.BC5;
                yield return TexImagePixelFmt.A8;
                yield return TexImagePixelFmt.LA8;
                yield return TexImagePixelFmt.BC3;
                yield return TexImagePixelFmt.BC7;
                yield return TexImagePixelFmt.RGB565_INDIRECT;

                if (LayoutEditorCore.AstcTextureEnabled)
                {
                    yield return TexImagePixelFmt.Unknown;

                    yield return TexImagePixelFmt.ASTC4x4;
                    yield return TexImagePixelFmt.ASTC5x4;
                    yield return TexImagePixelFmt.ASTC5x5;
                    yield return TexImagePixelFmt.ASTC6x5;
                    yield return TexImagePixelFmt.ASTC6x6;
                    yield return TexImagePixelFmt.ASTC8x5;
                    yield return TexImagePixelFmt.ASTC8x6;
                    yield return TexImagePixelFmt.ASTC10x5;
                    yield return TexImagePixelFmt.ASTC10x6;
                    yield return TexImagePixelFmt.ASTC8x8;
                    yield return TexImagePixelFmt.ASTC10x8;
                    yield return TexImagePixelFmt.ASTC10x10;
                    yield return TexImagePixelFmt.ASTC12x10;
                    yield return TexImagePixelFmt.ASTC12x12;
                }
            }
            else
            {
                yield return TexImagePixelFmt.BC1;
                yield return TexImagePixelFmt.BC4L;
                yield return TexImagePixelFmt.L8;
                yield return TexImagePixelFmt.RGB565;
                yield return TexImagePixelFmt.RGB8;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<TexImagePixelFmt> GetFormatList(bool hasAlpha, bool subFormat)
        {

            if (hasAlpha)
            {
                if (!subFormat)
                {
                    yield return TexImagePixelFmt.BC1;
                    yield return TexImagePixelFmt.BC2;
                    yield return TexImagePixelFmt.BC3;
                    yield return TexImagePixelFmt.BC7;
                    yield return TexImagePixelFmt.BC5;
                    yield return TexImagePixelFmt.BC4A;
                    yield return TexImagePixelFmt.RGBA8;
                    yield return TexImagePixelFmt.RGB565_INDIRECT;

                    yield return TexImagePixelFmt.Unknown;

                    yield return TexImagePixelFmt.A8;
                    yield return TexImagePixelFmt.LA4;
                    yield return TexImagePixelFmt.LA8;
                    yield return TexImagePixelFmt.RGB5A1;
                    yield return TexImagePixelFmt.RGBA4;

                    if (LayoutEditorCore.AstcTextureEnabled)
                    {
                        yield return TexImagePixelFmt.Unknown;

                        yield return TexImagePixelFmt.ASTC4x4;
                        yield return TexImagePixelFmt.ASTC5x4;
                        yield return TexImagePixelFmt.ASTC5x5;
                        yield return TexImagePixelFmt.ASTC6x5;
                        yield return TexImagePixelFmt.ASTC6x6;
                        yield return TexImagePixelFmt.ASTC8x5;
                        yield return TexImagePixelFmt.ASTC8x6;
                        yield return TexImagePixelFmt.ASTC10x5;
                        yield return TexImagePixelFmt.ASTC10x6;
                        yield return TexImagePixelFmt.ASTC8x8;
                        yield return TexImagePixelFmt.ASTC10x8;
                        yield return TexImagePixelFmt.ASTC10x10;
                        yield return TexImagePixelFmt.ASTC12x10;
                        yield return TexImagePixelFmt.ASTC12x12;
                    }
                }
                else
                {
                    yield return TexImagePixelFmt.BC4A;
                    yield return TexImagePixelFmt.BC5;
                    yield return TexImagePixelFmt.A8;
                    yield return TexImagePixelFmt.LA8;
                    yield return TexImagePixelFmt.RGBA8;
                }
            }
            else
            {
                if (!subFormat)
                {
                    yield return TexImagePixelFmt.BC1;
                    yield return TexImagePixelFmt.BC4L;

                    yield return TexImagePixelFmt.Unknown;

                    yield return TexImagePixelFmt.L8;
                    yield return TexImagePixelFmt.RGB565;
                    yield return TexImagePixelFmt.RGB8;
                    yield return TexImagePixelFmt.BC7;

                    if (LayoutEditorCore.AstcTextureEnabled)
                    {
                        yield return TexImagePixelFmt.Unknown;

                        yield return TexImagePixelFmt.ASTC4x4;
                        yield return TexImagePixelFmt.ASTC5x4;
                        yield return TexImagePixelFmt.ASTC5x5;
                        yield return TexImagePixelFmt.ASTC6x5;
                        yield return TexImagePixelFmt.ASTC6x6;
                        yield return TexImagePixelFmt.ASTC8x5;
                        yield return TexImagePixelFmt.ASTC8x6;
                        yield return TexImagePixelFmt.ASTC10x5;
                        yield return TexImagePixelFmt.ASTC10x6;
                        yield return TexImagePixelFmt.ASTC8x8;
                        yield return TexImagePixelFmt.ASTC10x8;
                        yield return TexImagePixelFmt.ASTC10x10;
                        yield return TexImagePixelFmt.ASTC12x10;
                        yield return TexImagePixelFmt.ASTC12x12;
                    }
                }
                else
                {
                    yield return TexImagePixelFmt.BC4A;
                    yield return TexImagePixelFmt.BC5;
                    yield return TexImagePixelFmt.A8;
                    yield return TexImagePixelFmt.LA8;
                    yield return TexImagePixelFmt.RGBA8;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public TexImagePixelFmt GetDefaultTextureFormat(
            bool isGrayScale, bool intendedToBeAlphaFormat, TexImagePixelFmt nw4cTgaFormat)
        {
            // フォーマット初期値を決定します。
            DefaultTextureFormat defaultFormat = LayoutEditorCore.DefaultTextureFormat;
            Debug.Assert(defaultFormat != null);

            TexImagePixelFmt fmt;
            if (isGrayScale)
            {
                if (intendedToBeAlphaFormat)
                {
                    fmt = defaultFormat.GrayScaleWithAlpha;
                }
                else
                {
                    fmt = defaultFormat.GrayScale;
                }
            }
            else
            {
                if (intendedToBeAlphaFormat)
                {
                    fmt = defaultFormat.RGBColorWithAlpha;
                }
                else
                {
                    fmt = defaultFormat.RGBColor;
                }
            }
            return fmt;
        }

        /// <summary>
        /// テクスチャコンバイナ設定の候補を取得します。
        /// </summary>
        public IEnumerable<TextureCombineModeDesc> GetTextreuCombinerInfoColor(int stageNo)
        {
            yield return new TextureCombineModeDesc(AttrCombineMode.Replace, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_REPLACE"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Modulate, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_MODULATE"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Add, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ADD"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Subtract, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_SUBTRACT"));

            yield return new TextureCombineModeDesc(AttrCombineMode.AddSigned, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ADDSIGNED"));
            yield return new TextureCombineModeDesc(AttrCombineMode.AddMultDMP, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ADDMULT"));
            yield return new TextureCombineModeDesc(AttrCombineMode.MultAddDMP, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_MULTADD"));

            yield return new TextureCombineModeDesc(AttrCombineMode.Overlay, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_OVERLAY"));

            yield return new TextureCombineModeDesc(AttrCombineMode.Lighten, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_LIGHTEN"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Darken, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_DARKEN"));

            yield return new TextureCombineModeDesc(AttrCombineMode.Indirect, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_INDIRECT"));
            yield return new TextureCombineModeDesc(AttrCombineMode.BlendIndirect, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_BLENDINDIRECT"));
            yield return new TextureCombineModeDesc(AttrCombineMode.EachIndirect, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_EACHINDIRECT"));
        }

        /// <summary>
        /// テクスチャコンバイナ設定の候補を取得します。
        /// </summary>
        public IEnumerable<TextureCombineModeDesc> GetTextreuCombinerInfoAlpha()
        {
            yield return new TextureCombineModeDesc(AttrCombineMode.Replace, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ALPHA_MAX"));
            yield return new TextureCombineModeDesc(AttrCombineMode.Modulate, LECore.LECoreStringResMgr.Get("PROP_MAT_TEXCOMB_STAGE_ALPHA_MIN"));
        }

        /// <summary>
        /// 利用可能なステージ数を取得します。
        /// </summary>
        public int GetNumAvailableStages(ITevStages tevStages)
        {
            return LayoutEditorCore.PlatformDetail.MaxTevStageCount;
        }

        /// <summary>
        /// 非２べき乗テクスチャでミラーやリピートが可能か？
        /// </summary>
        public bool CheckTextureMirrorRepeatEnabled(TexImagePixelFmt nw4cTgaFormat, int size)
        {
            return true;
        }

        /// <summary>
        /// エディタでのテクスチャフォーマット変更を許すか？
        /// </summary>
        public bool TextureFormatEditEnabled { get { return true; } }

        /// <summary>
        /// ウインドウペインのカスタムフレームサイズ設定が有効かどうか
        /// </summary>
        public bool CustomWindowFrameSizeEnabled { get { return true; } }

        /// <summary>
        /// インダイレクトテクスチャが有効か
        /// </summary>
        public bool IndirectTextureEnabled { get { return true; } }

        /// <summary>
        ///
        /// </summary>
        public bool MultiScreenEnabled { get { return false; } }

        /// <summary>
        /// 詳細なマテリアルの編集は可能か？
        /// </summary>
        public bool LowLevelCombinerSettingsEnabled { get { return true; } }

        /// <summary>
        /// 最大TEVステージ数を取得します。
        /// </summary>
        public int MaxTevStageCount { get { return 6; } }

        /// <summary>
        /// TEVコンスタントカラーの数を取得します。
        /// </summary>
        public int TevConstantColorCount { get { return 5; } }

        /// <summary>
        /// プラットフォームに対応しているか？
        /// </summary>
        public bool IsAcceptableBntxPlatform(string platform)
        {
            return LEFontHelper.IsAcceptableBntxPlatform(platform, PlatformPreference);
        }
    }
}
