﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;

namespace LECore.Structures
{

    using Core;
    using NWFont = NW4R.Font;

    /// <summary>
    /// 行そろえ位置
    /// (NotSpecifiedを指定すると過去のバージョンと同様の挙動)
    /// </summary>
    public enum LineAlignment
    {
        NotSpecified,
        Left,
        Center,
        Right
    }

    public enum PerCharTransformLoopType
    {
        OneTime,
        Loop
    }

    /// <summary>
    /// TextBox ペインの情報
    /// 外部モジュール公開インタフェース
    /// </summary>
    public interface ITextBox
        : IDrawable, IPaneExParamater, IVertexColor4Holder
    {
        //----------------------------------------------------------
        //
        //----------------------------------------------------------

        IMaterial IMaterial { get; }
        string ContentsText { get; }
        string TextID { get; }
        ILEFont ILEFont { get; }
        string FontName { get; }
        FVec2 FontSize { get; }
        FVec2 FontSizeOriginal { get; }
        FVec2 FontScale { get; }
        float LineSpace { get; }
        float CharSpace { get; }
        HorizontalLocation BasePosTypeH { get; }
        VerticalLocation BasePosTypeV { get; }
        LineAlignment LineAlignment { get; }
        bool CenterCeilingEnabled { get; }
        bool DrawFromRightToLeft { get; }
        uint AllocStringLength { get; }
        bool SpecifiedStringLength { get; }
        RGBAColor TopColor { get; }
        RGBAColor BottomColor { get; }

        /// <summary>
        /// 斜体傾き係数
        /// </summary>
        float ItalicFactor { get; }

        //----------------------------------------------------------
        // 影関連
        //----------------------------------------------------------

        /// <summary>
        /// 影が有効かどうか
        /// </summary>
        bool ShadowEnabled { get; }

        /// <summary>
        /// 影のオフセット
        /// </summary>
        FVec2 ShadowOffset { get; }

        /// <summary>
        /// 影のスケール
        /// </summary>
        FVec2 ShadowScale { get; }

        /// <summary>
        /// 影斜体傾き係数
        /// </summary>
        float ShadowItalicFactor { get; }

        /// <summary>
        /// 影上端色
        /// </summary>
        RGBAColor ShadowTopColor { get; }

        /// <summary>
        /// 影下端色
        /// </summary>
        RGBAColor ShadowBottomColor { get; }

        /// <summary>
        /// 影黒カラー補間
        /// </summary>
        RGBAColor ShadowBlackColor { get; }

        /// <summary>
        /// 影白カラー補間
        /// </summary>
        RGBAColor ShadowWhiteColor { get; }

        /// <summary>
        /// 文字変化のオフセット時間
        /// </summary>
        float PerCharacterTransformTimeOffset { get; }

        /// <summary>
        /// 文字変化のオフセット時間
        /// </summary>
        float PerCharacterTransformTimeWidth { get; }

        /// <summary>
        /// 文字変化のループ種類
        /// </summary>
        PerCharTransformLoopType PerCharacterTransformLoopType { get; }

        /// <summary>
        /// 文字変化の原点（垂直方向）
        /// </summary>
        VerticalLocation PerCharacterTransformOriginV { get; }

        /// <summary>
        /// 文字変化の原点のオフセット（垂直方向）
        /// </summary>
        float PerCharacterTransformOriginVOffset { get; }

        /// <summary>
        /// 文字変化の文字幅による分割
        /// </summary>
        bool PerCharacterTransformSplitByCharWidth { get; }

        /// <summary>
        /// 文字変化の影のアルファを文字のアルファに自動追従させる
        /// </summary>
        bool PerCharacterTransformAutoShadowAlpha { get; }

        /// <summary>
        /// 文字変化の各文字の原点をテキストボックスの中心にする
        /// </summary>
        bool PerCharacterTransformOriginToCenter { get; }

        /// <summary>
        /// 文字変化の有効・無効
        /// </summary>
        bool IsPerCharTransformEnabled { get; }

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定機能の有効・無効
        /// </summary>
        bool IsPerCharTransformFixSpaceEnabled { get; }

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定の文字幅
        /// </summary>
        float PerCharTransformFixSpaceWidth { get; }

        /// <summary>
        /// 文字毎のアニメーションの文字幅固定の文字揃え
        /// </summary>
        HorizontalLocation PerCharTransformFixSpaceOrigin { get; }

        /// <summary>
        /// 文字毎のアニメーションの簡単設定
        /// </summary>
        EasySettingOfPerCharTransform[] EasySettingOfPerCharTransform { get; }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の開始角度
        /// </summary>
        float PerCharTransformEasyCircleBeginAngle { get; }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の終了角度
        /// </summary>
        float PerCharTransformEasyCircleEndAngle { get; }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の円の半径X
        /// </summary>
        float PerCharTransformEasyCircleRadiusX { get; }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の円の半径Y
        /// </summary>
        float PerCharTransformEasyCircleRadiusY { get; }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の分割数
        /// </summary>
        int PerCharTransformEasyCircleDivisionNumber { get; }

        /// <summary>
        /// 文字毎のアニメーションの円形配置の文字の向き
        /// </summary>
        float PerCharTransformEasyCircleCharAngle { get; }

        /// <summary>
        /// 縁取りOFF
        /// </summary>
        bool InvisibleBorderEnabled { get; }

        /// <summary>
        /// 2度書き縁取り
        /// </summary>
        bool DoubleDrawnBorderEnabled { get; }

        /// <summary>
        /// 行ごとの幅とオフセットの有効無効
        /// </summary>
        bool LineWidthOffsetEnabled { get; }

        /// <summary>
        /// 行ごとの幅
        /// </summary>
        float[] LineWidth { get; }

        /// <summary>
        /// 行ごとの幅の取得
        /// </summary>
        float GetLineWidth(int index);

        /// <summary>
        /// 行ごとの幅の設定
        /// </summary>
        void SetLineWidth(int index, float value);

        /// <summary>
        /// 行ごとのオフセット
        /// </summary>
        float[] LineOffset { get; }

        /// <summary>
        /// 行ごとのオフセット
        /// </summary>
        float GetLineOffset(int index);

        /// <summary>
        /// 行ごとのオフセットの設定
        /// </summary>
        void SetLineOffset(int index, float value);

        /// <summary>
        /// 拡張タグの解釈の有効無効
        /// </summary>
        bool IsExtendedTagEnabled { get; }

        //----------------------------------------------------------
        // アニメーション関連
        //----------------------------------------------------------

        IAnmAttribute TopColorIAnmAttr { get; }
        IAnmAttribute BottomColorIAnmAttr { get; }

        IAnmAttribute ShadowBlackColorIAnmAttr { get; }
        IAnmAttribute ShadowWhiteColorIAnmAttr { get; }

        IAnmAttribute PerCharTransformTimeOffsetIAnmAttr { get; }
        IAnmAttribute PerCharTransformTimeWidthIAnmAttr { get; }

        IAnmAttribute PerCharTransformTIAnmAttr { get; }
        IAnmAttribute PerCharTransformRIAnmAttr { get; }

        IAnmAttribute PerCharTransformTopColorIAnmAttr { get; }
        IAnmAttribute PerCharTransformBottomColorIAnmAttr { get; }
    }

    /// <summary>
    /// ヘルパ
    /// </summary>
    public static class ITextBoxHelper
    {
        /// <summary>
        /// 行ごとの幅とオフセットの数
        /// </summary>
        public const int LineOffsetWidthCountMax = 16;

        /// <summary>
        /// カラーが変更されているか、アニメーションを持つ場合は出力する。
        /// </summary>
        public static bool NeedToOutShadowParamaers(this ITextBox srcTextBox)
        {
            return
                srcTextBox.ShadowBlackColor != RGBAColor.Black ||
                srcTextBox.ShadowWhiteColor != RGBAColor.White ||
                HasShadowParamaerAnimation(srcTextBox);
        }

        /// <summary>
        /// 影アニメーションを持つか
        /// </summary>
        public static bool HasShadowParamaerAnimation(this ITextBox srcTextBox)
        {
            return
                srcTextBox.ShadowBlackColorIAnmAttr.CheckHaveKeyRecursiveInAllTag() ||
                srcTextBox.ShadowWhiteColorIAnmAttr.CheckHaveKeyRecursiveInAllTag();
        }

        /// <summary>
        /// 行ごとの幅とオフセットを左右の値に変換します
        /// </summary>
        public static void ChangeWidthOffsetToLeftRight(out float left, out float right, float offset, float width)
        {
            left = -offset;
            right = offset + width;
        }

        /// <summary>
        /// 行ごとのオフセットと幅を左右の値に変換します
        /// </summary>
        public static void ChangeLeftRightToWidthOffset(out float offset, out float width, float left, float right)
        {
            offset = -left;
            width = right + left;
        }
    }
}
