﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Diagnostics;
using System.Linq;
using System.Text.RegularExpressions;

namespace LECore.Structures
{
    using Core;
    using Core.Command;
    using Util;
    using System.Collections.Generic;
    using LECore.Structures.SerializableObject.Lyt;
    using System.Threading;

    /// <summary>
    /// レイアウトに対応する概念。
    /// 通常のController(View)は、本クラスに対して操作を行います。
    ///
    /// </summary>
    internal class SubScene :
        ISubScene,                             // 外部公開インタフェース
        ISelectedSetChangedEventListener,   // 選択セットの変更通知を監視します。
        IDisposable
    {
        /// <summary>
        /// ルートペイン名
        /// </summary>
        const string _RootPaneName = "RootPane";

        #region ------------ イベント ------------
        internal event Action<ISubScene, SceneModifyEventArgs.Kind> OnModified;
        #endregion ------------ イベント ------------


        #region ------------ フィールド ------------
        Scene _grobalScene = Scene.Instance;

        // Pane配列
        List<Pane> _paneSet = new List<Pane>();

        // グループセット
        LEGroupMgr _groupMgr = null;

        // 階層構造のルートを定義するペイン
        Pane _rootPane = new Pane();

        // 選択セット
        // TODO:Scene Manipulator に移動されます。=> されません。
        SelectedSet _selectedSet = null;

        // 大量のシーン更新メッセージを
        // 1つのメッセージに変換するクラス
        SceneModifyEventAggregater _sceneModifyEventAggregater = new SceneModifyEventAggregater();

        // キャプチャテクスチャの更新モード
        CaptureTextureUpdateMode _captureTextureUpdateMode = CaptureTextureUpdateMode.All;

        // キャプチャテクスチャの更新フラグ
        bool _updateCaptureTexture = true;

        // キャプチャテクスチャを順番に更新する際のインデックス
        UInt32 _captureTextureUpdateIndex = 0;

        // ペインからのペイン変更通知を受け取るデリゲータのインスタンス
        readonly Pane.OnPainModifyHandler _OnPainModifyHandler = null;
        readonly EventHandler _OnGroupModifyHandler = null;

        // 背景設定情報関連
        BackGround _backGround = new BackGround();

        /// <summary>
        /// Undo/Redo用コマンドキュー
        /// </summary>
        readonly CommandFactory _cmdFactory = null;
        readonly SubSceneCmdQueue _cmdQueue = null;


        // アニメーションフレーム区間セット
        readonly AnimFrameSectionSet _animFrameSectionSet = new AnimFrameSectionSet();

        /// <summary>
        /// テクスチャマネージャ
        /// </summary>
        readonly TextureMgr _textureMgr = null;

        /// <summary>
        /// フォントマネージャ
        /// </summary>
        readonly FontManager _fontMgr = new FontManager();

        /// <summary>
        /// アニメーション共有情報マネージャ
        /// </summary>
        readonly AnimShareInfomationManager _animShareInfomationMgr = new AnimShareInfomationManager();

        /// <summary>
        /// 部品情報
        /// </summary>
        readonly PartsSettings _partsSettings;

        /// <summary>
        /// コントロール定義セット
        /// </summary>
        readonly ControlSettingsSet _ctrlSettingSet;

        // アニメーション範囲
        readonly Range _animRange = new Range(0, 100);
        // アニメーション再生範囲
        readonly Range _animPlayRange = new Range(0, 100);

        // 拡張ユーザ情報
        readonly UserDataHolder _userDataHolder = new UserDataHolder();

        private IPane _currentPane = null;

        // TODO:画像リソース
        // TODO:フォントリソース
        #endregion // ------------ フィールド ------------


        #region ------------ プロパティ ------------
        public LayoutDocument OwnerDocument { get; internal set; }

        public IDocumentHeader DocumentHeader { get; set; }

        /// <summary>
        /// シーン中のペイン(読み取り専用)
        /// </summary>
        public IEnumerable<Pane> PaneSet
        {
            get { return _paneSet; }
        }

        public IEnumerable<IPane> IPaneArray
        {
            get { return _paneSet.ConvertAll((x) => x as IPane); }
        }

        /// <summary>
        /// 選択ペインセットを返します。
        /// </summary>
        public SelectedSet SelectedPaneSet
        {
            get { return _selectedSet; }
        }

        public IPane CurrentPane
        {
            get
            {
                return this._currentPane;
            }
            set
            {
                if (this._currentPane != value)
                {
                    this._currentPane = value;

                    NotifySceneModifyEvent_(
                    new SceneModifyEventArgs(
                    SceneModifyEventArgs.Kind.CurrentPaneModify, this));
                }
            }
        }

        /// <summary>
        /// 階層構造のルート
        /// </summary>
        public Pane RootPane
        {
            get { return _rootPane; }
        }

        /// <summary>
        /// 背景設定情報関連
        /// </summary>
        public BackGround BackGround
        {
            get { return _backGround; }
        }

        public IUserDataHolder IUserDataHolder { get { return _userDataHolder; } }

        /// <summary>
        /// サブシーンが管理するコマンド生成クラスを取得します。
        /// </summary>
        public CommandFactory CommandFactory
        {
            get { return _cmdFactory; }
        }

        public void Undo()
        {
            _cmdQueue.Undo();
        }

        public void Redo()
        {
            _cmdQueue.Redo();
        }

        /// <summary>
        /// アニメーション範囲(開始)
        /// </summary>
        public int AnimStartTime { get { return this.AnimRange.Min; } }

        /// <summary>
        /// アニメーション範囲(終了)
        /// </summary>
        public int AnimEndTime { get { return this.AnimRange.Max; } }

        /// <summary>
        /// アニメーション再生範囲(開始)
        /// </summary>
        public int AnimPlayStartTime { get { return this.AnimPlayRange.Min; } }

        /// <summary>
        /// アニメーション再生範囲(終了)
        /// </summary>
        public int AnimPlayEndTime { get { return this.AnimPlayRange.Max; } }

        /// <summary>
        /// 時間パラメータを設定します。
        /// </summary>
        public bool SetTimeRange( int start, int end, int playStart, int playEnd )
        {
            // 開始と終了が入れ替わっている場合、
            // 警告を表示して、設定を行いません。
            if( ( start >= end ) || ( playStart >= playEnd ) )
            {
                return false;
            }

            this.AnimRange.Max = Math.Max(end, playEnd);
            this.AnimRange.Min = Math.Min(start, playStart);

            this.AnimPlayRange.Max = playEnd;
            this.AnimPlayRange.Min = playStart;

            return true;
        }

        public Range AnimRange
        {
            get { return _animRange; }
        }

        // アニメーション再生範囲
        public Range AnimPlayRange
        {
            get { return _animPlayRange; }
        }

        /// <summary>
        /// アニメーションが分割編集モードか
        /// </summary>
        public bool IsAnimEditSeparate { get; set; }

        /// <summary>
        /// 選択中の区間タグ名
        /// </summary>
        public string CurrentTagName
        {
            get
            {
                return this.IsAnimEditSeparate ?
                    this.IAnimFrameSectionSet?.TargetIAnimFrameSection?.Name ?? String.Empty :
                    String.Empty;
            }
        }

        /// <summary>
        /// 分割モード時のアニメーション再評価中かどうか
        /// </summary>
        public bool EvaluateMultiAnimation
        {
            get;
            set;
        }

        public IStateMachine IStateMachine
        {
            get;
            private set;
        }

        #endregion // ------------ プロパティ ------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SubScene()
        {
            _textureMgr = new TextureMgr(this);

            DocumentHeader = new DocumentHeader()
            {
                Comment = string.Empty,
                Title = string.Empty,
            };
            (DocumentHeader as DocumentHeader).OnModify += (s, e) => NotifySceneModifyEvent_(e);

            _groupMgr = new LEGroupMgr(this);

            _selectedSet = new SelectedSet(this);
            _rootPane.X = 0;
            _rootPane.Y = 0;
            _rootPane.Size = FVec3.Empty;

            _OnPainModifyHandler = new Pane.OnPainModifyHandler(OnPainModify_);
            _OnGroupModifyHandler = new EventHandler(OnGroupModify_);

            _rootPane.PaneName = _RootPaneName;
            _rootPane.OnPainModify += this._OnPainModifyHandler;
            _groupMgr.OnGroupMgrModify += this._OnGroupModifyHandler;
            _rootPane.OwnerSubScene = this;

            // 選択セットの変更イベントをシーン更新イベントとして受け取ります。
            _selectedSet.AddListener(this);

            // コマンドクラスを初期化します。
            _cmdQueue = new SubSceneCmdQueue();
            _cmdFactory = new CommandFactory(_cmdQueue);

            _partsSettings = new PartsSettings(this);
            _ctrlSettingSet = new ControlSettingsSet(this);

            _textureMgr.OnTextureMgrUpdate += _textureMgr_OnTextureMgrUpdate;
            _fontMgr.OnLEFontMgrUpdate += _fontMgr_OnLEFontMgrUpdate;
            _animFrameSectionSet.OnAnimSectionTagChanged += _animFrameSectionSet_OnAnimSectionTagSetUpdate;
            _animShareInfomationMgr.OnAnimShareManagerChanged += _animShareInfomationMgr_OnAnimShareManagerChanged;
            _partsSettings.OnChanged += _partsSettings_OnChanged;
            _ctrlSettingSet.OnChanged += _partsSettings_OnChanged;
            _backGround.OnBackGroundModify += _backGround_OnBackGroundModify;
            _userDataHolder.OnChanged += () => NotifySceneModifyEvent_(new SceneModifyEventArgs(SceneModifyEventArgs.Kind.SubScenePropertyModify, this));

            this.IStateMachine = new StateMachineData(this);
            (IStateMachine as StateMachineData).OnChanged += _StateMachine_OnChanged;

            this.OnModified += this.RegisterNewTextureToPartsSubScenes_;
        }

        void _backGround_OnBackGroundModify(BackGround sender, SceneModifyEventArgs args)
        {
            NotifySceneModifyEvent_(args);
        }

        void _StateMachine_OnChanged(SceneModifyEventArgs.Kind kind)
        {
            SceneModifyEventArgs args = new SceneModifyEventArgs(kind, null);
            NotifySceneModifyEvent_(args);
        }

        /// <summary>
        /// リソースを破棄します。
        /// </summary>
        public void Dispose()
        {
            ResetScene();
            _rootPane.Dispose();

            _textureMgr.Dispose();

            // TODO: LEFont の _fontReferenceMap が _fontMgr への参照を残している可能性がある。
            _fontMgr.OnLEFontMgrUpdate -= _fontMgr_OnLEFontMgrUpdate;
            _fontMgr.Dispose();

            _partsSettings.Reset();
        }

        /// <summary>
        /// 編集対象として設定される直前の初期化処理です。
        /// </summary>
        public void FirstTimeInitialize()
        {
            this._partsSettings.ResolveReference(this);
            foreach(var pane in this.PaneSet)
            {
                pane.IPaneExParamater?.FirstTimeInitialize();
            }
        }

        #region 自動命名処理


        /// <summary>
        /// 部品レイアウト以下から同名ペインを探す。
        /// </summary>
        IPane FindPaneFromPartPane_(IPane pane, string name)
        {
            if (pane != null && pane.PaneKind == PaneKind.Parts)
            {
                return SubSceneHelper.FindPaneByName(pane.IPartsLayout.PartsSubScene, name);
            }

            return null;
        }

        /// <summary>
        /// FindTargetByName を生成するためのラッパーメソッド
        /// </summary>
        object FindPaneByName_(string currentName, string newName)
        {
            // 名前が変更されているなら、衝突が無いか探す。
            object result = null;
            if (currentName != newName)
            {
                result = SubSceneHelper.FindPaneByName(this, newName);
            }

            // 衝突が発見されなかった場合は
            if(result == null)
            {
                // 実行時には、部品ペインがルートペインとなる。
                // そのため、部品レイアウト中に、部品ペインと同名のペインが存在しないようにしないと
                // 名前の重複が起こってしまう。
                // これを防ぐために部品ペイン配下のレイアウトについても調査する。
                IPane currentPane = SubSceneHelper.FindPaneByName( this, currentName );
                return FindPaneFromPartPane_(currentPane, newName);
            }

            return result;
        }

        /// <summary>
        /// シーン内で重複しない適切なペイン名前を取得します。
        /// </summary>
        public string GetUniqueNewPaneName(IPane pane, string currentName, string newName)
        {
            return SubSceneHelper.GetValidName(
                (name) =>
                {
                    // シーン中のペイン名の重複をチェックする。
                    object result = FindPaneByName_(currentName, name);

                    // ペイン名の重複がない場合は、キャプチャテクスチャ系のペイン用にテクスチャ名との重複をチェックする。
                    if (result == null)
                    {
                        if (PaneHelper.IsCaptureTextureNameDuplicated(pane.OwnerSubScene, pane, name))
                        {
                            // 重複する何かが存在することを返すため object を作成する。
                            result = new object();
                        }
                    }

                    return result;
                },
                AppConstants.MaxPaneNameLength, newName);
        }

#endregion 自動命名処理

#region シーンに対する大量の変更の通知
        /// <summary>
        /// 大量に変更操作を行うことをシーンに通知します。
        ///
        /// 本関数を実行することによって、
        /// 多数のシーン変更コマンドを集約することが可能です。
        ///
        /// 本関数は、Undo/Redoの管理を行う、コマンドマネージャの
        /// コマンド集約機能を同時に設定します。
        /// このことによって、大量の変換は、
        /// 1つのUndo単位として取り扱われるようになります。
        /// </summary>
        public void BeginMassiveModify()
        {
            // イベント集約機を有効にします。
            _sceneModifyEventAggregater.BeginRegister();
            // コマンド集約を開始します。
            _cmdQueue.BeginCmdPacking( this );

            GlobalTime.Inst.SuspendEventNotification();
        }

        /// <summary>
        /// 大量に変更操作が完了したことをシーンに通知します。
        /// </summary>
        public void EndMassiveModify()
        {
            EndMassiveModify(false, false);
        }

        /// <summary>
        /// 大量に変更操作が完了したことをシーンに通知します。
        ///
        /// BeginMassiveModify()が呼ばれてから、
        /// 本関数メソッドコールまでのシーン変更を集約したイベントが送信されます。
        /// </summary>
        public void EndMassiveModify(bool forceDropCommand, bool forceDropEvent)
        {
            // コマンド集約を終了します。(内部で、複合コマンドが生成され登録されます。)
            _cmdQueue.EndCmdPacking(forceDropCommand);

            // イベント集約機を無効にし、集約されたコマンドを取り出します。
            SceneModifyEventArgs[] aggregatedCmdSet = _sceneModifyEventAggregater.EndRegister();
            if (!forceDropEvent)
            {
                // サブシーン変更をシーンに通知
                foreach (var arg in aggregatedCmdSet)
                {
                    // _grobalScene.OnSubSceneChanged( this, arg );
                    NotifySceneModifyEvent_(arg);
                }
            }

            // ペイン削除イベントが含まれる場合は、上書き設定の参照再評価を行う。
            if(aggregatedCmdSet.Any((cmd) => cmd.Target == SceneModifyEventArgs.Kind.PaneRemove))
            {
                this._partsSettings.ResolveReference(this);
            }

            GlobalTime.Inst.ResumeEventNotification(true);
        }
#endregion シーンに対する大量の変更の通知

#region ペインの追加
        /// <summary>
        /// ペインを追加します。
        /// Undoが反映されないAPIです。
        /// モジュール外部には公開されません。
        /// </summary>
        /// <param name="newPane"></param>
        public void AddPane(Pane newPane)
        {
            // 登録済みではないペインを登録しているはず。
            Debug.Assert(_paneSet.IndexOf(newPane) == -1);

            // 名前が重複しないように、チェックします。
            newPane.PaneName = GetUniqueNewPaneName(newPane, string.Empty, newPane.PaneName);

            Debug.Assert(newPane != null);
            newPane.OnPainModify += this._OnPainModifyHandler;
            newPane.OwnerSubScene = this;
            newPane.OnJoinSceneInitialize();

            _paneSet.Add(newPane);

            // ルートペインの子供とする
            if (newPane.Parent == null)
            {
                _rootPane.AddChildPane(newPane);
            }

            // 新たに追加されたペインのテクスチャキャッシュをリセットして新たに取り直します。
            foreach (var mat in newPane.IMaterial)
            {
                foreach (var itexMap in mat.IMaterialTexMapSet)
                {
                    MaterialTexMap texMap = itexMap as MaterialTexMap;
                    texMap.ImageCache.Reset();
                }
            }

            // パーツ内のキャプチャテクスチャ参照を更新するためにテクスチャキャッシュをリセットします。
            PaneHelper.ResetCaptureTextureImageCacheInPartsTree(newPane);

            TryPrepareTextboxScFontUpdate_(newPane);

            // Pane追加イベントの通知
            SceneModifyEventArgs ea =
                new SceneModifyEventArgs(SceneModifyEventArgs.Kind.PaneAdd, this);

            NotifySceneModifyEvent_(ea);
        }

        /// <summary>
        /// ペインを削除します。
        /// Undoが反映されないAPIです。
        /// モジュール外部には公開されません。
        ///
        /// 以前のバージョンで行っていた、以下の処理は、
        /// 本メソッド内では、行わないことにしました。
        ///
        /// ・階層構造を破壊する。
        /// removedPane.DestroyHierarchyConnection();
        /// ・グループメンバから削除する。
        /// _groupMgr.RemoveMemberFromAll( removedPane );
        ///
        /// このような処理は、本メソッドを使用する
        /// マニュピレータクラスで実行するようにしました。
        /// Undo/Redoなどのコマンドクラスが本メソッドを利用するため、
        ///
        /// 余計な処理が入っていると求める挙動と異なってしまうためです。
        ///
        /// </summary>
        /// <param name="newPane"></param>
        public void RemovePain( Pane removedPane )
        {
            _selectedSet.Reset();

            removedPane.OnPainModify -= this._OnPainModifyHandler;
            removedPane.OwnerSubScene = null;

            _paneSet.Remove( removedPane );

            // 親がルートペインならば、ルートの子供から削除します。
            if( removedPane.Parent == RootPane )
            {
                removedPane.Parent.RemoveChildNode( removedPane );
            }

            // Pane削除イベントの通知
            SceneModifyEventArgs ea =
                new SceneModifyEventArgs( SceneModifyEventArgs.Kind.PaneRemove, this );
            NotifySceneModifyEvent_( ea );
        }
#endregion ペインの追加

#region アニメーション編集モードの変更
        /// <summary>
        /// アニメーション編集モードを変更します。
        /// </summary>
        public void ChangeAnimEditMode( bool val)
        {
            if(this.IsAnimEditSeparate == val)
            {
                return;
            }

            this.IsAnimEditSeparate = val;

            // 分割モード・統合モードの変更
            if (this.IsAnimEditSeparateMode())
            {
                // 結合モード→分割モードの場合
                SubSceneHelper.EvaluateAnimation(this, GlobalTime.Inst.Time); // カーブを再評価する
            }
            else
            {
                // 分割モード→結合モードの場合
                SubSceneHelper.ResetValueAsBaseValue(this); // アトリビュートの現在値をbaseValueに戻す
                SubSceneHelper.EvaluateAnimation(this, GlobalTime.Inst.Time); // カーブを再評価する
            }

            NotifyAnimationEditModeChanged();
        }

        /// <summary>
        /// アニメーション編集モードを変更します。
        /// </summary>
        public void NotifyAnimationEditModeChanged()
        {
            var ea = new SceneModifyEventArgs(SceneModifyEventArgs.Kind.PaneAnimModeModify, this);
            NotifySceneModifyEvent_(ea);
        }
#endregion

#region 部品レイアウトの設定
        /// <summary>
        /// 部品レイアウトを設定します。
        /// </summary>
        /// <param name="partsLayout">部品レイアウト</param>
        public void SetPartsLayout(PartsLayout partsLayout)
        {
            this.IPartsLayout = partsLayout;
        }

#endregion

#region シーンのリセット
        /// <summary>
        /// シーンをリセットします。
        /// </summary>
        public void ResetScene()
        {
            Pane[] paneSet = _paneSet.ToArray();

            bool bMassiveModify = _sceneModifyEventAggregater.Active;
            if( !bMassiveModify )
            {
                BeginMassiveModify();
                foreach (Pane pane in paneSet)
                {
                    pane.Dispose();
                    RemovePain( pane );
                    _groupMgr.RemoveMemberFromAll( pane );
                    pane.Dispose();
                }
                EndMassiveModify();
            }
            else
            {
                foreach( Pane pane in paneSet )
                {
                    pane.Dispose();
                    RemovePain( pane );
                    _groupMgr.RemoveMemberFromAll( pane );
                    pane.Dispose();
                }
            }

            ResetUndoState();
        }

        /// <summary>
        /// Undoコマンドの状態をリセットします。
        /// </summary>
        public void ResetUndoState()
        {
            _cmdQueue.Reset();
        }


        #endregion シーンのリセット

        #region メッセージ通知

        /// <summary>
        /// シーン変更イベントを通知します。
        /// </summary>
        /// <param name="ea"></param>
        void NotifySceneModifyEvent_( SceneModifyEventArgs ea )
        {
            // コマンド集約機が有効ならば...
            if( _sceneModifyEventAggregater.Active )
            {
                // イベントの伝播を行わず、集約します。
                _sceneModifyEventAggregater.AddCommand( ea );
            }
            else
            {
                if (OnModified != null)
                {
                    OnModified(this, ea.Target);
                }

                // サブシーン変更をシーンに通知
                _grobalScene.OnSubSceneChanged( this, ea );
            }
        }

        /// <summary>
        /// ペイン変更イベントを起こす
        /// </summary>>
        public void RaisePaneModifyEvent()
        {
            OnPainModify_(null, new SceneModifyEventArgs(SceneModifyEventArgs.Kind.PaneModify, new Pane()));
        }

        /// <summary>
        /// ペイン変更イベントを起こす
        /// </summary>>
        public void RaisePaneModifyEvent(IEnumerable<IPane> args)
        {
            SceneModifyEventArgs eventArgs = new SceneModifyEventArgs(SceneModifyEventArgs.Kind.PaneModify);
            eventArgs.AddParameters(args);

            OnPainModify_(null, eventArgs);
        }

        /// <summary>
        /// スケーラブルフォントのセットアップをします。
        /// </summary>
        public void PrepareTextboxScFontUpdate()
        {
            foreach (Pane pane in this.IPaneArray)
            {
                TryPrepareTextboxScFontUpdate_(pane);
            }
        }

        /// <summary>
        /// スケーラブルフォントのセットアップをします。
        /// </summary>
        void TryPrepareTextboxScFontUpdate_(Pane pane)
        {
            if (pane == null || pane.PaneKind != PaneKind.Textbox)
            {
                return;
            }

            ITextBox textBox = pane.ITextBox;
            if (textBox.ILEFont == null)
            {
                return;
            }

            var scfonts = LEFontHelper.GetLeafs(textBox.ILEFont.NWFont).OfType<NW4R.Font.ScFont>().ToArray();
            if (scfonts.Length == 0)
            {
                return;
            }

            foreach (var scFont in scfonts)
            {
                bool newCharacterRegisterd = false;
                if (textBox.ContentsText.Length == 0)
                {
                    IEnumerable<ITextBox> sameFontTextboxSet = new List<ITextBox>();
                    foreach (var subScene in Scene.Instance.SubSceneSet)
                    {
                        sameFontTextboxSet = sameFontTextboxSet.Concat(GetTextboxUsingSameFontInAllSubScenes_(subScene, textBox.ILEFont));
                    }
                    bool fontUsedNoGlyph = sameFontTextboxSet.All((text) => text.ContentsText.Count() == 0);
                    if (fontUsedNoGlyph)
                    {
                        lock (textBox.ILEFont.GetLockHandle())
                        {
                            //DbgConsole.WriteLine("Reset --------------- ");
                            scFont.ResetTextureCache();
                            scFont.RegisterAlternateCharGlyph();
                        }

                        newCharacterRegisterd = true;
                    }
                }
                else
                {
                    var sameFontTextboxSet = GetTextboxUsingSameFont_(pane.OwnerSubScene, textBox.ILEFont);
                    foreach (var sameFontTextbox in sameFontTextboxSet)
                    {
                        foreach (char c in sameFontTextbox.ContentsText.Distinct().ToArray())
                        {
                            if (!scFont.IsGlyphExistInFont(c))
                            {
                                continue;
                            }

                            lock (textBox.ILEFont.GetLockHandle())
                            {
                                if (!scFont.IsGlyphPlot(c))
                                {
                                    //DbgConsole.WriteLine("RegisterGlyph --------------- {0}", c);
                                    scFont.RegisterGlyph(c, 1);

                                    newCharacterRegisterd = true;
                                }
                                else
                                {
                                    // 未使用と判定され削除されてしまわないように、利用中マークをする。
                                    scFont.SetGlyphUsed(c);
                                }
                            }
                        }
                    }
                }

                if (newCharacterRegisterd)
                {
                    // フォント初期化終了時アクション。更新イベントをGUIスレッドで発行する。
                    var guiThreadContext = SynchronizationContext.Current;
                    Action onPreUpdate = new Action(() =>
                    {
                        lock (textBox.ILEFont.GetLockHandle())
                        {
                            //DbgConsole.WriteLine("UpdateAndComplete ---------------");
                            scFont.UpdateAndCompleteTheTextureCache();
                        }
                    });
                    Action onFinishUpdate = new Action(() =>
                    {
                        if (guiThreadContext != null)
                        {
                            guiThreadContext.Post((arg) => this.RaisePaneModifyEvent(), null);
                        }
                    });

                    if (textBox.ILEFont.IsBitmapReady)
                    {
                        textBox.ILEFont.UpdateFontAsync(onPreUpdate, onFinishUpdate);
                    }
                    else
                    {
                        textBox.ILEFont.InitFontAsync(() => textBox.ILEFont.UpdateFontAsync(onPreUpdate, onFinishUpdate));
                    }
                }
            }
        }



        /// <summary>
        ///
        /// </summary>
        static IEnumerable<ITextBox> GetTextboxUsingSameFont_(ISubScene subScene, ILEFont font)
        {
            return subScene.IPaneArray.Where((p) => p.PaneKind == PaneKind.Textbox && object.ReferenceEquals(p.ITextBox.ILEFont, font)).Select<IPane, ITextBox>((p) => p.ITextBox);
        }

        /// <summary>
        ///
        /// </summary>
        static IEnumerable<ITextBox> GetTextboxUsingSameFontInAllSubScenes_(ISubScene subScene, ILEFont font)
        {
            List<ITextBox> textBoxList = new List<ITextBox>();
            foreach (var pane in subScene.IPaneArray)
            {
                PaneHelper.CollectTextBoxesUsingSpecificFont(pane, font, textBoxList);
            }
            return textBoxList;
        }

        /// <summary>
        /// ペイン変更ハンドラ
        /// </summary>>
        void OnPainModify_( Pane sender, SceneModifyEventArgs args )
        {
            if (args.Target == SceneModifyEventArgs.Kind.TextboxFontChanged ||
                args.Target == SceneModifyEventArgs.Kind.TextboxContentsChanged)
            {
                TryPrepareTextboxScFontUpdate_(sender);
            }

            NotifySceneModifyEvent_( args );
        }

        /// <summary>
        /// グループ更新イベントハンドラ
        /// </summary>
        void OnGroupModify_( object sender, EventArgs args )
        {
            LEGroupMgrModifyEventArgs groupArg = args as LEGroupMgrModifyEventArgs;
            if( groupArg == null )
            {
                return;
            }
            // イベントを生成して...
            SceneModifyEventArgs ea
                = new SceneModifyEventArgs( groupArg.EventKind, groupArg.GroupMgrParams.Group );

            NotifySceneModifyEvent_( ea );
        }

        /// <summary>
        /// テクスチャ登録時ハンドラ。
        /// すべての部品ペインについて、未登録テクスチャを登録します。
        /// </summary>
        private void RegisterNewTextureToPartsSubScenes_(
            ISubScene subScene, SceneModifyEventArgs.Kind kind)
        {
            Ensure.Argument.True(object.ReferenceEquals(subScene, this));

            if (kind != SceneModifyEventArgs.Kind.TextureManager)
            {
                return;
            }

            var partsPanes = this.GetPaneSet(
                    (pane) =>
                        pane.PaneKind == PaneKind.Parts &&
                        pane.IPartsLayout.PartsPropaerties.Count() > 0);

            foreach (var partsPane in partsPanes)
            {
                var partsSubScene = partsPane.IPartsLayout.PartsSubScene as ISubScene;
                var partsTexMgr = partsSubScene.ITextureMgr as TextureMgr;

                var unRegisteredTextures = subScene.ITextureMgr.ITextureImageSet.Where(
                    (tex) => partsTexMgr.FindITextureImageByName(tex.Name) == null);
                {
                    if (unRegisteredTextures.Count() > 0)
                    {

                        try
                        {
                            partsSubScene.BeginMassiveModify();
                            partsTexMgr.BeginUpdate();
                            foreach (var unRegisteredTexture in unRegisteredTextures)
                            {
                                partsTexMgr.SyncTextureImageNoEvent(unRegisteredTexture);
                            }
                        }
                        finally
                        {
                            partsTexMgr.EndUpdate();
                            partsSubScene.EndMassiveModify(true, true);
                        }
                    }
                }
            }
        }

#endregion メッセージ通知

#region IDrawable 実装

        /// <summary>
        /// 木構造に沿ってペインを描画します。
        /// </summary>
        void DrawByTreeOrder_(IRenderer renderer, Pane pane, DrawableOption option, Func<IPane, bool> paneDrawCondition)
        {
            // スケールが有効ならば...
            if( pane.ValidScaleForDraw )
            {
                renderer.PushMtx();

                // 変換を乗算
                Matrix34 mtxPane = pane.LocalMtx34;
                Matrix34 mtxCurr = renderer.CurrentMtx34;

                renderer.CurrentMtx34 = mtxCurr * mtxPane;

                int transparency = renderer.Transparency;

                if(pane.PaneKind == PaneKind.Parts)
                {
                    if (pane.InfluenceChildrenTransparency)
                    {
                        renderer.Transparency = (int)(transparency * PaneHelper.GetNormalizedTransparency(pane.Transparency));
                    }
                }else{
                    renderer.Transparency = (int)(transparency * PaneHelper.GetNormalizedTransparency( pane.Transparency ));
                }

                // ペインの描画
                if (paneDrawCondition(pane))
                {
                    pane.Draw(renderer, option);
                }

                // 子供に再帰
                {
                    if (!option.TerminateDrawTraverse)
                    {
                        if (!pane.InfluenceChildrenTransparency)
                        {
                            renderer.Transparency = transparency;
                        }

                        if (!pane.Visible)
                        {
                            renderer.Transparency = 0;
                        }

                        foreach (Pane childPane in pane.Children)
                        {
                            DrawByTreeOrder_(renderer, childPane, option, paneDrawCondition);
                        }

                        renderer.Transparency = transparency;
                    }

                    option.TerminateDrawTraverse = false;
                }

                renderer.PopMtx();
            }
        }

        /// <summary>
        /// シーンを描画します。
        /// </summary>
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            // キャプチャテクスチャのビットマップが更新されていたらテクスチャも作り直します。
            TextureMgr texMgr = ITextureMgr as TextureMgr;

            if (texMgr != null)
            {
                foreach (var itex in texMgr.ITextureImageSet)
                {
                    CaptureTexture tex = itex as CaptureTexture;
                    if (tex != null)
                    {
                        //_ownTexture.GDIBit
                        renderer.TryUpdateTextureIfOld(tex.GDIBitmap, DateTime.MinValue, RendererTextrureFormat.ARGB);
                    }
                }
            }

            this.Draw(renderer, option, (pane) => true);
        }

        /// <summary>
        /// シーンを描画します。
        /// </summary>
        public void Draw(IRenderer renderer, DrawableOption option, Func<IPane, bool> paneDrawCondition)
        {
            // root 以下のペインを描画します。
            foreach( Pane childPane in _rootPane.Children )
            {
                DrawByTreeOrder_(renderer, childPane, option, paneDrawCondition);
            }
        }
#endregion

#region ISubScene メンバ
        /// <summary>
        /// 描画準備。
        /// </summary>
        public void PrepareDraw(IRenderer renderer)
        {
        }

        /// <summary>
        /// キャプチャテクスチャの描画更新処理
        /// </summary>
        /// <param name="renderer">キャプチャテクスチャのレンダリングに使用するレンダラー</param>
        public void RenderCaptureTexture(IRenderer renderer)
        {
            if (_updateCaptureTexture)
            {
                switch (_captureTextureUpdateMode)
                {
                    case CaptureTextureUpdateMode.All:
                        {
                            // 全キャプチャテクスチャを一度に更新する。
                            // パーツペインのキャプチャテクスチャを先に更新する
                            foreach (var partsPane in SubSceneHelper.FindPanesByKindRecursively(this, PaneKind.Parts))
                            {
                                if (partsPane.IPartsLayout != null)
                                {
                                    foreach (var tex in TextureMgrHelper.GetCaptureTextureArray(partsPane.IPartsLayout.PartsSubScene.ITextureMgr))
                                    {
                                        tex.ICaptureTexture.RenderCaptureTexture(renderer, BackGround.BackgroundImage);
                                    }
                                }
                            }

                            foreach (var tex in TextureMgrHelper.GetCaptureTextureArray(ITextureMgr))
                            {
                                tex.ICaptureTexture.RenderCaptureTexture(renderer, BackGround.BackgroundImage);
                            }

                            // 一度全体を更新した後はシーケンシャル更新モードに移行する。
                            _captureTextureUpdateMode = CaptureTextureUpdateMode.Sequential;
                        }
                        break;
                    case CaptureTextureUpdateMode.Sequential:
                        {
                            // キャプチャテクスチャを 1 paint に 1 つづつ更新して負荷を分散する。
                            List<ITextureImage> captureTextures = new List<ITextureImage>();
                            // パーツペインのキャプチャテクスチャを先に更新する
                            foreach (var partsPane in SubSceneHelper.FindPanesByKindRecursively(this, PaneKind.Parts))
                            {
                                if (partsPane.IPartsLayout != null)
                                {
                                    captureTextures.AddRange(TextureMgrHelper.GetCaptureTextureArray(partsPane.IPartsLayout.PartsSubScene.ITextureMgr));
                                }
                            }
                            captureTextures.AddRange(TextureMgrHelper.GetCaptureTextureArray(ITextureMgr));
                            if (captureTextures.Count > 0)
                            {
                                captureTextures[(int)(_captureTextureUpdateIndex % captureTextures.Count)].ICaptureTexture.RenderCaptureTexture(renderer, BackGround.BackgroundImage);
                            }
                        }
                        break;
                }
                _updateCaptureTexture = false;
            }
        }

        /// <summary>
        /// キャプチャテクスチャの描画更新リクエスト
        /// </summary>
        /// <param name="mode">キャプチャテクスチャの更新モードです</param>
        public void RequestRenderCaptureTexture(CaptureTextureUpdateMode mode)
        {
            _captureTextureUpdateMode = mode;
            _updateCaptureTexture = true;
            _captureTextureUpdateIndex = 0;
        }

        /// <summary>
        /// キャプチャテクスチャの状態を更新します。
        /// </summary>
        public void UpdateCaptureTextureState()
        {
            if (!_updateCaptureTexture)
            {
                _updateCaptureTexture = true;
                ++_captureTextureUpdateIndex;
            }
        }

        public IPane RootIPane
        {
            get { return _rootPane; }
        }

        /// <summary>
        /// グループマネージャを取得します。
        /// </summary>
        public ILEGroupMgr ILEGroupMgr
        {
            get { return _groupMgr; }
        }

        /// <summary>
        /// グループマネージャを取得します。
        /// </summary>
        public LEGroupMgr LEGroupMgr
        {
            get { return _groupMgr; }
        }

        /// <summary>
        /// テクスチャマネージャ
        /// </summary>
        public ITextureMgr ITextureMgr
        {
            get { return _textureMgr; }
        }

        /// <summary>
        /// フォントマネージャ
        /// </summary>
        public ILEFontManager ILEFontManager
        {
            get { return FontManager; }
        }

        /// <summary>
        /// アニメーション共有マネージャ
        /// </summary>
        public IAnimShareInfomationManager IAnimShareInfomationManager
        {
            get { return _animShareInfomationMgr; }
        }

        /// <summary>
        /// フォントマネージャ
        /// </summary>
        public FontManager FontManager
        {
            get { return _fontMgr; }
        }

        /// <summary>
        /// コマンドキューの内部状態を取得します。
        /// </summary>
        public SubSceneCmdQueue.InternalState CmdQueueInternalState
        {
            get { return _cmdQueue.InternalStateData; }
        }

        /// <summary>
        /// コマンドキューを取得します。
        /// </summary>
        public ISubSceneCmdQueue ISubSceneCmdQueue
        {
            get { return _cmdQueue; }
        }

        /// <summary>
        /// 選択セットを返します。
        /// </summary>
        public ISelectedSet ISelectedSet
        {
            get { return _selectedSet; }
        }

        /// <summary>
        /// アニメーション区間セットを返します。
        /// </summary>
        public IAnimFrameSectionSet IAnimFrameSectionSet { get { return _animFrameSectionSet; } }

        /// <summary>
        /// パーツ設定を返します。
        /// </summary>
        public IPartsSettings IPartsSettings { get { return _partsSettings; } }

        /// <summary>
        /// 部品レイアウトを返します。
        /// </summary>
        public IPartsLayout IPartsLayout { get; private set; }

        /// <summary>
        /// コントロール設定
        /// </summary>
        public IEnumerable<IControlSettings> IControlSettings { get { return _ctrlSettingSet.Items; } }

        /// <summary>
        /// コントロール設定
        /// </summary>
        public IControlSettingsSet IControlSettingsSet { get { return _ctrlSettingSet; } }

        /// <summary>
        /// コントロール設定セット
        /// </summary>
        public ControlSettingsSet ControlSettingsSet { get { return _ctrlSettingSet; } }

        /// <summary>
        /// Undo可能か
        /// </summary>
        public bool UndoEnable { get { return _cmdQueue.UndoEnable; } }
        /// <summary>
        /// Redo可能か
        /// </summary>
        public bool RedoEnable { get { return _cmdQueue.RedoEnable; } }

#endregion ISubScene メンバ

        //-------------------------------------------------------
#region ISelectedSetChangedEventListener メンバ

        /// <summary>
        /// 選択セット更新イベントハンドラ
        /// シーン更新イベントとして伝播します。
        /// </summary>
        /// <param name="selSet"></param>
        public void OnSelectedSetChangedEventHandler( IPane[] selectedPaneSet )
        {
            // イベントを生成して...
            SceneModifyEventArgs ea
                = new SceneModifyEventArgs(
                    SceneModifyEventArgs.Kind.SelectedSetModify,
                    selectedPaneSet );

            NotifySceneModifyEvent_( ea );
        }

        /// <summary>
        ///
        /// </summary>
        public void OnSelectedSetClickedEventHandler( IPane selSet, PointF posClicked )
        {
            // イベントを生成して...
            SceneModifyEventArgs ea =
                new SceneModifyEventArgs(
                    SceneModifyEventArgs.Kind.SelectedSetClicked,
                    new SceneModifyEventArgs.SelectedPaneClickedEventArgs(selSet, posClicked));

            NotifySceneModifyEvent_(ea);
        }

#endregion

        /// <summary>
        /// テクスチャマネージャ更新ハンドラ
        /// </summary>
        void _textureMgr_OnTextureMgrUpdate()
        {
            // すべての水平ウインドウのサイズを調整する
            foreach (IPane horizontalWndPane in this.IPaneArray.Where(
                (pane) => pane.PaneKind == PaneKind.Window && pane.ILEWindow.WindowKind != WindowKind.Around))
            {
                (horizontalWndPane.ILEWindow as LEWindow).AdjustPaneSize();
            }

            // テクスチャマネージャ更新イベント
            SceneModifyEventArgs ea = new SceneModifyEventArgs( SceneModifyEventArgs.Kind.TextureManager, this );
            NotifySceneModifyEvent_( ea );
        }

        /// <summary>
        /// フォントマネージャ更新ハンドラ
        /// </summary>
        void _fontMgr_OnLEFontMgrUpdate()
        {
            // テクスチャマネージャ更新イベント
            SceneModifyEventArgs ea = new SceneModifyEventArgs( SceneModifyEventArgs.Kind.FontManager, this );
            NotifySceneModifyEvent_( ea );

            PrepareTextboxScFontUpdate();
        }

        /// <summary>
        /// アニメーションフレーム区間タグ更新ハンドラ
        /// </summary>
        void _animFrameSectionSet_OnAnimSectionTagSetUpdate(SceneModifyEventArgs.Kind kind)
        {
            // 分割モードなら、基本値の更新と、アニメーションの再評価を行う。
            if(kind == SceneModifyEventArgs.Kind.AnimSectionTagTarget)
            {
                if (this.IsAnimEditSeparateMode())
                {
                    this.BeginMassiveModify();

                    // タイムラインの時間を0にします
                    GlobalTime.Inst.SetTimeForcibly(0);

                    SubSceneHelper.ResetValueAsBaseValue(this);
                    SubSceneHelper.EvaluateAnimation(this, GlobalTime.Inst.Time);

                    this.EndMassiveModify();
                }
            }

            SceneModifyEventArgs ea = new SceneModifyEventArgs(kind, this);
            NotifySceneModifyEvent_( ea );
        }

        /// <summary>
        /// アニメーション共有情報マネージャ更新ハンドラ
        /// </summary>
        void _animShareInfomationMgr_OnAnimShareManagerChanged()
        {
            // テクスチャマネージャ更新イベント
            SceneModifyEventArgs ea = new SceneModifyEventArgs( SceneModifyEventArgs.Kind.AnimShareInfomarionManager, this );
            NotifySceneModifyEvent_( ea );
        }

        /// <summary>
        /// 部品情報更新ハンドラ
        /// </summary>
        void _partsSettings_OnChanged()
        {
            // 部品情報更新イベント
            SceneModifyEventArgs ea = new SceneModifyEventArgs(SceneModifyEventArgs.Kind.PartsSettings, this);
            NotifySceneModifyEvent_(ea);
        }
    }
}
