﻿using LECore.Structures.Core;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace LECore.Structures
{
    //----------------------------------------------------------
    public interface IStateMachine
    {
        StateMachineRunMode RunMode { get; }
        ISubScene OwnerSubScene { get; }

        bool IsEnabled { get; }
        string Name { get; }
        bool IsSelected { get; }
        IEnumerable<IStateLayer> Layers { get; }

        IEnumerable<IStateMachineVariable> Variables { get; }

        void RunUpdate(float stepTime, FVec2 mousePos, bool isLButton);

        void Draw(IRenderer renderer);
    }

    //----------------------------------------------------------

    public static class IStateMachineHelper
    {
        public static bool NeedToInitialSetup(this IStateMachine stateMachine)
        {
            if (!stateMachine.Layers.Any())
            {
                return true;
            }

            if (!stateMachine.Layers.Any(l => l.GetBaseState() == null))
            {
                return true;
            }

            return false;
        }

        public static IStateLayer GetSelectedIStateLayer(this IStateMachine stateMachine)
        {
            if (!stateMachine.IsEnabled)
            {
                return null;
            }

            var selected = stateMachine.Layers.FirstOrDefault(l => l.IsSelected);
            if (selected != null)
            {
                return selected;
            }

            return stateMachine.Layers.FirstOrDefault();
        }

        // 選択されたプロパティ（ステート、もしくは トランジション）
        public static IStateLayerEditItem GetSelectedEditItem(this IStateMachine stateMachine)
        {
            var layer = GetSelectedIStateLayer(stateMachine);
            if (layer == null)
            {
                return null;
            }

            return layer.GetSelectedEditItem();
        }

        public static IEnumerable<IStateMachine> EnumrateChildStateMachine(this IStateMachine stateMachine)
        {
            ISubScene subScene = stateMachine?.OwnerSubScene;
            if (subScene == null)
            {
                return new IStateMachine[0];
            }

            return subScene.FindPanesByKind(PaneKind.Parts)
                .Where(partsPane =>
                {
                    IStateMachine partsStateMachine = partsPane?.IPartsLayout?.PartsSubScene?.IStateMachine;
                    return partsStateMachine != null && partsStateMachine.IsEnabled;
                }).Select(
                    partsPane => partsPane?.IPartsLayout?.PartsSubScene?.IStateMachine
                );
        }
    }

    /// <summary>
    /// StateLayer が持つ編集プロパティ（≒ State or Transition）
    /// </summary>
    public interface IStateLayerEditItem
    {
        bool IsSelected { get; }
        string EditName { get; }

        IEnumerable<IFeatureParamater> GetFreatureParamaters();
    }

    public static class IStateLayerEditItemHelper
    {
        public static bool IsDeletable(this IStateLayerEditItem item)
        {
            if (item is IState)
            {
                return !(item as IState).IsBaseState();
            }

            return true;
        }

        public static IFeatureParamater FindFreatureParamaterByIndex(this IStateLayerEditItem item, int index)
        {
            // 注意：すこし重い処理が走る（特に Transition の GetFreatureParamaters）ので必要に応じて最適化する。
            return item.GetFreatureParamaters().ElementAt(index);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public static class IStateLayerHelper
    {
        public static IState GetBaseState(this IStateLayer stateLayer)
        {
            return stateLayer.States.FirstOrDefault(s => s.IsBaseState());
        }

        public static IState FindStateByName(this IStateLayer stateLayer, string name)
        {
            return stateLayer.States.FirstOrDefault(s => s.Name == name);
        }

        public static IStateTransition FindTransitionByName(this IStateLayer stateLayer, string startName, string endName)
        {
            return stateLayer.Transitions.FirstOrDefault(t => t.StartStateName == startName && t.EndStateName == endName);
        }

        public static IEnumerable<IFeatureParamater> FindFeatureParamaterByTargetName(this IStateLayer stateLayer, string stateName, string targetName)
        {
            IState state = stateLayer.States.FirstOrDefault(s => s.Name == stateName);
            if (state == null)
            {
                return new List<IFeatureParamater>();
            }

            return state.Paramaters.Where(p => p.Name == targetName);
        }

        public static IEnumerable<IStateLayerEditItem> EnumrateEditItems(this IStateLayer stateLayer)
        {
            if (stateLayer == null)
            {
                yield break;
            }

            foreach (var state in stateLayer.States)
            {
                yield return state;
            }

            foreach (var transition in stateLayer.Transitions)
            {
                yield return transition;
            }
        }

        public static IStateLayerEditItem GetSelectedEditItem(this IStateLayer stateLayer)
        {
            return stateLayer.EnumrateEditItems().FirstOrDefault(s => s.IsSelected);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public static class IStateHelper
    {
        public static bool IsBaseState(this IState state)
        {
            return state.Name == "Base";
        }

        public static bool HasSameParamater(this IState state, string targetName, FeatureParamaterKind kind)
        {
            return FindParamater(state, targetName, kind) != null;
        }

        public static IFeatureParamater FindParamater(this IState state, string targetName, FeatureParamaterKind kind)
        {
            return state.Paramaters.FirstOrDefault(param => param.Name == targetName && param.Kind == kind);
        }

        internal static void CloneAllParamaters(this StateData state, IState source)
        {
            state.ParamaterList.Clear();

            foreach (var sourceParam in source.Paramaters)
            {
                var newParam = new FeatureParamaterData();
                newParam.SetFeatureParamaterValueByOther(sourceParam);
                state.AddNewFeature(newParam);
            }
        }

        internal static void AddNewFeature(this StateData stateData, FeatureParamaterData newParam)
        {
            stateData.ParamaterList.Add(newParam);
            stateData.ParamaterList.Sort(CompareFeatureParamater_);
        }

        private static int CompareFeatureParamater_(IFeatureParamater p1, IFeatureParamater p2)
        {
            int result = p1.Name.CompareTo(p2.Name);
            if (result != 0)
            {
                return result;
            }

            result = p1.Kind.CompareTo(p2.Kind);

            return result;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public static class IStatePropertyHelper
    {
        public static void SetStateEditItemSelectFlag(this IStateLayerEditItem prop, bool flag)
        {
            if (prop is StateData)
            {
                (prop as StateData).IsSelected = flag;
            }
            else if (prop is StateTransitionData)
            {
                (prop as StateTransitionData).IsSelected = flag;
            }
            else
            {
                Debug.Assert(false);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal static class FeatureParamaterHelper
    {
        /// <summary>
        /// ISubScene ⇒ FeatureParamaterData
        /// </summary>
        internal static void StoreFeatureParamaterValueFromScene(this FeatureParamaterData param, ISubScene subScene, string targetName, FeatureParamaterKind kind)
        {
            param.Name = targetName;
            param.Kind = kind;

            var targetPane = subScene.FindPaneByName(targetName);
            if (targetPane == null)
            {
                Debug.Assert(false);
                return;
            }

            switch (kind)
            {
                case FeatureParamaterKind.Position:
                    param.X = targetPane.X;
                    param.Y = targetPane.Y;
                    param.Z = targetPane.Z;
                    param.W = 0.0f;
                    break;
                case FeatureParamaterKind.Scale:
                    param.X = targetPane.Scale.X;
                    param.Y = targetPane.Scale.Y;
                    param.Z = targetPane.Scale.Z;
                    param.W = 0.0f;
                    break;
                case FeatureParamaterKind.RotateZ:
                    param.X = targetPane.RotAng.X;
                    param.Y = targetPane.RotAng.Y;
                    param.Z = targetPane.RotAng.Z;
                    param.W = 0.0f;
                    break;
                case FeatureParamaterKind.WhiteColor:
                    // TODO:Float対応
                    var colW = targetPane.IMaterial[0].WhiteColor.ToRGBAColor();
                    param.X = colW.R;
                    param.Y = colW.G;
                    param.Z = colW.B;
                    param.W = colW.A;
                    break;
                case FeatureParamaterKind.BlackColor:
                    var colB = targetPane.IMaterial[0].WhiteColor.ToRGBAColor();
                    param.X = colB.R;
                    param.Y = colB.G;
                    param.Z = colB.B;
                    param.W = colB.A;
                    break;
            }
        }

        internal static void ApplayFeatureParamaterToPane(this IFeatureParamater paramater, Pane targetPane)
        {
            Debug.Assert(targetPane != null);
            Debug.Assert(paramater != null);

            switch (paramater.Kind)
            {
                case FeatureParamaterKind.Position:
                    targetPane.Trans = new FVec3(paramater.X, paramater.Y, paramater.Z); break;
                case FeatureParamaterKind.Scale:
                    targetPane.Scale = new FVec3(paramater.X, paramater.Y, paramater.Z); break;
                case FeatureParamaterKind.RotateZ:
                    targetPane.RotAng = new FVec3(paramater.X, paramater.Y, paramater.Z); break;
                case FeatureParamaterKind.BlackColor:
                    targetPane.Material[0].BlackColor = new FloatColor(new RGBAColor((byte)paramater.X, (byte)paramater.Y, (byte)paramater.Z, (byte)paramater.W)); break;
                case FeatureParamaterKind.WhiteColor:
                    targetPane.Material[0].WhiteColor = new FloatColor(new RGBAColor((byte)paramater.X, (byte)paramater.Y, (byte)paramater.Z, (byte)paramater.W)); break;
            }
        }

        internal static void SetFeatureParamaterValueByOther(this FeatureParamaterData param, IFeatureParamater source)
        {
            param.Name = source.Name;
            param.Kind = source.Kind;
            param.X = source.X;
            param.Y = source.Y;
            param.Z = source.Z;
            param.W = source.W;
        }
    }

    //----------------------------------------------------------

    public interface IStateMachineEvent
    {
        StateMachineEventKind kind { get; }
        object param1 { get; }
        object param2 { get; }
        uint delayFrames { get; }
    };

    internal class StateMachineEvent : IStateMachineEvent
    {
        public StateMachineEventKind kind { get; set; }
        public object param1 { get; set; }
        public object param2 { get; set; }
        public uint delayFrames { get; set; }
    };

    internal static class StateMachineEventHelper
    {
        public static bool IsSame(this IStateMachineEvent stateMachineEvent, StateMachineEventKind kind, object param1, object param2, uint delayFrames)
        {
            return
                stateMachineEvent.kind == kind &&
                stateMachineEvent.param1 == param1 &&
                stateMachineEvent.param2 == param2 &&
                stateMachineEvent.delayFrames == delayFrames;
        }

        internal static void Set(this StateMachineEvent stateMachineEvent, StateMachineEventKind kind, object param1, object param2, uint delayFrames)
        {
            stateMachineEvent.kind = kind;
            stateMachineEvent.param1 = param1;
            stateMachineEvent.param2 = param2;
            stateMachineEvent.delayFrames = delayFrames;
        }
    }

    internal class EventQueue
    {
        List<StateMachineEvent> _eventQueue { get; set; } = new List<StateMachineEvent>();

        public EventQueue()
        {
        }

        //----------------------------------------------------------
        public void Initialize()
        {
            Reset();
        }

        //----------------------------------------------------------
        public void Reset()
        {
            _eventQueue.Clear();
        }

        //----------------------------------------------------------
        public void Push(StateMachineEventKind kind, object param1, object param2, uint delayFrames)
        {
            // フリーリストから取って、後ろに追加
            StateMachineEvent newItem = new StateMachineEvent();

            newItem.kind = kind;
            newItem.param1 = param1;
            newItem.param2 = param2;
            newItem.delayFrames = delayFrames;

            _eventQueue.Add(newItem);
        }

        //----------------------------------------------------------
        public StateMachineEvent Peek()
        {
            return _eventQueue.FirstOrDefault();
        }

        //----------------------------------------------------------
        public StateMachineEvent Bottom()
        {
            return _eventQueue.LastOrDefault();
        }

        //----------------------------------------------------------
        public void Pop()
        {
            if (!_eventQueue.Any())
            {
                return;
            }

            _eventQueue = new List<StateMachineEvent>(_eventQueue.Skip(1));
        }

        //----------------------------------------------------------
        public void MoveTopToTail()
        {
            StateMachineEvent head = Peek();
            if (head != null)
            {
                this.Pop();
                _eventQueue.Add(head);
            }
        }
    }

    internal static class StateMachineDataHelper
    {
        internal static void Set(this StateMachineData target, IStateMachine source)
        {
            target.IsEnabled = source.IsEnabled;
            target.Name = source.Name;

            // Layer
            foreach (IStateLayer srcLayer in source.Layers)
            {
                target.AddLayer(srcLayer.Name);

                IState srcBaseState = srcLayer.States.First();
                foreach (var srcParam in srcBaseState.Paramaters)
                {
                    target.AddNewFeatureParamaterToLayer(srcLayer.Name, srcParam.Name, srcParam.Kind);
                }

                // State
                foreach (var srcState in srcLayer.States)
                {
                    target.AddStateToLayer(srcLayer.Name, srcState.Name);

                    target.SetState(srcLayer.Name, srcState.Name, (s) =>
                    {
                        // 今のところ設定するものなし
                        return true;
                    });

                    // state - featureParamaters
                    if (srcState.Paramaters != null)
                    {
                        foreach (var srcStateParam in srcState.Paramaters)
                        {
                            target.SetStateFeatureParamater(srcLayer.Name, srcState.Name, srcStateParam.Name, srcStateParam.Kind, (stateParam) =>
                            {
                                stateParam.X = srcStateParam.X;
                                stateParam.Y = srcStateParam.Y;
                                stateParam.Z = srcStateParam.Z;
                                stateParam.W = srcStateParam.W;
                                return true;
                            });
                        }
                    }
                }

                // IStateTransition
                foreach (IStateTransition srcTransition in srcLayer.Transitions)
                {
                    target.AddTransitionToLayer(srcLayer.Name, srcTransition.StartStateName, srcTransition.EndStateName);

                    target.SetTransition(srcLayer.Name, srcTransition.StartStateName, srcTransition.EndStateName, (t) =>
                    {
                        t.Offset = srcTransition.Offset;
                        t.TotalDuration = srcTransition.TotalDuration;

                        t.StartStateName = srcTransition.StartStateName;
                        t.EndStateName = srcTransition.EndStateName;

                        t.IsLoop = srcTransition.IsLoop;
                        t.IsCancelable = srcTransition.IsCancelable;
                        (t.Trigger as StateTransitionTriggerData).StateMachineOperator = srcTransition.Trigger.StateMachineOperator;
                        (t.Trigger as StateTransitionTriggerData).TriggerKind = srcTransition.Trigger.TriggerKind;
                        (t.Trigger as StateTransitionTriggerData).StateMachineVariableList.AddRange(srcTransition.Trigger.StateMachineVariables);

                        foreach (IFeatureParamater srcTrackParam in srcTransition.GetFreatureParamaters())
                        {
                            var srcTrack = srcTransition.FindIStateTransitionTrackDataByName(srcTrackParam.Name, srcTrackParam.Kind);

                            t.SetStateTransitionTrackDataByName(
                                srcTrackParam.Name, srcTrackParam.Kind,
                                srcTrack.Offset, srcTrack.Duration,
                                srcTrack.StateEasing);

                            foreach (var key in srcTrack.KeyParamaters)
                            {
                                if(key.FeatureParamaterData.Kind != FeatureParamaterKind.StateMachineEvent)
                                {
                                    t.SetStateTransitionTrackKey(
                                        srcTrackParam.Name, srcTrackParam.Kind,
                                        key.Time,
                                        key.StateEasing,
                                        key.FeatureParamaterData.X, key.FeatureParamaterData.Y, key.FeatureParamaterData.Z, key.FeatureParamaterData.W);
                                }
                                else
                                {
                                    var smd = key.StateMachineEvent;
                                    t.SetStateTransitionTrackEvent(
                                        srcTrackParam.Name,
                                        key.Time,
                                        smd.kind, smd.param1, smd.param2, smd.delayFrames);

                                }
                            }
                        }

                        return true;
                    });
                }
                // TODO : target.Variables
            }

            target.SetRunModeInitialState();
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal class StateMachineData : IStateMachine
    {
        Font _fontForDebug = new Font("Tahoma", 8.25F, FontStyle.Regular);

        bool _IsEnabled;
        ISubScene _owner;
        StateMachineRunMode _runMode = StateMachineRunMode.Edit;
        EventQueue _eventQueue = new EventQueue();
        bool _isLastFrameLButtonClicked = false;

        //----------------------------------------------------------

        public StateMachineRunMode RunMode
        {
            get { return _runMode; }
            set { _runMode = value; }
        }

        public ISubScene OwnerSubScene
        {
            get { Debug.Assert(_owner != null); return _owner; }
        }

        public bool IsEnabled
        {
            get { return _IsEnabled; }
            set
            {
                _IsEnabled = value;
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            }
        }

        public Action<SceneModifyEventArgs.Kind> OnChanged;

        public string Name { get; set; } = string.Empty;

        public bool IsSelected { get; set; }

        public IEnumerable<IStateLayer> Layers { get; } = new List<IStateLayer>();

        public IEnumerable<IStateMachineVariable> Variables { get; } = new List<IStateMachineVariable>();

        //----------------------------------------------------------

        public StateMachineData(ISubScene owner)
        {
            Debug.Assert(owner != null);
            _owner = owner;
        }

        //----------------------------------------------------------
        void UpdateStateGroupTransitions_(StateLayerData stateLayer, StateMachineEvent eventMsg)
        {
            if (!stateLayer.IsCurrentTransitionCancelable)
            {
                if (!stateLayer.IsStateTransitionCompleted)
                {
                    return;
                }
            }

            if (stateLayer.RunModeCurrentState == null)
            {
                Debug.Assert(false);
                return;
            }

            foreach (IStateTransition transition in stateLayer.Transitions)
            {
                // 現在のステートを対象としているトリガーかどうか？
                if (transition.StartStateName != stateLayer.RunModeCurrentState.Name)
                {
                    continue;
                }

                // 状態遷移するか調査
                if ((transition.Trigger as StateTransitionTriggerData).IsTriggered(eventMsg))
                {
                    // 状態を遷移する
                    stateLayer.ChangeState(transition);
                    break;
                }
            }
        }

        //----------------------------------------------------------
        // StateGroup の更新（StateGroupのメソッドに引っ越します）
        void UpdateStateGroup_(StateLayerData stateLayer)
        {
            // 更新します
            stateLayer.RunUpdate();

            // ステート完了イベントの送信
            if (stateLayer.IsStateTransitionJustCompleted)
            {
                _eventQueue.Push(StateMachineEventKind.StateTransitionCompleted, stateLayer.RunModeNextStateName, 0, 0);
            }
        }

        public void SetRunModeInitialState()
        {
            foreach (StateLayerData stateLayer in this.Layers)
            {
                stateLayer.SetRunModeInitialState();
                ApplayStateToCurrentSubScene_(stateLayer.GetBaseState().GetFreatureParamaters());
            }

            _eventQueue.Reset();
        }

        /// <summary>
        /// 子供を探す。子供のキューにPushする。
        /// </summary>
        /// <param name="trackEvent"></param>
        private void PostEventToChild_(KeyParamater trackEvent)
        {
            IStateMachineEvent stateEvent = trackEvent.StateMachineEvent;

            string targetChildState = stateEvent.param1 as string;
            IPane pane =  this._owner.FindPaneByName(targetChildState);
            StateMachineData childStateMachine = pane?.IPartsLayout?.PartsSubScene?.IStateMachine as StateMachineData;

            if (childStateMachine == null || !childStateMachine.IsEnabled)
            {
                Debug.Assert(false);
                return;
            }

            childStateMachine.PostStateMachineEvent(stateEvent);
        }

        private void PostStateMachineEvent(IStateMachineEvent stateMachineEvent)
        {
            _eventQueue.Push(stateMachineEvent.kind, stateMachineEvent.param1, stateMachineEvent.param2, stateMachineEvent.delayFrames);
        }

        internal void PostTrackEvent(StateTransitionTrackData sender, IEnumerable<KeyParamater> trackEvents)
        {
            foreach(KeyParamater trackEvent in trackEvents)
            {
                PostEventToChild_(trackEvent);
            }
        }

        public void Draw(IRenderer renderer)
        {
            if(this.RunMode != StateMachineRunMode.Run)
            {
                return;
            }

            int layerIndex = 0;
            foreach (StateLayerData stateLayer in this.Layers)
            {
                float baseY = 0 + _fontForDebug.Height * layerIndex;
                RendererHelper.DrawStringWithoutTransform(
                       renderer,
                       _fontForDebug,
                       Color.Blue,
                       string.Format("[{0}: {1}]", stateLayer.Name, stateLayer.RunModeCurrentState.Name), 0, baseY);

                if (stateLayer.RunModeActiveTransition != null)
                {
                    const float BarWidth = 200.0f;
                    float normalizeValue = stateLayer.RunModeActiveTransition.LocalTime.Time / stateLayer.RunModeActiveTransition.TotalDuration;

                    renderer.Color = Color.Red;
                    renderer.FillRectangle(0, baseY, 0, BarWidth, 4);
                    renderer.Color = Color.GreenYellow;
                    renderer.FillRectangle(0, baseY, 0, normalizeValue * BarWidth, 4);
                }

                layerIndex++;
            }
        }

        public void RunUpdate(float stepTime, FVec2 mousePos, bool isLButtonClick)
        {
            // 更新
            foreach (StateLayerData stateLayer in this.Layers)
            {
                UpdateStateGroup_(stateLayer);
            }

            // Tick イベントを積む
            _eventQueue.Push(StateMachineEventKind.Tick, null, null, 0);

            // Lトリガー時に決定イベントを発行
            if (isLButtonClick && _isLastFrameLButtonClicked != true)
            {
                _eventQueue.Push(StateMachineEventKind.Decided, null, null, 0);
            }

            _isLastFrameLButtonClicked = isLButtonClick;

            // メッセージキューを舐め、StateGroup にメッセージを振り分ける。。。
            StateMachineEvent eventMsg = _eventQueue.Peek();
            StateMachineEvent endEventMsg = _eventQueue.Bottom();

            while (eventMsg != null)
            {
                // イベントのディレイフレーム対応
                if (eventMsg.delayFrames > 0)
                {
                    // ディレイフレームが完了するまで、処理しません。
                    eventMsg.delayFrames--;
                    _eventQueue.MoveTopToTail();
                }
                else
                {
                    foreach (StateLayerData stateLayer in this.Layers)
                    {
                        UpdateStateGroupTransitions_(stateLayer, eventMsg);
                    }

                    _eventQueue.Pop();
                }

                // 終端まで走査したら抜ける
                if (eventMsg == endEventMsg)
                {
                    break;
                }

                eventMsg = _eventQueue.Peek();
            }

            foreach(IStateMachine child in IStateMachineHelper.EnumrateChildStateMachine(this))
            {
                child.RunUpdate(stepTime, mousePos, isLButtonClick);
            }
        }

        void RaiseModifyEvent(SceneModifyEventArgs.Kind kind)
        {
            if (OnChanged != null) { OnChanged(kind); }
        }

        public StateLayerData AddLayer(string name)
        {
            StateLayerData layer = new StateLayerData();
            layer.Name = name;

            (Layers as List<IStateLayer>).Add(layer);

            this.AddStateToLayer(layer.Name, "Base");

            RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);

            return layer;
        }

        public void RemoveLayerByName(string name)
        {
            var removed = Layers.FirstOrDefault(l => l.Name == name);
            if (removed != null)
            {
                (Layers as List<IStateLayer>).Remove(removed);
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            }
        }

        public StateData AddStateToLayer(string layerName, string name)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return null;
            }

            if (targetLayer.States.Any(s => s.Name == name))
            {
                return null;
            }

            var state = new StateData();
            state.Name = name;

            (targetLayer.States as List<IState>).Add(state);

            // パラメーターの同期
            var baseState = targetLayer.GetBaseState();
            if (baseState != null)
            {
                foreach (var param in baseState.Paramaters)
                {
                    AddNewFeatureToAllStates_(targetLayer, param.Name, param.Kind);
                }
            } else
            {
                Debug.Assert(!targetLayer.States.Any());
            }

            RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);

            return state;
        }

        public void RemoveStateFromLayer(string layerName, string name)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                Debug.Assert(false);
                return;
            }

            var state = targetLayer.FindStateByName(name);
            if (state == null)
            {
                Debug.Assert(false);
                return;
            }

            (targetLayer.States as List<IState>).Remove(state);

            // 関連する Transition の削除をします。
            {
                var relatedTransitions = targetLayer.Transitions.Where(t => t.StartStateName == name || t.EndStateName == name).ToArray();
                foreach (var relatedTransition in relatedTransitions)
                {
                    RemoveTransitionFromLayer(layerName, relatedTransition.StartStateName, relatedTransition.EndStateName);
                }
            }

            RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
        }

        /// <summary>
        /// 指定レイヤー以下に、 FeatureParamater を追加する。
        /// </summary>
        private bool AddNewFeatureToAllStates_(IStateLayer targetLayer, string targetName, FeatureParamaterKind kind)
        {
            bool isModified = false;
            // 全てのステートについて足す。
            // 値はすべてベースと同じ値に設定をする。
            foreach (StateData state in targetLayer.States)
            {
                if (state.HasSameParamater(targetName, kind))
                {
                    continue;
                }

                var newParam = new FeatureParamaterData();
                newParam.StoreFeatureParamaterValueFromScene(_owner, targetName, kind);
                state.AddNewFeature(newParam);

                isModified = true;
            }

            return isModified;
        }

        /// <summary>
        /// 指定レイヤー以下から、 FeatureParamater を削除する。
        /// </summary>
        private bool RemoveNewFeatureFromAllStates_(IStateLayer targetLayer, string targetName, FeatureParamaterKind kind)
        {
            bool isModified = false;

            foreach (StateData state in targetLayer.States)
            {
                int oldCount = state.ParamaterList.Count;
                state.ParamaterList.RemoveAll(p => p.Name == targetName && p.Kind == kind);

                isModified = oldCount != state.ParamaterList.Count;
            }

            return isModified;
        }

        public void AddNewFeatureParamaterToLayer(string layerName, string targetName, FeatureParamaterKind kind)
        {
            if (kind == FeatureParamaterKind.None)
            {
                return;
            }

            // TODO : 異なるLayer間でコンフリクトしていないか
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return;
            }

            bool isModified = AddNewFeatureToAllStates_(targetLayer, targetName, kind);
            if (isModified)
            {
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            }
        }

        public void RemoveFeatureParamaterToLayer(string layerName, string targetName, FeatureParamaterKind kind)
        {
            if (kind == FeatureParamaterKind.None)
            {
                return;
            }

            // TODO : 異なるLayer間でコンフリクトしていないか
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return;
            }

            // 全てのステートについて足す。
            // 値はすべてベースと同じ値に設定をする。
            bool isModified = RemoveNewFeatureFromAllStates_(targetLayer, targetName, kind);
            if (isModified)
            {
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            }
        }

        public StateTransitionData AddTransitionToLayer(string layerName, string nameStart, string nameEnd)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return null;
            }

            if (targetLayer.Transitions.Any(t => t.StartStateName == nameStart && t.EndStateName == nameEnd))
            {
                return null;
            }

            var transition = new StateTransitionData(this, (t) => ApplayStateToCurrentSubScene_(t.GetFreatureParamaters()));
            transition.StartStateName = nameStart;
            transition.EndStateName = nameEnd;
            transition.TotalDuration = 60;
            (targetLayer.Transitions as List<IStateTransition>).Add(transition);

            RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            return transition;
        }

        public void RemoveTransitionFromLayer(string layerName, string nameStart, string nameEnd)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return;
            }

            if (!targetLayer.Transitions.Any(t => t.StartStateName == nameStart && t.EndStateName == nameEnd))
            {
                return;
            }

            (targetLayer.Transitions as List<IStateTransition>).RemoveAll(t => t.StartStateName == nameStart && t.EndStateName == nameEnd);

            RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
        }

        public void SetLayer(string layerName, Func<StateLayerData, bool> modifyAction)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return;
            }

            bool isModified = modifyAction(targetLayer as StateLayerData);
            if (isModified)
            {
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            }
        }

        public void SetState(string layerName, string stateName, Func<StateData, bool> modifyAction)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return;
            }

            var state = targetLayer.States.FirstOrDefault(s => s.Name == stateName);
            if (state == null)
            {
                return;
            }

            bool isModified = modifyAction(state as StateData);
            if (isModified)
            {
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            }
        }

        public void SetTransition(string layerName, string nameStart, string nameEnd, Func<StateTransitionData, bool> modifyAction)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return;
            }

            var transition = targetLayer.Transitions.FirstOrDefault(t => t.StartStateName == nameStart && t.EndStateName == nameEnd);
            if (transition == null)
            {
                return;
            }

            bool isModified = modifyAction(transition as StateTransitionData);
            if (isModified)
            {
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            }
        }

        public void SetStateFeatureParamater(string layerName, string stateName, string targetName, FeatureParamaterKind kind, Func<FeatureParamaterData, bool> modifyAction)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return;
            }

            var state = targetLayer.FindStateByName(stateName);
            if (state == null)
            {
                return;
            }

            var param = state.FindParamater(targetName, kind);
            if (param == null)
            {
                return;
            }

            SetFeatureParamater(param, modifyAction);
        }

        public void SetFeatureParamater(IFeatureParamater paramater, Func<FeatureParamaterData, bool> modifyAction)
        {
            bool isModified = modifyAction(paramater as FeatureParamaterData);
            if (isModified)
            {
                ApplayFeatureParamaterToPane_(paramater);
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            }
        }

        public void SetProperty(string layerName, string editName, Func<IStateLayerEditItem, bool> modifyAction)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return;
            }

            var editItem = targetLayer.EnumrateEditItems().FirstOrDefault(sp => sp.EditName == editName);
            if (editItem == null)
            {
                return;
            }

            bool isModified = modifyAction(editItem as IStateLayerEditItem);
            if (isModified)
            {
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineModify);
            }
        }

        //----------------------------------------------------------
        // 選択変更

        public void SetPropertySelection(string layerName, string editName, bool selectFlag)
        {
            var targetLayer = Layers.FirstOrDefault(l => l.Name == layerName);
            if (targetLayer == null)
            {
                return;
            }

            var editItem = targetLayer.EnumrateEditItems().FirstOrDefault(sp => sp.EditName == editName);
            if (editItem == null)
            {
                return;
            }

            if (editItem.IsSelected != selectFlag)
            {
                editItem.SetStateEditItemSelectFlag(selectFlag);
                if (selectFlag)
                {
                    ApplayStateToCurrentSubScene_(editItem.GetFreatureParamaters());
                }
                RaiseModifyEvent(SceneModifyEventArgs.Kind.StateMachineSelectionChanged);
            }
        }

        //----------------------------------------------------------
        // シーンに反映する処理

        void ApplayStateToCurrentSubScene_(IEnumerable<IFeatureParamater> paramaters)
        {
            _owner.BeginMassiveModify();
            foreach (var paramater in paramaters)
            {
                var targetPane = _owner.FindPaneByName(paramater.Name);
                if (targetPane == null)
                {
                    continue;
                }

                paramater.ApplayFeatureParamaterToPane(targetPane as Pane);
            }
            _owner.EndMassiveModify();
        }

        void ApplayFeatureParamaterToPane_(IFeatureParamater paramater)
        {
            var targetPane = _owner.FindPaneByName(paramater.Name) as Pane;
            if (targetPane == null)
            {
                return;
            }
            paramater.ApplayFeatureParamaterToPane(targetPane);
        }
    }

    /// <summary>
    ///
    /// </summary>
    //----------------------------------------------------------
    public interface IStateLayer
    {
        string Name { get; }

        bool IsSelected { get; }

        IEnumerable<IState> States { get; }

        IEnumerable<IStateTransition> Transitions { get; }
    }

    /// <summary>
    ///
    /// </summary>
    //----------------------------------------------------------
    internal class StateLayerData : IStateLayer
    {
        public string Name { get; set; } = string.Empty;

        public bool IsSelected { get; set; }

        public IEnumerable<IState> States { get; } = new List<IState>();

        public IEnumerable<IStateTransition> Transitions { get; } = new List<IStateTransition>();

        //----------------------------------------------------------
        // Run Mode
        internal IState RunModeCurrentState { get; private set; }

        internal IStateTransition RunModeActiveTransition { get { return _activeTransition; } }

        public string RunModeNextStateName
        {
            get { return _activeTransition != null ? _activeTransition.EndStateName : string.Empty; }
        }

        public void ChangeState(IStateTransition nextTransition)
        {
            // ステート変更を行う
            IState nextState = this.FindStateByName(nextTransition.EndStateName);
            Debug.Assert(nextState != null);
            RunModeCurrentState = nextState;

            ResetStateFlags_();
            IsCurrentTransitionCancelable = nextTransition.IsCancelable;

            // 動的に構築する
            {
                IStateTransition prevTransition = _activeTransition;
                float prevAnimFrame = _activeTransition != null ? _activeTransition.LocalTime.Time : 0.0f;

                // 開始値を算出する
                // Transition を作る
                // 開始値を設定する
                // ローカル時間を適切にリセット
                //
                SetupTransition_(nextTransition, prevTransition, prevAnimFrame);
            }
        }

        void SetupTransition_(IStateTransition nextTransition, IStateTransition prevTransition, float prevAnimFrame)
        {
            _activeTransition = nextTransition as StateTransitionData;
            (_activeTransition.LocalTime as StateTransitionLocalTime).Time = 0;
        }

        public void SetRunModeInitialState()
        {
            this.RunModeCurrentState = States.FirstOrDefault(s => s.IsBaseState());
            this._activeTransition = null;

            ResetStateFlags_();

            Debug.Assert(RunModeCurrentState != null);
        }

        public void RunUpdate()
        {
            if (_activeTransition != null)
            {
                // ローカル時間の更新
                _activeTransition.RunUpdate();
            }

            // TransitionCompleted 関連
            {
                bool current = IsStateTransitionCompleted;
                IsStateTransitionJustCompleted = LastIsStateTransitionCompleted != current;
                LastIsStateTransitionCompleted = current;
            }
        }

        void ResetStateFlags_()
        {
            IsStateTransitionJustCompleted = false;
            LastIsStateTransitionCompleted = false;
        }

        private StateTransitionData _activeTransition { get; set; }

        public bool IsCurrentTransitionCancelable { get; private set; }

        public bool LastIsStateTransitionCompleted { get; private set; }

        public bool IsStateTransitionCompleted
        {
            get
            {
                if (_activeTransition == null)
                {
                    return true;
                }

                //if (!_activeTransitionIsEnabled())
                //{
                //    return true;
                //}

                if (_activeTransition.LocalTime.Time >= _activeTransition.Offset + _activeTransition.TotalDuration)
                {
                    return true;
                }

                return false;
            }
        }

        public bool IsStateTransitionJustCompleted { get; private set; }
    }

    /// <summary>
    ///
    /// </summary>
    public class StateTransitionLocalTime : ITime
    {
        IStateTransition _owner;
        int _time = 0;

        public StateTransitionLocalTime(IStateTransition owner)
        {
            _owner = owner;
        }

        public int AnimPlayEndTime
        {
            get { return (int)_owner.TotalDuration; }
        }

        public int AnimPlayStartTime
        {
            get { return 0; }
        }

        public bool IsPlaying
        {
            get { return GlobalTime.Inst.IsPlaying; }
            set
            {
                if (GlobalTime.Inst.IsPlaying != value)
                {
                    GlobalTime.Inst.IsPlaying = value;
                    GlobalTime.Inst.SettingTimeForcibly = value;
                }
            }
        }

        public int Time
        {
            get
            {
                return _time;
            }

            set
            {
                if (_time != value)
                {
                    _time = value;
                    (_owner as StateTransitionData).RaiseOnTimeLineUpdatedEvent();
                    PostTickEvent();
                }
            }
        }

        public void PostTickEvent()
        {
            GlobalTime.Inst.PostTickEvent();
        }

        public void Play()
        {
            this.IsPlaying = true;
            PostTickEvent();
        }

        public void Stop()
        {
            this.IsPlaying = false;
            (_owner as StateTransitionData).RaiseOnTimeLineUpdatedEvent();
            PostTickEvent();
        }
    }

    public interface IStateEasing
    {
        EasingType EasingType { get; }

        float EasingExtraParamater { get; }
    }

    static public class IStateEasingHelper
    {
        static public bool IsSame(this IStateEasing e1, IStateEasing e2)
        {
            return e1.EasingType == e2.EasingType && e1.EasingExtraParamater == e2.EasingExtraParamater;
        }

        static public bool IsSame(this IStateEasing e1, EasingType type, float paramater)
        {
            return e1.EasingType == type && e1.EasingExtraParamater == paramater;
        }

        static public IStateEasing Create(EasingType type, float easingExtraParamater)
        {
            return new StateEasing { EasingType = type, EasingExtraParamater = easingExtraParamater };
        }

        static public IStateEasing Empty { get; } = new StateEasing();
    }

    internal class StateEasing : IStateEasing
    {
        public EasingType EasingType { get; set; }

        public float EasingExtraParamater { get; set; }
    }

    /// <summary>
    ///
    /// </summary>
    //----------------------------------------------------------
    public interface IStateTransition : IStateLayerEditItem
    {
        ITime LocalTime { get; }

        float Offset { get; }

        float TotalDuration { get; }

        bool IsCancelable { get; }

        IStateEasing StateEasing { get; }

        bool IsLoop { get; }

        string StartStateName { get; }

        string EndStateName { get; }

        IStateTransitionTrigger Trigger { get; }

        IStateTransitionTrack FindIStateTransitionTrackDataByName(string name, FeatureParamaterKind kind);
    }

    /// <summary>
    ///
    /// </summary>
    public class KeyParamater : ITimeEditableItems
    {
        public float Time { get; set; } = 0.0f;
        public IFeatureParamater FeatureParamaterData { get; } = new FeatureParamaterData();


        public IStateMachineEvent StateMachineEvent { get; } = new StateMachineEvent();

        public IStateEasing StateEasing { get; } = new StateEasing();


        public KeyParamater(float time, IStateEasing stateEasing)
        {
            this.Time = time;
            (this.StateEasing as StateEasing).EasingType = stateEasing.EasingType;
            (this.StateEasing as StateEasing).EasingExtraParamater = stateEasing.EasingExtraParamater;
        }

        public float EditTime
        {
            get { return this.Time; }
            set { this.Time = value; }
        }

        public float EditDuraiton
        {
            get { return 1.0f; }
            set { }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public static class KeyParamaterHelper
    {
        internal static bool IsSame(this KeyParamater key, KeyParamater other)
        {
            var od = other.FeatureParamaterData;
            return
                key.Time == other.Time &&
                key.StateEasing.IsSame(other.StateEasing) &&
                key.FeatureParamaterData.IsSame(od.X, od.Y, od.Z, od.W);
        }
    }

    /// <summary>
    ///
    /// </summary>
    public interface IStateTransitionTrack
    {
        float Offset { get; }

        float Duration { get; }

        IStateEasing StateEasing { get; }

        // 区間外リピートの設定はここに入るはず。

        // 親の変更に追従してスケーリングするかどうか（自動スケーリングでは、値を比率で持つようにする。そうすることで自動追従ができる。）

        // Track上の任意位置にイベント発火を設定できるようにする。（ランタイムでは、Track はインデックスだけ持ち、実体は ステートマシンで保持するようにする）
        // 最初は固定数（16bit * 4個）にする

        // キーフレームアニメーションへのフォールバックは、この単位で行われるはず。
        // ランタイムは、名前さえ紐づければ何とでもなりそう。
        // ツール側は、ペイン複製の実体をステート側で持ち、そこにキーを打つようにする。
        // 編集にはインプレイスで表示される、特注のカーブエディタを作る。

        // 開始の値、終了の値、はそれぞれ別に必要になるはず。そうしないと任意の位置アニメーションを簡単に表現できない。
        // これができたら、ほとんどキーフレームアニメーションと同じような自由度でアニメーションが作れるはず。
        // 中間経由値 を 設定できるようにする（イベント発火と同じように ４個固定として、実体は別で管理する）

        IEnumerable<KeyParamater> KeyParamaters { get; }
    }

    /// <summary>
    ///
    /// </summary>
    internal class StateTransitionTrackData : IStateTransitionTrack, ITimeEditableItems
    {
        public float Offset { get; set; }

        public float Duration { get; set; }

        public IStateEasing StateEasing { get; } = new StateEasing();

        public IEnumerable<KeyParamater> KeyParamaters { get { return KeyParamaterList; } }

        internal List<KeyParamater> KeyParamaterList { get; } = new List<KeyParamater>();

        public float EditTime
        {
            get { return this.Offset; }
            set { this.Offset = value; }
        }

        public float EditDuraiton
        {
            get { return this.Duration; }
            set { this.Duration = value; }
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal static class IStateTransitionTrackHelper
    {
        internal static bool IsSame(this IStateTransitionTrack track, float offset, float duration, IStateEasing stateEasing)
        {
            return IsSame(track, offset, duration, stateEasing.EasingType, stateEasing.EasingExtraParamater);
        }

        internal static bool IsSame(this IStateTransitionTrack track, float offset, float duration, EasingType type, float easingExtraParamater)
        {
            return
                track.Offset == offset &&
                track.Duration == duration &&
                track.StateEasing.IsSame(type, easingExtraParamater);
        }

        internal static bool HasSameKeies(this IStateTransitionTrack track, IEnumerable<KeyParamater> keies)
        {
            var td = track as StateTransitionTrackData;

            int keyCount = td.KeyParamaterList.Count;
            if (keyCount != keies.Count())
            {
                return false;
            }

            int index = 0;
            foreach(var key in keies)
            {
                if(!td.KeyParamaterList[index].IsSame(key))
                {
                    return false;
                }
            }

            return true;
        }

        internal static IEnumerable<KeyParamater> GatherEventByRange(this IStateTransitionTrack track, float offset, float duration)
        {
            // TODO : track から Kind を取りたい！
            return  track.KeyParamaters.Where(e =>
            (e.FeatureParamaterData.Kind == FeatureParamaterKind.StateMachineEvent) &&
            e.Time >= offset && e.Time < (offset + duration));
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal interface ITimeEditableItems
    {
        float EditTime { get; set; }
        float EditDuraiton { get; set; }
    }

    /// <summary>
    ///
    /// </summary>
    public static class TimeEditableItemsHelper
    {
        internal static void Scale(IEnumerable<ITimeEditableItems> items, float origin, float offset, float scale, bool notScaleDuration)
        {
            foreach (var item in items)
            {
                // 時間
                item.EditTime = (item.EditTime - origin) * scale + origin;

                // 期間
                if(!notScaleDuration)
                {
                    item.EditDuraiton = item.EditDuraiton * scale;
                }

                // オフセット加算
                item.EditTime = item.EditTime + offset;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal class StateTransitionData : IStateTransition
    {
        readonly IStateMachine _owner;
        readonly Dictionary<string, StateTransitionTrackData> _tracks = new Dictionary<string, StateTransitionTrackData>();

        Action<IStateTransition> _OnTimeLineUpdated { get; set; }

        //----------------------------------------------------------

        public ITime LocalTime { get; private set; }

        public string EditName { get { return string.Format("{0}_to_{1}", this.StartStateName, this.EndStateName); } }
        public bool IsSelected { get; set; }

        // ------------ パラメーター（FeatureParamaterData）個別での設定がありうる要素

        public float Offset { get; set; }
        public IStateEasing StateEasing { get; } = new StateEasing();

        // ------------

        public float TotalDuration { get; set; }

        public bool IsCancelable { get; set; }

        public bool IsLoop { get; set; }

        public string StartStateName { get; set; }

        public string EndStateName { get; set; }

        public IStateTransitionTrigger Trigger { get; }

        //----------------------------------------------------------

        public void RunUpdate()
        {
            this.LocalTime.Time += 1;

            // イベント発火を調査・処理します。
            foreach(StateTransitionTrackData track in this._tracks.Values)
            {
                IEnumerable<KeyParamater> trackEvents = track.GatherEventByRange(this.LocalTime.Time, 1.0f);
                (_owner as StateMachineData).PostTrackEvent(track, trackEvents);
            }

            // ループ再生・再生ストップをハンドリングします。
            if (this.LocalTime.Time >= this.LocalTime.AnimPlayEndTime)
            {
                if (this.IsLoop)
                {
                    this.LocalTime.Time = 0;
                }
                else
                {
                    this.LocalTime.Time = this.LocalTime.AnimPlayEndTime;
                }
            }
        }

        //----------------------------------------------------------

        public void Scale(float offset, float scale, bool notScaleDuration)
        {
            foreach(var track in _tracks.Values)
            {
                float origin = track.Offset;

                TimeEditableItemsHelper.Scale(new[] { track }, 0.0f, offset, scale, notScaleDuration);

                if (!notScaleDuration)
                {
                    TimeEditableItemsHelper.Scale(track.KeyParamaters.Cast<ITimeEditableItems>(), origin, offset, scale, notScaleDuration);
                }
            }
        }

        //----------------------------------------------------------

        public StateTransitionData(IStateMachine owner, Action<IStateTransition> onTimeChangedAction)
        {
            _owner = owner;
            LocalTime = new StateTransitionLocalTime(this);
            _OnTimeLineUpdated = onTimeChangedAction;

            Trigger = new StateTransitionTriggerData(this);
        }

        //----------------------------------------------------------

        public void RaiseOnTimeLineUpdatedEvent()
        {
            if (this._OnTimeLineUpdated != null)
            {
                this._OnTimeLineUpdated(this);
            }
        }

        //----------------------------------------------------------

        string GetKeyString_(string name, FeatureParamaterKind kind)
        {
            Debug.Assert(!String.IsNullOrEmpty(name));
            Debug.Assert(kind != FeatureParamaterKind.None);

            return name + kind.ToString();
        }

        //----------------------------------------------------------

        public IStateTransitionTrack FindIStateTransitionTrackDataByName(string name, FeatureParamaterKind kind)
        {
            return FindStateTransitionTrackDataByName(name, kind);
        }

        //----------------------------------------------------------

        StateTransitionTrackData CreateANewTrackData_()
        {
            var track = new StateTransitionTrackData();
            SetCommonSettingToTrack_(track);
            return track;
        }

        //----------------------------------------------------------

        void SetCommonSettingToTrack_(StateTransitionTrackData track)
        {
            track.Duration = this.Offset;
            track.Duration = this.TotalDuration;
            (track.StateEasing as StateEasing).EasingType = this.StateEasing.EasingType;
            (track.StateEasing as StateEasing).EasingExtraParamater = this.StateEasing.EasingExtraParamater;
        }

        //----------------------------------------------------------

        internal StateTransitionTrackData FindStateTransitionTrackDataByName(string name, FeatureParamaterKind kind)
        {
            StateTransitionTrackData result;
            if(!_tracks.TryGetValue(GetKeyString_(name, kind), out result))
            {
                // 個別設定がない場合は共通設定を返す。
                result = CreateANewTrackData_();
                // FindOrCreateTrack_ と異なり、こちらは 登録済みの内容を返すだけ。新規追加はしません。
            }

            return result;
        }

        //----------------------------------------------------------

        private StateTransitionTrackData FindOrCreateTrack_(string name, FeatureParamaterKind kind)
        {
            StateTransitionTrackData result;
            if (!_tracks.TryGetValue(GetKeyString_(name, kind), out result))
            {
                result = CreateANewTrackData_();
                _tracks.Add(GetKeyString_(name, kind), result);
            }

            return result;
        }

        //----------------------------------------------------------

        internal void SetStateTransitionTrackDataByName(string name, FeatureParamaterKind kind, float offset, float duration, IStateEasing stateEasing)
        {
            StateTransitionTrackData track = FindOrCreateTrack_(name, kind);

            float offsetDiff = offset - track.Offset;
            float durationScale = duration / track.Duration;
            float origin = track.Offset;

            track.Offset = offset;
            track.Duration = duration;
            (track.StateEasing as StateEasing).EasingType = stateEasing.EasingType;
            (track.StateEasing as StateEasing).EasingExtraParamater = stateEasing.EasingExtraParamater;

            // キーの位置調整。
            TimeEditableItemsHelper.Scale(track.KeyParamaters.Cast<ITimeEditableItems>(), origin, offsetDiff, durationScale, false);
        }

        //----------------------------------------------------------

        internal bool SetStateTransitionTrackKey(string name, FeatureParamaterKind kind, float time, IStateEasing stateEasing,  float x, float y, float z, float w)
        {
            StateTransitionTrackData track = FindOrCreateTrack_(name, kind);

            bool isModified = false;

            KeyParamater keyParamater = track.KeyParamaters.FirstOrDefault(key => key.Time == time);
            if (keyParamater == null)
            {
                keyParamater = new KeyParamater(time, stateEasing);
                track.KeyParamaterList.Add(keyParamater);
                track.KeyParamaterList.Sort((k1, k2) => Math.Sign(k1.Time - k2.Time));
                isModified |= true;
            }

            {
                if(keyParamater.FeatureParamaterData.Kind != kind)
                {
                    (keyParamater.FeatureParamaterData as FeatureParamaterData).Kind = kind;
                    isModified |= true;
                }

                if (!keyParamater.FeatureParamaterData.IsSame(x, y, z, w))
                {
                    keyParamater.FeatureParamaterData.Set(x, y, z, w);
                    isModified |= true;
                }

                if (!keyParamater.StateEasing.IsSame(stateEasing))
                {
                    (keyParamater.StateEasing as StateEasing).EasingType = stateEasing.EasingType;
                    (keyParamater.StateEasing as StateEasing).EasingExtraParamater = stateEasing.EasingExtraParamater;
                    isModified |= true;
                }
            }

            return isModified;
        }

        internal bool SetStateTransitionTrackEvent(string name, float time, StateMachineEventKind eventkind, object param1, object param2, uint delayFrames)
        {
            FeatureParamaterKind kind = FeatureParamaterKind.StateMachineEvent;
            StateTransitionTrackData track = FindOrCreateTrack_(name, kind);
            Debug.Assert(track != null);

            bool isModified = SetStateTransitionTrackKey(name, kind, time, new StateEasing(), 0.0f, 0.0f, 0.0f, 0.0f);
            KeyParamater keyParamater = track.KeyParamaters.FirstOrDefault(key => key.Time == time);
            {
                if (!keyParamater.StateMachineEvent.IsSame(eventkind, param1, param2, delayFrames))
                {
                    (keyParamater.StateMachineEvent as StateMachineEvent).Set(eventkind, param1, param2, delayFrames);
                    isModified |= true;
                }
            }

            return isModified;
        }

        //----------------------------------------------------------

        /// <summary>
        ///
        /// </summary>
        public IEnumerable<IFeatureParamater> GetFreatureParamaters()
        {
            var layer = _owner.GetSelectedIStateLayer();
            if(layer == null)
            {
                Debug.Assert(false);
                yield break;
            }

            var startState = layer.FindStateByName(this.StartStateName);
            var endState = layer.FindStateByName(this.EndStateName);
            if(startState == null || endState == null)
            {
                Debug.Assert(false);
                yield break;
            }

            if(startState.GetFreatureParamaters().Count() != endState.GetFreatureParamaters().Count())
            {
                Debug.Assert(false);
                yield break;
            }

            // 現在の(Transitionローカル時間)で評価をした、補間結果を返すようにする
            int count = startState.GetFreatureParamaters().Count();
            for (int index = 0; index < count; index++)
            {
                var paramS = startState.Paramaters.ElementAt(index);
                var paramE = endState.Paramaters.ElementAt(index);

                var track = this.FindIStateTransitionTrackDataByName(paramS.Name, paramS.Kind);

                float startTime = track.Offset;
                float endTime = track.Offset + track.Duration;
                IStateEasing stateEasing = track.StateEasing;

                FeatureParamaterData result = new FeatureParamaterData();
                result.SetFeatureParamaterValueByOther(paramS);

                StateAnimationHelper.TryFindStartAndEndPair((track as StateTransitionTrackData).KeyParamaterList,
                    LocalTime.Time, ref paramS, ref paramE, ref startTime, ref endTime, ref stateEasing);

                StateAnimationHelper.CalcInterp(result, startTime, endTime, LocalTime.Time, stateEasing, paramS, paramE);

                yield return result;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal static class StateAnimationHelper
    {
        static Random _random = new Random();

        public static bool TryFindStartAndEndPair(List<KeyParamater> keyParamaterList,
            float time,
            ref IFeatureParamater startParam, ref IFeatureParamater endParam,
            ref float startTime, ref float endTime, ref IStateEasing stateEasing)
        {
            if (keyParamaterList.Count() <= 0)
            {
                return false;
            }
            else
            {
                var foundKey = keyParamaterList.FirstOrDefault(key => key.Time > time);
                if (foundKey == null)
                {
                    var lastKey = keyParamaterList.Last();

                    startParam = lastKey.FeatureParamaterData;
                    startTime = lastKey.Time;
                    stateEasing = lastKey.StateEasing;
                }
                else
                {
                    // 終端値は、発見されたキー
                    endParam = foundKey.FeatureParamaterData;
                    endTime = foundKey.Time;

                    // 一つ前があれば開始値として採用
                    int index = keyParamaterList.IndexOf(foundKey);
                    if (index - 1 >= 0)
                    {
                        var prevKey = keyParamaterList[index - 1];

                        startParam = prevKey.FeatureParamaterData;
                        startTime = prevKey.Time;
                        stateEasing = prevKey.StateEasing;
                    }
                }

                return true;
            }
        }

        static public void CalcInterp(FeatureParamaterData result,
            float startTime, float endTime, float time, IStateEasing stateEasing, IFeatureParamater paramS, IFeatureParamater paramE)
        {
            float weight = ITimeHelper.GetNormalizedTime((int)startTime, (int)endTime, (int)time);

            // 0.0 - 1.0 にクランプ(TODO：区間外リピートにも対応する)
            weight = Math.Max(Math.Min(weight, 1.0f), 0.0f);

            // テスト用の仮実装
            switch (stateEasing.EasingType)
            {
                case EasingType.Linear:
                    {
                        result.X = LinearInterp_(paramS.X, paramE.X, weight);
                        result.Y = LinearInterp_(paramS.Y, paramE.Y, weight);
                        result.Z = LinearInterp_(paramS.Z, paramE.Z, weight);
                        result.W = LinearInterp_(paramS.W, paramE.W, weight);
                        break;
                    }
                case EasingType.Square:
                    {
                        result.X = SquareInterp_(paramS.X, paramE.X, weight);
                        result.Y = SquareInterp_(paramS.Y, paramE.Y, weight);
                        result.Z = SquareInterp_(paramS.Z, paramE.Z, weight);
                        result.W = SquareInterp_(paramS.W, paramE.W, weight);
                        break;
                    }
                case EasingType.SinOffset:
                    {
                        result.X = SinOffsetInterp_(paramS.X, paramE.X, stateEasing.EasingExtraParamater, weight);
                        result.Y = SinOffsetInterp_(paramS.Y, paramE.Y, stateEasing.EasingExtraParamater, weight);
                        result.Z = SinOffsetInterp_(paramS.Z, paramE.Z, stateEasing.EasingExtraParamater, weight);
                        result.W = SinOffsetInterp_(paramS.W, paramE.W, stateEasing.EasingExtraParamater, weight);
                        break;
                    }
                case EasingType.RandomOffset:
                    {
                        result.X = RandomOffsetInterp_(paramS.X, paramE.X, stateEasing.EasingExtraParamater, weight);
                        result.Y = RandomOffsetInterp_(paramS.Y, paramE.Y, stateEasing.EasingExtraParamater, weight);
                        result.Z = RandomOffsetInterp_(paramS.Z, paramE.Z, stateEasing.EasingExtraParamater, weight);
                        result.W = RandomOffsetInterp_(paramS.W, paramE.W, stateEasing.EasingExtraParamater, weight);
                        break;
                    }
                case EasingType.LinearOffset:
                    {
                        result.X = LinearOffsetInterp_(paramS.X, paramE.X, stateEasing.EasingExtraParamater, weight);
                        result.Y = LinearOffsetInterp_(paramS.Y, paramE.Y, stateEasing.EasingExtraParamater, weight);
                        result.Z = LinearOffsetInterp_(paramS.Z, paramE.Z, stateEasing.EasingExtraParamater, weight);
                        result.W = LinearOffsetInterp_(paramS.W, paramE.W, stateEasing.EasingExtraParamater, weight);
                        break;
                    }
            }
        }

        static float LinearInterp_(float start, float end, float weight)
        {
            return start + (end - start) * weight;
        }

        static float SquareInterp_(float start, float end, float weight)
        {
            return start + (end - start) * (1.0f - (1.0f - weight) * (1.0f - weight));
        }

        static float SinOffsetInterp_(float start, float end, float param, float weight)
        {
            return start + param * (float)Math.Sin(2.0 * Math.PI * weight);
        }

        static float RandomOffsetInterp_(float start, float end, float param, float weight)
        {
            return start + param * (float)_random.NextDouble();
        }

        static float LinearOffsetInterp_(float start, float end, float param, float weight)
        {
            return start + param * weight;
        }
    }

    //----------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IStateTransitionTrigger
    {
        IStateTransition OwnerTransiton { get; }

        bool IsSelected { get; }

        TriggerKind TriggerKind { get; }

        StateMachineOperator StateMachineOperator { get; }

        IEnumerable<IStateMachineVariable> StateMachineVariables { get; }
    }

    //----------------------------------------------------------

    /// <summary>
    ///
    /// </summary>
    internal class StateTransitionTriggerData : IStateTransitionTrigger
    {
        public IStateTransition OwnerTransiton { get; private set; }

        public bool IsSelected { get; set; }

        public TriggerKind TriggerKind { get; set; }

        public StateMachineOperator StateMachineOperator { get; set; }

        public List<IStateMachineVariable> StateMachineVariableList { get; } = new List<IStateMachineVariable>();

        public StateTransitionTriggerData(IStateTransition owner)
        {
            OwnerTransiton = owner;
        }

        public bool IsTriggered(StateMachineEvent eventMsg)
        {
            switch(this.TriggerKind)
            {
                case TriggerKind.IsAuto:
                    return true;
                case TriggerKind.IsHit:
                    return eventMsg.kind == StateMachineEventKind.Hit;
                case TriggerKind.IsDecided:
                    return eventMsg.kind == StateMachineEventKind.Decided;
                case TriggerKind.IsStateTransitionCompleted:
                    if (eventMsg.kind == StateMachineEventKind.StateTransitionCompleted)
                    {
                        return (string)eventMsg.param1 == this.OwnerTransiton.StartStateName;
                    }
                    return false;
                case TriggerKind.IsVariableChanged:
                    return false;
                case TriggerKind.IsStateChangeRequested:
                    if(eventMsg.kind == StateMachineEventKind.StateChangeRequested)
                    {
                        return eventMsg.param2.ToString() == this.OwnerTransiton.EndStateName;
                    }
                    return false;
                default:
                    Debug.Assert(false);
                    return false;
            }
        }

        public IEnumerable<IStateMachineVariable> StateMachineVariables { get { return this.StateMachineVariableList; } }
    }

    //----------------------------------------------------------

    /// <summary>
    ///
    /// </summary>
    public interface IState : IStateLayerEditItem
    {
        string Name { get; }

        IEnumerable<IFeatureParamater> Paramaters { get; }
    }

    //----------------------------------------------------------

    /// <summary>
    ///
    /// </summary>
    internal class StateData : IState
    {
        public string EditName { get { return this.Name; } }

        public string Name { get; set; } = string.Empty;
        public bool IsSelected { get; set; }

        public IEnumerable<IFeatureParamater> GetFreatureParamaters()
        {
            return this.Paramaters;
        }

        internal List<IFeatureParamater> ParamaterList { get; } = new List<IFeatureParamater>();

        public IEnumerable<IFeatureParamater> Paramaters { get { return ParamaterList; } }
    }

    //----------------------------------------------------------

    /// <summary>
    ///
    /// </summary>
    public interface IFeatureParamater
    {
        string Name { get; }

        bool IsSelected { get; }

        FeatureParamaterKind Kind { get; }

        float X { get; }
        float Y { get; }
        float Z { get; }
        float W { get; }
    }

    /// <summary>
    ///
    /// </summary>
    internal static class FeatureParamaterDataHelper
    {
        internal static bool IsSame(this IFeatureParamater fp, float x, float y, float z, float w)
        {
            return fp.X == x && fp.Y == y && fp.Z == z && fp.W == w;
        }

        internal static void Set(this IFeatureParamater fp, float x, float y, float z, float w)
        {
            var fpd = fp as FeatureParamaterData;
            fpd.X = x;
            fpd.Y = y;
            fpd.Z = z;
            fpd.W = w;
        }
    }


    //----------------------------------------------------------

    /// <summary>
    ///
    /// </summary>
    internal class FeatureParamaterData : IFeatureParamater
    {
        public string Name { get; set; }

        public bool IsSelected { get; set; }

        public FeatureParamaterKind Kind { get; set; }

        public float X { get; set; }
        public float Y { get; set; }
        public float Z { get; set; }
        public float W { get; set; }
    }

    //----------------------------------------------------------

    /// <summary>
    ///
    /// </summary>
    public interface IStateMachineVariable
    {
        string Name { get; }

        float Value { get; }

        StateMachineVariableType VariableType { get; }

        bool IsSelected { get; }
    }

    //----------------------------------------------------------

    /// <summary>
    ///
    /// </summary>
    internal class StateMachineVariableData : IStateMachineVariable
    {
        public string Name { get; set; } = string.Empty;

        public float Value { get; set; }

        public StateMachineVariableType VariableType { get; set; }

        public bool IsSelected { get; set; }
    }

    public static class EasingTypeHelper
    {
        public static bool UseExtraParamater(this EasingType easingType)
        {
            switch(easingType)
            {
                case EasingType.Linear:
                case EasingType.Square:
                case EasingType.Smooth:
                    return false;
                case EasingType.SinOffset:
                case EasingType.RandomOffset:
                case EasingType.LinearOffset:
                    return true;
                default:
                    Debug.Assert(false);
                    return false;
            }
        }
    }

    //----------------------------------------------------------

    public enum TriggerKind
    {
        IsAuto,
        IsHit,
        IsDecided,
        IsStateTransitionCompleted,
        IsVariableChanged,
        IsStateChangeRequested
    };

    public enum StateMachineVariableType
    {
        Float,
    };

    public enum StateMachineOperator
    {
        GreaterThan,
        GreaterEqual,
        Equal,
        NotEqual,
        LessEqual,
        LessThan,
        AlwaysPass,
        AlwaysFail,
    };

    [Flags]
    public enum FeatureParamaterKind
    {
        None = 0x0,
        Position = 0x1,
        Scale = 0x2,
        RotateZ = 0x4,
        WhiteColor = 0x8,
        BlackColor = 0x10,
        StateMachineEvent = 0x20
    }

    /// <summary>
    /// ParameterizedAnimType に対応した値を持ちます。
    /// </summary>
    public enum EasingType
    {
        Linear,
        Square,
        Smooth,
        SinOffset,
        RandomOffset,
        LinearOffset
    }

    public enum StateMachineRunMode
    {
        Edit,
        Run,
    }

    public enum StateMachineEventKind
    {
        None,
        Tick,
        Hit,
        Decided,
        StateTransitionCompleted,
        VariableChanged,
        StateChangeRequested
    };

    public enum StateTrackKeyEventKind
    {
        PostEventToChild,
    };
}
