﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;

namespace LECore.Structures
{
    using Core;
    using LECore.Save_Load;
    using System.Threading.Tasks;

    /// <summary>
    /// レイアウト空間を表すクラスです。
    /// シングルトンクラスです。
    /// </summary>
    internal sealed class Scene :
        IDrawable,
        IScene
    {

        #region フィールド
        private readonly List<SubScene> _subSceneSet = new List<SubScene>();
        private readonly List<IPartsControlSetting> partsControlSettings = new List<IPartsControlSetting>();
        private readonly List<IPartsSubScene> partsSubScenes = new List<IPartsSubScene>();
        private readonly List<ScalableFontSettings> _scalableFontSettingsSet = new List<ScalableFontSettings>();
        private SubScene _currentSubScene = null;

        // シングルトン・インスタンス
        private static Scene _theInstance = null;

        #endregion フィールド

        #region プロパティ

        /// <summary>
        /// イベント通知を強制的に無効化する
        /// </summary>
        internal bool ForceDisableEventNotifications
        {
            get;
            set;
        }

        /// <summary>
        /// シングルトン実体を取得します。
        /// </summary>
        public static Scene Instance
        {
            get
            {
                if (_theInstance == null)
                {
                    _theInstance = new Scene();
                    InitializePartsSubScenes_(_theInstance);
                }
                return _theInstance;
            }
        }

        /// <summary>
        /// サブシーンのセットを取得します。
        /// </summary>
        public SubScene[] SubSceneSet
        {
            get
            {
                return _subSceneSet.ToArray();
            }
        }

        /// <summary>
        /// 現在の操作対象のサブシーンを取得します。
        /// </summary>
        public static SubScene CurrentSubScene
        {
            get
            {
                return Scene.Instance._currentSubScene;
            }
        }

        /// <summary>
        /// 現在のシーンを取得、設定します。
        /// </summary>
        public ISubScene CurrentISubScene
        {
            get
            {
                return Scene.Instance._currentSubScene;
            }

            set
            {
                SubScene newCurrentSubScene = value as SubScene;
                if (Scene.Instance._currentSubScene != newCurrentSubScene)
                {
                    Scene.Instance._currentSubScene = newCurrentSubScene;
                    if (OnSceneModify != null)
                    {
                        // カレントサブシーンが変更になったことを通知します。
                        SceneModifyEventArgs currentSubSceneChangedArgs = new SceneModifyEventArgs
                            (SceneModifyEventArgs.Kind.CurrentSubSceneChanged,
                              null);
                        OnSceneModify(this, currentSubSceneChangedArgs);


                        // 選択セットも変更になるので、通知します。
                        // サブシーンが存在しない場合は、仮の空選択セットを通知します。
                        IPane[] selectedPaneArray =
                            newCurrentSubScene != null ? newCurrentSubScene.ISelectedSet.IPaneArray : new IPane[0];

                        SceneModifyEventArgs selSetChangedArgs = new SceneModifyEventArgs(SceneModifyEventArgs.Kind.SelectedSetModify, selectedPaneArray);
                        OnSceneModify(this, selSetChangedArgs);

                    }
                }
            }
        }

        /// <summary>
        /// 部品シーンです。
        /// </summary>
        public IEnumerable<IPartsSubScene> PartsSubScenes
        {
            get { return partsSubScenes; }
        }

        #endregion プロパティ

        #region イベント

        /// <summary>
        /// シーン変更時に発生するイベント
        /// </summary>
        public event OnSceneModifyHandler OnSceneModify;

        /// <summary>
        /// 部品レイアウトが更新イベントです。
        /// </summary>
        public event Action<string> OnPartsSubSceneModified;

        #endregion イベント

        /// <summary>
        /// 非公開・コンストラクタ
        /// </summary>
        private Scene()
        {
            // テスト用に仮に、部品フォルダをデスクトップ上の特定フォルダとしておく。
            this.PartsRootPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.Desktop), "parts_layout_root");
        }

        /// <summary>
        /// コントロール定義の初期化を行います。
        /// </summary>
        private void InitializePartsControlSettings_()
        {
            List<string> filePaths;
            try
            {
                partsControlSettings.Clear();

                // 部品ルートが無い場合。
                if (!Directory.Exists(this.PartsRootPath))
                {
                    InitializeDefaultPartsControlSettings_();
                    return;
                }

                // 部品ルート以下にある、コントロール定義を読み込む。
                // デフォルト設定を上書きするファイル指定があるか？
                filePaths = new List<string>(Directory.GetFiles(
                    this.PartsRootPath,
                    AppConstants.ContorlFileExtPattern,
                    SearchOption.AllDirectories));

                // ファイル名昇順でソートします。
                filePaths.Sort((f1, f2) => string.Compare(Path.GetFileName(f1), Path.GetFileName(f2)));

                var defaultOverrideFilePath = filePaths.FirstOrDefault(
                    (filePath) => Path.GetFileName(filePath) == AppConstants.DefaultContorlFileName);
                if (defaultOverrideFilePath == null)
                {
                    // 指定がなければ
                    // 組み込みのコントロール定義をハードコードで生成しておく。
                    InitializeDefaultPartsControlSettings_();
                }
                else
                {
                    // デフォルト設定が一番最初にロードされるように先頭に持ってくる。
                    filePaths.Remove(defaultOverrideFilePath);
                    filePaths.Insert(0, defaultOverrideFilePath);
                }
            }
            catch (Exception e)
            {
                LayoutEditorCore.MsgReporter.ReportError(
                    LECoreStringResMgr.Get("LAYOUT_ERROR_DLG_TITLE"),
                    LECoreStringResMgr.Get("LAYOUT_ERROR_DLG_INITIALIZE_CONTROL") + e.Message);

                return;
            }


            // すべての定義を読み込む
            if (filePaths.Count() > 0)
            {
                try
                {
                    string category = LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSTEMPLATE_INIT", this.PartsRootPath);
                    LayoutEditorCore.MsgReporter.BeginPacking(category);

                    // 無効化を先頭に
                    RegisterPartsControlSetting(GetNoneControl_());

                    ClctImporter importer = new ClctImporter();
                    foreach (var filePath in filePaths)
                    {
                        string fileCategory = string.Format(
                            "{0}[{1}]",
                            LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSTEMPLATE"),
                            Path.GetFileName(filePath));
                        LayoutEditorCore.MsgReporter.BeginPacking(fileCategory);
                        {
                            LayoutEditorCore.ImportFromFile(_currentSubScene, importer, filePath);
                        }
                        LayoutEditorCore.MsgReporter.EndPacking(false);
                    }
                }
                catch (Exception e)
                {
                    LayoutEditorCore.MsgReporter.ReportError(
                        LECoreStringResMgr.Get("LAYOUT_ERROR_DLG_TITLE"),
                        LECoreStringResMgr.Get("LAYOUT_ERROR_DLG_INITIALIZE_CONTROL") + e.Message);
                }
                finally
                {
                    LayoutEditorCore.MsgReporter.EndPacking(false);
                }
            }
        }

        /// <summary>
        /// 標準レイアウトにもどすための空のコントロールを設定です。
        /// </summary>
        private PartsControlSetting GetNoneControl_()
        {
            return new PartsControlSetting(
                "None",
                LECoreStringResMgr.Get("PARTS_NORMAL_LAYOUT"),
                LECoreStringResMgr.Get("PARTS_NORMAL_LAYOUT_DESC"),
                new PartsParamaterTemplate[0],
                new PartsParamaterTemplate[0]);
        }

        /// <summary>
        /// 組み込みのコントロール定義の初期化を行います。
        /// </summary>
        private void InitializeDefaultPartsControlSettings_()
        {
            RegisterPartsControlSetting(GetNoneControl_());

            PartsControlSetting locatorTemplate = new PartsControlSetting(
                "Locator",
                LECoreStringResMgr.Get("PARTS_LOCATOR"),
                LECoreStringResMgr.Get("PARTS_LOCATOR_DESC"),
                new PartsParamaterTemplate[0],
                new PartsParamaterTemplate[0]);
            RegisterPartsControlSetting(locatorTemplate);

            PartsControlSetting buttonTemplate = new PartsControlSetting(
                "NormalButton",
                LECoreStringResMgr.Get("PARTS_NRMBUTTON"),
                LECoreStringResMgr.Get("PARTS_NRMBUTTON_DESC"),
                new PartsParamaterTemplate[]{
                    new PartsParamaterTemplate("OnOff", LECoreStringResMgr.Get("PARTS_NRMBUTTON_ONOFF_DESC"), true),
                    new PartsParamaterTemplate("Down", LECoreStringResMgr.Get("PARTS_NRMBUTTON_DOWN_DESC"), true),
                    new PartsParamaterTemplate("Disable", LECoreStringResMgr.Get("PARTS_NRMBUTTON_DISABLE_DESC"), false)},
                 new PartsParamaterTemplate[]{
                     new PartsParamaterTemplate("Hit", LECoreStringResMgr.Get("PARTS_HITPANE_DESC"), true),
                 });
            RegisterPartsControlSetting(buttonTemplate);

            PartsControlSetting buttonTemplate3 = new PartsControlSetting(
                "CheckButton",
                LECoreStringResMgr.Get("PARTS_CHKBUTTON"),
                LECoreStringResMgr.Get("PARTS_CHKBUTTON_DESC"),
                new PartsParamaterTemplate[]{
                    new PartsParamaterTemplate("OnOff", LECoreStringResMgr.Get("PARTS_CHKBUTTON_ONOFF_DESC"), true),
                    new PartsParamaterTemplate("Down", LECoreStringResMgr.Get("PARTS_BUTTON_DOWN_DESC"), true),
                    new PartsParamaterTemplate("Check", LECoreStringResMgr.Get("PARTS_CHKBUTTON_CHECK_DESC"), true),
                    new PartsParamaterTemplate("Disable", LECoreStringResMgr.Get("PARTS_NRMBUTTON_DISABLE_DESC"), false)},
                 new PartsParamaterTemplate[]{
                     new PartsParamaterTemplate("Hit", LECoreStringResMgr.Get("PARTS_HITPANE_DESC"), true),
                 });
            RegisterPartsControlSetting(buttonTemplate3);

            PartsControlSetting buttonTemplate4 = new PartsControlSetting(
                "DecisionButton",
                LECoreStringResMgr.Get("PARTS_DECISIONBUTTON"),
                LECoreStringResMgr.Get("PARTS_DECISIONBUTTON_DESC"),
                new PartsParamaterTemplate[]{
                    new PartsParamaterTemplate("OnOff", LECoreStringResMgr.Get("PARTS_NRMBUTTON_ONOFF_DESC"), true),
                    new PartsParamaterTemplate("Down", LECoreStringResMgr.Get("PARTS_BUTTON_DOWN_DESC"), true),
                    new PartsParamaterTemplate("Disable", LECoreStringResMgr.Get("PARTS_NRMBUTTON_DISABLE_DESC"), false)},
                 new PartsParamaterTemplate[] {
                     new PartsParamaterTemplate("Hit", LECoreStringResMgr.Get("PARTS_HITPANE_DESC"), true),});
            RegisterPartsControlSetting(buttonTemplate4);

            PartsControlSetting buttonTemplate5 = new PartsControlSetting(
                "SelectButton",
                LECoreStringResMgr.Get("PARTS_SELECTBUTTON"),
                LECoreStringResMgr.Get("PARTS_SELECTBUTTON_DESC"),
                new PartsParamaterTemplate[]{
                    new PartsParamaterTemplate("OnOff", LECoreStringResMgr.Get("PARTS_NRMBUTTON_ONOFF_DESC"), true),
                    new PartsParamaterTemplate("Down", LECoreStringResMgr.Get("PARTS_BUTTON_DOWN_DESC"), true),
                    new PartsParamaterTemplate("Cancel", LECoreStringResMgr.Get("PARTS_SELECTBUTTON_CANSEL_DESC"), true),
                    new PartsParamaterTemplate("Disable", LECoreStringResMgr.Get("PARTS_NRMBUTTON_DISABLE_DESC"), false)},
                 new PartsParamaterTemplate[] {
                     new PartsParamaterTemplate("Hit", LECoreStringResMgr.Get("PARTS_HITPANE_DESC"), true),});
            RegisterPartsControlSetting(buttonTemplate5);

            PartsControlSetting buttonTemplate6 = new PartsControlSetting(
                "TouchOffButton",
                LECoreStringResMgr.Get("PARTS_TOUCHOFFBUTTON"),
                LECoreStringResMgr.Get("PARTS_TOUCHOFFBUTTON_DESC"),
                new PartsParamaterTemplate[]{
                    new PartsParamaterTemplate("OnOff", LECoreStringResMgr.Get("PARTS_NRMBUTTON_ONOFF_DESC"), true),
                    new PartsParamaterTemplate("Down", LECoreStringResMgr.Get("PARTS_BUTTON_DOWN_DESC")+LECoreStringResMgr.Get("PARTS_BUTTON_DOWN2_DESC"), true),
                    new PartsParamaterTemplate("Disable", LECoreStringResMgr.Get("PARTS_NRMBUTTON_DISABLE_DESC"), false)},
                 new PartsParamaterTemplate[] {
                     new PartsParamaterTemplate("Hit", LECoreStringResMgr.Get("PARTS_HITPANE_DESC"), true),});
            RegisterPartsControlSetting(buttonTemplate6);

            PartsControlSetting buttonTemplate7 = new PartsControlSetting(
                "TouchOffCheckButton",
                LECoreStringResMgr.Get("PARTS_TOUCHOFFCHECKBUTTON"),
                LECoreStringResMgr.Get("PARTS_TOUCHOFFCHECKBUTTON_DESC"),
                new PartsParamaterTemplate[]{
                    new PartsParamaterTemplate("OnOff", LECoreStringResMgr.Get("PARTS_NRMBUTTON_ONOFF_DESC"), true),
                    new PartsParamaterTemplate("Down", LECoreStringResMgr.Get("PARTS_BUTTON_DOWN_DESC")+LECoreStringResMgr.Get("PARTS_BUTTON_DOWN2_DESC"), true),
                    new PartsParamaterTemplate("Check", LECoreStringResMgr.Get("PARTS_CHKBUTTON_CHECK_DESC"), true),
                    new PartsParamaterTemplate("Disable", LECoreStringResMgr.Get("PARTS_NRMBUTTON_DISABLE_DESC"), false)},
                 new PartsParamaterTemplate[] {
                     new PartsParamaterTemplate("Hit", LECoreStringResMgr.Get("PARTS_HITPANE_DESC"), true),});
            RegisterPartsControlSetting(buttonTemplate7);

            PartsControlSetting buttonTemplate8 = new PartsControlSetting(
                "DragButton",
                LECoreStringResMgr.Get("PARTS_DRAGBUTTON"),
                LECoreStringResMgr.Get("PARTS_DRAGBUTTON_DESC"),
                new PartsParamaterTemplate[]{
                    new PartsParamaterTemplate("OnOff", LECoreStringResMgr.Get("PARTS_DRAGBUTTON_ONOFF_DESC"), true),
                    new PartsParamaterTemplate("Down", LECoreStringResMgr.Get("PARTS_DRAGBUTTON_DOWN_DESC"), true),
                    new PartsParamaterTemplate("Release", LECoreStringResMgr.Get("PARTS_DRAGBUTTON_RELEASE_DESC"), true),
                    new PartsParamaterTemplate("Disable", LECoreStringResMgr.Get("PARTS_NRMBUTTON_DISABLE_DESC"), false)},
                 new PartsParamaterTemplate[] {
                     new PartsParamaterTemplate("Hit", LECoreStringResMgr.Get("PARTS_HITPANE_DESC"), true),});
            RegisterPartsControlSetting(buttonTemplate8);

            PartsControlSetting buttonTemplate9 = new PartsControlSetting(
                "TouchDragButton",
                LECoreStringResMgr.Get("PARTS_TOUCHDRAGBUTTON"),
                LECoreStringResMgr.Get("PARTS_TOUCHDRAGBUTTON_DESC"),
                new PartsParamaterTemplate[]{
                    new PartsParamaterTemplate("Touch", LECoreStringResMgr.Get("PARTS_TOUCHDRAGBUTTON_TOUCH_DESC"), true),
                    new PartsParamaterTemplate("Release", LECoreStringResMgr.Get("PARTS_TOUCHDRAGBUTTON_RELEASE_DESC"), true),
                    new PartsParamaterTemplate("Disable", LECoreStringResMgr.Get("PARTS_NRMBUTTON_DISABLE_DESC"), false)},
                 new PartsParamaterTemplate[] {
                     new PartsParamaterTemplate("Hit", LECoreStringResMgr.Get("PARTS_HITPANE_DESC"), true),});
            RegisterPartsControlSetting(buttonTemplate9);
        }

        /// <summary>
        /// シーンの部品レイアウト情報を初期化します。
        /// </summary>
        private static void InitializePartsSubScenes_(Scene scene)
        {
            ReleasePartsSubScenes_(scene);

            try
            {
                // 部品レイアウトの初期化
                LayoutEditorCore.MsgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSLAYOUTLOADING"));
                var filePaths = scene.EnumrateLayoutFiles().ToArray();
                foreach (var filePath in filePaths)
                {
                    // ファイルの読み込み
                    var partsSubScene = new PartsSubScene(filePath);

                    // 同名部品ファイルがないかチェックして警告を表示します。
                    // 通常レイアウト同士については警告を表示しません。
                    string partsFileName = Path.GetFileName(filePath);
                    var sameNameParts = scene.partsSubScenes.FirstOrDefault(
                        (pss) => Path.GetFileName(pss.FilePath) == partsFileName &&
                        // 部品レイアウトかどうかの確認
                        (!string.IsNullOrEmpty(pss.DescriptionName) || !string.IsNullOrEmpty(partsSubScene.DescriptionName)));
                    if (sameNameParts != null)
                    {
                        string errorTilte = LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSLAYOUT_SAMEERROR", partsFileName);
                        LayoutEditorCore.MsgReporter.BeginPacking(errorTilte);
                        LayoutEditorCore.MsgReporter.ReportError("", "[{0}]", sameNameParts.FilePath);
                        LayoutEditorCore.MsgReporter.ReportError("", "[{0}]", filePath);
                        LayoutEditorCore.MsgReporter.OutLog(AppConstants.WarnigLogPrefix + errorTilte, filePath);
                        LayoutEditorCore.MsgReporter.EndPacking();
                    }

                    // 要素の追加
                    scene.partsSubScenes.Add(partsSubScene);
                }
            }
            catch (Exception e)
            {
                LayoutEditorCore.MsgReporter.ReportError(
                    LECoreStringResMgr.Get("LAYOUT_ERROR_DLG_TITLE"),
                    LECoreStringResMgr.Get("LAYOUT_ERROR_DLG_INITIALIZE_PARTS") + e.Message);

                return;
            }
            finally
            {
                LayoutEditorCore.MsgReporter.EndPacking();
            }
        }

        /// <summary>
        /// シーンの部品レイアウト情報を解放します。
        /// </summary>
        private static void ReleasePartsSubScenes_(Scene scene)
        {
            if (scene == null)
            {
                return;
            }

            foreach (PartsSubScene partsSubScene in scene.partsSubScenes)
            {
                DoUnloadOnePartsSubScene_(partsSubScene);
            }
            scene.partsSubScenes.Clear();

            // フォントキャッシュをリセット。
            if (scene._subSceneSet.Count <= 0)
            {
                FontFactory.ResetFontChacheAll();
            }
        }

        /// <summary>
        /// シーンの部品レイアウトを再読み込みします。
        /// </summary>
        public void ReloadOnePartsSubScene(string fileName)
        {
            var partsSubScene = this.FindPartsSubSceneByFileName(fileName);
            DoUnloadOnePartsSubScene_(partsSubScene);
            LoadPartsSubScene(partsSubScene);
        }

        /// <summary>
        /// 部品サブシーンを追加します。
        /// </summary>
        public IPartsSubScene AddPartsSubScene(string filePath)
        {
            var partsSubScene = this.FindPartsSubSceneByFileName(Path.GetFileName(filePath));
            if (partsSubScene == null)
            {
                partsSubScene = new PartsSubScene(filePath);
                this.partsSubScenes.Add(partsSubScene);
            }

            return partsSubScene;
        }

        /// <summary>
        /// シーンの部品レイアウトを再読み込みします。
        /// </summary>
        private static void DoUnloadOnePartsSubScene_(IPartsSubScene partsSubScene)
        {
            if (partsSubScene == null || !partsSubScene.IsLoaded)
            {
                return;
            }

            // 一旦削除して、再読み込みします。
            (partsSubScene.SubScene as SubScene).OnModified -= Scene.Instance.OnPartsSubSceneModify_;
            Scene.Instance.RemoveSubScene(partsSubScene.SubScene);
            (partsSubScene as PartsSubScene).SubScene = null;
        }

        /// <summary>
        /// 部品シーンを読み込みます。
        /// </summary>
        /// <param name="partsSubScene"></param>
        public void LoadPartsSubScene(IPartsSubScene partsSubScene)
        {
            if(partsSubScene == null || partsSubScene.IsLoaded)
            {
                return;
            }

            SubScene loadSubScene = LayoutEditorCore.CreateSubScene(
                partsSubScene.FilePath, LayoutEditorCore.LoadOption.TryToOpenRlan) as SubScene;
            if (loadSubScene != null)
            {
                loadSubScene.OnModified += this.OnPartsSubSceneModify_;
                (partsSubScene as PartsSubScene).UpdateInfo();
                (partsSubScene as PartsSubScene).SubScene = loadSubScene;
                OnPartsSubSceneModify_(loadSubScene, SceneModifyEventArgs.Kind.PartsLayout);
            }
        }

        /// <summary>
        /// スケーラブルフォントの登録
        /// </summary>
        public void RegisterScalableFont(ScalableFontSettings settings)
        {
            _scalableFontSettingsSet.Add(settings);
        }

        /// <summary>
        /// スケーラブルフォントの削除
        /// </summary>
        public void ClearScalableFont()
        {
            _scalableFontSettingsSet.Clear();
        }

        /// <summary>
        /// 部品レイアウトの更新ハンドラ
        /// </summary>
        private void OnPartsSubSceneModify_(ISubScene sender, SceneModifyEventArgs.Kind kind)
        {
            if (this.ForceDisableEventNotifications)
            {
                return;
            }

            if (this.OnPartsSubSceneModified == null)
            {
                return;
            }

            // シーン外部に伝搬します。
            if (kind == SceneModifyEventArgs.Kind.PartsLayout)
            {
                this.OnPartsSubSceneModified(this.FindPartsSubSceneFilePath(sender));
            }
        }

        /// <summary>
        /// シーンの部品レイアウト情報を初期化します。
        /// </summary>
        public void RefreshPartsSubScenes()
        {
            InitializePartsControlSettings_();
            InitializePartsSubScenes_(this);
        }

        public void RegisterPartsControlSetting(IPartsControlSetting newTemplate)
        {
            if (!partsControlSettings.Exists((controlSetting) => controlSetting.Name == newTemplate.Name))
            {
                partsControlSettings.Add(newTemplate);
            }
        }

        /// <summary>
        /// コントロール定義を登録します。
        /// </summary>
        /// <param name="newTemplate"></param>
        public void RegisterPartsControlSetting(IEnumerable<IPartsControlSetting> newTemplates)
        {
            foreach (var newTemplate in newTemplates)
            {
                RegisterPartsControlSetting(newTemplate);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public SubScene CreateSubSceneInstance()
        {
            SubScene subScene = new SubScene();

            foreach (IScalableFontSettings desc in _scalableFontSettingsSet)
            {
                subScene.ILEFontManager.CreateILEFont(desc.FontName, desc.FontFilePath, desc);
            }

            return subScene;
        }

        /// <summary>
        /// サブシーンを生成します。
        /// サブシーンを編集対象に追加しません。
        /// </summary>
        /// <returns></returns>
        public ISubScene CreateSubScene()
        {
            SubScene subScene = CreateSubSceneInstance();
            _subSceneSet.Add(subScene);

            DbgConsole.WriteLine("CreateSubScene() --- current_SubScene_num=[{0}]", _subSceneSet.Count);

            return subScene;
        }

        /// <summary>
        /// シーンにサブシーンを追加します。
        /// 追加したシーンが _currentSubScene に設定され、
        /// 操作対象となります。
        /// </summary>
        /// <returns></returns>
        public ISubScene AddSubScene()
        {
            CurrentISubScene = CreateSubScene();
            return CurrentISubScene;
        }

        /// <summary>
        /// サブシーンが変更されたときに、サブシーンから呼び出されます。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="ea"></param>
        public void OnSubSceneChanged(SubScene subScene, SceneModifyEventArgs ea)
        {
            Debug.Assert(subScene != null);
            Debug.Assert(ea != null);

            if (this.ForceDisableEventNotifications)
            {
                return;
            }

            // 登録前のシーンについては、メッセージ通知をしません。
            if (!this._subSceneSet.Contains(subScene))
            {
                return;
            }

            if (OnSceneModify != null)
            {
                OnSceneModify(subScene, ea);
            }
        }

        /// <summary>
        /// シーンが（サブシーンではなく）管理するリソース
        /// (例：テクスチャマネージャ、フォントマネージャ、クリップボード)
        /// に更新が起こった際に、呼び出します。
        ///
        /// TODO:
        /// 現在はテクスチャマネージャ、フォントマネージャに関しては、個別のイベントを
        /// 使用している。これは、シーン更新イベントと、統合する。
        ///
        /// MEMO:現在は統合されている、クリップボードのみが残されている。
        /// </summary>
        public void OnSceneResourceChanged(SceneModifyEventArgs ea)
        {
            if (this.ForceDisableEventNotifications)
            {
                return;
            }

            if (OnSceneModify != null && _currentSubScene != null)
            {
                OnSceneModify(_currentSubScene, ea);
            }
        }

        /// <summary>
        /// シーンからサブシーンを取り除きます。
        /// </summary>
        public void RemoveSubScene(ISubScene isubScene)
        {
            SubScene subScene = isubScene as SubScene;

            // サブシーン中の不要なオブジェクトを破棄します。
            subScene.Dispose();

            _subSceneSet.Remove(subScene);

            DbgConsole.WriteLine("RemoveSubScene() --- current_SubScene_num=[{0}]", _subSceneSet.Count);

            if (_currentSubScene == subScene)
            {
                CurrentISubScene = null;
            }

            // サブシーンが削除されたことを通知します。
            if (OnSceneModify != null)
            {
                SceneModifyEventArgs args = new SceneModifyEventArgs
                    (SceneModifyEventArgs.Kind.SubSceneRemoved, subScene);
                OnSceneModify(this, args);
            }
        }

        #region IDrawable 実装

        /// <summary>
        /// シーン中のすべてのサブシーンを描画します。
        /// </summary>
        public void Draw(IRenderer renderer, DrawableOption option)
        {
            foreach (SubScene subScene in _subSceneSet)
            {
                subScene.Draw(renderer, option);
            }
        }

        #endregion

        #region IScene メンバ (外部モジュール公開メソッド)

        public ISubScene[] ISubSceneSet
        {
            get { return _subSceneSet.ToArray(); }
        }

        /// <summary>
        /// クリップボードを取得します。
        /// </summary>
        public ILEClipboard ILEClipboard
        {
            get { return LECore.Structures.LEClipboard.Instance; }
        }

        /// <summary>
        /// ツールが利用できる、部品コントロール定義リストです。
        /// </summary>
        public IEnumerable<IPartsControlSetting> PartsControlSettings
        {
            get { return this.partsControlSettings; }
        }

        /// <summary>
        /// 部品ファイルが格納されたフォルダです。
        /// </summary>
        public string PartsRootPath { get; internal set; }

        #endregion IScene メンバ
    }
}
