﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;

namespace LECore.Structures
{
    /// <summary>
    /// 部品サブシーンヘルパークラス
    /// </summary>
    public static class PartsSubSceneHelper
    {
        /// <summary>
        /// ファイルが依存するファイル名を取得します。
        /// </summary>
        public static IEnumerable<string> GetDependsFiles(string filePath)
        {
            if (File.Exists(filePath))
            {
                // 参照部品レイアウト
                string contet = File.ReadAllText(filePath);
                foreach (Match m in Regex.Matches(contet, "<parts path=\"(.+?)\""))
                {
                    string dependFileName = m.Groups[1].Value;
                    if (Path.GetFileName(filePath) == dependFileName)
                    {
                        continue;
                    }

                    yield return dependFileName;
                }
            }
        }

        /// <summary>
        /// 部品ペインが依存する部品レイアウト名を収集します。
        /// </summary>
        public static IEnumerable<string> GetDependsPartsNameRecurcively(ISubScene partsLayoutSubScene)
        {
            if (partsLayoutSubScene == null)
            {
                yield break;
            }

            foreach (IPane partsPane in partsLayoutSubScene.FindPanesByKind(PaneKind.Parts))
            {
                yield return partsPane.IPartsLayout.PartsLayoutName;

                foreach (var subPartsName in GetDependsPartsNameRecurcively(partsPane.IPartsLayout.PartsSubScene))
                {
                    yield return subPartsName;
                }
            }
        }

        /// <summary>
        /// 部品がボディレイアウトに配置するものとして適切か判定します。
        /// </summary>
        public static bool IsPartsSubSceneValidForBodyLayout(IPartsSubScene partsLayoutSubScene, string bodyLayoutFilePath)
        {
            string bodyLayoutFileName = Path.GetFileName(bodyLayoutFilePath);
            var relatedPartsNames = PartsSubSceneHelper.GetDependsPartsNameRecurcively(partsLayoutSubScene.SubScene);
            if (bodyLayoutFileName == Path.GetFileName(partsLayoutSubScene.FilePath))
            {
                return false;
            }

            if (relatedPartsNames.Any((partsRelatedFile) => Path.GetFileName(partsRelatedFile) == bodyLayoutFileName))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// ファイルが依存するファイル名を取得します。
        /// </summary>
        public static string GetBasePartsFile(string filePath)
        {
            if (File.Exists(filePath))
            {
                // 基底部品レイアウト
                string contet = File.ReadAllText(filePath);
                var match = Regex.Match(contet, "<partsDerivativeBaseName>(.*?)</partsDerivativeBaseName>", RegexOptions.Singleline);
                if (match.Success)
                {
                    string baseFileName = match.Groups[1].Value;
                    if (baseFileName != Path.GetFileName(filePath))
                    {
                        return baseFileName;
                    }
                }
            }

            return string.Empty;
        }

        /// <summary>
        /// 派生部品かどうか判定します。
        /// </summary>
        public static bool IsDerivativeParts(this IPartsSubScene partsSubScene)
        {
            if (partsSubScene == null)
            {
                return false;
            }

            return !string.IsNullOrEmpty(partsSubScene.BasePartsName);
        }

        /// <summary>
        /// 基底部品が変更されたか判定します。
        /// </summary>
        public static void UpdateBasePartsSubSceneModified(this IPartsSubScene iPartsSubScene)
        {
            PartsSubScene partsSubScene = iPartsSubScene as PartsSubScene;
            var baseParts = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(partsSubScene.BasePartsName);
            if (baseParts == null)
            {
                partsSubScene.IsBasePartsSubSceneModified = false;
                return;
            }

            if (!File.Exists(baseParts.FilePath))
            {
                partsSubScene.IsBasePartsSubSceneModified = false;
                return;
            }

            var hashValue = FileUpdateHelper.ComputeSha1FromFlytAndFlan(baseParts.FilePath);
            partsSubScene.IsBasePartsSubSceneModified = partsSubScene.BasePartsHashValue != hashValue;
        }
    }

    internal class PartsSubScene : IPartsSubScene
    {
        /// <summary>
        ///
        /// </summary>
        public PartsSubScene(string filePath)
        {
            this.FilePath = filePath;
            UpdateInfo();
        }

        /// <summary>
        /// 情報を更新します。
        /// </summary>
        public void UpdateInfo()
        {
            var ctrlNames = new List<string>();
            this.ControlNames = ctrlNames;

            if (File.Exists(this.FilePath))
            {
                string content = File.ReadAllText(this.FilePath);

                // DescriptionName
                {
                    var match = Regex.Match(content, "<partsName>(.*?)</partsName>");
                    if (match.Success)
                    {
                        this.DescriptionName = match.Groups[1].Value;
                    }
                }

                // Name
                {
                    var matches = Regex.Matches(content, "<control name=\"(.*?)\"");
                    if(matches.Count > 0)
                    {
                        // 部品レイアウトなら、先頭のコントロールを部品種類として記録する。
                        if (!string.IsNullOrEmpty(this.DescriptionName))
                        {
                            this.PartsKind = matches[0].Groups[1].Value;
                        }

                        foreach (Match match in matches )
                        {
                            ctrlNames.Add(match.Groups[1].Value);
                        }
                    }
                }

                // Description
                {
                    var match = Regex.Match(content, "<partsDescription>(.*?)</partsDescription>", RegexOptions.Singleline);
                    if (match.Success)
                    {
                        this.Description = match.Groups[1].Value;
                    }
                }

                // BasePartsName
                {
                    var match = Regex.Match(content, "<partsDerivativeBaseName>(.*?)</partsDerivativeBaseName>", RegexOptions.Singleline);
                    if (match.Success)
                    {
                        this.BasePartsName = match.Groups[1].Value;
                    }
                }

                // BasePartsLastModify
                {
                    var match = Regex.Match(content, "<partsDerivativeBaseLastModifyDate>(.*?)</partsDerivativeBaseLastModifyDate>", RegexOptions.Singleline);
                    if (match.Success)
                    {
                        this.BasePartsLastModify = DateTime.Parse(match.Groups[1].Value);
                    }
                }

                // BasePartsHash
                {
                    var match = Regex.Match(content, "<partsDerivativeBaseHashValue>(.*?)</partsDerivativeBaseHashValue>", RegexOptions.Singleline);
                    if (match.Success)
                    {
                        this.BasePartsHashValue = match.Groups[1].Value;
                    }
                }

                // totalPaneCount
                {
                    var match = Regex.Match(content, "<totalPaneCount>(.*?)</totalPaneCount>", RegexOptions.Singleline);
                    if (match.Success)
                    {
                        this.TotalPaneCount = Int32.Parse(match.Groups[1].Value);
                    }
                }

                // totalPixelCount
                {
                    var match = Regex.Match(content, "<totalPixelCount>(.*?)</totalPixelCount>", RegexOptions.Singleline);
                    if (match.Success)
                    {
                        this.TotalPixelCount = Int32.Parse(match.Groups[1].Value);
                    }
                }
            }
        }

        public string FilePath { get; internal set; }
        public bool IsLoaded { get { return this.SubScene != null; } }
        public ISubScene SubScene { get; internal set; }

        public string PartsKind { get; internal set; }
        public string Description { get; internal set; }
        public string DescriptionName { get; internal set; }

        public int TotalPaneCount { get; internal set; }
        public int TotalPixelCount { get; internal set; }

        public IEnumerable<string> ControlNames { get; internal set; }

        public bool IsValidPartsSubScene
        {
            get { return !string.IsNullOrEmpty(this.DescriptionName) && !string.IsNullOrEmpty(this.PartsKind); }
        }

        public string BasePartsName { get; private set; }

        public DateTime BasePartsLastModify { get; private set; }

        public string BasePartsHashValue { get; private set; }

        /// <summary>
        /// 基底レイアウトが変更されたかどうか
        /// </summary>
        public bool IsBasePartsSubSceneModified { get; internal set; }
    }
}
