﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Linq;
using System.Collections.Generic;

namespace LECore.Structures
{

    using LECore.Save_Load;
    using LECore.Structures.Core;


    #region PaneParamaterCopyOption

    /// <summary>
    /// コピーオプションクラス
    /// </summary>
    public class PaneParamaterCopyOption
    {
        ParamaterKind _optionFlag = ParamaterKind.All;
        VectorElement _vectorElementOption = VectorElement.All;
        bool _overwriteAnimation = true;
        bool _blackColorAlpha = true;
        bool _copyBaseValue = false;
        bool _copyAnimOnCurrentTagOnly = false;
        bool _skipExtUserDataBasicParamCopy = false;

        string _targetTagName = null;

        public PaneParamaterCopyOption()
        {

        }

        public PaneParamaterCopyOption(PaneParamaterCopyOption instance)
        {
            _optionFlag = instance._optionFlag;
            _vectorElementOption = instance._vectorElementOption;
            _overwriteAnimation = instance._overwriteAnimation;
            _blackColorAlpha = instance._blackColorAlpha;
            _copyBaseValue = instance._copyBaseValue;
            _copyAnimOnCurrentTagOnly = instance._copyAnimOnCurrentTagOnly;
            _skipExtUserDataBasicParamCopy = instance._skipExtUserDataBasicParamCopy;

            _targetTagName = instance._targetTagName;
        }

        #region 有効パラメータ設定の取得関連

        /// <summary>
        /// 有効パラメータフラグを取得、設定します。
        /// </summary>
        public ParamaterKind Option
        {
            get { return _optionFlag; }
            private set { _optionFlag = value; }
        }

        /// <summary>
        /// アニメーションを上書きするか
        /// </summary>
        public bool OverwriteAnimation
        {
            get { return _overwriteAnimation; }
            set { _overwriteAnimation = value; }
        }

        /// <summary>
        /// 黒カラーのアルファが有効か
        /// </summary>
        public bool BlackColorAlpha
        {
            get { return _blackColorAlpha; }
            set { _blackColorAlpha = value; }
        }

        /// <summary>
        /// baseValueをコピーするか
        ///
        /// baseValueをコピーすると、分割モード時の区間未選択時のアトリビュート値(基準値)が変更されます
        /// ペインの複製時はbaseValueをコピーするようにしてください
        /// 値の貼り付け等で現在値のみを変更したい場合はbaseValueをコピーしないでください
        /// </summary>
        public bool CopyBaseValue
        {
            get { return _copyBaseValue; }
            set { _copyBaseValue = value; }
        }

        /// <summary>
        /// 選択中のタグについてのみアニメーションをコピーするか
        /// </summary>
        public bool CopyAnimOnCurrentTagOnly
        {
            get { return _copyAnimOnCurrentTagOnly; }
        }
        public void EnableCopyAnimTargetTagOnly(string tagName)
        {
            _copyAnimOnCurrentTagOnly = true;
            _targetTagName = tagName;
        }
        public void DisableCopyAnimTargetTagOnly()
        {
            _copyAnimOnCurrentTagOnly = false;
            _targetTagName = null;
        }

        /// <summary>
        /// 拡張ユーザー情報の基本情報コピーをスキップします
        /// </summary>
        public bool SkipExtUserDataBasicParamCopy
        {
            get { return _skipExtUserDataBasicParamCopy; }
            set { _skipExtUserDataBasicParamCopy = value; }
        }

        /// <summary>
        /// 選択タグ名
        /// </summary>
        public string TargetTagName
        {
            get { return _targetTagName; }
        }

        /// <summary>
        /// オプションフラグを設定します。
        /// </summary>
        public void SetOption( ParamaterKind flag, bool newValue )
        {
            if( newValue )
            {
                EnableOption( flag );
            }
            else
            {
                DisableOption( flag );
            }
        }

        /// <summary>
        /// ベクトル要素オプションを設定します。
        /// </summary>
        public void SetVectorElementOption( VectorElement flag, bool newValue )
        {
            if( newValue )
            {
                _vectorElementOption |= flag;
            }
            else
            {
                VectorElement mask = ~flag;
                _vectorElementOption &= mask;
            }
        }

        /// <summary>
        /// 不正な状態を調整します。
        /// 調整が起こったかどうかを返します。
        /// </summary>
        public bool RepairInvalidState()
        {
            bool bRepaired = false;

            // テクスチャ関連のパラメータが指定された場合は、
            // テクスチャコピーを有効にします。
            if( IsParamaterOptionEnable( ParamaterKind.Animation_TextureSRT ) ||
          IsParamaterOptionEnable( ParamaterKind.Animation_TexturePattern ))
            {
                if( !IsParamaterOptionEnable( ParamaterKind.Material_Textures ) )
                {
                    _optionFlag |= ParamaterKind.Material_Textures;
                    bRepaired |= true;
                }
            }
            return bRepaired;
        }

        /// <summary>
        /// パラメータオプションが有効か判定します。
        /// </summary>
        public bool IsParamaterOptionEnable( ParamaterKind flag )
        {
            return ( _optionFlag & flag ) != 0;
        }

        /// <summary>
        /// ベクトルオプションが有効か判定します。
        /// </summary>
        public bool IsVectorElementEnable( VectorElement flag )
        {
            return ( _vectorElementOption & flag ) != 0;
        }

        /// <summary>
        /// パラメータオプションを設定します。
        /// </summary>
        public void EnableOption( ParamaterKind flag )
        {
            _optionFlag |= flag;

        }

        /// <summary>
        /// パラメータオプションを無効にします。
        /// </summary>
        public void DisableOption( ParamaterKind flag )
        {
            ParamaterKind mask = ~flag;
            _optionFlag &= mask;
        }



        #endregion 有効パラメータ設定の取得関連
    }


    #endregion PaneParamaterCopyOption


    /// <summary>
    /// Paneの複製を行うクラスです。
    /// 複製するパラメータを指定して、
    /// 部分的にコピーを行うことができます。
    /// </summary>
    public class PaneParamaterPaster
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        private PaneParamaterPaster()
        {

        }

        #region ペイン基本パラメータ
        /// <summary>
        /// ペイン基本パラメータ をコピーします。
        /// </summary>
        static void PasteParamatersPaneBase_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane )
        {
            Pane dstPane = dstIPane as Pane;

            //----------- ペイン基本パラメータの場合
            // 位置
            if ( option.IsParamaterOptionEnable( ParamaterKind.Position ) )
            {
                dstPane.Trans = ParamaterPasterCommon.CopyFVec3(
                    option, dstPane.Trans, srcPane.Trans );
            }
            // サイズ
            if( option.IsParamaterOptionEnable( ParamaterKind.Size ) )
            {
                dstPane.Size = ParamaterPasterCommon.CopyFVec3(
                    option, dstPane.Size, srcPane.Size );
            }
            // スケール
            if( option.IsParamaterOptionEnable( ParamaterKind.Scale ) )
            {
                dstPane.Scale = ParamaterPasterCommon.CopyFVec3(
                    option, dstPane.Scale, srcPane.Scale );
            }
            // 回転
            if( option.IsParamaterOptionEnable( ParamaterKind.Rotate ) )
            {
                dstPane.RotAng = ParamaterPasterCommon.CopyFVec3(
                    option, dstPane.RotAng, srcPane.RotAng );
            }
            // ユーザコメント
            if( option.IsParamaterOptionEnable( ParamaterKind.UserComment ) )
            {
                dstPane.UserCommentString = srcPane.UserCommentString;
            }
            // ユーザデータ
            if( option.IsParamaterOptionEnable( ParamaterKind.UserData ) )
            {
                dstPane.UserData = srcPane.UserData;
            }
            // 拡張ユーザデータ
            if (!option.SkipExtUserDataBasicParamCopy &&
                (option.IsParamaterOptionEnable(ParamaterKind.UserDataEx) || option.IsParamaterOptionEnable(ParamaterKind.Animation_ExtUserData)))
            {
                dstPane.UserDataHolder.RemoveUserDataElementAll();
                foreach (IUserDataElement ude in srcPane.IUserDataHolder.UserDataElementSet)
                {
                    dstPane.UserDataHolder.AddUserDataElement(ude.Name, ude.UserDataKind, ude.Overwrite, ude.Value);
                }

                // 拡張ユーザーデータはUndo/Redoでデータセットの差し替えが発生するため、このタイミングでbaseValueの書き換えを行う必要がある
                if (option.CopyBaseValue)
                {
                    IAnmAttribute[] dstAttrSet = PaneAnimationHelper.GetAnimationTargetAttributeSet(dstIPane, ParamaterKind.Animation_ExtUserData);
                    IAnmAttribute[] srcAttrSet = PaneAnimationHelper.GetAnimationTargetAttributeSet(srcPane, ParamaterKind.Animation_ExtUserData);
                    CopyBaseValue(srcAttrSet, dstAttrSet, false);
                }

                // 拡張ユーザー情報の再構築をする際にアニメーションキーが失われてしまうため、明示的にコピーする
                PaneParamaterCopyOption udeCopyOption = new PaneParamaterCopyOption();
                udeCopyOption.DisableOption(ParamaterKind.All); // 一旦すべてのオプションをOFFにする
                udeCopyOption.EnableOption(ParamaterKind.Animation_ExtUserData); // 拡張ユーザー情報のアニメをコピーする
                udeCopyOption.SkipExtUserDataBasicParamCopy = true; // 再帰呼び出しを抑制する
                PasteParamaters(udeCopyOption, dstIPane, srcPane);
            }

            // 階層
            if( option.IsParamaterOptionEnable( ParamaterKind.Hierarchy ) )
            {
                // TODO:
            }
            // グループ
            if( option.IsParamaterOptionEnable( ParamaterKind.Group ) )
            {
                // TODO:
            }
            // 透明度
            if( option.IsParamaterOptionEnable( ParamaterKind.Transparency ) )
            {
                dstPane.Transparency = srcPane.Transparency;
            }
            // その他フラグ
            if( option.IsParamaterOptionEnable( ParamaterKind.PaneBasicFlags ) )
            {
                dstPane.BasePosTypeH = srcPane.BasePosTypeH;
                dstPane.BasePosTypeV = srcPane.BasePosTypeV;
                dstPane.ParentBasePosTypeH = srcPane.ParentBasePosTypeH;
                dstPane.ParentBasePosTypeV = srcPane.ParentBasePosTypeV;
                dstPane.Visible = srcPane.Visible;
                dstPane.ExtUserDataAnimEnabled = srcPane.ExtUserDataAnimEnabled;
                dstPane.InfluenceChildrenTransparency = srcPane.InfluenceChildrenTransparency;
                dstPane.LocationAdjust = srcPane.LocationAdjust;

                dstPane.IsAlignmentHorizontalMarginEnabled = srcPane.IsAlignmentHorizontalMarginEnabled;
                dstPane.AlignmentHorizontalMargin = srcPane.AlignmentHorizontalMargin;
                dstPane.AlignmentVerticalMargin = srcPane.AlignmentVerticalMargin;
                dstPane.IsAlignmentIgnore = srcPane.IsAlignmentIgnore;
                dstPane.IsAlignmentVerticalMarginEnabled = srcPane.IsAlignmentVerticalMarginEnabled;
                dstPane.AlignmentVerticalMargin = srcPane.AlignmentVerticalMargin;

                dstPane.PartsMagnifyInfluence = srcPane.PartsMagnifyInfluence;
            }

            bool hasUserDataAnm = false;

            // 拡張ユーザー情報アニメーションを持つか調べる
            foreach ( IUserDataElement element in srcPane.IUserDataHolder.UserDataElementSet )
            {
                if ( element.IAnmAttribute != null)
                {
                    hasUserDataAnm = true;
                    break;
                }
            }

            // 拡張ユーザー情報アニメーションをコピーするときは強制的に有効フラグをオンにする
            if ( option.IsParamaterOptionEnable( ParamaterKind.Animation_ExtUserData ) && hasUserDataAnm )
            {
                dstPane.ExtUserDataAnimEnabled = true;
            }
        }
        #endregion ペイン基本パラメータ

        #region マスク
        static void PasteParamatersPaneMask_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            if (option.IsParamaterOptionEnable(ParamaterKind.MaskAll))
            {
                Pane dstPane = dstIPane as Pane;
                IMask srcMask = srcPane.IMask;
                Mask dstMask = dstPane.IMask as Mask;

                dstMask.IsMaskEnabled = srcMask.IsMaskEnabled;
                dstMask.IsMaskWithPane = srcMask.IsMaskWithPane;
                PaneParamaterCopyOption material_option = new PaneParamaterCopyOption();
                material_option.SetOption(ParamaterKind.All, false);
                material_option.SetOption(ParamaterKind.Material_Textures, true);
                MaterialParamaterPaster.CopyMaterialParamater(material_option, dstMask.InternalMaterial, srcMask.InternalMaterial);
                dstMask.UpdateForChangeOfInternalMaterial();
            }
        }
        #endregion マスク

        #region ドロップシャドウ効果
        static void PasteParamatersPaneDropShadow_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            if (option.IsParamaterOptionEnable(ParamaterKind.DropShadowAll))
            {
                Pane dstPane = dstIPane as Pane;
                IDropShadow srcDropShadow = srcPane.IDropShadow;
                DropShadow dstMask = dstPane.IDropShadow as DropShadow;

                dstMask.IsKnockoutEnabled = srcDropShadow.IsKnockoutEnabled;
                dstMask.IsOnlyEffectEnabled = srcDropShadow.IsOnlyEffectEnabled;
                dstMask.IsStaticRenderingEnabled = srcDropShadow.IsStaticRenderingEnabled;
                dstMask.IsStrokeEnabled = srcDropShadow.IsStrokeEnabled;
                dstMask.strokeSize = srcDropShadow.strokeSize;
                dstMask.strokeBlendMode = srcDropShadow.strokeBlendMode;
                dstMask.strokeColor = srcDropShadow.strokeColor;
                dstMask.IsOuterGlowEnabled = srcDropShadow.IsOuterGlowEnabled;
                dstMask.outerGlowBlendMode = srcDropShadow.outerGlowBlendMode;
                dstMask.outerGlowColor = srcDropShadow.outerGlowColor;
                dstMask.outerGlowSpread = srcDropShadow.outerGlowSpread;
                dstMask.outerGlowSize = srcDropShadow.outerGlowSize;
                dstMask.IsDropShadowEnabled = srcDropShadow.IsDropShadowEnabled;
                dstMask.dropShadowBlendMode = srcDropShadow.dropShadowBlendMode;
                dstMask.dropShadowColor = srcDropShadow.dropShadowColor;
                dstMask.dropShadowAngle = srcDropShadow.dropShadowAngle;
                dstMask.dropShadowDistance = srcDropShadow.dropShadowDistance;
                dstMask.dropShadowSpread = srcDropShadow.dropShadowSpread;
                dstMask.dropShadowSize = srcDropShadow.dropShadowSize;
            }
        }
        #endregion マスク

        #region ピクチャペインパラメータ

        /// <summary>
        /// ピクチャペインパラメータをコピーします。
        /// </summary>
        static void PasteParamatersPicture_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane )
        {
            Pane dstPane = dstIPane as Pane;

            //----------- Picture ペインの場合
            if( dstPane.PaneKind == PaneKind.Picture &&
                srcPane.PaneKind == PaneKind.Picture )
            {
                Picture dstPicture = dstPane.Picture;
                IPicture srcPicture = srcPane.IPicture;
                Debug.Assert( dstPicture != null );

                // 形状
                if (option.IsParamaterOptionEnable(ParamaterKind.PictureShapeType))
                {
                    dstPicture.RoundRadius = srcPicture.RoundRadius;
                    dstPicture.RoundSlice = srcPicture.RoundSlice;
                    dstPicture.CircleSlice = srcPicture.CircleSlice;
                    dstPicture.SphereSlice = srcPicture.SphereSlice;
                    dstPicture.SphereStack = srcPicture.SphereStack;
                    dstPicture.ShapeType = srcPicture.ShapeType;
                }

                // 頂点カラー座標
                if( option.IsParamaterOptionEnable( ParamaterKind.VtxColors ) )
                {
                    dstPicture.VtxCol_LB = ParamaterPasterCommon.CopyRGBA( option, dstPicture.VtxCol_LB, srcPicture.VtxCol_LB );
                    dstPicture.VtxCol_LT = ParamaterPasterCommon.CopyRGBA( option, dstPicture.VtxCol_LT, srcPicture.VtxCol_LT );
                    dstPicture.VtxCol_RB = ParamaterPasterCommon.CopyRGBA( option, dstPicture.VtxCol_RB, srcPicture.VtxCol_RB );
                    dstPicture.VtxCol_RT = ParamaterPasterCommon.CopyRGBA( option, dstPicture.VtxCol_RT, srcPicture.VtxCol_RT );
                }

                // テクスチャ座標
                if( option.IsParamaterOptionEnable( ParamaterKind.TexCoords ) )
                {
                    dstPicture.NumTexCoord = srcPicture.NumTexCoord;
                    for( int i = 0 ; i < srcPicture.NumTexCoord ; i++ )
                    {
                        dstPicture.SetTexCoord4( srcPicture.GetTexCoord4( i ), i );
                    }
                }

                // 角丸関連情報
                PasteParamatersPaneProceduralShape_(option, dstPicture.IProceduralShape, srcPicture.IProceduralShape);

                // その他
                if( option.IsParamaterOptionEnable( ParamaterKind.PictureEtc ) )
                {
                    // 詳細なマテリアルを使用するか？
                    dstPicture.UseDetailedMaterial = srcPicture.UseDetailedMaterial;
                }
            }
        }

        #endregion ピクチャペインパラメータ

        #region 角丸データ
        static void PasteParamatersPaneProceduralShape_(
            PaneParamaterCopyOption option,
            IProceduralShape dstIProceduralShape,
            IProceduralShape srcProceduralShape)
        {
            if (option.IsParamaterOptionEnable(ParamaterKind.ProceduralShapeAll))
            {
                ProceduralShape dstProceduralShape = dstIProceduralShape as ProceduralShape;

                dstProceduralShape.IsProceduralShapeEnabled = srcProceduralShape.IsProceduralShapeEnabled;

                dstProceduralShape.Exp = srcProceduralShape.Exp;
                dstProceduralShape.Radius = srcProceduralShape.Radius;
                dstProceduralShape.IsInnerStrokeEnabled = srcProceduralShape.IsInnerStrokeEnabled;
                dstProceduralShape.InnerStrokeSize = srcProceduralShape.InnerStrokeSize;
                dstProceduralShape.InnerStrokeBlendMode = srcProceduralShape.InnerStrokeBlendMode;
                dstProceduralShape.InnerStrokeColor = srcProceduralShape.InnerStrokeColor;
                dstProceduralShape.IsInnerShadowEnabled = srcProceduralShape.IsInnerShadowEnabled;
                dstProceduralShape.InnerShadowBlendMode = srcProceduralShape.InnerShadowBlendMode;
                dstProceduralShape.InnerShadowColor = srcProceduralShape.InnerShadowColor;
                dstProceduralShape.InnerShadowAngle = srcProceduralShape.InnerShadowAngle;
                dstProceduralShape.InnerShadowDistance = srcProceduralShape.InnerShadowDistance;
                dstProceduralShape.InnerShadowSize = srcProceduralShape.InnerShadowSize;
                dstProceduralShape.InnerShadowType = srcProceduralShape.InnerShadowType;
                dstProceduralShape.IsColorOverayEnabled = srcProceduralShape.IsColorOverayEnabled;
                dstProceduralShape.ColorOverlayBlendMode = srcProceduralShape.ColorOverlayBlendMode;
                dstProceduralShape.ColorOverlayColor = srcProceduralShape.ColorOverlayColor;
                dstProceduralShape.IsGradationOverayEnabled = srcProceduralShape.IsGradationOverayEnabled;
                dstProceduralShape.GradationOverlayBlendMode = srcProceduralShape.GradationOverlayBlendMode;
                dstProceduralShape.GradationOverlayControlPointCount = srcProceduralShape.GradationOverlayControlPointCount;
                for(int i = 0; i < srcProceduralShape.GradationOverlayControlPointCount;++i)
                {
                    dstProceduralShape.SetGradationOverlayColor(i, srcProceduralShape.GetGradationOverlayColor(i));
                    dstProceduralShape.SetGradationOverlayControlPoint(i, srcProceduralShape.GetGradationOverlayControlPoint(i));
                }
                dstProceduralShape.GradationOverlayAngle = srcProceduralShape.GradationOverlayAngle;
                dstProceduralShape.IsOuterShadowEnabled = srcProceduralShape.IsOuterShadowEnabled;
                dstProceduralShape.OuterShadowBlendMode = srcProceduralShape.OuterShadowBlendMode;
                dstProceduralShape.OuterShadowColor = srcProceduralShape.OuterShadowColor;
                dstProceduralShape.OuterShadowAngle = srcProceduralShape.OuterShadowAngle;
                dstProceduralShape.OuterShadowDistance = srcProceduralShape.OuterShadowDistance;
                dstProceduralShape.OuterShadowSize = srcProceduralShape.OuterShadowSize;
                dstProceduralShape.OuterShadowType = srcProceduralShape.OuterShadowType;
            }
        }
        #endregion


        #region TextBoxペイン
        /// <summary>
        /// テキストボックスペインをコピーします。
        /// </summary>
        static void PasteParamatersTextBox_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane )
        {
            Pane dstPane = dstIPane as Pane;

            if( dstPane.PaneKind == PaneKind.Textbox &&
                srcPane.PaneKind == PaneKind.Textbox )
            {
                TextBox dstTextBox = dstPane.TextBox;
                ITextBox srcTextBox = srcPane.ITextBox;
                Debug.Assert(dstTextBox != null);

                // フォント名
                if (option.IsParamaterOptionEnable(ParamaterKind.FontFileName))
                {
                    // Debug.Assert( srcTextBox.ILEFont is LEFont );
                    dstTextBox.FontName = srcTextBox.FontName;
                }
                // 内容文字列
                if (option.IsParamaterOptionEnable(ParamaterKind.ContentsText))
                {
                    dstTextBox.ContentsText = srcTextBox.ContentsText;
                    dstTextBox.TextID = srcTextBox.TextID;
                }
                // サイズ
                if (option.IsParamaterOptionEnable(ParamaterKind.FontSize))
                {
                    // フォントサイズ
                    dstTextBox.FontSize = ParamaterPasterCommon.CopyFVec2(option, dstTextBox.FontSize, srcTextBox.FontSize);
                }
                // 間隔
                if (option.IsParamaterOptionEnable(ParamaterKind.FontBlank))
                {
                    // フォント間隔
                    FVec2 srcBlank = new FVec2(srcTextBox.CharSpace, srcTextBox.LineSpace);
                    FVec2 dstBlank = new FVec2(dstTextBox.CharSpace, dstTextBox.LineSpace);
                    dstBlank = ParamaterPasterCommon.CopyFVec2(option, dstBlank, srcBlank);
                    dstTextBox.CharSpace = dstBlank.X;
                    dstTextBox.LineSpace = dstBlank.Y;
                }
                // 文字列位置
                if (option.IsParamaterOptionEnable(ParamaterKind.TextBasePosition))
                {
                    dstTextBox.BasePosTypeH = srcTextBox.BasePosTypeH;
                    dstTextBox.BasePosTypeV = srcTextBox.BasePosTypeV;
                    dstTextBox.LineAlignment = srcTextBox.LineAlignment;
                }
                // 文字色
                if (option.IsParamaterOptionEnable(ParamaterKind.FontColors))
                {
                    dstTextBox.TopColor = ParamaterPasterCommon.CopyRGBA(option, dstTextBox.TopColor, srcTextBox.TopColor);
                    dstTextBox.BottomColor = ParamaterPasterCommon.CopyRGBA(option, dstTextBox.BottomColor, srcTextBox.BottomColor);
                }
                // 影設定
                if (option.IsParamaterOptionEnable(ParamaterKind.TextShadow))
                {
                    dstTextBox.ShadowEnabled = srcTextBox.ShadowEnabled;
                    dstTextBox.ShadowOffset = srcTextBox.ShadowOffset;
                    dstTextBox.ShadowScale = srcTextBox.ShadowScale;

                    dstTextBox.ShadowItalicFactor = srcTextBox.ShadowItalicFactor;

                    dstTextBox.ShadowTopColor = ParamaterPasterCommon.CopyRGBA(option, dstTextBox.ShadowTopColor, srcTextBox.ShadowTopColor);
                    dstTextBox.ShadowBottomColor = ParamaterPasterCommon.CopyRGBA(option, dstTextBox.ShadowBottomColor, srcTextBox.ShadowBottomColor);
                    dstTextBox.ShadowBlackColor = ParamaterPasterCommon.CopyRGBA(option, dstTextBox.ShadowBlackColor, srcTextBox.ShadowBlackColor);
                    dstTextBox.ShadowWhiteColor = ParamaterPasterCommon.CopyRGBA(option, dstTextBox.ShadowWhiteColor, srcTextBox.ShadowWhiteColor);
                }

                // 文字変換
                if (option.IsParamaterOptionEnable(ParamaterKind.PerCharTransformParam))
                {
                    // 文字変換
                    dstTextBox.PerCharacterTransformTimeOffset = srcTextBox.PerCharacterTransformTimeOffset;
                    dstTextBox.PerCharacterTransformTimeWidth = srcTextBox.PerCharacterTransformTimeWidth;
                    dstTextBox.PerCharacterTransformLoopType = srcTextBox.PerCharacterTransformLoopType;
                    dstTextBox.PerCharacterTransformOriginV = srcTextBox.PerCharacterTransformOriginV;
                    dstTextBox.PerCharacterTransformOriginVOffset = srcTextBox.PerCharacterTransformOriginVOffset;
                    dstTextBox.PerCharacterTransformSplitByCharWidth = srcTextBox.PerCharacterTransformSplitByCharWidth;
                    dstTextBox.PerCharacterTransformAutoShadowAlpha = srcTextBox.PerCharacterTransformAutoShadowAlpha;
                    dstTextBox.PerCharacterTransformOriginToCenter = srcTextBox.PerCharacterTransformOriginToCenter;
                    dstTextBox.IsPerCharTransformEnabled = srcTextBox.IsPerCharTransformEnabled;

                    dstTextBox.IsPerCharTransformFixSpaceEnabled = srcTextBox.IsPerCharTransformFixSpaceEnabled;
                    dstTextBox.PerCharTransformFixSpaceWidth = srcTextBox.PerCharTransformFixSpaceWidth;
                    dstTextBox.PerCharTransformFixSpaceOrigin = srcTextBox.PerCharTransformFixSpaceOrigin;

                    dstTextBox.PerCharTransformEasyCircleBeginAngle = srcTextBox.PerCharTransformEasyCircleBeginAngle;
                    dstTextBox.PerCharTransformEasyCircleEndAngle = srcTextBox.PerCharTransformEasyCircleEndAngle;
                    dstTextBox.PerCharTransformEasyCircleRadiusX = srcTextBox.PerCharTransformEasyCircleRadiusX;
                    dstTextBox.PerCharTransformEasyCircleRadiusY = srcTextBox.PerCharTransformEasyCircleRadiusY;
                    dstTextBox.PerCharTransformEasyCircleDivisionNumber = srcTextBox.PerCharTransformEasyCircleDivisionNumber;
                    dstTextBox.PerCharTransformEasyCircleCharAngle = srcTextBox.PerCharTransformEasyCircleCharAngle;

                    // 文字毎のアニメーションの簡単設定
                    if(srcTextBox.EasySettingOfPerCharTransform != null)
                    {
                        dstTextBox.EasySettingOfPerCharTransform = new EasySettingOfPerCharTransform[srcTextBox.EasySettingOfPerCharTransform.Length];
                        for (int i = 0; i < srcTextBox.EasySettingOfPerCharTransform.Length; i++)
                        {
                            dstTextBox.EasySettingOfPerCharTransform[i] = srcTextBox.EasySettingOfPerCharTransform[i];
                        }
                    }
                }

                // その他のパラメータ
                if (option.IsParamaterOptionEnable(ParamaterKind.TextExtra))
                {
                    // 斜体係数
                    dstTextBox.ItalicFactor = srcTextBox.ItalicFactor;

                    // 縁OFF
                    dstTextBox.InvisibleBorderEnabled = srcTextBox.InvisibleBorderEnabled;
                    // 二度書き縁
                    dstTextBox.DoubleDrawnBorderEnabled = srcTextBox.DoubleDrawnBorderEnabled;

                    // 文字列長さの指定
                    dstTextBox.SpecifiedStringLength = srcTextBox.SpecifiedStringLength;
                    dstTextBox.AllocStringLength = srcTextBox.AllocStringLength;

                    // 中央揃えのにじみ防止機能を有効にする
                    dstTextBox.CenterCeilingEnabled = srcTextBox.CenterCeilingEnabled;

                    // 左側の文字を前面に描画する
                    dstTextBox.DrawFromRightToLeft = srcTextBox.DrawFromRightToLeft;

                    // 拡張タグの解釈の有効無効
                    dstTextBox.IsExtendedTagEnabled = srcTextBox.IsExtendedTagEnabled;

                    // 行ごとの幅とオフセット
                    {
                        dstTextBox.LineWidthOffsetEnabled = srcTextBox.LineWidthOffsetEnabled;

                        if(srcTextBox.LineWidth != null)
                        {
                            dstTextBox.LineWidth = new float[srcTextBox.LineWidth.Length];
                            Array.Copy(srcTextBox.LineWidth, dstTextBox.LineWidth, srcTextBox.LineWidth.Length);
                        }

                        if(srcTextBox.LineOffset != null)
                        {
                            dstTextBox.LineOffset = new float[srcTextBox.LineOffset.Length];
                            Array.Copy(srcTextBox.LineOffset, dstTextBox.LineOffset, srcTextBox.LineOffset.Length);
                        }
                    }
                }

                dstTextBox.UpdateFontReference();
            }
        }
        #endregion TextBoxペイン

        #region Windowペインパラメータ
        static private ISubScene ChoiceOwnerSubScene_(IPane srcPane, IPane dstPane)
        {
            ISubScene ownerSubScene = srcPane?.OwnerSubScene != null ? srcPane?.OwnerSubScene : dstPane?.OwnerSubScene;
            if(ownerSubScene == null)
            {
                Debug.Assert(false);
                ownerSubScene = LayoutEditorCore.Scene.CurrentISubScene;
            }

            return ownerSubScene;
        }

        static void PasteParamatersWindow_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            Pane dstPane = dstIPane as Pane;

            //----------- Window ペインの場合
            if (dstPane.PaneKind == PaneKind.Window && srcPane.PaneKind == PaneKind.Window)
            {
                LEWindow dstWindow = dstPane.ILEWindow as LEWindow;
                ILEWindow srcWindow = srcPane.ILEWindow;
                Debug.Assert(dstWindow != null);

                // ウインドウ種類
                dstWindow.WindowKind = srcWindow.WindowKind;
                // 頂点カラーの種類
                dstWindow.UseVtxColorForAllWindow = srcWindow.UseVtxColorForAllWindow;
                // コンテンツを書くか
                dstWindow.NotDrawContent = srcWindow.NotDrawContent;

                // 使用するマテリアル数
                if (option.IsParamaterOptionEnable(ParamaterKind.WindowFrame))
                {
                    // カスタムフレームサイズ
                    dstWindow.FrameSize = srcWindow.FrameSize;

                    // テクスチャ数
                    dstWindow.NumTexture = srcWindow.NumTexture;

                    foreach (LEWindowPartsID partsID in Enum.GetValues(typeof(LEWindowPartsID)))
                    {
                        // フリップ設定
                        LEWindowFrameFlipRotate flipRot =
                            srcWindow.GetMaterialFlipRotateState(partsID);
                        dstWindow.SetMaterialFlipState(partsID, flipRot);

                        // 詳細なマテリアルを使用するか？
                        dstWindow.SetUseDetailedMaterialFlag(
                            partsID,
                            srcWindow.GetUseDetailedMaterialFlag(partsID));
                    }

                    //
                    dstWindow.UseLTMaterial = srcWindow.UseLTMaterial;
                }

                // コンテンツのパラメータ
                if (option.IsParamaterOptionEnable(ParamaterKind.WindowContentsParamaters))
                {
                    // テクスチャ座標数
                    dstWindow.NumContentsTexCoord = srcWindow.NumContentsTexCoord;

                    // テクスチャ座標値
                    for (int i = 0; i < dstWindow.NumContentsTexCoord; i++)
                    {
                        TexCoord4 texcoord4 = srcWindow.GetContentsTexCoord4(i);
                        dstWindow.SetContentsTexCood4(i, texcoord4);
                    }

                    // 頂点色
                    dstWindow.VtxCol_LT = srcWindow.VtxCol_LT;
                    dstWindow.VtxCol_RT = srcWindow.VtxCol_RT;
                    dstWindow.VtxCol_LB = srcWindow.VtxCol_LB;
                    dstWindow.VtxCol_RB = srcWindow.VtxCol_RB;

                    // コンテンツ拡張量
                    dstWindow.ContentsInflation = srcWindow.ContentsInflation;
                }

                // アニメーション
                if (option.IsParamaterOptionEnable(ParamaterKind.Animation_Window))
                {
                    ParamaterKind kind = ParamaterKind.Animation_Window;
                    IAnmAttribute[] dstAttrSet = PaneAnimationHelper.GetAnimationTargetAttributeSet(dstIPane, kind);
                    IAnmAttribute[] srcAttrSet = PaneAnimationHelper.GetAnimationTargetAttributeSet(srcPane, kind);

                    CopyAttribute_(
                        srcAttrSet,
                        dstAttrSet,
                        ChoiceOwnerSubScene_(srcPane, dstIPane),
                        option.OverwriteAnimation,
                        option.CopyAnimOnCurrentTagOnly,
                        option.TargetTagName);
                }
            }
        }

        #endregion Windowペインパラメータ

        #region Capture ペインパラメータ
        static void PasteParamatersCapture_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            Pane dstPane = dstIPane as Pane;

            //----------- Capture ペインの場合
            if (dstPane.PaneKind == PaneKind.Capture && srcPane.PaneKind == PaneKind.Capture)
            {
                Capture dstCapture = dstPane.ICapture as Capture;
                ICapture srcCapture = srcPane.ICapture;
                Debug.Assert(dstCapture != null);
            }
        }

        #endregion Capture ペインパラメータ

        #region Alignment ペインパラメータ
        static void PasteParamatersAlignment_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            Pane dstPane = dstIPane as Pane;

            if (dstPane.PaneKind == PaneKind.Alignment && srcPane.PaneKind == PaneKind.Alignment)
            {
                Alignment dstCapture = dstPane.IAlignment as Alignment;

                if (option.IsParamaterOptionEnable(ParamaterKind.AlignmentAll))
                {
                    dstCapture.IsExtendEdgeHorizontalEnabled = srcPane.IAlignment.IsExtendEdgeHorizontalEnabled;
                    dstCapture.DefaultHorizontalMargin = srcPane.IAlignment.DefaultHorizontalMargin;
                    dstCapture.Horizontal = srcPane.IAlignment.Horizontal;
                    dstCapture.IsAlignmentHorizontalEnabled = srcPane.IAlignment.IsAlignmentHorizontalEnabled;

                    dstCapture.IsExtendEdgeVerticalEnabled = srcPane.IAlignment.IsExtendEdgeVerticalEnabled;
                    dstCapture.DefaultVerticalMargin = srcPane.IAlignment.DefaultVerticalMargin;
                    dstCapture.Vertical = srcPane.IAlignment.Vertical;
                    dstCapture.IsAlignmentVerticalEnabled = srcPane.IAlignment.IsAlignmentVerticalEnabled;
                }
            }
        }

        #endregion Alignment ペインパラメータ

        #region Scissor ペインパラメータ
        static void PasteParamatersScissor_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            Pane dstPane = dstIPane as Pane;

            if (dstPane.PaneKind == PaneKind.Scissor && srcPane.PaneKind == PaneKind.Scissor)
            {
                Scissor dstScissor = dstPane.IScissor as Scissor;

                if (option.IsParamaterOptionEnable(ParamaterKind.ScissorAll))
                {
                    // 今のところパラメータはない
                }
            }
        }
        #endregion Scissor ペインパラメータ

        /// <summary>
        /// 部品のコピー
        /// </summary>
        static void PasteParamatersParts_(
            PaneParamaterCopyOption option,
            IPane dstPane,
            IPane srcPane)
        {
            //----------- Parts ペインの場合
            if (dstPane.PaneKind == PaneKind.Parts && srcPane.PaneKind == PaneKind.Parts)
            {
                var dstParts = dstPane.IPartsLayout as PartsLayout;
                var srcParts = srcPane.IPartsLayout;
                if (option.IsParamaterOptionEnable(ParamaterKind.PartsAll))
                {
                    // 上書きを実行しているペインは、再帰的にコピーを実行する
                    dstParts.SetAllProperties(srcParts.PartsPropaerties);

                    dstParts.Magnify = srcParts.Magnify;
                    dstParts.SizeConstraint = srcParts.SizeConstraint;
                }
            }
        }

#region マテリアル関連

        /// <summary>
        /// すべてのマテリアルをコピーします。
        /// </summary>
        static public void CopyAllMatareials_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            Pane dstPane = dstIPane as Pane;

            // すべてのマテリアルについて...
            for( int i = 0 ; i < srcPane.IMaterial.Length ; i++ )
            {
                Material dstMat = dstPane.Material[i];
                IMaterial srcMat = srcPane.IMaterial[i];

                MaterialParamaterPaster.CopyMaterialParamater( option, dstMat, srcMat );
            }

            // 詳細情報
            {
                IRevHWMaterialHolder srcHolder = srcPane as IRevHWMaterialHolder;

                IRevHWMaterial[] dstMatSet = PaneHelper.GetRevHWMatFromPane( dstPane );
                IRevHWMaterial[] srcMatSet = PaneHelper.GetRevHWMatFromPane( srcPane );
                Debug.Assert( dstMatSet.Length == srcMatSet.Length );

                for( int i = 0 ; i < dstMatSet.Length ; i++ )
                {
                    MaterialParamaterPaster.CopyMaterialDetailParamaters(
                        option,
                        dstMatSet[i],
                        srcMatSet[i],
                        option.IsParamaterOptionEnable(ParamaterKind.Material_Details));
                }
            }
        }

        /// <summary>
        /// 種類違いのマテリアルをコピーします。
        /// </summary>
        static public void CopyDifferentKindMaterials_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            // 種類ごとにコピー方法が異なる
            switch (srcPane.PaneKind)
            {
                case PaneKind.Picture: CopyMaterialsFromPicturePane_(option, dstIPane, srcPane); break;
                case PaneKind.Textbox: CopyMaterialsFromTextBoxPane_(option, dstIPane, srcPane); break;
                case PaneKind.Window:  CopyMaterialsFromWindowPane_(option, dstIPane, srcPane); break;
            }
        }

        /// <summary>
        /// ピクチャペインのマテリアルをコピーします。
        /// </summary>
        static private void CopyMaterialsFromPicturePane_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            CopyMaterialsSingleToMulti_(option, dstIPane, srcPane);
        }

        /// <summary>
        /// テキストボックスペインのマテリアルをコピーします。
        /// </summary>
        static private void CopyMaterialsFromTextBoxPane_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            CopyMaterialsSingleToMulti_(option, dstIPane, srcPane);
        }

        /// <summary>
        /// ウインドウペインのマテリアルをコピーします。
        /// </summary>
        static private void CopyMaterialsFromWindowPane_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            CopyMaterialsMultiToSingle_(option, dstIPane, srcPane);
        }

        /// <summary>
        /// 単数のマテリアルを複数のマテリアルにコピーします。
        /// </summary>
        static private void CopyMaterialsSingleToMulti_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            Pane dstPane = dstIPane as Pane;

            // マテリアル[0]をターゲットの全マテリアルにコピーする
            for (int i = 0; i < dstPane.IMaterial.Length; i++)
            {
                Material dstMat = dstPane.Material[i];
                IMaterial srcMat = srcPane.IMaterial[0];

                MaterialParamaterPaster.CopyMaterialParamater(option, dstMat, srcMat);
            }

            // マテリアルカラー/インダイレクト/アルファテスト
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                if (ParamaterKindHelper.MaterialAnimationKindSet[1].Contains(kind) &&
                    option.IsParamaterOptionEnable(kind))
                {
                    IAnmAttribute[] dstAttrSet = PaneAnimationHelper.GetAnimationTargetAttributeSet(dstIPane, kind);
                    IAnmAttribute[] srcAttrSet = PaneAnimationHelper.GetAnimationTargetAttributeSet(srcPane, kind);

                    if (kind == ParamaterKind.Animation_IndirectTextureSRT)
                    {
                        CopyAttribute_(
                            srcAttrSet,
                            dstAttrSet,
                            ChoiceOwnerSubScene_(srcPane, dstIPane),
                            option.OverwriteAnimation,
                            option.CopyAnimOnCurrentTagOnly,
                            option.TargetTagName);
                    }
                    else
                    {
                        CopyAttribute_(
                            srcAttrSet[0],
                            dstAttrSet,
                            ChoiceOwnerSubScene_(srcPane, dstIPane),
                            option.OverwriteAnimation,
                            option.CopyAnimOnCurrentTagOnly,
                            option.TargetTagName);
                    }
                }
            }

            // 詳細情報
            {
                IRevHWMaterialHolder srcHolder = srcPane as IRevHWMaterialHolder;

                IRevHWMaterial[] dstMatSet = PaneHelper.GetRevHWMatFromPane(dstPane);
                IRevHWMaterial[] srcMatSet = PaneHelper.GetRevHWMatFromPane(srcPane);

                for (int i = 0; i < dstMatSet.Length; i++)
                {
                    MaterialParamaterPaster.CopyMaterialDetailParamaters(
                        option,
                        dstMatSet[i],
                        srcMatSet[0],
                        option.IsParamaterOptionEnable(ParamaterKind.Material_Details));
                }
            }
        }

        /// <summary>
        /// 複数のマテリアルを単数のマテリアルにコピーします。
        /// </summary>
        static private void CopyMaterialsMultiToSingle_(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            Pane dstPane = dstIPane as Pane;

            // マテリアル[1]をターゲットにコピーする
            {
                Material dstMat = dstPane.Material[0];
                IMaterial srcMat = srcPane.IMaterial[1];

                MaterialParamaterPaster.CopyMaterialParamater(option, dstMat, srcMat);
            }

            // 詳細情報
            {
                IRevHWMaterialHolder srcHolder = srcPane as IRevHWMaterialHolder;

                IRevHWMaterial[] dstMatSet = PaneHelper.GetRevHWMatFromPane(dstPane);
                IRevHWMaterial[] srcMatSet = PaneHelper.GetRevHWMatFromPane(srcPane);

                MaterialParamaterPaster.CopyMaterialDetailParamaters(
                    option,
                    dstMatSet[0],
                    srcMatSet[1],
                    option.IsParamaterOptionEnable(ParamaterKind.Material_Details));
            }
        }
#endregion マテリアル関連


        /// <summary>
        /// コピー対象のアトリビュート
        /// </summary>
        private class CopyTargetAttrEntry
        {
            /// <summary>
            /// どの位コピー対象として適切かを表す値
            /// </summary>
            public int Score { get; private set; }
            public IAnmAttribute Attr { get; private set; }

            public CopyTargetAttrEntry(int score, IAnmAttribute attr)
            {
                this.Score = score;
                this.Attr = attr;
            }
        }

        /// <summary>
        /// コピー対象アトリビュートを選択します。
        /// </summary>
        static IEnumerable<IAnmAttribute> SelectTargetAttr_(IAnmAttribute srcAttr, IAnmAttribute[] dstAttrSet)
        {
            List<CopyTargetAttrEntry> attrEntrySet = new List<CopyTargetAttrEntry>();
            foreach(IAnmAttribute dstAttr in dstAttrSet )
            {
                // 階層を親方向にたどりながら、一致数をスコアとしてカウントする。
                int score = 0;
                IAnmAttribute tempSrcAttr = srcAttr;
                IAnmAttribute tempDstAttr = dstAttr;
                while( tempSrcAttr != null && tempDstAttr != null )
                {
                    // 異なる名称のノードが発見された ...
                    if( tempSrcAttr.DescriptionName != tempDstAttr.DescriptionName )
                    {
                        break;
                    }

                    score++;
                    tempSrcAttr = tempSrcAttr.OwnerNode;
                    tempDstAttr = tempDstAttr.OwnerNode;
                }

                if(score > 0)
                {
                    attrEntrySet.Add(new CopyTargetAttrEntry(score, dstAttr));
                }
            }

            // スコアが最大のグループをターゲットとして返す。
            if(attrEntrySet.Count > 0)
            {
                attrEntrySet.Sort((lhs, rhs) => rhs.Score - lhs.Score);
                foreach (var sameScoreEntry in attrEntrySet.TakeWhile((entry) => entry.Score == attrEntrySet[0].Score))
                {
                    yield return sameScoreEntry.Attr;
                }
            }

            yield break;
        }

        /// <summary>
        /// アトリビュートをコピーします。
        /// </summary>
        static void CopyAttribute_(
            IAnmAttribute[] srcAttrSet,
            IAnmAttribute[] dstAttrSet,
            ISubScene ownerSubScene,
            bool bOverwriteAnimation,
            bool copyTargetTag,
            string targetTagName)
        {
            for ( int i = 0 ; i < srcAttrSet.Length ; i++ )
            {
                // アトリビュート名で、コピー対象を検索します。
                IAnmAttribute srcAttribute = srcAttrSet[i];
                IEnumerable<IAnmAttribute> dstAttributes = SelectTargetAttr_(srcAttribute, dstAttrSet);

                // コピー対象が発見されれば、コピーします。
                foreach(IAnmAttribute dstAttribute in dstAttributes)
                {
                    ParamaterPasterCommon.CopyAllAnimAttribute(
                        dstAttribute,
                        srcAttribute,
                        ownerSubScene,
                        bOverwriteAnimation,
                        copyTargetTag,
                        targetTagName);
                }
            }
        }

        /// <summary>
        /// アトリビュートをコピーします。
        /// </summary>
        static void CopyAttribute_(
            IAnmAttribute srcAttribute,
            IAnmAttribute[] dstAttrSet,
            ISubScene ownerSubScene,
            bool bOverwriteAnimation,
            bool copyTargetTag,
            string targetTagName)
        {
            // アトリビュート名で、コピー対象を検索します。
            IEnumerable<IAnmAttribute> dstAttributes = SelectTargetAttr_(srcAttribute, dstAttrSet);

            // コピー対象が発見されれば、コピーします。
            foreach (IAnmAttribute dstAttribute in dstAttributes)
            {
                ParamaterPasterCommon.CopyAllAnimAttribute(
                    dstAttribute,
                    srcAttribute,
                    ownerSubScene,
                    bOverwriteAnimation,
                    copyTargetTag,
                    targetTagName);
            }
        }

        /// <summary>
        /// baseValueをコピーします。
        /// srcにキーフレームが存在する場合のみbaseBalueが更新されます
        /// </summary>
        /// <remarks>
        /// カーブにキーが無いアトリビュートに対してCopyBaseValueを単体で呼び出すと
        /// valueとbaseValueの値が乖離する可能性があります
        /// カーブと値のコピーと同時に使用するようにしてください
        /// </remarks>
        static public void CopyBaseValue(
            IAnmAttribute[] srcAttrSet,
            IAnmAttribute[] dstAttrSet,
            bool keyCheck)
        {
            for (int i = 0; i < srcAttrSet.Length; i++)
            {
                // アトリビュート名で、コピー対象を検索します。
                AnmAttribute srcAttribute = srcAttrSet[i] as AnmAttribute;
                if (srcAttribute != null)
                {
                    IEnumerable<IAnmAttribute> dstAttributes = SelectTargetAttr_(srcAttribute, dstAttrSet);

                    // コピー対象が発見されれば、コピーします。
                    foreach (IAnmAttribute dstAttribute in dstAttributes)
                    {
                        CopyBaseValueRecursive_(srcAttribute, dstAttribute, keyCheck);
                    }
                }
            }
        }

        static private void CopyBaseValueRecursive_(
            IAnmAttribute srcIAttribute,
            IAnmAttribute dstIAttribute,
            bool keyCheck)
        {
            AnmAttribute srcAttribute = srcIAttribute as AnmAttribute;
            AnmAttribute dstAttribute = dstIAttribute as AnmAttribute;

            if ((!keyCheck || (keyCheck && srcAttribute.HasAnyKey)) &&
                srcAttribute.NumSubAttribute == 0 && // 末端のみ処理する
                srcAttribute.CanGetOrSetValue())
            {
                // 通常baseValueを変更する場合はvalueも一緒に変更すること
                // ここでは例外的にbaseValueのみを変更する
                object val;
                srcAttribute.GetBaseValue(out val);
                dstAttribute.SetBaseValue(val);
            }

            if (srcAttribute.NumSubAttribute == dstIAttribute.NumSubAttribute)
            {
                for (int i = 0; i < srcAttribute.NumSubAttribute; i++)
                {
                    CopyBaseValueRecursive_(
                        srcAttribute.FindSubAttributeByIdx(i),
                        dstIAttribute.FindSubAttributeByIdx(i),
                        keyCheck);
                }
            }
        }

        /// <summary>
        /// baseValueをコピーします。
        /// </summary>
        static void PasteParamatersBaseValue_(
            IPane dstIPane,
            IPane srcPane)
        {
            // 対象ペインにキーがある場合、PaneParamaterPaster.PasteParamatersで
            // baseValueの更新が行なわれないため、baseValueを明示的にコピーします。
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                // マテリアルのbaseValueコピーはここでは行なわない
                // 拡張ユーザーデータのbaseValueコピーはここでは行なわない
                if (ParamaterKindHelper.MaterialAnimationKindSet[0].Contains(kind) ||
                    ParamaterKindHelper.MaterialAnimationKindSet[1].Contains(kind) ||
                    kind == ParamaterKind.Animation_ExtUserData)
                {
                    continue;
                }


                IAnmAttribute[] dstAttrSet = dstIPane.GetAnimationTargetAttributeSet(kind);
                IAnmAttribute[] srcAttrSet = srcPane.GetAnimationTargetAttributeSet(kind);

                CopyBaseValue(srcAttrSet, dstAttrSet, false);
            }
        }

        /// <summary>
        /// 貼り付けを実行します。
        /// </summary>
        /// <param name="srcPane"></param>
        static public void PasteParamaters(
            PaneParamaterCopyOption option,
            IPane dstIPane,
            IPane srcPane)
        {
            Debug.Assert( dstIPane != null && srcPane != null );

            // メソッド内でパラメータの書き換えが行なわれる場合があるので、
            // オプションのコピーを渡すようにする
            PaneParamaterCopyOption tmpOption = new PaneParamaterCopyOption(option);

            // --------------- baseValue
            if (tmpOption.CopyBaseValue)
            {
                PasteParamatersBaseValue_(dstIPane, srcPane);
            }

            // --------------- ペイン基本
            PasteParamatersPaneBase_( tmpOption, dstIPane, srcPane );

            // マスク
            PasteParamatersPaneMask_(tmpOption, dstIPane, srcPane);

            // ドロップシャドウ
            PasteParamatersPaneDropShadow_(tmpOption, dstIPane, srcPane);

            // --------------- マテリアルのコピー
            // ペイン固有情報の前に、マテリアルをコピーしておく。
            // マテリアルが持つテクスチャ数に制限を受けて、UV座標のコピーが
            // 動作しないなどの問題がおこるためである。
            if (srcPane.IMaterial.Length != 0 && dstIPane.IMaterial.Length != 0)
            {
                if (dstIPane.PaneKind == PaneKind.Textbox || srcPane.PaneKind == PaneKind.Textbox)
                {
                    tmpOption.BlackColorAlpha = false; // テキストボックスが対象の場合黒カラーのアルファを無効にする
                }

                if (srcPane.IMaterial.Length == dstIPane.IMaterial.Length)
                {
                    CopyAllMatareials_(tmpOption, dstIPane, srcPane);
                }
                else
                {
                    CopyDifferentKindMaterials_(tmpOption, dstIPane, srcPane);
                }
            }

            // --------------- ペイン種類ごとのパラメータのコピー
            // 同一種類であれば...
            if ( dstIPane.PaneKind == srcPane.PaneKind )
            {
                // 種類ごとに...
                switch( dstIPane.PaneKind )
                {
                    case PaneKind.Picture: PasteParamatersPicture_( tmpOption, dstIPane, srcPane ); break;
                    case PaneKind.Textbox: PasteParamatersTextBox_( tmpOption, dstIPane, srcPane ); break;
                    case PaneKind.Window: PasteParamatersWindow_( tmpOption, dstIPane, srcPane); break;
                    case PaneKind.Capture: PasteParamatersCapture_( tmpOption, dstIPane, srcPane ); break;
                    case PaneKind.Alignment: PasteParamatersAlignment_(tmpOption, dstIPane, srcPane); break;
                    case PaneKind.Scissor: PasteParamatersScissor_(tmpOption, dstIPane, srcPane); break;
                    case PaneKind.Parts: PasteParamatersParts_( tmpOption, dstIPane, srcPane); break;
                }
            }

            // --------------- 頂点カラーのコピー
            // 頂点カラー座標
            if (tmpOption.IsParamaterOptionEnable(ParamaterKind.AllVtxColor))
            {
                var srcVtx = srcPane.IPaneExParamater as IVertexColor4Holder;
                var dstVtxOrig = dstIPane.IPaneExParamater as IVertexColor4Holder;
                var dstVtx = dstIPane.IPaneExParamater as IVertexColor4Settable;

                if (srcVtx != null && dstVtxOrig != null && dstVtx != null)
                {
                    dstVtx.VtxCol_LB = ParamaterPasterCommon.CopyRGBA(tmpOption, dstVtxOrig.VtxCol_LB, srcVtx.VtxCol_LB);
                    dstVtx.VtxCol_LT = ParamaterPasterCommon.CopyRGBA(tmpOption, dstVtxOrig.VtxCol_LT, srcVtx.VtxCol_LT);
                    dstVtx.VtxCol_RB = ParamaterPasterCommon.CopyRGBA(tmpOption, dstVtxOrig.VtxCol_RB, srcVtx.VtxCol_RB);
                    dstVtx.VtxCol_RT = ParamaterPasterCommon.CopyRGBA(tmpOption, dstVtxOrig.VtxCol_RT, srcVtx.VtxCol_RT);
                }
            }

            // --------------- アニメーション情報のコピー
            // すべての種類のアニメーションについて...
            foreach( ParamaterKind kind in ParamaterKindHelper.AnimationKindSet )
            {
                if( !tmpOption.IsParamaterOptionEnable( kind ) )
                {
                    continue;
                }

                // マテリアルのアニメーションはコピー済みなので除外する
                if (ParamaterKindHelper.MaterialAnimationKindSet[0].Contains(kind) ||
                    ParamaterKindHelper.MaterialAnimationKindSet[1].Contains(kind))
                {
                    continue;
                }

                // コピー指定されていれば...
                IAnmAttribute[] dstAttrSet = PaneAnimationHelper.GetAnimationTargetAttributeSet(dstIPane, kind);
                IAnmAttribute[] srcAttrSet = PaneAnimationHelper.GetAnimationTargetAttributeSet(srcPane, kind);

                CopyAttribute_(
                    srcAttrSet,
                    dstAttrSet,
                    ChoiceOwnerSubScene_(srcPane, dstIPane),
                    tmpOption.OverwriteAnimation,
                    tmpOption.CopyAnimOnCurrentTagOnly,
                    tmpOption.TargetTagName);
            }
        }
    }
}
