﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.Linq;

namespace LECore.Structures
{
    using Core;
    /// <summary>
    /// 外部公開用インタフェース
    /// </summary>
    public interface ITextureSubSet
    {
        /// <summary>
        /// テクスチャ名前配列を取得します。
        /// </summary>
        string[] TextureSubSetNameSet { get;}
    }

    /// <summary>
    /// テクスチャサブセット
    /// テクスチャパターンアニメーションに使用される、
    /// テクスチャの部分セットを表現します。
    ///
    /// テクスチャパターンアニメーションはTextureSubSet内の
    /// テクスチャインデックス値のアニメーションとして管理されます。
    ///
    /// テクスチャパターンアニメーションを実現する、
    /// TextureChanger が 参照します。
    ///
    /// 複数の TextureChanger が 1つの TextureSubSet を共有参照する
    /// モデルも想定されます。
    /// </summary>
    internal class TextureSubSet
        : ITextureSubSet
    {
        /// <summary>
        /// テクスチャ名の配列
        /// </summary>
        ArrayList        _textureSet = new ArrayList();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureSubSet(){}


        /// <summary>
        /// テクスチャを登録します。
        /// </summary>
        public void AddTexture( string textureName )
        {
            if( !_textureSet.Contains( textureName ) )
            {
                _textureSet.Add( textureName );
            }
        }

        /// <summary>
        /// 複数のテクスチャを登録します。
        /// </summary>
        public void AddTextures( string[] textureNames )
        {
            foreach( string texName in textureNames )
            {
                this.AddTexture( texName );
            }
        }

        /// <summary>
        /// テクスチャ削除します。
        /// </summary>
        public void RemoveTexture( string textureName )
        {
            if( _textureSet.Contains( textureName ) )
            {
                _textureSet.Remove( textureName );
            }
        }

        /// <summary>
        /// テクスチャをすべて消去します。
        /// </summary>
        public void Clear()
        {
           _textureSet.Clear();
        }

        /// <summary>
        /// テクスチャを名前で検索します。
        /// </summary>
        public bool FindTextureByName( string textureName, out int index )
        {
            index = _textureSet.IndexOf( textureName );
            return (index != -1);
        }

        /// <summary>
        /// テクスチャを番号で検索します。
        /// </summary>
        public bool FindTextureByIndex( int index, out string texName )
        {
            if( index < _textureSet.Count && index >= 0 )
            {
                texName = _textureSet[index] as string;
                return true;
            }
            else
            {
                texName = string.Empty;
                return false;
            }
        }

        /// <summary>
        /// テクスチャ名のリストから、
        /// テクスチャサブセットが等値な内容をもつか判定します。
        /// </summary>
        public bool IsSameSet( string[] textureNameSet )
        {
            // 総数が同一か？
            if( textureNameSet.Length == _textureSet.Count )
            {
                for( int idx = 0; idx < textureNameSet.Length; idx++ )
                {
                    string serchTex = textureNameSet[idx];
                    string matchTexName;

                    // 同一順で、同一名のテクスチャが発見されるか確認します。
                    if( FindTextureByIndex( idx, out matchTexName ) &&
                        ( matchTexName == serchTex ) )
                    {
                        continue;
                    }
                    else
                    {
                        return false;
                    }
                }
                // すべてのテクスチャが発見された
                return true;
            }
            return false;
        }

        #region --------------- ITextureSubSet メンバ ---------------

        /// <summary>
        /// テクスチャ名前配列を取得します。
        /// </summary>
        public string[] TextureSubSetNameSet
        {
            get
            {
                return _textureSet.ToArray( typeof( string ) ) as string[];
            }
        }

        #endregion

    }

    /// <summary>
    /// テクスチャサブセット番号アニメーションアトリビュート
    /// </summary>
    internal class TextureChangerIndexAttr :
        AnmAttribute
    {
        static AnmAttrDescripter MakeAnmAttrDescripter_(int slotIndex)
        {
            return new AnmAttrDescripter(AttributeType.Int,
                string.Format("TexPatternAnm_{0}", slotIndex),
                null,
                0,
                AnmAttrDescripter.AnimationTypeLock.True,
                false);
        }

        public TextureChangerIndexAttr(LEDataNode ownerAttr, int slotIndex)
            : base( ownerAttr, null, MakeAnmAttrDescripter_( slotIndex ) )
        {
            this.CurrentAnimationCurve.DefaultInterpType = InterporationType.Step;
        }
    }

    /// <summary>
    /// 外部公開インタフェース
    /// </summary>
    public interface ITextureChanger
    {
        IAnmAttribute  TextureIndexIAnmAttr    {get;}
        ITextureSubSet ITextureSubSet          {get;}
    }

    /// <summary>
    /// ITextureChangerHelperのヘルパークラス
    /// </summary>
    static public class ITextureChangerHelper
    {
        static public bool HasAnyTextureSubSet(this ITextureChanger textureChanger)
        {
            return textureChanger != null && textureChanger.ITextureSubSet != null && textureChanger.ITextureSubSet.TextureSubSetNameSet.Any();
        }

        static public bool HasAnyTexturePatternAnimationOrTextureSubSet(this ITextureChanger textureChanger)
        {
            return textureChanger.TextureIndexIAnmAttr.CheckHaveKeyRecursiveInAllTag() || textureChanger.HasAnyTextureSubSet();
        }
    }

    /// <summary>
    /// テクスチャパターンアニメーションを実現するクラスです。
    ///
    /// テクスチャサブセット番号アニメーション
    /// を管理します。
    /// </summary>
    internal class TextureChanger
        : ITextureChanger,
        IDisposable
    {
        const int                              InvalidSlotIndex = -1;
        LEDataNode.LEDataNodeModifyHandler    _OnAnmAttrModifyHandler = null;


        TextureChangerIndexAttr         _textureChangerIndexAttr = null;
        readonly TextureSubSet	        _textureSubSet           = null;
        MaterialTexMap                  _targetMaterialTex       = null;
        int                             _slotIndex               = InvalidSlotIndex;

        #region ------------- プロパティ -------------
        public IAnmAttribute TextureIndexIAnmAttr
        {
            get{ return _textureChangerIndexAttr;}
        }

        public AnmAttribute TextureIndexAnmAttr
        {
            get{ return _textureChangerIndexAttr;}
        }

        /// <summary>
        /// 現在のテクスチャ番号を取得します。
        /// </summary>
        public int _TextureIndex
        {
            get
            {
                Debug.Assert( _textureChangerIndexAttr != null );

                object textureIndexRef = null;
                _textureChangerIndexAttr.GetValue( out textureIndexRef );

                return Convert.ToInt32( textureIndexRef );
            }
        }

        public MaterialTexMap TragetMaterialTexMap
        {
            get{ return  _targetMaterialTex;}
        }

        public TextureSubSet TextureSubSet
        {
            get{ return _textureSubSet;}
        }

        public ITextureSubSet ITextureSubSet
        {
            get{ return _textureSubSet;}
        }

        public string[] TextureSubsetNameSet
        {
            get{return TextureSubSet.TextureSubSetNameSet;}
        }

        public int SlotIndex
        {
            get{ return _slotIndex;}
        }

        public bool IsActive
        {
            get { return this._textureChangerIndexAttr.IsActiveAttribute; }
            set { this._textureChangerIndexAttr.IsActiveAttribute = value; }
        }

        bool _ReadyToChangeTex
        {
            get { return _targetMaterialTex != null;}
        }


        #endregion ------------- プロパティ -------------


        /// <summary>
        /// リセットします。
        /// </summary>
        public void Reset()
        {
            this.IsActive = false;
            _textureSubSet.Clear();
            _slotIndex = InvalidSlotIndex;
        }

        #region --------------- コンストラクタ ---------------
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureChanger( MaterialTexMap materialTexMap, int slotIndex )
        {
            _textureSubSet = new TextureSubSet();

            // アニメーション変更ハンドラの設定
            _OnAnmAttrModifyHandler =
                new LEDataNode.LEDataNodeModifyHandler( OnTextureIndexChanged_ );


            BindMaterialTexMap_( materialTexMap, slotIndex );
        }

        #region IDisposable メンバ

        public void Dispose()
        {
            _textureChangerIndexAttr.Dispose();
        }

        #endregion

        #endregion --------------- コンストラクタ ---------------




        #region ---------------  操作対象 MaterialTexMap への関連付け、解除 ---------------




         /// <summary>
         /// 操作対象 MaterialTexMap の 関連付け
         /// </summary>
         /// <param name="materialTexMap"></param>
        void BindMaterialTexMap_( MaterialTexMap materialTexMap, int slotIndex )
        {
            Debug.Assert( _targetMaterialTex == null );
            Debug.Assert( _textureChangerIndexAttr == null );

            _targetMaterialTex = materialTexMap;

            // アニメーションアトリビュートの設定をします。
            _textureChangerIndexAttr = new TextureChangerIndexAttr( materialTexMap.OwnerMaterialNode, slotIndex );

            _slotIndex = slotIndex;

            // テクスチャインデックの更新イベントハンドラを設定します。
            _textureChangerIndexAttr.OnModify += _OnAnmAttrModifyHandler;
        }

        /// <summary>
        /// 操作対象 MaterialTexMap の解除
        /// 現在は使用されていません。
        /// </summary>
        void UnbindMaterialTexMap_()
        {
            Debug.Assert( _targetMaterialTex != null );
            Debug.Assert( _textureChangerIndexAttr != null );


            _targetMaterialTex          = null;

            _slotIndex = InvalidSlotIndex;

            // アニメーションアトリビュートの後始末をします。
            _textureChangerIndexAttr.OnModify    -= _OnAnmAttrModifyHandler;
            _textureChangerIndexAttr = null;
        }

        #endregion ---------------  操作対象 MaterialTexMap への関連付け、解除 ---------------






        /// <summary>
        /// アニメーションなどによって、
        /// テクスチャ番号が変更になったときに呼ばれるハンドラです。
        /// </summary>
        /// <param name="sender"></param>
        void OnTextureIndexChanged_( LEDataNode sender, int kind )
        {
            Debug.Assert( sender == _textureChangerIndexAttr );

            // 設定できない場合は、処理しません。
            // また、アトリビュートが無効なら処理しません。（無効化の通知も含まれる）
            if (!_ReadyToChangeTex || !_textureChangerIndexAttr.IsActiveAttribute)
            {
                return;
            }

            // アトリビュートからテクスチャ番号を取得します。
            int textureIndex = _TextureIndex;

            // テクスチャ番号に対応するテクスチャを操作対象に対して設定します。
            string newTexName = string.Empty;
            if( _textureSubSet.FindTextureByIndex( textureIndex, out newTexName ) )
            {
                _targetMaterialTex.TexImgName = newTexName;
            }
        }

        /// <summary>
        /// 再評価します。
        /// </summary>
        public void Evaluate()
        {
            if( !_ReadyToChangeTex )
            {
                return;
            }

            // 一旦、画像をリセットし、再設定します。
            //
            // 画像変更イベントを送信するため、変更がない場合も
            // 必ずリセットしています。
            //
            string oldTextureName = _targetMaterialTex.TexImgName;
            _targetMaterialTex.TexImgName = string.Empty;

            int currentTextureIndex = _TextureIndex;
            if( currentTextureIndex >= 0 &&
                _textureSubSet.TextureSubSetNameSet.Length > currentTextureIndex )
            {
                OnTextureIndexChanged_(_textureChangerIndexAttr, (int)SceneModifyEventArgs.Kind.PaneModify);
            }
            else
            {
                _targetMaterialTex.TexImgName = oldTextureName;
            }
        }
    }
}
