﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

namespace LECore.Structures
{
    using Core;
    using Nsrif.Attributes;

    #region ＰＥデータ

    #region PEData

    /// <summary>
    /// ＰＥデータクラス。外部公開インタフェース
    /// </summary>
    public interface IPEData
    {
        /// <summary>
        /// 描画モード。
        /// </summary>
        AttrTransparencyMode DrawMode {get;}


        /// <summary>
        /// アルファコンペア。
        /// </summary>
        PEACompare ACompare {get;}


        /// <summary>
        /// Ｚコンペア。
        /// </summary>
        PEZCompare ZCompare {get;}


        /// <summary>
        /// ブレンド。
        /// </summary>
        PEBlend Blend {get;}

        /// <summary>
        /// ブレンド。
        /// </summary>
        PEBlend BlendAlpha { get; }

        /// <summary>
        /// デフォルトを使うかどうか
        /// </summary>
        bool UseDefaultBlendSettings { get; }

        /// <summary>
        /// デフォルトを使うかどうか
        /// </summary>
        bool UseDefaultAlphaTestSettings { get; }
    }

    /// <summary>
    /// ＰＥデータクラス。
    /// </summary>
    public  sealed class PEData :
        IPEData
    {
        //----------------------------------------------------------
        public const string FragmentOperationDataNodeName = "FragmentOperationData";

        //----------------------------------------------------------
        // プリセット用設定
        public static readonly PEData None;
        public static readonly PEData Mask;
        public static readonly PEData AlphaBlend;
        public static readonly PEData Add;
        public static readonly PEData Mul;
        public static readonly PEData Sub;
        public static readonly PEData AlphaBlendMaxA;

        //----------------------------------------------------------
        private AttrTransparencyMode _drawMode = AttrTransparencyMode.Opa;
        private readonly PEACompare  _acompare;
        private readonly PEZCompare  _zcompare;
        private readonly PEBlend     _blend;
        private readonly PEBlend _blendAlpha;
        private readonly LEDataNode _ownerNode;

        //----------------------------------------------------------
        #region コンストラクタ

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PEData(IAnmAttribute owner)
        {
            _ownerNode = owner as LEDataNode;
            _acompare = new PEACompare(owner);
            _zcompare = new PEZCompare();
            _blend = new PEBlend(AttrBlendType.Blend);
            _blendAlpha = new PEBlend(AttrBlendType.None);

            this.UseDefaultBlendSettings = true;
            this.UseDefaultAlphaTestSettings = true;
        }

        /// <summary>
        /// 静的コンストラクタ
        /// </summary>
        static PEData()
        {
            None = new PEData(null) { UseDefaultBlendSettings = false, UseDefaultAlphaTestSettings = true };
            None.Blend.Type = AttrBlendType.None;

            Mask = new PEData(null) { UseDefaultBlendSettings = false, UseDefaultAlphaTestSettings = false };
            Mask.Blend.Type = AttrBlendType.None;
            Mask.ACompare.Ref = 0.5f;
            Mask.ACompare.Comp = AttrCompareFunc.Greater;

            AlphaBlend = new PEData(null) { UseDefaultBlendSettings = true, UseDefaultAlphaTestSettings = true };

            Add = new PEData(null) { UseDefaultBlendSettings = false, UseDefaultAlphaTestSettings = true };
            Add.Blend.Type = AttrBlendType.Blend;
            Add.Blend.BlendOp = AttrBlendOp.Add;
            Add.Blend.SrcFactor = AttrBlendFactor.SrcAlpha;
            Add.Blend.DstFactor = AttrBlendFactor.V1_0;

            Mul = new PEData(null) { UseDefaultBlendSettings = false, UseDefaultAlphaTestSettings = true };
            Mul.Blend.Type = AttrBlendType.Blend;
            Mul.Blend.BlendOp = AttrBlendOp.Add;
            Mul.Blend.SrcFactor = AttrBlendFactor.V0;
            Mul.Blend.DstFactor = AttrBlendFactor.SrcClr;

            Sub = new PEData(null) { UseDefaultBlendSettings = false, UseDefaultAlphaTestSettings = true };
            Sub.Blend.Type = AttrBlendType.Blend;
            Sub.Blend.BlendOp = AttrBlendOp.ReverseSubtract;
            Sub.Blend.SrcFactor = AttrBlendFactor.SrcAlpha;
            Sub.Blend.DstFactor = AttrBlendFactor.V1_0;

            AlphaBlendMaxA = new PEData(null) { UseDefaultBlendSettings = false, UseDefaultAlphaTestSettings = true };
            AlphaBlendMaxA.BlendAlpha.Type = AttrBlendType.Blend;
            AlphaBlendMaxA.BlendAlpha.BlendOp = AttrBlendOp.SelectMax;
            AlphaBlendMaxA.BlendAlpha.SrcFactor = AttrBlendFactor.V1_0;
            AlphaBlendMaxA.BlendAlpha.DstFactor = AttrBlendFactor.V1_0;
        }

        #endregion

        //----------------------------------------------------------
        #region アトリビュート
        /// <summary>
        /// 描画モード。
        /// </summary>
        public AttrTransparencyMode DrawMode
        {
            get { return _drawMode;  }
            set
            {
                _drawMode = value;
                if (_ownerNode != null)
                {
                    _ownerNode.NotifyModifyEvent(_ownerNode, (int)SceneModifyEventArgs.Kind.PaneModify);
                }
            }
        }

        /// <summary>
        /// アルファコンペア。
        /// </summary>
        public PEACompare ACompare
        {
            get { return _acompare; }
        }

        /// <summary>
        /// Ｚコンペア。
        /// </summary>
        public PEZCompare ZCompare
        {
            get { return _zcompare; }
        }

        /// <summary>
        /// ブレンド。
        /// </summary>
        public PEBlend Blend
        {
            get { return _blend; }
        }

        /// <summary>
        /// ブレンド。
        /// </summary>
        public PEBlend BlendAlpha
        {
            get { return _blendAlpha; }
        }

        /// <summary>
        /// デフォルトブレンドを使うかどうか
        /// </summary>
        public bool UseDefaultBlendSettings { get; set; }

        /// <summary>
        /// デフォルトアルファテストを使うかどうか
        /// </summary>
        public bool UseDefaultAlphaTestSettings { get; set; }

        /// <summary>
        /// カラーブレンド設定を、アルファブレンド設定に利用するかどうか
        /// </summary>
        public bool UseColorBlendSettingsForAlpha
        {
            get{ return this.BlendAlpha.Type == AttrBlendType.None;}
        }

        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(PEData src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _drawMode = src._drawMode;
            _acompare.Set(src._acompare);
            _zcompare.Set(src._zcompare);
            _blend.Set(src._blend);
            _blendAlpha.Set(src._blendAlpha);
            UseDefaultBlendSettings = src.UseDefaultBlendSettings;
            UseDefaultAlphaTestSettings = src.UseDefaultAlphaTestSettings;

            if (_ownerNode != null)
            {
                _ownerNode.NotifyModifyEvent(_ownerNode, (int)SceneModifyEventArgs.Kind.PaneModify);
            }
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(PEData src)
        {
            if (src == null) return false;
            if (src == this) return false;

            return
                _drawMode == src._drawMode      &&
                _acompare.Equals(src._acompare) &&
                _zcompare.Equals(src._zcompare) &&
                _blend.Equals(src._blend) &&
                _blendAlpha.Equals(src._blendAlpha) &&
                UseDefaultBlendSettings == src.UseDefaultBlendSettings &&
                UseDefaultAlphaTestSettings == src.UseDefaultAlphaTestSettings;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as PEData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        #endregion
    }
    #endregion

    #region PEState
    /// <summary>
    /// ＰＥ設定状態。
    /// </summary>
    public enum PEState
    {
        /// <summary>不透明設定。</summary>
        Opa = 0,
        /// <summary>抜き設定。</summary>
        Edge,
        /// <summary>半透明設定。</summary>
        Xlu,
        /// <summary>加算合成設定。</summary>
        Add,
        /// <summary>乗算合成設定。</summary>
        Mul,
        /// <summary>減算合成設定。</summary>
        Sub,
        /// <summary>カスタム設定。</summary>
        Custom,
    }
    #endregion

    #region PEACompare
    /// <summary>
    /// ＰＥアルファコンペアクラス。
    /// </summary>
    public sealed class PEACompare
    {
        private AttrAlphaOp     _op    = AttrAlphaOp.Or;
        private AttrCompareFunc _comp = AttrCompareFunc.Always;
        private readonly AnmAttribute _refAttribute;
        #region コンストラクタ


        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PEACompare(IAnmAttribute ownerNode)
        {
            _refAttribute = new AnmAttribute(ownerNode as LEDataNode, null, new AnmAttrDescripter(AttributeType.Float, "AlphaTestValue", null, 0.0f));
            _refAttribute.MinValue = 0.0f;
            _refAttribute.MaxValue = 1.0f;
        }

        #endregion

        #region アトリビュート
        /// <summary>
        /// 比較方法０。
        /// </summary>
        public AttrCompareFunc Comp
        {
            get { return _comp;  }
            set { _comp = value; }
        }

        /// <summary>
        /// 参照値０。
        /// </summary>
        public float Ref
        {
            get { return _refAttribute.GetAsFloat(); }
            set { _refAttribute.SetAsFloat(value); }
        }

        /// <summary>
        /// アニメーションアトリビュートです。
        /// </summary>
        public IAnmAttribute RefAnmAttr
        {
            get { return _refAttribute; }
        }

        /// <summary>
        /// 演算方法。
        /// </summary>
        public AttrAlphaOp Op
        {
            get { return _op;  }
            set { _op = value; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(PEACompare src)
        {
            if (object.ReferenceEquals(this, src)) return;

            this._comp = src._comp;
            this.Ref = src.Ref;
            this._op = src._op;
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(PEACompare src)
        {
            if (src == null) return false;
            if (src == this) return false;

            return
                this._comp == src._comp &&
                this.Ref == src.Ref &&
                this._op == src._op;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as PEACompare);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        #endregion
    }
    #endregion

    #region PEZCompare
    /// <summary>
    /// ＰＥＺコンペアクラス。
    /// </summary>
    public sealed class PEZCompare
    {
        private bool            _enable    = true;
        private AttrCompareFunc _func      = AttrCompareFunc.Lequal;
        private bool            _update    = true;
        private bool            _beforeTex = false;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PEZCompare() {}

        #endregion

        #region アトリビュート
        /// <summary>
        /// 有効フラグ。
        /// </summary>
        public bool Enable
        {
            get { return _enable;  }
            set { _enable = value; }
        }

        /// <summary>
        /// 比較方法。
        /// </summary>
        public AttrCompareFunc Func
        {
            get { return _func;  }
            set { _func = value; }
        }

        /// <summary>
        /// 更新フラグ。
        /// </summary>
        public bool Update
        {
            get { return _update;  }
            set { _update = value; }
        }

        /// <summary>
        /// テクスチャ処理前フラグ。
        /// </summary>
        public bool BeforeTex
        {
            get { return _beforeTex;  }
            set { _beforeTex = value; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(PEZCompare src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _enable    = src._enable;
            _func      = src._func;
            _update    = src._update;
            _beforeTex = src._beforeTex;
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(PEZCompare src)
        {
            if (src == null) return false;
            if (src == this) return false;

            return
                _enable    == src._enable &&
                _func      == src._func   &&
                _update    == src._update &&
                _beforeTex == src._beforeTex;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as PEZCompare);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        #endregion

    }
    #endregion

    #region PEBlend
    /// <summary>
    /// ＰＥブレンドクラス。
    /// </summary>
    public sealed class PEBlend
    {
        private AttrBlendType   _type      = AttrBlendType.None;
        private AttrBlendFactor _srcFactor = AttrBlendFactor.SrcAlpha;
        private AttrBlendFactor _dstFactor = AttrBlendFactor.InvSrcAlpha;
        private AttrBlendOp      _blendOperation = AttrBlendOp.Add;
        private AttrLogicOp     _logicOp   = AttrLogicOp.Copy;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PEBlend() {}

        public PEBlend(AttrBlendType defaultBlendType)
        {
            _type = defaultBlendType;
        }

        public static readonly PEBlend DefaultBlendColor = new PEBlend(AttrBlendType.Blend);
        public static readonly PEBlend DefaultBlendAlpha = new PEBlend(AttrBlendType.Blend);
        #endregion

        #region アトリビュート
        /// <summary>
        /// タイプ。
        /// </summary>
        public AttrBlendType Type
        {
            get { return _type;  }
            set { _type = value; }
        }

        /// <summary>
        /// ソース引数。
        /// </summary>
        public AttrBlendFactor SrcFactor
        {
            get { return _srcFactor;  }
            set { _srcFactor = value; }
        }

        /// <summary>
        /// ディスティネーション引数。
        /// </summary>
        public AttrBlendFactor DstFactor
        {
            get { return _dstFactor;  }
            set { _dstFactor = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public AttrBlendOp BlendOp
        {
            get { return _blendOperation;  }
            set { _blendOperation = value; }
        }

        /// <summary>
        /// 論理演算。
        /// </summary>
        public AttrLogicOp LogicOp
        {
            get { return _logicOp;  }
            set { _logicOp = value; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(PEBlend src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _type           = src._type;
            _srcFactor      = src._srcFactor;
            _dstFactor      = src._dstFactor;
            _blendOperation = src._blendOperation;
            _logicOp        = src._logicOp;
        }
        #endregion

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(PEBlend src)
        {
            if (src == null) return false;
            if (src == this) return false;

            return
                _type      == src._type      &&
                _srcFactor == src._srcFactor &&
                _dstFactor == src._dstFactor &&
                _blendOperation == src._blendOperation &&
                _logicOp   == src._logicOp;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as PEBlend);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        #endregion

    }
    #endregion
    #endregion
}
