﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace LECore.Structures
{
    using Core;
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Manipulator;

    internal class Mask :
        LEDataNode,
        IMask,
        IDisposable
    {
        public const string MaskNodeName = "Mask";

        readonly MaskAttribute _attribute = null;

        readonly Pane _ownerPane = null;
        bool _isMaskEnabled = false;
        bool _isMaskWithPane = false;
        Material _internalMaterial; // マスク内部で保持するマテリアル

        public Mask(Pane ownerPane)
            : base(ownerPane, MaskNodeName)
        {
            System.Diagnostics.Debug.Assert(ownerPane != null);

            _ownerPane = ownerPane;
            _ownerPane.BindPaneMask(this);
            _internalMaterial = new Material(null, "Mask_InternalMaterial");

            _attribute = new Structures.Mask.MaskAttribute(this);
        }

        public void Dispose()
        {
            _attribute.Dispose();
        }

        public MaskAttribute MaskAttrRef
        {
            get
            {
                return _attribute;
            }
        }

        /// <summary>
        /// 変更をシーンに通知します。
        /// </summary>
        void NotifyChangeToScene_()
        {
            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }

        MaterialTexMap _TexMap
        {
            get
            {
                return _internalMaterial.GetIMaterialTexMapByIndex(0) as MaterialTexMap;
            }
        }

        #region IMask


        public IPane OwnerPane
        {
            get
            {
                return _ownerPane;
            }
        }

        public bool IsMaskEnabled
        {
            set
            {
                if (_isMaskEnabled != value)
                {
                    _isMaskEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isMaskEnabled;
            }
        }

        public bool IsMaskWithPane
        {
            set
            {
                if (_isMaskWithPane != value)
                {
                    _isMaskWithPane = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isMaskWithPane;
            }
        }

        public IMaterialTexMap MaskTexMap
        {
            get
            {
                return _internalMaterial.GetIMaterialTexMapByIndex(0);
            }
        }

        public ITexMtx MaskTexMatrix
        {
            get
            {
                return _internalMaterial.TexMtxSet[0];
            }
        }

        public AttrTexWrap WrapS
        {
            set
            {
                if (_TexMap.WrapS != value)
                {
                    _TexMap.WrapS = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return MaskTexMap.WrapS;
            }
        }

        public AttrTexWrap WrapT
        {
            set
            {
                if (_TexMap.WrapT != value)
                {
                    _TexMap.WrapT = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return MaskTexMap.WrapT;
            }
        }

        public AttrTexFilterMag MagFilter
        {
            set
            {
                if (_TexMap.MagFilter != value)
                {
                    _TexMap.MagFilter = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return MaskTexMap.MagFilter;
            }
        }

        public AttrTexFilterMin MinFilter
        {
            set
            {
                if (_TexMap.MinFilter != value)
                {
                    _TexMap.MinFilter = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return MaskTexMap.MinFilter;
            }
        }

        public FVec2 MtxTrans
        {
            set
            {
                MtxTransAttr.SetValue(value);
            }

            get
            {
                return MtxTransAttr.GetAsFVec2();
            }
        }

        public FVec2 MtxScale
        {
            set
            {
                MtxScaleAttr.SetValue(value);
            }

            get
            {
                return MtxScaleAttr.GetAsFVec2();
            }
        }

        public float MtxRotate
        {
            set
            {
                MtxRotateAttr.SetValue(value);
            }

            get
            {
                return MtxRotateAttr.GetAsFloat();
            }
        }

        public AnmAttribute MtxTransAttr
        {
            get
            {
                return _attribute.TransAttr;
            }
        }

        public AnmAttribute MtxScaleAttr
        {
            get
            {
                return _attribute.ScaleAttr;
            }
        }

        public AnmAttribute MtxRotateAttr
        {
            get
            {
                return _attribute.RotateAttr;
            }
        }

        public IMaterial InternalMaterial
        {
            get
            {
                return _internalMaterial;
            }
        }

        public void UpdateForChangeOfInternalMaterial()
        {
            NotifyChangeToScene_();
        }

        /// <summary>
        /// マスク用のキャプチャテクスチャを作成します。
        /// </summary>
        public void CreateCaptureTextureForMask(IPane pane, ITextureMgr textureMgr)
        {
            TextureMgrManipulator mnp = new TextureMgrManipulator();
            mnp.BindTarget(textureMgr);
            CaptureTextureManipulator captureTextureMnp = new CaptureTextureManipulator();
            ICaptureTexture captureTexture = mnp.RegisterCaptureTexture(pane, pane.PaneName, true, CaptureTextureUsage.Mask).ICaptureTexture;
            captureTextureMnp.BindTarget(captureTexture);
            captureTextureMnp.ClearColor = new FloatColor(0.0f, 0.0f, 0.0f, 0.0f);
            captureTextureMnp.FrameBufferCaptureEnabled = false;
        }
        #endregion

        internal class MaskAttribute :
            AnmAttribute
        {
            Mask _ownerMask = null;

            static readonly AnmAttrDescripter SelfAttrDesc =
                new AnmAttrDescripter(AttributeType.Combined, "Mask", null, null);

            public MaskAttribute(Mask ownerMask)
                : base(ownerMask, null, SelfAttrDesc)
            {
                _ownerMask = ownerMask;

                AnmAttrDescripter desc =
                    new AnmAttrDescripter(AttributeType.RefAttribute, "Texture_Mtx_", null, ownerMask.MaskTexMatrix);
                new AnmAttribute(this, null, desc);

                (ownerMask.InternalMaterial as Material).OwnerNode = this;
            }

            public AnmAttribute TransAttr
            {
                get
                {
                    return _ownerMask.MaskTexMatrix.TransIAnmAttr as AnmAttribute;
                }
            }

            public AnmAttribute ScaleAttr
            {
                get
                {
                    return _ownerMask.MaskTexMatrix.ScaleIAnmAttr as AnmAttribute;
                }
            }

            public AnmAttribute RotateAttr
            {
                get
                {
                    return _ownerMask.MaskTexMatrix.RotateIAnmAttr as AnmAttribute;
                }
            }
        }
    }
}
