﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.Text;
using System.Linq;
using System.Xml.Serialization;

using LECore.Structures.SerializableObject.Lyt;

namespace LECore.Structures
{
    using LECore.Structures.Core.Command;
    using NWFont = NW4R.Font;
    using FontManagerCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<FontManager>;
    using System.Windows.Forms;
    using System.Threading;
    using System.Runtime.InteropServices;
    using System.Text.RegularExpressions;
    using System.Xml.Linq;

    ///
    internal delegate void OnLEFontMgrUpdateHandler();

    /// <summary>
    ///
    /// </summary>
    public class AutoFilteredFontState
    {
        /// <summary>
        /// メッセージの種類
        /// </summary>
        private enum MessageType
        {
            Info, // 重複は消す
            Report,
            Error,
        }

        private List<Tuple<string, MessageType>> Messages { get; set; }
        public List<AutoFilteredFontEntry> Entry { get; private set; }

        public AutoFilteredFontState()
        {
            this.Entry = new List<AutoFilteredFontEntry>();
            this.Messages = new List<Tuple<string,MessageType>>();
        }

        /// <summary>
        /// メッセージの消去
        /// </summary>
        public void ClearMessages()
        {
            Messages.Clear();
        }

        /// <summary>
        /// メッセージの取得
        /// </summary>
        public IEnumerable<string> GetMessages(bool error)
        {
            if (error)
            {
                return Messages.Where(x => x.Item2 == MessageType.Error).Select(x => x.Item1);
            }
            else
            {
                return Messages
                    .Where(x => x.Item2 != MessageType.Error)
                    .Select((x, i) => new Tuple<string, int>(x.Item1, x.Item2 == MessageType.Info ? -1 : i))
                    .Distinct()
                    .Select(x => x.Item1);
            }
        }

        /// <summary>
        /// エラーメッセージの追加
        /// </summary>
        public void AddError(string message)
        {
            Messages.Add(new Tuple<string, MessageType>(message, MessageType.Error));
        }

        /// <summary>
        /// 情報の追加
        /// </summary>
        public void AddInformation(string message)
        {
            Messages.Add(new Tuple<string, MessageType>(message, MessageType.Info));
        }


        /// <summary>
        /// 報告の追加
        /// </summary>
        public void AddReport(string message)
        {
            Messages.Add(new Tuple<string, MessageType>(message, MessageType.Report));
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class AutoFilteredFontEntry
    {
        public bool IsFontGenerated { get; set; }
        public string SrcFontFilePath { get; set; }
        public string DstFontFilePath { get; set; }

        // スケーラブルフォントをビットマップフォント変換後のサイズを測るためのフォント
        public string LinearDstFontFilePath { get; set; }

        public long SrcBmpState { get; set; }
        public long SrcXlorState { get; set; }

        public string FilterContent { get; set; }

        public string AdditionalArguments { get; set; }

        public string FontConverterPath { get; set; }

        public bool IsSrcFileSame(string srcXlorPath, string srcBmpPath)
        {
            return SrcXlorState == GetFileState(srcXlorPath) && SrcBmpState == GetFileState(srcBmpPath);
        }

        public static long GetFileState(string filePath)
        {
            return File.GetLastWriteTime(filePath).ToBinary();
        }

        public bool IsSrcScalableFont()
        {
            return LEFontHelper.IsScalableFontFile(this.SrcFontFilePath);
        }
    }

    /// <summary>
    /// 指定フォルダ以下のレイアウトファイルが参照しているフォントを探し、
    /// 自動フィルタフォントを作ります。
    /// </summary>
    static public class AutoFilteredFontUtil
    {
        public class FontCharUsageData
        {
            private ILEFont LEFont { get; set; }
            public string FontPath { get; private set; }
            public string FontName { get; private set; }

            // 別のスレッドで fontSettings の内容が書き換わらない前提
            // TODO: fontSettings を imutable にして上の前提を保証したい。
            public IFontSettings FontSettings { get; private set; }

            public List<string> Words { get; private set; }

            public FontCharUsageData(ILEFont font)
            {
                LEFont = font;
                FontPath = font.FontPath;
                FontSettings = font.FontSettings;
                FontName = font.FontName;
                Words = new List<string>();
            }

            // 注意: UI スレッドから呼ぶこと
            public bool IsSameFont(ILEFont font)
            {
                if (font == null)
                {
                    return false;
                }

                if (this.LEFont.FontPath != font.FontPath)
                {
                    return false;
                }

                if (this.LEFont.IsScalableFont() != font.IsScalableFont())
                {
                    return false;
                }

                if (this.LEFont.IsScalableFont())
                {
                    return font.IsScalableFont() && (this.LEFont.FontSettings as IScalableFontSettings).IsSame(font.FontSettings as IScalableFontSettings);
                }
                else
                {
                    return !font.IsScalableFont();
                }
            }
        }

        public const int ViewerMaxWaitTimeMillisec = 10000;

        static public string IsGpuEncodingEnalbed { get; set; }

        /// <summary>
        /// xlor や bmp のようなソースファイルを探す。
        /// </summary>
        static string FindSourceFile_(string fontPath, string relativePath, string[] exts)
        {
            string fontDir = Path.GetDirectoryName(fontPath);
            string fontFileName = Path.GetFileName(fontPath);

            string sourceDir = Path.Combine(fontDir, relativePath);

            sourceDir = Directory.Exists(sourceDir) ? sourceDir : fontDir;
            foreach (string ext in exts)
            {
                var srcFiles = Directory.GetFiles(sourceDir, Path.ChangeExtension(fontFileName, ext));
                if (srcFiles.Length > 0)
                {
                    return srcFiles[0];
                }
            }

            return string.Empty;
        }

        /// <summary>
        /// テキストボックスペインを列挙する。(部品ペインも再帰的に列挙する)
        /// </summary>
        static IEnumerable<IPane> EnumerateAllTextBox_(ISubScene subScene)
        {
            if (subScene != null)
            {
                foreach (IPane textBox in subScene.FindPanesByKind(PaneKind.Textbox))
                {
                    yield return textBox;
                }

                foreach (var parts in subScene.FindPanesByKind(PaneKind.Parts))
                {
                    foreach (var subTextBox in EnumerateAllTextBox_(parts.IPartsLayout.PartsSubScene))
                    {
                        yield return subTextBox;
                    }
                }
            }
        }

        static public List<FontCharUsageData> CreateFontUsageData(IEnumerable<ISubScene> subSceneSet)
        {
            // 情報の収集、フォント毎に使っている語句を収集していく。
            List<FontCharUsageData> fontTable = new List<FontCharUsageData>();
            foreach (ISubScene subScene in subSceneSet)
            {
                foreach (var textBox in EnumerateAllTextBox_(subScene))
                {
                    if (textBox.ITextBox.ILEFont == null)
                    {
                        continue;
                    }

                    foreach (var font in GetFilterSourceFonts(textBox.ITextBox.ILEFont))
                    {
                        var fontUsageData = fontTable.Find((fud) => fud.IsSameFont(font));
                        if (fontUsageData == null)
                        {
                            fontUsageData = new FontCharUsageData(font);
                            fontTable.Add(fontUsageData);
                        }

                        if (fontUsageData.Words.Find((word) => word.Contains(textBox.ITextBox.ContentsText)) == null)
                        {
                            fontUsageData.Words.Add(textBox.ITextBox.ContentsText);
                        }
                    }
                }

                // テキストボックスペインで利用されない fcpx で利用される ffnt も必要なので収集する
                foreach (var font in subScene.ILEFontManager.ILEFontSet)
                {
                    if (LEFontHelper.IsComplexFont(font) && (font.NWFont is NW4R.Font.PairFont || font.NWFont is NW4R.Font.NW4RResFont))
                    {
                        foreach (var leaf in GetFilterSourceFonts(font))
                        {
                            if (LEFontHelper.IsFontConvertSettingsFile(leaf.FontPath))
                            {
                                var fontUsageData = fontTable.Find((fud) => fud.IsSameFont(leaf));
                                if (fontUsageData == null)
                                {
                                    fontUsageData = new FontCharUsageData(leaf);
                                    fontTable.Add(fontUsageData);
                                }
                            }
                        }
                    }
                }
            }

            return fontTable;
        }

        /// <summary>
        /// フィルタフォントの生成
        /// </summary>
        static public void Create(IEnumerable<FontCharUsageData> fontTable, AutoFilteredFontState fontState, ViewerPreviewParam previewParam)
        {
            List<AutoFilteredFontEntry> fontEntries = fontState.Entry;

            Stopwatch sw = new Stopwatch();
            sw.Start();

            string autoGenFontDir = LECore.Util.TemporaryFileUtil.AutoFilteredFontDir;

            // フォント毎に、コンバータを駆動してフィルタ済みフォントを生成する。
            bool createAutoFilterFont = false;
            foreach (FontCharUsageData fontUsageData in fontTable)
            {
                // フィルタ文字列
                string filterContent = string.Join(" ", fontUsageData.Words);

                if (fontUsageData.FontSettings is IScalableFontSettings)
                {
                    if (LEFontHelper.IsComplexFontFile(fontUsageData.FontPath)) { continue; }
                    createAutoFilterFont |= DoCretateAutoFilteredFontFromScalableFont_(fontUsageData.FontPath, fontUsageData.FontSettings as IScalableFontSettings, autoGenFontDir, filterContent, fontState, previewParam.PlatformPreference);
                }
                else if (LEFontHelper.IsFontConvertSettingsFile(fontUsageData.FontPath))
                {
                    createAutoFilterFont |= DoCretateAutoFilteredFontFromFfnt_(fontUsageData.FontPath, autoGenFontDir, filterContent, fontState, previewParam.PlatformPreference, null, null);
                }
                else if (previewParam.AutoFilteredFontGenerationEnabled)
                {
                    if (!File.Exists(fontUsageData.FontPath)) { continue; }

                    string ffntPath = Path.ChangeExtension(fontUsageData.FontPath, "ffnt");
                    if (previewParam.AutoFilteredFontGenerationFromSameNameFfntEnabled)
                    {
                        if (File.Exists(ffntPath))
                        {
                            createAutoFilterFont |= DoCretateAutoFilteredFontFromFfnt_(ffntPath, autoGenFontDir, filterContent, fontState, previewParam.PlatformPreference, fontUsageData.FontPath, AppConstants.FontFileAutoFilteredSuffix);
                        }
                    }
                    else
                    {
                        createAutoFilterFont |= DoCretateAutoFilteredFont_(fontUsageData.FontPath, autoGenFontDir, filterContent, fontState, previewParam.PlatformPreference);
                    }
                }
            }

            sw.Stop();

            if (createAutoFilterFont)
            {
                fontState.AddReport(string.Format(LECoreStringResMgr.Get("LECORE_FONT_AUTOFILTER_REPORT"), sw.ElapsedMilliseconds));
            }
        }

        static public IEnumerable<ILEFont> GetFilterSourceFonts(ILEFont root)
        {
            if (LEFontHelper.IsComplexFont(root))
            {
                return LEFontHelper.GetLeafs(root.NWFont).OfType<NW4R.Font.NW4RResFont>().Select(x => new AbstractLEFont() { FontName = Path.GetFileNameWithoutExtension(x.SourcePath), FontPath = x.SourcePath });
            }
            else
            {
                return Enumerable.Repeat<ILEFont>(root, 1);
            }

        }

        public static IEnumerable<string> FileNamesOfImageFontWithConvertSettings(IEnumerable<ISubScene> subSceneSet)
        {
            return (from subScene in subSceneSet
                    from textBox in EnumerateAllTextBox_(subScene)
                    let font = textBox.ITextBox.ILEFont
                    where font != null && LEFontHelper.IsFontConvertSettingsFile(font.FontPath)
                    select font.FontName + AppConstants.BFFNTFileExt).Distinct();

        }

        public static IEnumerable<string> FileNamesOfImageFontWithConvertSettingsInFcpx(IEnumerable<ISubScene> subSceneSet)
        {
            return (from subScene in subSceneSet
                    from font in subScene.ILEFontManager.ILEFontSet // テキストボックスペインで使われていないものも含める
                    where LEFontHelper.IsComplexFont(font) && (font.NWFont is NW4R.Font.PairFont || font.NWFont is NW4R.Font.NW4RResFont)
                    from filterSource in GetFilterSourceFonts(font)
                    where LEFontHelper.IsFontConvertSettingsFile(filterSource.FontPath)
                    select filterSource.FontName + AppConstants.BFFNTFileExt).Distinct();
        }

        /// <summary>
        /// xllt （フィルタ定義）ファイルを作る。
        /// </summary>
        private static string CreateXlltFile_(ref string filterContent)
        {
            // フィルタファイルの作成
            // 完全な空にならないようにダミー文字を挿入する。最低1文字を含むフォントが作られる。
            string filterPath = Path.Combine(Path.GetTempPath(), "auto_gen.xllt");
            filterContent += "a";

            XDocument doc = new XDocument(
                new XElement("letter-list",
                    new XAttribute("version", "1.0"),
                    new XElement("body",
                        new XElement("letter", filterContent)
                    )
                ));
            try
            {
                doc.Save(filterPath);
                return filterPath;
            }
            catch
            {
                return null;
            }
        }

        /// <summary>
        /// フィルタされたフォントの出力パスを取得する。
        /// </summary>
        private static string GetDstFilterdFontPath_(string autoGenFontDir, string fontPath)
        {
            return Path.Combine(autoGenFontDir, Path.GetFileNameWithoutExtension(fontPath) + AppConstants.FontFileAutoFilteredSuffix + AppConstants.BFFNTFileExt);
        }

        /// <summary>
        /// フィルタされたフォントの出力パスを取得する。
        /// </summary>
        private static string GetDstFilterdScFontPath_(string autoGenFontDir, string fontPath, IScalableFontSettings scfontSettings)
        {
            return Path.Combine(autoGenFontDir, scfontSettings.FontName + AppConstants.BFFNTFileExt);
        }

        /// <summary>
        /// ttf ファイルから 最小限にフィルタされた、bffnt を作る
        /// </summary>
        private static bool DoCretateAutoFilteredFontFromScalableFont_(string fontPath, IScalableFontSettings scfontSettings, string autoGenFontDir, string filterContent, AutoFilteredFontState fontState, PlatformPreference platformPreference)
        {
            List<AutoFilteredFontEntry> fontEntries = fontState.Entry;

            // スケーラブルフォントのパラメータによって、修飾された ファイル名を取得します。
            string dstFontFilePath = GetDstFilterdScFontPath_(autoGenFontDir, fontPath, scfontSettings);

            if (!File.Exists(fontPath))
            {
                // 以前の結果の削除
                if (File.Exists(dstFontFilePath))
                {
                    try
                    {
                        File.Delete(dstFontFilePath);
                    }
                    catch
                    {
                        return false;
                    }
                }
            }

            // フォントエントリを探す。
            var fontEntry = fontEntries.Find((ent) => ent.DstFontFilePath == dstFontFilePath);
            if (fontEntry == null)
            {
                fontEntry = new AutoFilteredFontEntry();
                fontEntries.Add(fontEntry);

                fontEntry.SrcFontFilePath = fontPath;
            }

            fontEntry.DstFontFilePath = dstFontFilePath;

            // フォントの高さが変わってしまう問題を回避するため、背の高そうな文字一式を追加します。
            // 下記のリストから適当に選びました。
            // http://www.fileformat.info/info/unicode/category/Lu/list.htm
            // http://www.fileformat.info/info/unicode/category/Lt/list.htm
            filterContent += "ÅÊÍÖÜÝĂĈĎĚĜĨĴḜṜṨṸṺẄỮỸἎἮᾎᾞᾯ";

            var additionalArguments = GetFontConverterAdditionalArguments(platformPreference);
            var fontConverterPath = AppConstants.FontConverterConsolePath;
            // 作り直す必要がない！
            if (fontEntry.FilterContent == filterContent
                && fontEntry.AdditionalArguments == additionalArguments
                && fontEntry.FontConverterPath == fontConverterPath
                && fontEntry.IsFontGenerated)
            {
                return false;
            }

            fontEntry.IsFontGenerated = false;

            fontEntry.FilterContent = filterContent;
            fontEntry.AdditionalArguments = additionalArguments;
            fontEntry.FontConverterPath = fontConverterPath;

            string filterPath = CreateXlltFile_(ref filterContent);
            if (string.IsNullOrEmpty(filterPath))
            {
                return false;
            }

            if (!AppConstants.IsGfxMode)
            {
                // サイズの計算専用のフィルタフォントを作成する。
                var linearFontDir = LECore.Util.TemporaryFileUtil.AutoFilteredLinearFontDir;

                string linearFontFilePath = GetDstFilterdScFontPath_(linearFontDir, fontPath, scfontSettings);

                fontEntry.LinearDstFontFilePath = linearFontFilePath;
                if (!DoExecConverterToMakeBffntFromTTF_(AppConstants.FontConverterConsolePathForToolInternalPreview, fontPath, fontEntry.LinearDstFontFilePath, scfontSettings, fontState, fontEntries, fontEntry, filterPath, GetPcFontConverterArguments(), "-in"))
                {
                    return false;
                }
            }

            string inputOption = "-in";

            // NW4F のフォントコンバータは、bfttf を -in で正しく処理できないので、-sysfont に置き換える。
            if (!AppConstants.IsGfxMode)
            {
                if (Path.GetExtension(fontPath).ToLower() == ".bfttf")
                {
                    inputOption = "-isysfont";
                }
            }

            fontEntry.IsFontGenerated = DoExecConverterToMakeBffntFromTTF_(fontEntry.FontConverterPath, fontPath, fontEntry.DstFontFilePath, scfontSettings, fontState, fontEntries, fontEntry, filterPath, fontEntry.AdditionalArguments, inputOption);
            return fontEntry.IsFontGenerated;
        }

        private static string GetFontConverterAdditionalArguments(PlatformPreference platformPreference)
        {
            return platformPreference != null ? platformPreference.FontConverterAdditionalArguments : string.Empty;
        }

        private static string GetPcFontConverterArguments()
        {
            return "-tile-mode linear";
        }

        /// <summary>
        /// ttf から bffnt を作るために コンバータを実行する。
        /// </summary>
        private static bool DoExecConverterToMakeBffntFromTTF_(string fontConverter, string fontPath, string dstFontPath, IScalableFontSettings scfontSettings, AutoFilteredFontState fontState, List<AutoFilteredFontEntry> fontEntries, AutoFilteredFontEntry fontEntry, string filterPath, string additionalArgs, string inputOption)
        {
            // 以前の結果の削除
            if (File.Exists(dstFontPath))
            {
                try
                {
                    File.Delete(dstFontPath);
                }
                catch
                {
                    return false;
                }
            }

            // フォントの生成
            // -isck カーニングに iType を使う ON
            string arg = string.Format("-skipvalidation -i win {6} \"{0}\" -ic {1} -is {4} -ia -isck -f \"{3}\" -o bffnt -of \"{2}\" -ok {5}", fontPath, "A8", dstFontPath, filterPath, scfontSettings.FontSize, additionalArgs, inputOption);
            string error;
            if (!ExecCvtr_(fontConverter, arg, out error, 60))
            {
                fontState.AddError(string.Format("ERROR : Can't create a bffnt for previewing the layout that contains scalable-font(ttf/otf).[{0}]", error));
                fontEntries.Remove(fontEntry);
                return false;
            }

            return true;
        }

        /// <summary>
        /// bffnt から 最小限にフィルタされた、bffnt を作る
        /// </summary>
        private static bool DoCretateAutoFilteredFont_(string fontPath, string autoGenFontDir, string filterContent, AutoFilteredFontState fontState, PlatformPreference platformPreference)
        {
            List<AutoFilteredFontEntry> fontEntries = fontState.Entry;
            // 変換ソースファイル取得
            string xlorFile = FindSourceFile_(fontPath, "../xlor", new string[] { ".xlor" });
            string imgFile = FindSourceFile_(fontPath, "../bmp", new string[] { ".bmp", ".tga", ".tiff", ".tif" });

            var dstFontFilePath = GetDstFilterdFontPath_(autoGenFontDir, fontPath);
            if (xlorFile == string.Empty || imgFile == string.Empty)
            {
                fontState.AddInformation(LECoreStringResMgr.Get("LECORE_FONT_AUTOFILTER_INFO"));

                // 以前の結果の削除
                if (File.Exists(dstFontFilePath))
                {
                    try
                    {
                        File.Delete(dstFontFilePath);
                    }
                    catch
                    {

                    }
                }

                return false;
            }

            // フォントエントリを探す。
            var fontEntry = fontEntries.Find((ent) => ent.SrcFontFilePath == fontPath);
            if (fontEntry == null)
            {
                fontEntry = new AutoFilteredFontEntry();
                fontEntries.Add(fontEntry);

                fontEntry.SrcFontFilePath = fontPath;
            }

            // ソースファイルに変更があるかどうかチェック。あるなら必ず変換する。
            bool isSrcSame = fontEntry.IsSrcFileSame(xlorFile, imgFile);
            if (!isSrcSame)
            {
                fontEntry.SrcXlorState = AutoFilteredFontEntry.GetFileState(xlorFile);
                fontEntry.SrcBmpState = AutoFilteredFontEntry.GetFileState(imgFile);
            }

            var additionalArguments = GetFontConverterAdditionalArguments(platformPreference);
            var fontConverterPath = AppConstants.FontConverterConsolePath;

            // 作り直す必要がない！
            if (isSrcSame &&
                fontEntry.FilterContent == filterContent &&
                fontEntry.AdditionalArguments == additionalArguments &&
                fontEntry.FontConverterPath == fontConverterPath &&
                fontEntry.IsFontGenerated)
            {
                return false;
            }

            fontEntry.IsFontGenerated = false;

            fontEntry.FilterContent = filterContent;
            fontEntry.AdditionalArguments = additionalArguments;
            fontEntry.FontConverterPath = fontConverterPath;

            fontEntry.DstFontFilePath = dstFontFilePath;

            string filterPath = CreateXlltFile_(ref filterContent);
            if (string.IsNullOrEmpty(filterPath))
            {
                return false;
            }

            // 以前の結果の削除
            if (File.Exists(fontEntry.DstFontFilePath))
            {
                try
                {
                    File.Delete(fontEntry.DstFontFilePath);
                }
                catch (Exception e)
                {
                    fontState.AddError(LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "\r\n" + e.Message);
                    return false;
                }
            }

            string error;

            // フォーマットを抽出
            bool cvtrResult = false;
            string formatFilePath = Path.Combine(Path.GetTempPath(), "auto_gen_fmt.txt");
            string optionString = string.Format("-i bffnt -if {0} -o format -of {1} {2}", fontPath, formatFilePath, fontEntry.AdditionalArguments);
            cvtrResult = ExecCvtr_(AppConstants.FontConverterConsolePath, optionString, out error, 60);
            if (!cvtrResult)
            {
                fontState.AddError(LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "\r\n" + AppConstants.FontConverterConsolePath + " " + optionString + "\r\n" + error);
                return false;
            }

            string formatString = File.ReadAllText(formatFilePath);

            // フォントの生成
            optionString = string.Format("-skipvalidation -i image -if \"{0}\" -io \"{1}\" -ic {2}  -f \"{4}\" -o bffnt -of \"{3}\" {5} -ok",
                imgFile, xlorFile, formatString, fontEntry.DstFontFilePath, filterPath, fontEntry.AdditionalArguments);
            cvtrResult = ExecCvtr_(AppConstants.FontConverterConsolePath, optionString, out error, 60);
            if (!cvtrResult)
            {
                fontState.AddError(LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "\r\n" + AppConstants.FontConverterConsolePath + " " + optionString + "\r\n" + error);
                return false;
            }

            // ★暫定対処
            // テクスチャサイズが小さすぎる場合に問題が発生する。(ランタイム側で正しくテクスチャを復元できない問題がある)
            // 出力されたフォントのサイズがかなり小さいことを検出して、シートサイズを指定して再度フォント出力をおこなう。
            // ランタイム側の問題が解消したら、この処理は不要になるので削除する。
            // TODO: PlatformDetail への依存を消したい。現在は値が不変なので問題ない
            if (LayoutEditorCore.PlatformDetail.SmallSizeFontProblemWorkaroundEnabled)
            {
                if ((new FileInfo(fontEntry.DstFontFilePath).Length <= 16 * 1024))
                {
                    cvtrResult = ExecCvtr_(AppConstants.FontConverterConsolePath, optionString + " -ox 64x64", out error, 60);
                    if (!cvtrResult)
                    {
                        fontState.AddError(LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "\r\n" + AppConstants.FontConverterConsolePath + " " + optionString + "\r\n" + error);
                        return false;
                    }
                }
            }

            fontEntry.IsFontGenerated = true;
            return true;
        }

        /// <summary>
        /// ffnt から 最小限にフィルタされた、bffnt を作る
        /// </summary>
        private static bool DoCretateAutoFilteredFontFromFfnt_(string fontPath, string autoGenFontDir, string filterContent, AutoFilteredFontState fontState, PlatformPreference platformPreference, string fontEntrySrcFontPath, string dstFontSuffix)
        {
            var dstFontFilePath = Path.Combine(autoGenFontDir, Path.GetFileNameWithoutExtension(fontPath) + (dstFontSuffix ?? string.Empty) + AppConstants.BFFNTFileExt);

            if (string.IsNullOrEmpty(fontEntrySrcFontPath))
            {
                // fontEntrySrcFontPath が指定されていなければ、fontPath を使う
                fontEntrySrcFontPath = fontPath;
            }

            if (!File.Exists(fontPath))
            {
                // 以前の結果の削除
                if (File.Exists(dstFontFilePath))
                {
                    try
                    {
                        File.Delete(dstFontFilePath);
                    }
                    catch
                    {
                    }
                }

                fontState.AddError(string.Format(LECore.LECoreStringResMgr.Get("FILE_NOT_EXISTS"), fontPath));
                return false;
            }

            List<AutoFilteredFontEntry> fontEntries = fontState.Entry;

            // フォントエントリを探す。
            var fontEntry = fontEntries.Find((ent) => ent.SrcFontFilePath == fontEntrySrcFontPath);
            if (fontEntry == null)
            {
                fontEntry = new AutoFilteredFontEntry();
                fontEntries.Add(fontEntry);

                fontEntry.SrcFontFilePath = fontEntrySrcFontPath;
            }

            var additionalArguments = GetFontConverterAdditionalArguments(platformPreference);
            var fontConverterPath = AppConstants.FontConverterConsolePath;

            // 作り直す必要がない！
            if (fontEntry.FilterContent == filterContent &&
                fontEntry.AdditionalArguments == additionalArguments &&
                fontEntry.FontConverterPath == fontConverterPath &&
                fontEntry.IsFontGenerated)
            {
                return false;
            }

            fontEntry.IsFontGenerated = false;

            fontEntry.FilterContent = filterContent;
            fontEntry.AdditionalArguments = additionalArguments;
            fontEntry.FontConverterPath = fontConverterPath;

            fontEntry.DstFontFilePath = dstFontFilePath;

            string filterPath = CreateXlltFile_(ref filterContent);
            if (string.IsNullOrEmpty(filterPath))
            {
                return false;
            }

            // 以前の結果の削除
            if (File.Exists(fontEntry.DstFontFilePath))
            {
                try
                {
                    File.Delete(fontEntry.DstFontFilePath);
                }
                catch (Exception e)
                {
                    fontState.AddError(e.Message);
                    return false;
                }
            }

            // フォントの生成
            string optionString = string.Format("-skipvalidation -i ffnt -if \"{0}\" -o bffnt -of \"{1}\" -f {2} {3} -ok -nocache",
                fontPath, fontEntry.DstFontFilePath, filterPath, fontEntry.AdditionalArguments);

            string error;

            // TODO: PlatformDetail への依存を消したい。現在は値が不変なので問題ない
            if (!LayoutEditorCore.PlatformDetail.IsGpuEncodingAvailable)
            {
                // 問い合わせ
                string formatFile = Path.ChangeExtension(fontEntry.DstFontFilePath, ".txt");
                string query = optionString + string.Format(" -o format -of \"{0}\"", formatFile);

                if (!ExecCvtrForToolInternalPreview_(query, out error))
                {
                    fontState.AddError(LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "; " + error + "; " + query);
                    return false;
                }

                try
                {
                    // BC3 に置き換え
                    var content = File.ReadAllText(formatFile);
                    GlyphImageFormat format;
                    if (Enum.TryParse(content, out format))
                    {
                        switch (format)
                        {
                            case GlyphImageFormat.RGBA8_BC7:
                                optionString += " -ic RGB5A3";
                                break;
                            case GlyphImageFormat.RGBA8_BC7_SRGB:
                                optionString += " -ic RGB5A3_SRGB";
                                break;
                        }
                    }
                }
                catch (Exception e)
                {
                    fontState.AddError(LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "; " + e.Message);
                    return false;
                }
            }

            bool cvtrResult = ExecCvtr_(fontEntry.FontConverterPath, optionString, out error, 60);
            if (!cvtrResult)
            {
                fontState.AddError(LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "\r\n" + fontEntry.FontConverterPath + " " + optionString + "\r\n" + error);
                return false;
            }

            // ★暫定対処
            // テクスチャサイズが小さすぎる場合に問題が発生する。(ランタイム側で正しくテクスチャを復元できない問題がある)
            // 出力されたフォントのサイズがかなり小さいことを検出して、シートサイズを指定して再度フォント出力をおこなう。
            // ランタイム側の問題が解消したら、この処理は不要になるので削除する。
            // TODO: PlatformDetail 依存をなくす。現在は常に true だから問題ない
            if (LayoutEditorCore.PlatformDetail.SmallSizeFontProblemWorkaroundEnabled)
            {
                if ((new FileInfo(fontEntry.DstFontFilePath).Length <= 16 * 1024))
                {
                    cvtrResult = ExecCvtr_(fontEntry.FontConverterPath, optionString + " -ox 64x64", out error, 60);
                    if (!cvtrResult)
                    {
                        fontState.AddError(error);
                        return false;
                    }
                }
            }

            fontEntry.IsFontGenerated = true;
            return true;
        }

        /// <summary>
        /// コンバータの実行
        /// </summary>
        static bool ExecCvtr_0(string arg)
        {
            // フォントの生成
            Process proc = new Process();
            proc.StartInfo.FileName = AppConstants.FontConverterConsolePath;
            proc.StartInfo.CreateNoWindow = true;
            proc.StartInfo.UseShellExecute = false;
            proc.StartInfo.Arguments = arg;
            proc.Start();
            if (!proc.WaitForExit(60000))
            {
                return false;
            }

            if (proc.ExitCode != 0)
            {
                return false;
            }

            return true;
        }

        static bool ExecCvtr_(string converterPath, string arg, out string error, int timeoutSec = 0)
        {
            // メモ：
            // ツール内の表示に利用するため一時的に生成するフォントは 最新のコンバーターを使ってリニアで作成する必要がある。
            // LayoutEditor は NW4F のフォントを解釈できない。
            // 一時フォント生成用に、FontConverterConsolePathForToolInternalPreview を別途用意した。
            //if (!AppConstants.IsGfxMode)
            //{
            //    arg = Regex.Replace(arg, "-p .+? ", "");
            //    arg += " -tile-mode linear";
            //}

            if(!string.IsNullOrEmpty(IsGpuEncodingEnalbed))
            {
                arg += " -gpu-encoding " + IsGpuEncodingEnalbed;
            }

            var errorBuilder = new StringBuilder();
            var errorType = "";
            var ret = ProcessUtil.ProcessStart(converterPath, arg, (s, e) => { errorBuilder.Append((e != null && e.Data != null) ? e.Data : ""); },
                (s, e) =>
                {
                    errorBuilder.Append((e != null && e.Data != null) ? e.Data : "");
                },
                    out errorType, timeoutSec);
            error = errorType + errorBuilder.ToString();
            return ret == 0;
        }

        static bool ExecCvtrForToolInternalPreview_(string arg, out string error, int timeoutSec = 0)
        {
            var errorBuilder = new StringBuilder();
            var errorType = "";

            if (!string.IsNullOrEmpty(IsGpuEncodingEnalbed))
            {
                arg += " -gpu-encoding " + IsGpuEncodingEnalbed;
            }

            var ret = ProcessUtil.ProcessStart(AppConstants.FontConverterConsolePathForToolInternalPreview, arg, null,
                (s, e) =>
                {
                    errorBuilder.Append((e != null && e.Data != null) ? e.Data : "");
                },
                    out errorType, timeoutSec);
            error = errorType + errorBuilder.ToString();
            return ret == 0;
        }

        public enum GlyphImageFormat
        {
            A4,
            A8,

            LA4,
            LA8,
            LA4Packed,
            LA4PackedPassThrough, // LA4Packed形式の bffnt を読み込んで、LA4Packed固有の画像スキップしてファイル出力する場合に使う。
            LA4PackedNoCompress,
            LA4PackedNoCompressPassThrough, // LA4PackedNoCompress形式の bffnt を読み込んで、LA4PackedNoCompress固有の画像スキップしてファイル出力する場合に使う。

            RGB565,
            RGB5A1,
            RGB5A3,
            RGBA4,

            RGB8,
            RGBA8,

            RGBA8_SRGB, // リニア色空間用 SRGBフェッチ形式
            RGB565_SRGB, //リニア色空間用 SRGBフェッチ形式
            RGB5A3_SRGB, // リニア色空間用 SRGBフェッチ形式

            RGBA8_BC7, // リニア色空間用 SRGBフェッチ形式
            RGBA8_BC7_SRGB, // リニア色空間用 SRGBフェッチ形式

            Max
        }

        internal static bool DoExecConverterToMakeBffntFromFfntForToolInternalPreview(string filePath, string outputPath, string platform, string filterFilePath, bool nocache, out string error, out bool replacedToBC3)
        {
            replacedToBC3 = false;
            error = string.Empty;

            string arg = string.Format("-i ffnt -if \"{0}\" -o bffnt -of \"{1}\" -tile-mode linear", filePath, outputPath);

            if (!string.IsNullOrEmpty(filterFilePath))
            {
                arg += string.Format(" -f \"{0}\"", filterFilePath);
            }
            else
            {
                // LayoutEditor 上で表示される ffnt に関しては、ffnt 内のフィルタは無視する。
                arg += " -ignore-filter";
            }

            if (nocache)
            {
                arg += " -nocache";
            }

            // TODO: PlatformDetail への依存を消したい。現在は値が不変なので問題ない
            if (!LayoutEditorCore.PlatformDetail.IsGpuEncodingAvailable)
            {
                // 問い合わせ
                string formatFile = Path.ChangeExtension(outputPath, ".txt");
                string query = string.Format("-i ffnt -if \"{0}\" -o format -of \"{1}\" -tile-mode linear", filePath, formatFile);

                if (!ExecCvtrForToolInternalPreview_(query, out error))
                {
                    error = LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "; " + error + "; " + arg;
                    return false;
                }

                try
                {
                    // BC3 に置き換え
                    var content = File.ReadAllText(formatFile);
                    GlyphImageFormat format;
                    if (Enum.TryParse(content, out format))
                    {
                        switch (format)
                        {
                            case GlyphImageFormat.RGBA8_BC7:
                                arg += " -ic RGB5A3";
                                replacedToBC3 = true;
                                break;
                            case GlyphImageFormat.RGBA8_BC7_SRGB:
                                arg += " -ic RGB5A3_SRGB";
                                replacedToBC3 = true;
                                break;
                        }
                    }
                }
                catch (Exception e)
                {
                    error = LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "; " + e.Message;
                    return false;
                }
            }

            var sw = new Stopwatch();
            sw.Start();
            if (!ExecCvtrForToolInternalPreview_(arg, out error))
            {
                error = LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "; " + error + "; " + arg;
                sw.Stop();
                LECore.DbgConsole.WriteLine("FontConvert {0} : {1} ms", arg, sw.ElapsedMilliseconds);
                return false;
            }
            sw.Stop();
            LECore.DbgConsole.WriteLine("FontConvert {0} : {1} ms", arg, sw.ElapsedMilliseconds);

            // ★暫定対処
            // テクスチャサイズが小さすぎる場合に問題が発生する。(ランタイム側で正しくテクスチャを復元できない問題がある)
            // 出力されたフォントのサイズがかなり小さいことを検出して、シートサイズを指定して再度フォント出力をおこなう。
            // ランタイム側の問題が解消したら、この処理は不要になるので削除する。
            // TODO: PlatformDetail への依存を消したい。現在は値が不変なので問題ない
            if (LayoutEditorCore.PlatformDetail.SmallSizeFontProblemWorkaroundEnabled)
            {
                if ((new FileInfo(outputPath).Length <= 16 * 1024))
                {
                    if (!ExecCvtrForToolInternalPreview_(arg + " -ox 64x64", out error))
                    {
                        error = LECore.LECoreStringResMgr.Get("CONVERTER_ERROR_FAILCONVERT_FONT") + "; " + error + "; " + arg;
                        return false;
                    }
                }
            }

            return true;
        }
    }

    /// <summary>
    /// フォントマネージャ変更通知を受け取るクラスが実装するインタフェースです。
    /// </summary>
    internal interface ILEFontMgrUpdateListener
    {
        /// <summary>
        /// デリゲータ OnLEFontMgrUpdateHandler を取得します。
        /// </summary>
        OnLEFontMgrUpdateHandler OnLEFontMgrUpdateHandler { get; }
    }

    /// <summary>
    /// フォント生成クラス。
    ///
    /// フォントクラスを生成する静的メソッドを集約しているクラスです。
    /// </summary>
    internal class FontFactory
    {
        /// <summary>
        /// フォント名の同一判定で、大文字小文字を区別するか。
        /// </summary>
        const bool _IgnoreCase = true;

        static List<LEFont> _fontSet = new List<LEFont>();
        static Dictionary<LEFont, List<FontManager>> _fontReferenceMap = new Dictionary<LEFont, List<FontManager>>();
        static NW4R.Font.ITextureConvertUtility _textureConvertUtility = null;
        public static NW4R.Font.ITextureConvertUtility TextureConverterUtility {
            get {
                Debug.Assert(_textureConvertUtility != null);
                return _textureConvertUtility;
            }
        }
        static bool _endianSwapEnabled;
        static Size _textureCacheSizeForScalableFont = new Size(1024, 2048);
        public static bool ForceConvertBffntFromSameNameFfnt { get; set; }

        /// <summary>
        /// 静的コンストラクタ
        /// </summary>
        static FontFactory()
        {
            // TODO: 分かりづらく非常に危ない気がする
            // Visual Studio のデザイナで例外が起きることがある
            if (!InitializeNativeFontConverter_(LayoutEditorCore.PlatformDetail.PlatformName))
            {
                throw new Exception(LECoreStringResMgr.Get("LECORE_FONT_ERROR_INITIALIZE", LayoutEditorCore.PlatformDetail.PlatformName));
            }
        }

        public static bool InitializeNativeFontConverter_(string platformName)
        {
            string fontPlatformDllName = "NWFont" + platformName;
            try
            {
                Type type = LayoutEditorCore.EnumrateInterfaceTypesFromDLL((dllFileName) => dllFileName == fontPlatformDllName, typeof(NW4R.Font.ITextureConvertUtility)).First();

                _textureConvertUtility = Activator.CreateInstance(type) as NW4R.Font.ITextureConvertUtility;

                // フォント読み込み用にプラットフォーム名を渡す
                if (_textureConvertUtility != null)
                {
                    var grahicsToolPath = Path.Combine(LayoutEditorCore.PlatformDetail.NwToolsRootPath, "GraphicsTools");
                    var nameMem = Marshal.StringToCoTaskMemUni(LayoutEditorCore.PlatformDetail.UIPlatformName);
                    var pathMem = Marshal.StringToCoTaskMemUni(grahicsToolPath);
                    _textureConvertUtility.SetPlatform(nameMem, pathMem);
                    Marshal.FreeCoTaskMem(nameMem);
                    Marshal.FreeCoTaskMem(pathMem);
                }
                _endianSwapEnabled = _textureConvertUtility.IsFontResourceBigEndian();
                return true;
            }
            catch (Exception e)
            {
                // 静的コンストラクタで例外がおきると分かりづらいので、アサートしておく
                // なお TypeInitalizationException の InnerException でも例外発生場所は分かります。
                Debug.Assert(false, e.ToString());
                throw e;
            }
        }

        /// <summary>
        /// OpenGLコンテキストの初期化です。
        /// </summary>
        public static bool InitializeOpenGLContext(IntPtr hdc)
        {
            return _textureConvertUtility.InitializeOpenGL(hdc);
        }

        /// <summary>
        /// OpenGLコンテキストの破棄です。
        /// </summary>
        public static void FinalizeOpenGLContext()
        {
            _textureConvertUtility.FinalizeOpenGL();
        }

        /// <summary>
        /// スケーラブルフォント用のテクスチャキャッシュサイズを設定します。
        /// </summary>
        public static void SetTextureCacheSizeForScalableFont(Size size)
        {
            _textureCacheSizeForScalableFont = size;
        }

        /// <summary>
        /// フォントの内容を更新します。
        /// </summary>
        static public bool UpdateFont( ILEFont ifont, string newFilePath )
        {
            LEFont font = ifont as LEFont;

            if( string.Compare( Path.GetFileNameWithoutExtension( newFilePath ), font.FontName, _IgnoreCase ) == 0 )
            {
                if (LEFontHelper.IsBitmapFontFile(newFilePath))
                {
                    string ffntPath = Path.ChangeExtension(newFilePath, "ffnt");
                    if (ForceConvertBffntFromSameNameFfnt && File.Exists(ffntPath))
                    {
                        string error;
                        bool replacedToBC3;
                        var bffnt = ConvertFfntToBffnt(font.FontName, ffntPath, out error, out replacedToBC3);
                        if (!string.IsNullOrEmpty(bffnt))
                        {
                            InitializeTextureImage_(font, bffnt, font.FontName, newFilePath, true, replacedToBC3);
                        }
                    }
                    else
                    {
                        InitializeTextureImage_(font, newFilePath, null);
                    }
                }
                else if (LEFontHelper.IsFontConvertSettingsFile(newFilePath))
                {
                    string error;
                    bool replacedToBC3;
                    var bffntPath = ConvertFfntToBffnt(font.FontName, newFilePath, out error, out replacedToBC3);
                    if (!string.IsNullOrEmpty(bffntPath))
                    {
                        InitializeTextureImage_(font, bffntPath, font.FontName, newFilePath, true, replacedToBC3);
                    }
                    else
                    {
                        font.InitializeInternal(null, font.FontName, newFilePath, null, null, error);
                    }
                }
                else if (LEFontHelper.IsComplexFontFile(newFilePath))
                {
                    // 複合フォントの初期化
                    var settings = new ScalableFontSettings
                    {
                        FontName = font.FontName,
                        FontFilePath = newFilePath,
                    };
                    InitializeComplexFont(settings, font);
                }
                else
                {
                    return false;
                }

                return true;
            }

            return false;
        }

      /// <summary>
      ///
      /// </summary>
      private enum BinaryFileValidateCode
      {
          Valid = 0,
          InvalidSignature = 1,
          InvalidByteOrderMark = 2,
          InvalidVersion = 3,
          InvalidFileSize = 4,
          InvalidBlockSize = 5,
          InvalidPlatform = 6,
          UnknownInvalid = -1,
      };

      /// <summary>
      /// フォントを初期化します。
      /// </summary>
      private static void InitializeTextureImage_( LEFont newFont, string inputFilePath, string fontName, string filePath = null, bool skipBntxPlatformCheck = false, bool replacedToBC3 = false, IFontSettings fontSettings = null)
      {
            using (FileStream fontFile = new FileStream(inputFilePath, FileMode.Open, FileAccess.Read))
            {
                byte[] fileBuffer = new byte[fontFile.Length];
                fontFile.Read(fileBuffer, 0, (int)fileBuffer.Length);

                var nw4rFont = CreateNW4RResFont(fileBuffer, inputFilePath);

                // リソース設定後にプラットフォームを取得
                // メモ：ツール内描画のために一時的にりようする用途においては、対象プラットフォームと異なるプラットフォームを許容する必要がありそう。
                string platform = nw4rFont.GetPlatform();

                // 新規作成して、登録済みフォントに記録
                if (string.IsNullOrEmpty(fontName))
                {
                    fontName = Path.GetFileNameWithoutExtension(inputFilePath);
                }
                newFont.InitializeInternal(nw4rFont, fontName, filePath ?? inputFilePath, skipBntxPlatformCheck ? null : platform, fontSettings);

                if (replacedToBC3 && string.IsNullOrEmpty(newFont.InitializationErrorMessage))
                {
                    newFont.InitializationErrorMessage = LECoreStringResMgr.Get("WARNING_FONT_BC7_BC3");
                }
            }
        }

        private static NW4R.Font.NW4RResFont CreateNW4RResFont(byte[] fileBuffer, string inputFilePath)
        {
            // ファイルストリームから新しいフォントインスタンスの構築を試みます。
            NWFont.NW4RResFont nw4rFont = new NW4R.Font.NW4RResFont();

            // フォントのヘッダ情報の検証を行います。
            int resultCode = NW4R.Font.NW4RResFont.ValidateBinary(fileBuffer, _textureConvertUtility.GetFontResourceVersion());
            BinaryFileValidateCode result = (BinaryFileValidateCode)Enum.Parse(typeof(BinaryFileValidateCode), resultCode.ToString());
            switch (result)
            {
                case BinaryFileValidateCode.InvalidVersion:
                    throw new Exception(LECoreStringResMgr.Get("LECORE_FONT_ERROR_INVALID_VERSION", Path.GetFileName(inputFilePath)));
                case BinaryFileValidateCode.InvalidByteOrderMark:
                    throw new Exception(LECoreStringResMgr.Get("LECORE_FONT_ERROR_INVALID_BYTEORDER", Path.GetFileName(inputFilePath)));
                case BinaryFileValidateCode.InvalidSignature:
                case BinaryFileValidateCode.InvalidFileSize:
                case BinaryFileValidateCode.InvalidBlockSize:
                case BinaryFileValidateCode.UnknownInvalid:
                    throw new Exception(LECoreStringResMgr.Get("LECORE_SYS_ERROR_LOAD_FONT", Path.GetFileName(inputFilePath)));
            }

            // リソース設定に失敗した場合...
            bool bResult = nw4rFont.SetResource(fileBuffer, _textureConvertUtility, _endianSwapEnabled, null);
            if (!bResult)
            {
                throw new Exception(LECoreStringResMgr.Get("LECORE_SYS_ERROR_LOAD_FONT", Path.GetFileName(inputFilePath)));
            }

            return nw4rFont;
        }

        /// <summary>
        /// ファイルからフォントを生成します。
        ///
        /// 複数のフォントマネージャで、実体を共有するように試みます。
        /// 正しく、リソースの開放を行うため、参照カウントのような処理をおこなっています。
        ///
        ///
        ///
        /// 参照するフォントマネージャが無くなった場合に、フォントは自身のDisposeメソッドを
        /// 呼び出して、リソースの開放を実行します。
        /// </summary>
        static public LEFont BuildFontFromFile( FontManager owner, string fontName, string filePath, bool skipBntxPlatformCheck)
        {
            LEFont newFont = _fontSet.Find(font => string.Compare(font.FontPath, filePath, _IgnoreCase) == 0);

            if (newFont == null)
            {
                fontName = Path.GetFileNameWithoutExtension(fontName);
                if (LEFontHelper.IsComplexFontFile(filePath))
                {
                    // 複合フォントの初期化
                    var settings = new ScalableFontSettings
                                       {
                                           FontName = fontName,
                                           FontFilePath = filePath
                                       };
                    // BuildFontFromScalableFontSettings() で登録(DoRegister_)されるのでここでリターン
                    return BuildFontFromScalableFontSettings(owner, settings);
                }
                else if (LEFontHelper.IsFontConvertSettingsFile(filePath))
                {
                    // ffnt の読み込み
                    return BuildFontFromFontConvertSettingsFile(owner, fontName, filePath);
                }
                else
                {
                    newFont = new LEFont();
                    try
                    {
                        string ffntPath = Path.ChangeExtension(filePath, "ffnt");
                        if (ForceConvertBffntFromSameNameFfnt && File.Exists(ffntPath))
                        {
                            // bffnt の代わりに隣にある ffnt を使う
                            string error;
                            bool replacedToBC3;
                            var bffnt = ConvertFfntToBffnt(fontName, ffntPath, out error, out replacedToBC3);
                            if (!string.IsNullOrEmpty(bffnt))
                            {
                                InitializeTextureImage_(newFont, bffnt, fontName, filePath, true, replacedToBC3);
                            }
                            else
                            {
                                newFont.InitializeInternal(null, fontName, filePath, null, null, error);
                            }
                        }
                        else
                        {
                            InitializeTextureImage_(newFont, filePath, null, null, skipBntxPlatformCheck);
                        }
                    }
                    catch (Exception e)
                    {
                        newFont.InitializeInternal(null, fontName, filePath, null, null, e.Message);
                    }
                }
            }

            DoRegister_(owner, newFont);

            return newFont;
        }

        static public LEFont BuildFontForPreview(string fontName, string filePath)
        {
            var newFont = new LEFont();
            try
            {
                InitializeTextureImage_(newFont, filePath, null, null, true);
            }
            catch (Exception e)
            {
                newFont.InitializeInternal(null, fontName, filePath, null, null, e.Message);
            }
            return newFont;
        }

        private static LEFont BuildFontFromFontConvertSettingsFile(FontManager owner, string fontName, string filePath)
        {
            LEFont newFont = _fontSet.Find(font => string.Compare(font.FontPath, filePath, _IgnoreCase) == 0);

            if (newFont == null)
            {
                newFont = new LEFont();

                try
                {
                    string error;
                    bool replacedToBC3;
                    var bffntFilePath = ConvertFfntToBffnt(fontName, filePath, out error, out replacedToBC3);
                    if (!string.IsNullOrEmpty(bffntFilePath))
                    {
                        InitializeTextureImage_(newFont, bffntFilePath, fontName, filePath, true, replacedToBC3);
                    }
                    else
                    {
                        newFont.InitializeInternal(null, fontName, filePath, null, null, error);
                    }
                }
                catch (Exception e)
                {
                    newFont.InitializeInternal(null, fontName, filePath, null, null, e.Message);
                }
            }

            DoRegister_(owner, newFont);

            return newFont;
        }

        private static string ConvertFfntToBffnt(string fontName, string filePath, out string error, out bool replacedToBC3)
        {
            // AppConstants.PlatformName を使っているので UI スレッド以外からは呼ばない

            replacedToBC3 = false;
            var tempFontDir = LECore.Util.TemporaryFileUtil.TempImageFontDir;
            var platformFontDir = Path.Combine(tempFontDir, AppConstants.PlatformName);
            if (!Directory.Exists(platformFontDir))
            {
                Directory.CreateDirectory(platformFontDir);
            }

            var bffntFileName = fontName + AppConstants.BFFNTFileExt;
            var bffntFilePath = Path.Combine(platformFontDir, bffntFileName);

            if (AutoFilteredFontUtil.DoExecConverterToMakeBffntFromFfntForToolInternalPreview(filePath, bffntFilePath, AppConstants.PlatformName, string.Empty, false, out error, out replacedToBC3))
            {
                return bffntFilePath;
            }

            return string.Empty;
        }

        /// <summary>
        /// スケーラブルフォントを生成します。
        /// </summary>
        static public LEFont BuildFontFromScalableFontSettings(FontManager owner, IScalableFontSettings settings)
        {
            // まったく同じフォントがすでに作成されていれば使いまわします。
            LEFont newFont = _fontSet.Find(delegate(LEFont font)
            {
                if (!font.IsScalableFont())
                {
                    return false;
                }

                return (font.FontSettings as IScalableFontSettings).IsSame(settings);
            });

            if (newFont == null)
            {
                newFont = new LEFont();
                DbgConsole.WriteLine("New----ScFont-Created[{0}]", settings.FontName);
                var filePath = settings.FontFilePath;
                if (LEFontHelper.IsComplexFontFile(filePath))
                {
                    // 複合フォントの読み込み
                    InitializeComplexFont(settings, newFont);
                }
                else
                {
                    try
                    {
                        var scfont = new NW4R.Font.ScFont();

                        var initArg = new NW4R.Font.ScFont.InitializeArg
                                          {
                                              fontDatas = new IntPtr[1][],
                                              fontDataSizes = new int[1][],
                                              fontBoldWeights = new float[1][],
                                              fontBorderSizes = new int[1][],
                                              fontScaleWidths = new float[1][],
                                              fontScaleHeights = new float[1][],
                                              fontTypes = new int[1][],
                                              ignorePalt = new bool[1][],
                                              deleteBearingX = new bool[1][],
                                              bearingOffsetX = new int[1][],
                                              forceMonospacedEnabled = new bool[1][],
                                              forceMonospacedWidth = new int[1][],
                                              baselineOffset = new int[1][],
                                              charCodeRangeCount = new int[1][],
                                              charCodeRangeFirst = new uint[1][][],
                                              charCodeRangeLast = new uint[1][][]
                                          };

                        initArg.fontDatas[0] = new IntPtr[1];
                        initArg.fontDataSizes[0] = new int[1];
                        initArg.fontBoldWeights[0] = new float[1];
                        initArg.fontBorderSizes[0] = new int[1];
                        initArg.fontScaleWidths[0] = new float[1];
                        initArg.fontScaleHeights[0] = new float[1];
                        initArg.fontTypes[0] = new int[1];
                        initArg.ignorePalt[0] = new bool[1];
                        initArg.deleteBearingX[0] = new bool[1];
                        initArg.bearingOffsetX[0] = new int[1];
                        initArg.forceMonospacedEnabled[0] = new bool[1];
                        initArg.forceMonospacedWidth[0] = new int[1];
                        initArg.baselineOffset[0] = new int[1];
                        initArg.charCodeRangeCount[0] = new int[1];
                        initArg.charCodeRangeFirst[0] = new uint[1][];
                        initArg.charCodeRangeLast[0] = new uint[1][];

                        initArg.fontBoldWeights[0][0] = settings.BoldWeight;
                        initArg.fontBorderSizes[0][0] = settings.BorderWidth;
                        initArg.fontScaleWidths[0][0] = 1.0f;
                        initArg.fontScaleHeights[0][0] = 1.0f;
                        initArg.fontTypes[0][0] = 0;
                        initArg.ignorePalt[0][0] = false;
                        initArg.deleteBearingX[0][0] = false;
                        initArg.bearingOffsetX[0][0] = 0;
                        initArg.forceMonospacedEnabled[0][0] = false;
                        initArg.forceMonospacedWidth[0][0] = 0;
                        initArg.baselineOffset[0][0] = 0;
                        if (filePath.EndsWith("bfttf"))
                        {
                            initArg.fontTypes[0][0] = 2;
                        }
                        else if (filePath.EndsWith("bfotf"))
                        {
                            initArg.fontTypes[0][0] = 3;
                        }
                        else if (filePath.EndsWith("otf"))
                        {
                            initArg.fontTypes[0][0] = 1;
                        }
                        initArg.fontSize = settings.FontSize;
                        initArg.alternateChar = '?';
                        initArg.lineFeedOffset = 0;
                        initArg.charCodeRangeCount[0][0] = 0;
                        initArg.charCodeRangeFirst[0][0] = new uint[0];
                        initArg.charCodeRangeLast[0][0] = new uint[0];

                        initArg.textureCacheWidth = (uint)_textureCacheSizeForScalableFont.Width;
                        initArg.textureCacheHeight = (uint)_textureCacheSizeForScalableFont.Height;

                        scfont.SetResource(File.ReadAllBytes(filePath), _textureConvertUtility, _endianSwapEnabled, initArg);

                        scfont.UpdateAndCompleteTheTextureCache();
                        newFont.InitializeInternal(scfont, settings.FontName, filePath, null, settings);
                    }
                    catch (Exception e)
                    {
                        newFont.InitializeInternal(null, settings.FontName, filePath, null, settings, e.Message);
                    }
                }


            }

            DoRegister_(owner, newFont);

            return newFont;
        }

        // ComplexFont の初期化結果
        private class InitializeComplexFontTreeResult
        {
            public bool ReplacedToBC3 { get; set; } = false;
        }

        private static NW4R.Font.INWFont InitializeComplexFontTree(
            SubFont subfont,
            string fcpxPath,
            InitializeComplexFontTreeResult result
            )
        {
            if (subfont.Item is BitmapFont)
            {
                BitmapFont item = (BitmapFont)subfont.Item;
                var path = Path.GetFullPath(Path.Combine(Path.GetDirectoryName(fcpxPath), item.path));
                string bffntPath;
                bool replacedToBC3 = false;
                if (LEFontHelper.IsFontConvertSettingsFile(path))
                {
                    string error;
                    var name = Path.GetFileNameWithoutExtension(path);
                    bffntPath = ConvertFfntToBffnt(name, path, out error, out replacedToBC3);
                    if (string.IsNullOrEmpty(bffntPath))
                    {
                        throw new Exception(error);
                    }
                }
                else
                {
                    bffntPath = path;
                }

                using (FileStream fontFile = new FileStream(bffntPath, FileMode.Open, FileAccess.Read))
                {
                    byte[] fileBuffer = new byte[fontFile.Length];
                    fontFile.Read(fileBuffer, 0, (int)fileBuffer.Length);
                    var font = CreateNW4RResFont(fileBuffer, bffntPath);
                    font.SourcePath = path;
                    if (item.charCodeRangeSet != null && item.charCodeRangeSet.Length > 0)
                    {
                        uint[] first = new uint[item.charCodeRangeSet.Length];
                        uint[] last = new uint[item.charCodeRangeSet.Length];
                        for (int i = 0; i < item.charCodeRangeSet.Length; i++)
                        {
                            first[i] = item.charCodeRangeSet[i].first;
                            last[i] = item.charCodeRangeSet[i].last;
                        }
                        font.SetCharCodeRange(first, last);
                    }

                    if (replacedToBC3)
                    {
                        result.ReplacedToBC3 |= replacedToBC3;
                    }

                    return font;
                }
            }
            else if (subfont.Item is MultiScalableFont)
            {
                MultiScalableFont subfontItem = (MultiScalableFont)subfont.Item;
                return CreateScFont(subfontItem.size, subfontItem.scalableFontDescriptionSet, fcpxPath, subfontItem.alternateChar, subfontItem.lineFeedOffset);
            }
            else if (subfont.Item is ScalableFont)
            {
                ScalableFont subfontItem = (ScalableFont)subfont.Item;
                return CreateScFont(subfontItem.size, new[] { subfontItem.scalableFontDescription }, fcpxPath, subfontItem.alternateChar, subfontItem.lineFeedOffset);
            }
            else if (subfont.Item is PairFont)
            {
                var first = InitializeComplexFontTree(((PairFont)subfont.Item).firstFont, fcpxPath, result);
                var second = InitializeComplexFontTree(((PairFont)subfont.Item).secondFont, fcpxPath, result);
                return new NW4R.Font.PairFont(first, second);
            }

            throw new Exception(LECoreStringResMgr.Get("ERROR_INVALID_SUBFONT"));
        }

        private static void InitializeComplexFont(IScalableFontSettings settings, LEFont newFont)
        {
            var filePath = settings.FontFilePath;

            try
            {
                var serializer = LECore.Save_Load.XmlSerializerCache.GetXmlSerializer(typeof(SerializableObject.Lyt.ComplexFontDescription));
                ComplexFontDescription compFontDesc;

                using (var fs = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                {
                    compFontDesc = LECore.Save_Load.XmlSerializerCache.Deserialize(serializer, fs) as ComplexFontDescription;
                }

                if (compFontDesc != null && compFontDesc.subFont != null)
                {

                    var result = new InitializeComplexFontTreeResult();
                    var iNWFont = InitializeComplexFontTree(compFontDesc.subFont, filePath, result);
                    newFont.InitializeInternal(iNWFont, settings.FontName, filePath);
                    if (result.ReplacedToBC3 && string.IsNullOrEmpty(newFont.InitializationErrorMessage))
                    {
                        newFont.InitializationErrorMessage = LECoreStringResMgr.Get("WARNING_PAIR_FONT_BC7_BC3");
                    }

                    if (iNWFont is NW4R.Font.PairFont)
                    {
                        ValidatePairFont(newFont, (NW4R.Font.PairFont)iNWFont);
                    }
                }
                else
                {
                    var message = string.Format("Invalid fcpx file.{0}", filePath);
                    newFont.InitializeInternal(null, settings.FontName, filePath, null, settings, message);
                }
            }
            catch (Exception e)
            {
                newFont.InitializeInternal(null, settings.FontName, filePath, null, settings, e.Message);
            }
        }

        private static void ValidatePairFont(LEFont font, NW4R.Font.PairFont pairFont)
        {
            if (!string.IsNullOrEmpty(font.InitializationErrorMessage))
            {
                // 既にエラーが出ているならとりあえずそのままにする。
                return;
            }
            var leafs = LEFontHelper.GetLeafs(pairFont).ToArray();

            var platforms = leafs.OfType<NW4R.Font.NW4RResFont>().Where(x => !LEFontHelper.IsFontConvertSettingsFile(x.SourcePath))
                .Select(x => x.GetPlatform()).Distinct().ToArray();

            if (platforms.Length == 1)
            {
                font.BntxPlatform = platforms[0];
            }
            else if (platforms.Length > 1)
            {
                font.InitializationErrorMessage = LECoreStringResMgr.Get("DIFFERENT_PLATFORMIN_FCPX", string.Join(", ", platforms));
                return;
            }

            if (leafs.Select(x => x.IsBorderEffectEnabled()).Distinct().Count() > 1)
            {
                font.InitializationErrorMessage = LECoreStringResMgr.Get("INVALID_FCPX_BORDER_SETTING");
                return;
            }
        }

        /// <summary>
        /// ScFont を生成します。
        /// </summary>
        private static NW4R.Font.ScFont CreateScFont(float size, ScalableFontDescription[] fontDescs, string rootPath, uint alternateChar, int lineFeedOffset)
        {
            var scfont = new NW4R.Font.ScFont();
            var initArg = new NW4R.Font.ScFont.InitializeArg
            {
                fontDatas = new IntPtr[1][],
                fontDataSizes = new int[1][],
                fontBoldWeights = new float[1][],
                fontBorderSizes = new int[1][],
                fontScaleWidths = new float[1][],
                fontScaleHeights = new float[1][],
                fontTypes = new int[1][],
                ignorePalt = new bool[1][],
                deleteBearingX = new bool[1][],
                bearingOffsetX = new int[1][],
                forceMonospacedEnabled = new bool[1][],
                forceMonospacedWidth = new int[1][],
                baselineOffset = new int[1][],
                fontSize = (int)size,
                alternateChar = alternateChar,
                lineFeedOffset = lineFeedOffset,
                charCodeRangeCount = new int[1][],
                charCodeRangeFirst = new uint[1][][],
                charCodeRangeLast = new uint[1][][]
            };

            var fontCount = fontDescs.Length;
            initArg.fontDatas[0] = new IntPtr[fontCount];
            initArg.fontDataSizes[0] = new int[fontCount];
            initArg.fontBoldWeights[0] = new float[fontCount];
            initArg.fontBorderSizes[0] = new int[fontCount];
            initArg.fontScaleWidths[0] = new float[fontCount];
            initArg.fontScaleHeights[0] = new float[fontCount];
            initArg.fontTypes[0] = new int[fontCount];
            initArg.ignorePalt[0] = new bool[fontCount];
            initArg.deleteBearingX[0] = new bool[fontCount];
            initArg.bearingOffsetX[0] = new int[fontCount];
            initArg.forceMonospacedEnabled[0] = new bool[fontCount];
            initArg.forceMonospacedWidth[0] = new int[fontCount];
            initArg.baselineOffset[0] = new int[fontCount];
            initArg.charCodeRangeCount[0] = new int[fontCount];
            initArg.charCodeRangeFirst[0] = new uint[fontCount][];
            initArg.charCodeRangeLast[0] = new uint[fontCount][];

            initArg.textureCacheWidth = (uint)_textureCacheSizeForScalableFont.Width;
            initArg.textureCacheHeight = (uint)_textureCacheSizeForScalableFont.Height;

            var bitArrays = new byte[fontCount][];

            for (int i = 0; i < fontCount; i++)
            {
                var fontDesc = fontDescs[i];

                // 範囲のチェック
                if (fontDesc.boldWeight < 0 || 1.0 < fontDesc.boldWeight)
                {
                    throw new Exception(LECore.LECoreStringResMgr.Get("INVALID_BOLD_WEIGHT"));
                }

                if (fontDesc.borderWidth < 0 || 10.0 < fontDesc.borderWidth )
                {
                    throw new Exception(LECore.LECoreStringResMgr.Get("INVALID_BORDER_WIDTH"));
                }

                initArg.fontBoldWeights[0][i] = fontDesc.boldWeight;
                initArg.fontBorderSizes[0][i] = fontDesc.borderWidth;
                initArg.fontScaleWidths[0][i] = fontDesc.scaleWidth;
                initArg.fontScaleHeights[0][i] = fontDesc.scaleHeight;
                initArg.fontTypes[0][i] = 0;
                if (fontDesc.path.EndsWith("bfttf"))
                {
                    initArg.fontTypes[0][i] = 2;
                }
                else if (fontDesc.path.EndsWith("bfotf"))
                {
                    initArg.fontTypes[0][i] = 3;
                }
                else if (fontDesc.path.EndsWith("otf"))
                {
                    initArg.fontTypes[0][i] = 1;
                }
                initArg.ignorePalt[0][i] = fontDesc.ignorePalt;
                initArg.deleteBearingX[0][i] = fontDesc.deleteBearingX;
                initArg.bearingOffsetX[0][i] = fontDesc.bearingOffsetX;
                initArg.forceMonospacedEnabled[0][i] = fontDesc.forceMonospacedEnabled;
                initArg.forceMonospacedWidth[0][i] = fontDesc.forceMonospacedWidth;
                initArg.baselineOffset[0][i] = fontDesc.baselineOffset;

                initArg.charCodeRangeCount[0][i] = fontDesc.charCodeRangeSet.Length;
                initArg.charCodeRangeFirst[0][i] = new uint[fontDesc.charCodeRangeSet.Length];
                initArg.charCodeRangeLast[0][i] = new uint[fontDesc.charCodeRangeSet.Length];
                for (int j = 0; j < fontDesc.charCodeRangeSet.Length; j++)
                {
                    initArg.charCodeRangeFirst[0][i][j] = fontDesc.charCodeRangeSet[j].first;
                    initArg.charCodeRangeLast[0][i][j] = fontDesc.charCodeRangeSet[j].last;
                }

                string fontPath;
                fontPath = Path.GetFullPath(Path.Combine(Path.GetDirectoryName(rootPath), fontDesc.path));
                bitArrays[i] = File.ReadAllBytes(fontPath);
            }

            scfont.SetResources(bitArrays, _textureConvertUtility, _endianSwapEnabled, initArg);
            scfont.UpdateAndCompleteTheTextureCache();

            return scfont;
        }


        /// <summary>
        /// 登録します。（管理情報を更新します）
        /// </summary>
        private static void DoRegister_(FontManager owner, LEFont newFont)
        {
            if (!_fontSet.Contains(newFont))
            {
                _fontSet.Add(newFont);
            }

            // 参照情報を更新します。
            if (newFont != null)
            {
                List<FontManager> refSet = null;
                if (_fontReferenceMap.TryGetValue(newFont, out refSet))
                {
                    // 重複する登録は起こらないはず。
                    Debug.Assert(!refSet.Contains(owner));

                    // 参照FontManagerのリストに追加
                    refSet.Add(owner);
                }
                else
                {
                    // フォントを参照しているFontManagerのリストを新規作成。
                    List<FontManager> newRefSet = new List<FontManager>();
                    newRefSet.Add(owner);

                    _fontReferenceMap.Add(newFont, newRefSet);
                }
            }
        }

        /// <summary>
        /// フォントを破棄します。
        /// </summary>
        public static void DisposeFont( FontManager owner, LEFont font )
        {
            Debug.Assert( font != null );
            List<FontManager> refSet = _fontReferenceMap[font];
            Debug.Assert( refSet != null );
            Debug.Assert( refSet.Contains( owner ) );

            refSet.Remove( owner );

            // フォントを参照しているマネージャが無くなったら
            if( refSet.Count <= 0 )
            {
                // フォントを破棄します。
                _fontSet.Remove( font );
                font.Dispose();

                DbgConsole.WriteLine("FNT_CNT={0}", _fontSet.Count);
            }
        }

        /// <summary>
        /// 内部キャッシュを強制的にリセットします。
        /// </summary>
        public static void ResetFontChacheAll()
        {
            foreach (LEFont font in _fontReferenceMap.Keys)
            {
                font.Dispose();
            }

            _fontReferenceMap.Clear();
            _fontSet.Clear();
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class LEFontManagerManipulator
    {
        FontManagerCommandCreater commandCreator = null;

        FontManager _targetFontMgr = null;

        /// <summary>
        ///
        /// </summary>
        public LEFontManagerManipulator()
        {
            this.commandCreator = new FontManagerCommandCreater
                ( delegate( FontManager fontManager)
                        {
                            return new FontManagerMemento( fontManager);
                        });
        }

        /// <summary>
        ///
        /// </summary>
        public void BindTarget( ILEFontManager fontMgr )
        {
            _targetFontMgr = fontMgr as FontManager;
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateFontFilePath( ILEFont font, string newFilePath )
        {
            this.commandCreator.Modify
                ( _targetFontMgr,
                  delegate()
                      {
                          UpdateFont( font, newFilePath);
                      });
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateFont( ILEFont font, string newFilePath )
        {
            if( _targetFontMgr.FindFontByName( font.FontName ) != null )
            {
                _targetFontMgr.UpdateFont( font as LEFont, newFilePath );
            }
            else
            {
                Debug.Assert( false );
            }
        }
    }

    /// <summary>
    /// フォント マネージャクラス(シングルトンクラス)
    /// 外部モジュール公開インタフェース
    /// </summary>
    public interface ILEFontManager
    {
        /// <summary>
        /// ファイルからフォントをシーンに読み込みます。
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        ILEFont CreateILEFontFromBRFNT(string filePath);

        /// <summary>
        /// ファイルからフォントをシーンに読み込みます。
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        ILEFont CreateILEFont(string fontName, string filePath, IFontSettings fontDesc);

        /// <summary>
        /// 登録フォントセット
        /// </summary>
        IEnumerable<ILEFont> ILEFontSet { get;}

        /// <summary>
        /// ファイル名からフォントを検索します。
        /// </summary>
        /// <param name="fileName"></param>
        /// <returns></returns>
        ILEFont FindFontByName(string fontName);


        /// <summary>
        /// 読み取り専用か調べます。
        /// </summary>
        bool CheckReadOnlyByName(string fontName);
    }

    /// <summary>
    /// フォント マネージャクラス(シングルトンクラス)
    ///
    /// LECore外部に直接公開しません。
    ///
    /// </summary>
    internal class FontManager :
        ILEFontManager,                // 外部モジュール公開インタフェース
        IDisposable
    {
        /// <summary>
        /// 文字列の同一判定で、大文字、小文字の区別をするかどうか。
        /// </summary>
        const bool _IgnoreNoCase = true;

        /// <summary>
        /// マネージャが管理している、LEFontインスタンスのセット
        /// </summary>
        readonly List<ILEFont> _fontSet = new List<ILEFont>();
        readonly List<ILEFont> _readOnlyFontSet = new List<ILEFont>();

        /// <summary>
        /// 更新イベント
        /// </summary>
        public event OnLEFontMgrUpdateHandler OnLEFontMgrUpdate;

#region --------------- プロパティ ---------------

#endregion --------------- プロパティ ---------------

        /// <summary>
        /// フォントが有効か判定します。
        /// </summary>
        bool IsFontEncodingValid_(LEFont font)
        {
            if (font != null)
            {
                // フォントが発見できなかった場合は、判断ができないので有効としておく。
                if (font.IsFileNotFound())
                {
                    return true;
                }

                // 現在は、UTF16エンコーディングのフォントしか対応しません。
                if (font.NWFont != null && font.NWFont.GetEncoding() != NWFont.INWFont.FontEncoding.FONT_ENCODING_UTF16)
                {
                    // "レイアウトエディタでは、UTF16エンコーディングのフォントしかサポートしません。"
                    string msg = LECoreStringResMgr.Get("LECORE_SYS_ERROR_FONTENCODING");
                    throw new Exception(msg);
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// 登録フォントの中から、
        /// ファイルパス名で、フォントを検索します。
        /// </summary>
        public LEFont FindFontByFilePath(string filePath)
        {
            foreach (LEFont font in _fontSet)
            {
                if (string.Compare(font.FontPath, filePath, _IgnoreNoCase) == 0)
                {
                    return font;
                }
            }

            // 発見できなかった。
            return null;
        }

        /// <summary>
        /// フォントファイル名からフォントを検索します。
        /// fileNameは、拡張子無しのファイル名で指定してください。
        /// </summary>
        /// <param name="fileName"></param>
        /// <returns></returns>
        public ILEFont FindFontByName(string fileName)
        {
            if (string.IsNullOrEmpty(fileName))
            {
                return null;
            }

            int dotIndex = fileName.LastIndexOf('.');

            string fileNameWithoutExt;

            // ファイルパスから名前の部分を抜き出す
            if (dotIndex != -1)
            {
                switch (fileName.Substring(dotIndex).ToLower())
                {
                    case ".bffnt":
                    case ".bfttf":
                    case ".bfotf":
                    case ".ttf":
                    case ".fcpx":
                    case ".ffnt":
                        fileNameWithoutExt = Path.GetFileNameWithoutExtension(fileName);
                        break;
                    default:
                        // 想定した拡張子以外は何もしない
                        fileNameWithoutExt = fileName;
                        break;
                }
            }
            else
            {
                fileNameWithoutExt = Path.GetFileNameWithoutExtension(fileName);
            }

            foreach (LEFont font in _fontSet)
            {
                if (string.Compare(font.FontName, fileNameWithoutExt, _IgnoreNoCase) == 0)
                {
                    return font;
                }
            }

            // 発見できなかった。
            return null;
        }

        /// <summary>
        /// 拡張子つきのファイル名を取得します。
        /// </summary>
        string GetFontFileNameWithExt_(string fontName)
        {
            if (Path.GetExtension(fontName) == string.Empty)
            {
                fontName = fontName + AppConstants.BFFNTFileExt;
            }
            return fontName;
        }

        /// <summary>
        /// 変更イベントを通知します。
        /// </summary>
        void NotifyChangeEvent_()
        {
            if (OnLEFontMgrUpdate != null)
            {
                OnLEFontMgrUpdate();
            }
        }

        /// <summary>
        /// 更新イベントを強制的に起こす
        /// </summary>
        public void RaiseChangeEvent()
        {
            NotifyChangeEvent_();
        }

        /// <summary>
        /// スケーラブルフォント設定からLEFontを生成します。
        /// </summary>
        public LEFont CreateLEFontFromScalableFontSettings(IScalableFontSettings fontSettings)
        {
            LEFont font = FindFontByName(fontSettings.FontName) as LEFont;
            if (font != null)
            {
                return font;
            }

            LEFont newFont = FontFactory.BuildFontFromScalableFontSettings(this, fontSettings);
            if (IsFontEncodingValid_(newFont))
            {
                RegisterFont(newFont);
                return newFont;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// フォントバイナリファイルから、LEFontクラスを生成します。
        ///
        /// 同一ファイルからの生成がおこわなれた場合は、
        /// 以前の読み込み結果を返します。
        /// </summary>
        public LEFont CreateLEFont(string fontName, string filePath, IFontSettings fontDesc)
        {
            // まず、すでに登録されていないか確認します。
            LEFont font = null;

            // 第１に、同じフォントパスのフォントが無いか探します。
            if (!string.IsNullOrEmpty(filePath))
            {
                font = FindFontByFilePath(filePath) as LEFont;
                if (font != null)
                {
                    // フォント名まで一致しているなら、同一のフォントとみなせるので返します。
                    // (たとえば、同じ tff ファイルで パラメータ違いのスケーラブルフォントはフォント名で区別します。)
                    if (font.FontName == fontName)
                    {
                        return font;
                    }
                    else
                    {
                        font = null;
                    }
                }
            }

            // 第２に、フォント名で探します。
            // 検索の順番として、まず、フォントパスで探すことは重要で、同じフォントをディレクトリを分けて配置している場合に
            // 区別ができなくなることを防いでいます(フォント名が同一で、パスだけが違う状態)。
            font = FindFontByName(fontName) as LEFont;
            if (font != null)
            {
                return font;
            }

            // 不正なファイル名でないか、判定します。
            if (!string.IsNullOrEmpty(filePath) &&
                !SaveLoadHelper.IsValidStringForFileName(Path.GetFileNameWithoutExtension(filePath)))
            {
                string msg = LECoreStringResMgr.Get("LECORE_SYS_ERROR_NGFILENAME", Path.GetFileName(filePath));
                throw new Exception(msg);
            }

            // 新規登録します。
            if (fontDesc is IScalableFontSettings)
            {
                return CreateLEFontFromScalableFontSettings((IScalableFontSettings)fontDesc);
            }
            else
            {
                LEFont newFont = FontFactory.BuildFontFromFile(this, fontName, filePath, false);
                if (IsFontEncodingValid_(newFont))
                {
                    RegisterFont(newFont);
                    return newFont;
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// 登録します。
        /// </summary>
        private void RegisterFont(ILEFont font)
        {
            Ensure.Argument.True(FindFontByName(font.FontName) == null);

            _fontSet.Add(font);

            NotifyChangeEvent_();
        }

        /// <summary>
        /// フォントを削除します。
        /// </summary>
        public void RemoveFont(LEFont font)
        {
            Debug.Assert(_fontSet.Contains(font));

            if (_fontSet.Contains(font))
            {
                SetReadOnlyState(font, false);
                _fontSet.Remove(font);

                FontFactory.DisposeFont(this, font);

                // イベント通知
                NotifyChangeEvent_();
            }
        }

        /// <summary>
        /// ファイルを更新します。
        /// 新しいファイルパスを設定します。
        /// </summary>
        public bool UpdateFont(LEFont font, string newFilePath)
        {
            Debug.Assert(font != null);

            bool bUpdated = false;

            if (FindFontByName(font.FontName) != null)
            {
                bUpdated = FontFactory.UpdateFont(font, newFilePath);
                if (bUpdated)
                {
                    NotifyChangeEvent_();
                }
            }
            return bUpdated;
        }

        /// <summary>
        /// フォントが読み取り専用か調べます。
        /// </summary>
        public bool CheckReadOnlyByName(string fontName)
        {
            var font = FindFontByName(fontName);
            return (font == null) ? false : _readOnlyFontSet.Contains(font as LEFont);
        }

        /// <summary>
        /// テクスチャの読みとり専用状態を設定します。
        /// <remarks>
        /// ユーザのUI 操作によって実行されないメソッドなの
        /// で、UIへの変更通知(NotifyTextureMgrUpdateEvent_() )をしていません。
        /// </remarks>
        /// </summary>
        public void SetReadOnlyState(LEFont font, bool isReadonly)
        {
            if (!this._fontSet.Contains(font))
            {
                return;
            }

            if (isReadonly)
            {
                if (!_readOnlyFontSet.Contains(font)) { _readOnlyFontSet.Add(font); }
            }
            else
            {
                _readOnlyFontSet.Remove(font);
            }
        }

#region ILEFontManager メンバ（外部モジュール公開）

        public ILEFont CreateILEFontFromBRFNT(string filePath)
        {
            return CreateLEFont(Path.GetFileName(filePath), filePath, null);
        }

        /// <summary>
        /// ファイルからフォントを読み込みます。
        /// </summary>
        /// <param name="filePath"></param>
        /// <returns></returns>
        public ILEFont CreateILEFont(string fontName, string filePath, IFontSettings fontDesc)
        {
            return CreateLEFont(fontName, filePath, fontDesc);
        }

        /// <summary>
        /// 登録フォントセットを取得します。
        /// </summary>
        public IEnumerable<ILEFont> ILEFontSet
        {
            get { return _fontSet; }
        }

#endregion ILEFontManager メンバ

#region IDisposable メンバ
        /// <summary>
        /// リソースを破棄します。
        /// </summary>
        public void Dispose()
        {
            foreach (LEFont font in _fontSet)
            {
                FontFactory.DisposeFont(this, font);
            }
            _fontSet.Clear();
        }

#endregion
    }

    public interface IFontSettings
    {
        string FontName { get; }
    }

    public interface IScalableFontSettings : IFontSettings
    {
        string FontFilePath { get; }
        int FontSize { get; }
        float BoldWeight { get; }
        int BorderWidth { get; }
    }

    public static class IScalableFontSettingsHelper
    {
        public static bool IsSame(this IScalableFontSettings lhs, IScalableFontSettings rhs)
        {
            if (lhs == null)
            {
                return (rhs == null);
            }
            else
            {
                if (rhs == null)
                {
                    return false;
                }
            }

            return
                lhs.FontName == rhs.FontName &&
                lhs.FontFilePath == rhs.FontFilePath &&
                lhs.FontSize == rhs.FontSize &&
                lhs.BoldWeight == rhs.BoldWeight &&
                lhs.BorderWidth == rhs.BorderWidth;
        }
    }

    /// <summary>
    /// スケーラブルフォント設定
    /// </summary>
    public class ScalableFontSettings : IScalableFontSettings
    {
        public string FontName { get; set; }
        public string FontFilePath { get; set; }
        public int FontSize { get; set; }
        public float BoldWeight { get; set; }
        public int BorderWidth { get; set; }
    }

    public class SheetBitmap
    {
        public Bitmap Bitmap { get; set; }
        public DateTime LastModifyTime  { get; set; }
    }

    /// <summary>
    /// フォントクラス
    /// 外部モジュール公開インタフェース
    /// </summary>
    public interface ILEFont
    {
        string FontName { get; }
        string FontPath { get; }
        string BntxPlatform { get; }
        string InitializationErrorMessage { get; }

        bool IsBitmapReady { get; }
        bool IsReadyToDraw { get; }
        bool IsWaitingUpdate { get; }

        IFontSettings FontSettings { get; }

        /// <summary>
        /// フィルタの影響を受けない文字幅。
        /// CellWidth, CellHeight から変更になりました。
        /// </summary>
        int Width { get; }
        int Height { get; }

        NWFont.INWFont NWFont { get; }

        void InitFontAsync(Action onFinish);

        void UpdateFontAsync(Action onPreUpdate, Action onFinish);

        object GetLockHandle();

        SheetBitmap GetFontBitmapBySheetIdx(uint sheetIdx);
    }

    /// <summary>
    ///
    /// </summary>
    internal class AbstractLEFont : ILEFont
    {
        public string FontName { get; set; }
        public string FontPath { get; set; }
        public string BntxPlatform { get { throw new NotImplementedException(); } }
        public string InitializationErrorMessage { get { throw new NotImplementedException(); } }

        public bool IsBitmapReady { get { throw new NotImplementedException(); } }
        public bool IsReadyToDraw { get { throw new NotImplementedException(); } }
        public bool IsWaitingUpdate { get { throw new NotImplementedException(); } }
        public void BeforeUpdate() { throw new NotImplementedException(); }

        public IFontSettings FontSettings { get { return null; } }

        /// <summary>
        /// フィルタの影響を受けない文字幅。
        /// CellWidth, CellHeight から変更になりました。
        /// </summary>
        public int Width { get { throw new NotImplementedException(); } }
        public int Height { get { throw new NotImplementedException(); } }

        public NWFont.INWFont NWFont { get { throw new NotImplementedException(); } }

        public void InitFontAsync(Action onFinish)
        {
            throw new NotImplementedException();
        }

        public void UpdateFontAsync(Action onPreUpdate, Action onFinish)
        {
            throw new NotImplementedException();
        }

        public object GetLockHandle()
        {
            throw new NotImplementedException();
        }

        public SheetBitmap GetFontBitmapBySheetIdx(uint sheetIdx)
        {
            throw new NotImplementedException();
        }
    }

    /// <summary>
    /// フォントクラス
    /// </summary>
    sealed internal class LEFont :
        ILEFont,      // 外部モジュールに公開されるインタフェース
        IDisposable
    {
        // 不正なフォントなどのサイズに使われる初期フォントサイズ
        internal const int InitialFontSize = 16;

        /// <summary>
        /// バックグラウンドで、フォントを初期化するクラス
        /// </summary>
        private static class BackGroundFontInitializer
        {
            static private object _lockObject = new object();

            static public object LockObject { get { return _lockObject;} }

            /// <summary>
            ///
            /// </summary>
            private class GLContextProxy : IDisposable
            {
                const int MaxGLInitRetryCount = 5;

                private Graphics _gfx;
                private IntPtr _gfxHdc;

                /// <summary>
                /// 生成
                /// </summary>
                public static GLContextProxy TryCreate(Graphics gfx)
                {
                    GLContextProxy glContext = new GLContextProxy();
                    glContext._gfx = gfx;
                    glContext._gfxHdc = gfx.GetHdc();

                    int GLInitRetryCount = 0;
                    bool result = LayoutEditorCore.InitializeOpenGLContext(glContext._gfxHdc);
                    while (!result)
                    {
                        Thread.Sleep(1000);
                        result = LayoutEditorCore.InitializeOpenGLContext(glContext._gfxHdc);

                        // なんども失敗するときは、エラーを記録して断念する。
                        if (GLInitRetryCount++ > MaxGLInitRetryCount)
                        {
                            return null;
                        }
                    }

                    return glContext;
                }

                /// <summary>
                /// 破棄
                /// </summary>
                public void Dispose()
                {
                    LayoutEditorCore.FinalizeOpenGLContext();
                    _gfx.ReleaseHdc(_gfxHdc);
                }
            }

            /// <summary>
            ///
            /// </summary>
            static bool InitGLContextFromGraphics_(Graphics gfx)
            {
                IntPtr graphicsHdc = IntPtr.Zero;
                try
                {
                    graphicsHdc = gfx.GetHdc();
                    return LayoutEditorCore.InitializeOpenGLContext(graphicsHdc);
                }
                finally
                {
                    gfx.ReleaseHdc(graphicsHdc);
                }
            }

            /// <summary>
            /// 終了時コールバックを、GUIスレッドから呼び出す。
            /// GUI 再描画のトリガをひくような処理が実行されている。
            /// </summary>
            static void TryCallOnFinishAction_(Action onFinish)
            {
                if (onFinish != null)
                {
                    onFinish();
                }
            }

            /// <summary>
            ///
            /// </summary>
            static void OnErrorFinalize_(LEFont font, string errorMsg, Action onFinish)
            {
                font.InitializationErrorMessage = string.Format("Can't initialize font. ({0})", errorMsg);
                font._fontBitmapSet = new SheetBitmap[0];

                TryCallOnFinishAction_(onFinish);
            }

            /// <summary>
            /// 更新ジョブを登録する。
            /// </summary>
            public static void QueueUpdateJob(LEFont font, Action onPreUpdate, Action onFinish)
            {
                if (font._nwFont == null || font.IsFileNotFound())
                {
                    return;
                }

                System.Threading.Interlocked.Increment(ref font.UpdateRequestCount);
                System.Threading.ThreadPool.QueueUserWorkItem((arg) =>
                {
                    try
                    {
                        lock (_lockObject)
                        {
                            if (onPreUpdate != null)
                            {
                                onPreUpdate();
                            }

                            // 値のコピーだけおこなう。
                            NWFont.INWFont.SheetPropaties[] sheetSet = font._nwFont.GetSheetPropaties();
                            for (int i = 0; i < sheetSet.Length; i++)
                            {
                                NWFont.INWFont.SheetPropaties sheet = sheetSet[i];
                                CopyToFontBitmpSet_(font._fontBitmapSet[i].Bitmap, sheet);

                                font._fontBitmapSet[i].LastModifyTime = DateTime.Now;
                            }
                        }

                        TryCallOnFinishAction_(onFinish);
                    }
                    finally
                    {
                        System.Threading.Interlocked.Decrement(ref font.UpdateRequestCount);
                    }
                });
            }

            /// <summary>
            /// 初期化ジョブを登録する。
            /// </summary>
            public static void QueueInitJob(LEFont font, Action onFinish)
            {
                System.Threading.Interlocked.Increment(ref font.UpdateRequestCount);
                System.Threading.ThreadPool.QueueUserWorkItem((arg) =>
                {
                    try
                    {
                        // タイリング解除処理を行う TexUtils が マルチスレッド実行できないので、
                        // 同時にひとつのジョブしか実行しないようにする。
                        lock (_lockObject)
                        {
                            if (font._nwFont == null || font.IsFileNotFound())
                            {
                                return;
                            }

                            using (var pictureBox = new PictureBox())
                            {
                                Graphics graphics = Graphics.FromHwnd(pictureBox.Handle);

                                // 初期化処理と破棄処理で不整合が起こらないようにロックを取得してから処理をする。
                                // Dispose のコードも参照のこと。
                                lock (font._nwFont)
                                {
                                    // GL コンテキストの初期化。
                                    GLContextProxy glContextProxy = GLContextProxy.TryCreate(graphics);
                                    if (glContextProxy == null)
                                    {
                                        OnErrorFinalize_(font, "Can't initialize GL-context for font-initialization.", onFinish);
                                        return;
                                    }

                                    using (glContextProxy)
                                    {
                                        // 実際の初期化
                                        try
                                        {
                                            font._nwFont.InitializeAtTheFirstTimeDraw();
                                        }
                                        catch (System.Runtime.CompilerServices.RuntimeWrappedException e)
                                        {
                                            // フォントの初期化に失敗した。ユーザに報告するためにエラー情報を保持しておく。
                                            OnErrorFinalize_(font, e.WrappedException.ToString(), onFinish);
                                            return;
                                        }
                                        catch (Exception e)
                                        {
                                            // フォントの初期化に失敗した。ユーザに報告するためにエラー情報を保持しておく。
                                            OnErrorFinalize_(font, e.Message, onFinish);
                                            return;
                                        }

                                        // レンダリング用ビットマップの生成。
                                        {
                                            NWFont.NW4RResFont.SheetPropaties[] sheetSet = font._nwFont.GetSheetPropaties();
                                            SheetBitmap[] bmpSet = new SheetBitmap[sheetSet.Length];
                                            for (int i = 0; i < sheetSet.Length; i++)
                                            {
                                                NWFont.NW4RResFont.SheetPropaties sheet = sheetSet[i];
                                                bmpSet[i] = new SheetBitmap();
                                                bmpSet[i].Bitmap = new Bitmap(sheet.texWidth, sheet.texHeight, sheet.texWidth * 4, PixelFormat.Format32bppArgb, sheet.imgPtr);
                                                bmpSet[i].LastModifyTime = DateTime.Now;
                                            }

                                            font._fontBitmapSet = bmpSet;
                                        }
                                    }
                                }
                            }

                            TryCallOnFinishAction_(onFinish);
                        }
                    }
                    finally
                    {
                        System.Threading.Interlocked.Decrement(ref font.UpdateRequestCount);
                    }
                });
            }

            /// <summary>
            /// メモリコピー
            /// </summary>
            private static void CopyToFontBitmpSet_(Bitmap dstBitmap, NWFont.INWFont.SheetPropaties sheet)
            {
                lock (dstBitmap)
                {
                    BitmapData bd = null;
                    do
                    {
                        try
                        {
                            bd = dstBitmap.LockBits(new Rectangle(0, 0, dstBitmap.Width, dstBitmap.Height), ImageLockMode.ReadWrite, PixelFormat.Format32bppArgb);
                        }
                        catch (System.InvalidOperationException)
                        {
                            // アサートに引っかかるなら他のスレッド内で lock せずに操作している可能性がある
                            Debug.Assert(false);

                            // 別のスレッドが dstBitmap にアクセスしていて LockBits に失敗することがあるので、
                            // 間をおいて再試行する。
                            System.Threading.Thread.Sleep(50);
                        }
                    } while (bd == null);

                    LECore.Win32.Kernel32.CopyMemory(bd.Scan0, sheet.imgPtr, (uint)(bd.Stride * bd.Height));

                    dstBitmap.UnlockBits(bd);
                }
            }
        }

        string _fontFileName = "";
        string _fontFilePath = "";
        public string BntxPlatform { get; set; }
        //public bool ReplacedToBC3 { get; set; }

        NWFont.INWFont _nwFont = null;

        SheetBitmap[] _fontBitmapSet = null;

        /// <summary>
        /// INWFont を取得します。
        /// </summary>
        public NWFont.INWFont NWFont
        {
            get { return _nwFont; }
        }

        public LEFont()
        {
        }

        /// <summary>
        /// 内部情報を初期化します。
        /// </summary>
        public void InitializeInternal( NWFont.INWFont nw4rFont, string filePath )
        {
            InitializeInternal(nw4rFont, Path.GetFileNameWithoutExtension( filePath ), filePath);
        }

        /// <summary>
        /// 内部情報を初期化します。
        /// </summary>
        public void InitializeInternal(NWFont.INWFont nw4rFont, string fontName, string filePath, string bntxPlatform = null, IFontSettings settings = null, string errorMessage = null)
        {
            Dispose();

            _nwFont = nw4rFont;
            _fontFileName = fontName;
            _fontFilePath = filePath;
            BntxPlatform = bntxPlatform;
            FontSettings = settings;
            InitializationErrorMessage = errorMessage;
        }

        /// <summary>
        /// フォントのビットマップデータをシート番号を指定して、取得します。
        /// </summary>
        public void InitFontAsync(Action onFinish)
        {
            if (_fontBitmapSet == null)
            {
                _fontBitmapSet = new SheetBitmap[0];

                // 初期化ジョブを登録する。
                BackGroundFontInitializer.QueueInitJob(this, onFinish);
            }
        }

        /// <summary>
        /// フォントを更新します。
        /// </summary>
        public void UpdateFontAsync(Action onPreUpdate, Action onFinish)
        {
            if (_fontBitmapSet != null && _fontBitmapSet.Length > 0)
            {
                BackGroundFontInitializer.QueueUpdateJob(this, onPreUpdate, onFinish);
            }
        }

        public object GetLockHandle()
        {
            return  BackGroundFontInitializer.LockObject;
        }

        /// <summary>
        /// フォントのビットマップデータをシート番号を指定して、取得します。
        /// </summary>
        public SheetBitmap GetFontBitmapBySheetIdx(uint sheetIdx)
        {
            if (_fontBitmapSet == null || sheetIdx >= _fontBitmapSet.Length)
            {
                return null;
            }

            return _fontBitmapSet[sheetIdx];
        }

#region ILEFont メンバ
        public string FontName
        {
            get { return _fontFileName; }
        }

        public string FontPath
        {
            get { return _fontFilePath; }
        }

        string errorMessage_;
        public string InitializationErrorMessage
        {
            get { return errorMessage_; }
            internal set { errorMessage_ = value; }
        }

        /// <summary>
        /// フィルタの影響を受けない文字幅。
        /// </summary>
        public int Width
        {
            get { return this.IsFileNotFound() || _nwFont == null ? InitialFontSize : _nwFont.GetWidth(); }
        }

        public int Height
        {
            get { return this.IsFileNotFound() || _nwFont == null ? InitialFontSize : _nwFont.GetHeight(); }
        }

        public bool IsBitmapReady
        {
            get { return _fontBitmapSet != null && _fontBitmapSet.Length > 0; }
        }

        public bool IsReadyToDraw
        {
            get { return this.IsBitmapReady && this._nwFont != null && this._nwFont.IsReadyToDraw() && UpdateRequestCount == 0; }
        }

        public bool IsWaitingUpdate
        {
            get { return UpdateRequestCount > 0; }
        }

        public IFontSettings FontSettings { get; private set; }

        public volatile int UpdateRequestCount = 0;
        #endregion ILEFont メンバ

        #region IDisposable メンバ

        /// <summary>
        /// 破壊します。
        /// </summary>
        public void Dispose()
        {
            // バックグラウンドで初期化中は破棄しないようにロック
            if (_nwFont != null)
            {
                lock (_nwFont)
                {
                     _nwFont.RemoveResource();

                    _nwFont = null;

                    if (_fontBitmapSet != null)
                    {

                        for (int i = 0; i < _fontBitmapSet.Length; i++)
                        {
                            _fontBitmapSet[i].Bitmap.Dispose();
                        }
                        _fontBitmapSet = null;

                    }
                }
            }
        }

#endregion
    }
}
