﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;

namespace LECore.Structures.Core
{
    [Flags]
    public enum DrawableOptionFlag
    {
        None = 0x00000000,
        IgnoreBVDrawing = 0x00000001,
        IgnoreDummyObjectDrawing    = 0x00000002,
        IgnoreBoundingPaneDrawing   = 0x00000004,
        IgnoreCapturePaneDrawing    = 0x00000008,
        IgnorePartsPaneRectDrawing  = 0x00000010,
        IgnorePaneImageDrawing      = 0x00000020,
        DrawPartsPaneActiveProperty = 0x00000040,
        SyncTextureInitialization   = 0x00000080,
    };

    /// <summary>
    /// 描画オプション
    /// </summary>
    public class DrawableOption
    {
        DrawableOptionFlag _optionFlag;

        /// <summary>
        /// リニア色空間に変換
        /// </summary>
        Color CalcSystemColor_(Color sRGBColor, bool convertToLinerGamma)
        {
            return convertToLinerGamma ? ColorHelper.ConvertToLinerGamma(sRGBColor) : sRGBColor;
        }

        /// <summary>
        /// 色を初期化します。
        /// </summary>
        public void InitializeSystemColors(bool linerGammaMode, bool linerColorParamaterMode)
        {
            int halfAlpha = linerGammaMode ? 128 : 64;

            bool convertToLinerGamma = linerGammaMode && linerColorParamaterMode;

            this.SystemWhite = Color.White;
            this.SystemRed = CalcSystemColor_(Color.Red, convertToLinerGamma);
            this.SystemDarkRed = CalcSystemColor_(Color.DarkRed, convertToLinerGamma);
            this.SystemGreen = CalcSystemColor_(Color.Green, convertToLinerGamma);
            this.SystemBlue = CalcSystemColor_(Color.Blue, convertToLinerGamma);
            this.SystemGray = CalcSystemColor_(Color.Gray, convertToLinerGamma);
            this.SystemOpeGreen = CalcSystemColor_(Color.FromArgb(halfAlpha, Color.Green), convertToLinerGamma);
            this.SystemOpeBlue = CalcSystemColor_(Color.FromArgb(halfAlpha, Color.Blue), convertToLinerGamma);
            this.SystemOpeYellow = CalcSystemColor_(Color.FromArgb(halfAlpha, Color.Yellow), convertToLinerGamma);
            this.SystemOpeMagenta = CalcSystemColor_(Color.FromArgb(halfAlpha, Color.Magenta), convertToLinerGamma);
        }

        public bool TerminateDrawTraverse
        {
            get;
            set;
        }

        public DrawableOptionFlag DrawableOptionFlag
        {
            get { return _optionFlag; }
            set { _optionFlag = value; }
        }

        public bool CheckActive(DrawableOptionFlag flag)
        {
            return (flag & _optionFlag) != 0;
        }

        public void EnableFlag(DrawableOptionFlag flag)
        {
            _optionFlag |= flag;
        }

        public void DisableFlag(DrawableOptionFlag flag)
        {
            _optionFlag &= ~flag;
        }

        public void ToggleFlag(DrawableOptionFlag flag)
        {
            DrawableOptionFlag mask = flag;
            DrawableOptionFlag invMask = ~mask;

            _optionFlag = (~(_optionFlag & mask) & mask) | (_optionFlag & invMask);
        }

        public Color SystemWhite { get; set; }
        public Color SystemDarkRed { get; set; }
        public Color SystemRed { get; set; }
        public Color SystemGreen { get; set; }
        public Color SystemBlue { get; set; }
        public Color SystemGray { get; set; }

        public Color SystemOpeGreen { get; set; }
        public Color SystemOpeBlue { get; set; }
        public Color SystemOpeYellow { get; set; }
        public Color SystemOpeMagenta { get; set; }


        public static readonly DrawableOption Empty = new DrawableOption();
    };

    /// <summary>
    /// IDrawable の概要の説明です。
    /// 描画可能なオブジェクトが実装するべきインタフェースです。
    /// </summary>
    public interface IDrawable
    {
        /// <summary>
        /// レンダラインタフェースを実装するクラスによって描画を行います。
        /// 具象クラスでは、IRendererの提供メソッドを使用して
        /// 本関数のオーバーライド関数内で描画処理を実装します。
        ///
        /// 描画処理は Gof. Visitorパターンで処理されます。
        ///
        /// Acceptor を 表現するインタフェースと考えることも出来ます。
        /// (IRenderer は Visitor)
        /// </summary>
        /// <param name="renderer"></param>
        void Draw(IRenderer renderer, DrawableOption option);
    }
}
