﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;

namespace LECore.Structures.Core
{
    /// <summary>
    /// アニメーションカーブを表現するクラスです。
    /// キーフレーム列を管理し、キーフレームの補間計算の結果をAnmAttributeクラスに反映します。
    ///
    /// 本クラスは、ITimeChageEventListener インタフェースを実装し、
    /// グローバル時間変更イベントの通知を受け取ります。
    ///
    /// </summary>
    internal class AnmCurve :
        IAnmCurve,               // システムコア外部に公開されるAPI
        IDisposable
    {
        // AnmCurveクローン時のKeyFrameの有無
        public enum KeyCloneType
        {
            None,  // キーなし
            Copy,  // コピー(インスタンスが共有されます)
            Clone, // クローン
        }

        // アニメーション対象となる、アトリビュート
        // AnmAttributeは、subAttribute を 持たないアトリビュート型である必要があります。
        AnmAttribute            _targetAttr = null;

        // キーフレームのリスト
        List<IAnmKeyFrame> _KeyFrames = new List<IAnmKeyFrame>();
        List<IAnmKeyFrame> _SelectedKeyFrames = new List<IAnmKeyFrame>();

        bool                    _KeyShouldBeSorted      = false;
        bool                    _IsParameterizedAnim = false;

        LECore.Structures.SerializableObject.Lan.ParameterizedAnimParameter[] _parameterizedAnimParameters = null;

        AnmCurveInfinityType _preInfinityType = AnmCurveInfinityType.Constant;
        AnmCurveInfinityType _postInfinityType = AnmCurveInfinityType.Constant;

        #region ----------------------- 補間計算
        /// <summary>
        /// 線形補間
        /// </summary>
        /// <param name="val0"></param>
        /// <param name="val1"></param>
        /// <param name="t0"></param>
        /// <param name="t1"></param>
        /// <param name="time"></param>
        /// <returns></returns>
        float DoInterp_Liner_( float val0, float val1, float t0, float t1, float time )
        {
            float normalizedTime   = (float)( time - t0) / (t1 - t0);
            float diff             = (val1 - val0) * normalizedTime;
            return val0 + diff;
        }

        /// <summary>
        /// エルミート補間
        /// </summary>
        /// <param name="val0"></param>
        /// <param name="val1"></param>
        /// <param name="t0"></param>
        /// <param name="t1"></param>
        /// <param name="outTan"></param>
        /// <param name="inTan"></param>
        /// <param name="time"></param>
        /// <returns></returns>
        float DoInterp_Hermite_( float val0, float val1, float t0, float t1, float outTan, float inTan, float time )
        {
            double dT  = (double)( t1 - t0 );
            double T  = (double)(time - t0) / dT;
            double T2 = T * T;
            double T3 = T2 * T;

            // outTan は傾き(dy/dt)ではなく、 dy のようである。
            // 入力パラメータは、 dy/dt として、データを取り扱っている。
            // したがって、利用前に dt をここで乗じてあげる必要がある。
            // 数学的な意味を理解していない！
            outTan   = (float)(outTan * dT);
            inTan    = (float)(inTan * dT);


            double result = ( 2.0*T3 - 3.0*T2 + 1.0 ) * val0 +
                         ( T3   - 2.0*T2 + T ) * outTan +
                         ( T3   - T2 )       * inTan +
                         ( -2.0*T3 + 3.0*T2 )    * val1;

            return (float)result;
        }

        /// <summary>
        /// エルミート補間の傾きを計算
        /// (エルミート曲線式を微分した式にパラメータを代入して計算しています)
        /// </summary>
        /// <param name="val0"></param>
        /// <param name="val1"></param>
        /// <param name="t0"></param>
        /// <param name="t1"></param>
        /// <param name="outTan"></param>
        /// <param name="inTan"></param>
        /// <param name="time"></param>
        /// <returns></returns>
        float GetTangent_Hermite_( float val0, float val1, float t0, float t1, float outTan, float inTan, float time )
        {
            float T  = (float)(time - t0) / ( t1 - t0 );
            float T2 = T * T;

            float result = 3.0f*( 2.0f*val0 + outTan + inTan -2.0f*val1 )*T2 +
                           2.0f*( -3.0f*val0 + -2.0f*outTan - inTan +3.0f*val1 )*T+
                           outTan;

            return result;
        }
        #endregion // ----------------------- 補間計算

        /// <summary>
        /// ターゲットのアトリビュートを取得します。
        /// </summary>
        public AnmAttribute TargetAttribute { get { return _targetAttr; } }

        /// <summary>
        /// 現在のパラメータの値を取得します。
        /// </summary>
        public float TargetAttributeMaxValue { get { return _targetAttr.MaxValue; } }

        /// <summary>
        /// 現在のパラメータの値を取得します。
        /// </summary>
        public float TargetAttributeMinValue { get { return _targetAttr.MinValue; } }

        /// <summary>
        /// 現在のパラメータの値を取得します。
        /// </summary>
        public object TargetAttributeValue
        {
            get
            {
                object currentVal = null;
                _targetAttr.GetValue( out currentVal );
                return currentVal;
            }
        }

        // 読み取り専用かどうか
        public bool IsReadOnlyLocked
        {
            get { return _targetAttr.IsReadOnlyLocked; }
            set
            {
                if (_targetAttr.IsReadOnlyLocked != value)
                {
                    _targetAttr.IsReadOnlyLocked = value;
                }
            }
        }

        public bool HasCustomTimeSource
        {
            get { return _targetAttr.TimeReleyProvider != null; }
        }

        public int CustomTimeFrame
        {
            get { return _targetAttr.TimeReleyProvider != null ? _targetAttr.TimeReleyProvider.Time : 0; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmCurve()
        {
        }

        /// <summary>
        ///
        /// </summary>
        ITimeChangeEventProvider GetTimeSource_()
        {
            return _targetAttr.GetTimeSource();
        }

        /// <summary>
        /// リソースの開放
        /// </summary>
        public void Dispose()
        {

        }

        /// <summary>
        /// パラメタライズドアニメーションカーブを設定する
        /// </summary>
        public void SetParameterizedAnimCurve(SerializableObject.Lan.ParameterizedAnimParameter[] parameters)
        {
            _IsParameterizedAnim = true;

            _parameterizedAnimParameters = new SerializableObject.Lan.ParameterizedAnimParameter[parameters.Length];
            for (int i = 0; i < parameters.Length; i++)
            {
                _parameterizedAnimParameters[i] = new SerializableObject.Lan.ParameterizedAnimParameter();
                _parameterizedAnimParameters[i].parameterizedAnimType = parameters[i].parameterizedAnimType;
                _parameterizedAnimParameters[i].startValue = parameters[i].startValue;
                _parameterizedAnimParameters[i].targetValue = parameters[i].targetValue;
                _parameterizedAnimParameters[i].offset = parameters[i].offset;
                _parameterizedAnimParameters[i].duration = parameters[i].duration;
            }
            Array.Sort(_parameterizedAnimParameters, (x, y) => x.offset.CompareTo(y.offset));
            _targetAttr.SetTimeEventHandler(true);
        }

        /// <summary>
        /// アニメーションカーブをアトリビュートと関連付けます。
        /// </summary>
        /// <param name="target"></param>
        public void BindTarget( AnmAttribute target, bool lockInterpType, bool useCustomTimeSource)
        {
            Trace.Assert( target != null );
            Trace.Assert( !target.HasSubAttribute );

            _targetAttr = target;
            _targetAttr.InitializeITimeChageEventListener(useCustomTimeSource);
            _targetAttr.DefaultInterpType = AnmAttribute.GetDefaultInterpType( _targetAttr.Type );
            _targetAttr.LockInterpType = lockInterpType;
        }

        /// <summary>
        /// アニメーションカーブをアトリビュートと関連付けます。
        /// InterpTypeをデフォルトに戻しません。
        /// </summary>
        /// <param name="target"></param>
        public void BindTargetKeepInterpType(AnmAttribute target, bool lockInterpType, bool useCustomTimeSource)
        {
            Trace.Assert(target != null);
            Trace.Assert(!target.HasSubAttribute);

            _targetAttr = target;
            _targetAttr.InitializeITimeChageEventListener(useCustomTimeSource);
            _targetAttr.LockInterpType = lockInterpType;
        }

        /// <summary>
        /// キーフレームを直接登録します。
        /// Undoの実装にのみ利用されます。
        ///
        /// Undoコマンドが正しく動作するために、同一のインスタンスを
        /// 使い回して利用する必要があるためです。
        /// </summary>
        public void AddNewKeyFrame( AnmKeyFrame newKey )
        {
            if (_IsParameterizedAnim)
            {
                return;
            }

            // 同一時刻にキーが存在しなければ...
            if (FindKeyExistAt(newKey.Time) == null)
            {
                AddNewKeyFrame_(newKey, false);

                // 変更を通知します。
                Update(true);
            }
        }

        /// <summary>
        /// キーフレームを直接登録します。
        /// アニメーション管理モード切り替えにのみ利用されます。
        /// </summary>
        public void AddNewKeyFrameWithNoUpdate(AnmKeyFrame newKey)
        {
            if (_IsParameterizedAnim)
            {
                return;
            }

            // 同一時刻にキーが存在しなければ...
            if (FindKeyExistAt(newKey.Time) == null)
            {
                AddNewKeyFrameWithNoUpdate_(newKey, false);
            }
        }

        /// <summary>
        /// カーブにキーフレームを挿入します。
        /// 接線が適切に再計算されます。
        /// </summary>
        /// <param name="newKey"></param>
        public void InsertNewKeyFrameWithTangentAdjust( AnmKeyFrame newKey )
        {
            if (_IsParameterizedAnim)
            {
                return;
            }

            // 同一時刻にキーが存在しなければ...
            if ( FindKeyExistAt( newKey.Time ) == null )
            {
                AddNewKeyFrame_( newKey, true );

                // 変更を通知します。
                Update( true );
            }
        }

        /// <summary>
        /// キーフレームの作成。
        /// システム外部には公開されないAPIです。
        /// </summary>
        public AnmKeyFrame AddNewKeyFrame( float time, float val, bool bUnifiyTangent, float inTangent, float outTangent, InterporationType type, float viewScale )
        {
            if (_IsParameterizedAnim)
            {
                return null;
            }

            AnmKeyFrame keyFrame = null;
            if( FindKeyExistAt( time ) == null )
            {
                keyFrame = new AnmKeyFrame(this, time, val, bUnifiyTangent, inTangent, outTangent, type, viewScale, true);
                AddNewKeyFrame_( keyFrame, false );
            }
            return keyFrame;
        }


        /// <summary>
        /// キーフレームの作成。
        /// システム外部には公開されないAPIです。
        /// </summary>
        public AnmKeyFrame AddNewKeyFrame( float time, float val, bool bUnifiyTangent, float inTangent, float outTangent, InterporationType type )
        {
            if (_IsParameterizedAnim)
            {
                return null;
            }

            return AddNewKeyFrame(time, val, bUnifiyTangent, inTangent, outTangent, type, 1.0f);
        }

        /// <summary>
        ///
        /// </summary>
        public void RemoveKeyFrame( AnmKeyFrame newKey )
        {
            if (_IsParameterizedAnim)
            {
                return;
            }

            RemoveKeyFrame( newKey, true );
        }

        /// <summary>
        /// キーを削除します。
        /// Undo/Redo は処理されません。
        /// </summary>
        /// <param name="newKey"></param>
        public void RemoveKeyFrame( AnmKeyFrame newKey, bool bEvaluate )
        {
            if (_IsParameterizedAnim)
            {
                return;
            }

            // 参照アトリビュートが追加されたので、
            // すでに削除されているキーを削除するケースがありえます。
            // Undo/Rodoを考えると修正する必要がるかもしれません。
            // Debug.Assert( this.Contains( newKey ) );
            if( !this.Contains( newKey ) )
            {
                return;
            }

            // 選択を解除します。
            RemoveSelectedSet( newKey );

            _KeyFrames.Remove( newKey );

            // TODO:前後のキーについて接線を再計算する必要があります。

            // キーがなくなった場合、時間変更ハンドラを削除します。
            if( _KeyFrames.Count == 0 )
            {
                _targetAttr.SetTimeEventHandler(false);
            }

            // 変更を通知します。
            Update( bEvaluate );
        }

        /// <summary>
        /// キーフレームを全て削除します。
        /// </summary>
        public void ClearKeyFrames()
        {
            ClearKeyFrames_(true);
        }

        /// <summary>
        /// キーフレームを全て削除します。
        /// アニメーション管理モード切り替えにのみ利用されます。
        /// </summary>
        public void ClearKeyFramesWithNoUpdate()
        {
            ClearKeyFrames_(false);
        }

        /// <summary>
        /// キーフレームを全て削除します。
        /// </summary>
        private void ClearKeyFrames_(bool update)
        {
            if (_IsParameterizedAnim)
            {
                return;
            }

            // 選択キーをすべてリセットします。
            ResetSelectedSet();

            _KeyFrames.Clear();

            _targetAttr.SetTimeEventHandler(false);

            // 変更を通知します。
            if (update)
            {
                Update(true);
            }
        }

        /// <summary>
        /// アニメーションカーブの状態を更新が必要な状態に設定します。
        /// キーフレームの更新時などに呼び出されます。
        /// 本関数呼び出し後に、Update()が呼ばれると、キーのソート処理などが行われます。
        /// </summary>
        public void SetCurveStateDirty()
        {
            _KeyShouldBeSorted = true;
        }

        /// <summary>
        /// キーフレームをソートします。
        /// キーフレームの時間を変更した際などに、実行する必要があります。
        /// </summary>
        void SortKeyFramesByTimeOrder_()
        {
            var ordered = _KeyFrames.OrderBy(x => x.Time).ToArray();
            for (int i = 0; i < ordered.Length; i++)
            {
                _KeyFrames[i] = ordered[i];
            }

            _KeyShouldBeSorted = false;
        }

        /// <summary>
        /// カーブを無効化します。
        /// 変更イベントが発生します。
        ///
        /// bEvaluate が 設定された場合、カーブの再評価結果をアトリビュートに反映します。
        /// その結果、シーン変更イベントが発生します。
        /// シーン変更イベントは、ビュー全体の更新のトリガになります。
        /// </summary
        public void Update( bool bEvaluate )
        {
            // 必要があればソートします。
            if( _KeyShouldBeSorted )
            {
                SortKeyFramesByTimeOrder_();
            }

            // カーブの再評価し、結果をアトリビュートに反映します
            if( bEvaluate )
            {
                _targetAttr.UpdateAttributeValue( GlobalTime.Inst.Time );
                NotifyChangeEvent_();
            }
        }

        /// <summary>
        /// アニメーションカーブがキーフレームを含むか調査します。
        /// </summary>
        /// <param name="kf"></param>
        /// <returns></returns>
        public bool Contains( AnmKeyFrame kf )
        {
            if (_IsParameterizedAnim)
            {
                return false;
            }

            return _KeyFrames.Contains( kf );
        }

        /// <summary>
        /// 指定時間にキーが存在するか調べます。
        /// 存在した場合には、キーの参照をかえします。
        /// </summary>
        public IAnmKeyFrame FindKeyExistAt( float time )
        {
            if (_IsParameterizedAnim)
            {
                return null;
            }

            foreach ( AnmKeyFrame  kf in _KeyFrames )
            {
                if( kf.Time == time )
                {
                    return kf;
                }
            }

            return null;
        }

        #region ------------------ 選択関連 ------------------
        /// <summary>
        /// キーフレームを選択セットに追加します。
        /// 重複登録は無視されます。
        ///
        /// 自分の所有しているキーフレーム以外が指定された場合は、
        /// アサートに失敗します。
        /// </summary>
        public void AddSelectedSet( IAnmKeyFrame key )
        {
            if (_IsParameterizedAnim)
            {
                return;
            }

            if( _KeyFrames.Contains( key ) )
            {
                if( !_SelectedKeyFrames.Contains( key ) )
                {
                    _SelectedKeyFrames.Add( key );
                    NotifyAddRemoveEvent_();
                }
            }
            else
            {
                Debug.Assert( false, "Unexpected Invalid keyframe selection is detected." );
            }
        }

        /// <summary>
        /// キーフレームを選択セットから削除します。
        /// </summary>
        public void RemoveSelectedSet( IAnmKeyFrame key )
        {
            if (_IsParameterizedAnim)
            {
                return;
            }

            if ( _KeyFrames.Contains( key ) )
            {
                _SelectedKeyFrames.Remove( key );
                NotifyAddRemoveEvent_();
            }
            else
            {
                Debug.Assert( false, "Unexpected Invalid keyframe selection is detected." );
            }
        }

        /// <summary>
        /// 選択キーのリセット
        /// </summary>
        public void ResetSelectedSet()
        {
            if (_IsParameterizedAnim)
            {
                return;
            }

            if ( _SelectedKeyFrames.Count != 0 )
            {
                _SelectedKeyFrames.Clear();
                NotifyChangeEvent_();
            }
        }
        #endregion ------------------ 選択関連 ------------------

        #region IAnmCurve メンバ
        #region ----------------- IAnmCurve 非公開メソッド
        /// <summary>
        /// キーの登録
        /// </summary>
        /// <param name="newKey"></param>
        void AddNewKeyFrame_( AnmKeyFrame newKey, bool reCalcTangent )
        {
            AddNewKeyFrameCore_(newKey, reCalcTangent, true);
        }

        /// <summary>
        /// キーの登録
        /// アニメーション管理モード切り替えにのみ利用されます。
        /// </summary>
        /// <param name="newKey"></param>
        void AddNewKeyFrameWithNoUpdate_(AnmKeyFrame newKey, bool reCalcTangent)
        {
            AddNewKeyFrameCore_(newKey, reCalcTangent, false);
        }

        /// <summary>
        /// キーの登録
        /// </summary>
        /// <param name="newKey"></param>
        void AddNewKeyFrameCore_(AnmKeyFrame newKey, bool reCalcTangent, bool notify)
        {
            // 初めてのキー登録なら、時間変更ハンドラを登録します。// ドリブンカーブ指定があれば、つながないようにする。
            if (_KeyFrames.Count == 0)
            {
                _targetAttr.SetTimeEventHandler(true);
            }

            _KeyFrames.Add(newKey);

            // キーを時間順に、ソートします。
            SortKeyFramesByTimeOrder_();

            if (reCalcTangent)
            {
                // エルミート補間形式の場合は、
                // 新規に挿入されたキーおよび、挿入されたキーの前後のキーの接線情報を計算します。
                CalcTangentValue_(newKey);
            }

            // 変更を通知します。
            if (notify)
            {
                NotifyAddRemoveEvent_();
            }
        }

        /// <summary>
        /// 2つのキーの間の傾きを取得します。
        /// </summary>
        /// <param name="kf0"></param>
        /// <param name="kf1"></param>
        /// <returns></returns>
        float GetTangentBetweenTwoKeies_( AnmKeyFrame kf0, AnmKeyFrame kf1 )
        {
            float v0 = kf0.ValueAsFloat;
            float v1 = kf1.ValueAsFloat;

            return ( v1 - v0 ) / ( kf1.Time - kf0.Time );
        }

        /// <summary>
        /// 2つのキーの間の傾きを取得します。
        /// </summary>
        /// <param name="frame">挿入するキーの時間</param>
        /// <param name="lhs">前時間のキー</param>
        /// <param name="rhs">後時間のキー</param>
        /// <returns></returns>
        public float HermiteSlope(float frame, AnmKeyFrame lhs, AnmKeyFrame rhs)
        {
            double length = rhs.Time - (double)lhs.Time;
            if (length == 0f)
            {
                Debug.Assert(false);
                return 0;
            }

            double v0 = lhs.ValueAsFloat;
            double v1 = rhs.ValueAsFloat;
            double t0 = lhs.OutTangent;
            double t1 = rhs.InTangent;
            double d = length;
            // a, b, c は s をパラメータとするエルミット曲線の微分の係数です。a * s^2 + b * s + c
            double a = 6 * (v0 - v1) + 3 * d * (t1 + t0);
            double b = 6 * (v1 - v0) - 2 * d * t1 - 4 * d * t0;
            double c = d * t0;
            double s = (frame - (double)lhs.Time) / d;
            return (float)(((a * s + b) * s + c) / d);
        }

        /// <summary>
        /// キーの前のキーを取得します。
        /// </summary>
        AnmKeyFrame GetPrevKeyFrame_( AnmKeyFrame kf )
        {
            int keyIdx = _KeyFrames.IndexOf( kf );
            if( keyIdx  >= 1 )
            {
                return _KeyFrames[ keyIdx - 1 ] as AnmKeyFrame;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// キーの後のキーを取得します。
        /// </summary>
        AnmKeyFrame GetNextKeyFrame_( AnmKeyFrame kf )
        {
            int keyIdx = _KeyFrames.IndexOf( kf );
            if( keyIdx + 1 < _KeyFrames.Count )
            {
                return _KeyFrames[ keyIdx + 1 ] as AnmKeyFrame;
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// キーフレームの傾き情報を計算します。
        /// </summary>
        /// <param name="?"></param>
        void CalcTangentValue_( AnmKeyFrame kf )
        {
            if( _KeyFrames.Count <= 1 )
            {
                return;
            }

            AnmKeyFrame kfPrev = GetPrevKeyFrame_( kf );
            AnmKeyFrame kfNext = GetNextKeyFrame_( kf );

            float inTan = 0.0f, outTan = 0.0f;

            if (kfPrev != null && kfNext != null)
            {
                inTan = outTan = HermiteSlope(kf.Time, kfPrev, kfNext);
            }

            if( kf.UnifyTangents )
            {
                float avarage = (inTan + outTan) / 2.0f;
                inTan    = avarage;
                outTan   = avarage;
            }

            kf.InTangent       = inTan;
            kf.OutTangent      = outTan;
        }

        public float CurveMax { get; set; } = float.MaxValue;
        public float CurveMin { get; set; } = float.MinValue;

        /// <summary>
        /// float のオーバーフローを最大の値域でクランプします。
        /// </summary>
        private float ClampInfinityValueToLimit_(float val)
        {
            if (float.IsPositiveInfinity(val))
            {
                return float.MaxValue;
            }
            if (float.IsNegativeInfinity(val))
            {
                return float.MinValue;
            }

            return val;
        }

        /// <summary>
        /// クランプします。
        /// </summary>
        public float ClampTargetMinMax(float val)
        {
            // カーブ上の値は見かけ上のスケールを考慮した値でクランプする
            float maxValue = ClampInfinityValueToLimit_(this.TargetAttributeMaxValue * ViewScale);
            float minValue = ClampInfinityValueToLimit_(this.TargetAttributeMinValue * ViewScale);

            return Math.Max(Math.Min(val, maxValue), minValue);
        }

        /// <summary>
        /// アニメーション結果は、外部へは、float で出力されることとなりました。
        /// </summary>
        /// <param name="time"></param>
        /// <returns></returns>
        public float Evaluate( float time )
        {
            if (_IsParameterizedAnim)
            {
                return EvaluateParameterizedAnim(time);
            }

            // 新しい時間で、値を計算して、その結果を _targetAttrに設定します。
            AnmKeyFrame pkf = null;
            AnmKeyFrame nkf = null;
            float fTime = time;

            //-----------------------------------------------
            // 2つのキーが発見された場合
            // 現在の時間の前後のキーを検索します。
            FindPrevKey( fTime, out pkf, out nkf );

            if( pkf != null && nkf != null )
            {
                if( pkf.Time != nkf.Time )
                {
                    //
                    // キーが2つ発見できた時には、補間計算を試みます。
                    //
                    if( pkf.InterporationType == InterporationType.Step )
                    {
                        return pkf.ValueAsFloat;
                    }
                    else
                    {
                        // エルミート補間
                        return ClampTargetMinMax(DoInterp_Hermite_(
                            pkf.ValueAsFloat,
                            nkf.ValueAsFloat,
                            pkf.Time,
                            nkf.Time,
                            pkf.OutTangent,
                            nkf.InTangent,
                            time ));
                    }
                }
            }

            //-----------------------------------------------
            // 2つのキーが発見できなかった場合
            // キーが存在するなら...
            if( _KeyFrames.Count > 0 )
            {
                // 最初のキーよりも前なら、
                AnmKeyFrame firstKey = _KeyFrames[0] as AnmKeyFrame;
                if( fTime <= firstKey.Time )
                {
                    if( fTime == firstKey.Time )
                    {
                        return firstKey.ValueAsFloat;
                    }
                    else
                    {
                        return ClampTargetMinMax(EvaluatePreInfinity_( time, firstKey ));
                    }
                }
                else
                {
                    // 最後キーよりもあとなら、
                    AnmKeyFrame lastKey = _KeyFrames[_KeyFrames.Count - 1] as AnmKeyFrame;
                    if( lastKey.Time <= fTime )
                    {
                        if( fTime == lastKey.Time )
                        {
                            return lastKey.ValueAsFloat;
                        }
                        else
                        {
                            return ClampTargetMinMax(EvaluatePostInfinity_( time, lastKey ));
                        }
                    }
                }

                // キーが存在する場合は、上記条件にあてはまるはずです。
                Debug.Assert( false );
                return _targetAttr.GetAsFloat();
            }
            else
            {
                // キーが存在しない場合は、アトリビュートの値を返します。
                return _targetAttr.GetAsFloat();
            }
        }

        /// <summary>
        /// 現在時刻付近のキーフレームを検索します。(TODO:仮実装)
        /// </summary>
        private void FindPrevKey(float time, out AnmKeyFrame prevKey, out AnmKeyFrame nextKey)
        {
            prevKey = null;
            nextKey = null;

            // 後方キーを捜します。
            int idxNext = 0;
            for (idxNext = 0; idxNext < NumKeyFrame; idxNext++)
            {
                AnmKeyFrame kf = IKeyFrameSet[idxNext] as AnmKeyFrame;
                if (kf.Time > time)
                {
                    nextKey = kf;
                    break;
                }
            }

            // 後方キーの位置から前方にむかって、前方キーを捜します。
            for (int idxPrev = idxNext - 1; idxPrev >= 0; idxPrev--)
            {
                AnmKeyFrame kf = IKeyFrameSet[idxPrev] as AnmKeyFrame;
                if (kf.Time <= time)
                {
                    prevKey = kf;
                    break;
                }
            }
        }

        /// <summary>
        /// パラメタライズドアニメーションのカーブの計算
        /// </summary>
        float EvaluateParameterizedAnim(float frame)
        {
            for (int i = _parameterizedAnimParameters.Length - 1; i >= 0; i--)
            {
                SerializableObject.Lan.ParameterizedAnimParameter parameter = _parameterizedAnimParameters[i];
                // アニメーションの範囲外ならクランプする
                if (frame < parameter.offset)
                {
                    if (i == 0)
                    {
                        frame = parameter.offset;
                    }
                    else
                    {
                        continue;
                    }
                }
                if (parameter.offset + parameter.duration < frame)
                {
                    frame = parameter.offset + parameter.duration;
                }
                // アニメーションの計算
                float normalizedFrame = (frame - parameter.offset) / parameter.duration; // [0.0, 1.0] に正規化したフレーム
                switch (parameter.parameterizedAnimType)
                {
                    case SerializableObject.Lan.ParameterizedAnimType.Linear: // y = ax
                        return parameter.startValue + (parameter.targetValue - parameter.startValue) * normalizedFrame;
                    case SerializableObject.Lan.ParameterizedAnimType.Square: // y = a(1 - (1 - x) ^ 2)
                        return parameter.startValue + (parameter.targetValue - parameter.startValue) * (1.0f - (1.0f - normalizedFrame) * (1.0f - normalizedFrame));
                    case SerializableObject.Lan.ParameterizedAnimType.Smooth: // y = a * (1 - cos(πx)) / 2
                        return parameter.startValue + (parameter.targetValue - parameter.startValue) * ((1.0f - (float)Math.Cos(Math.PI * normalizedFrame)) / 2.0f);
                    default:
                        return 0.0f;
                }
            }
            return 0.0f;
        }

        /// <summary>
        /// 最初のキーを取得します。
        /// </summary>
        static IAnmKeyFrame GetFirstKey_( List<IAnmKeyFrame> keySet )
        {
            Debug.Assert( keySet.Count > 0 );
            return keySet[0];
        }

        /// <summary>
        /// 最後のキーを取得します。
        /// </summary>
        static IAnmKeyFrame GetLastKey_( List<IAnmKeyFrame> keySet )
        {
            Debug.Assert( keySet.Count > 0 );
            return keySet[keySet.Count - 1];
        }

        /// <summary>
        /// キーフレーム領域の長さを計算します。
        /// </summary>
        static float GetKeyedAreaLength_( List<IAnmKeyFrame> keySet )
        {
            if( keySet.Count <= 1 )
            {
                return 0;
            }
            else
            {
                return keySet[keySet.Count - 1].Time - keySet[0].Time;
            }
        }

        /// <summary>
        /// 最初のキー以前の値を評価します。
        /// </summary>
        float EvaluatePreInfinity_( float time, IAnmKeyFrame firstKey )
        {
            float keyFramedTimeRange = GetKeyedAreaLength_( _KeyFrames );

            if(_preInfinityType == AnmCurveInfinityType.Constant ||
                keyFramedTimeRange == 0 )
            {
                return firstKey.ValueAsFloat;
            }
            else
            {
                // Cycle の場合
                Debug.Assert(_preInfinityType == AnmCurveInfinityType.Cycle );

                float diffTime = (float)firstKey.Time - time;
                Debug.Assert( diffTime >= 0.0f );

                float remider = diffTime % keyFramedTimeRange;

                diffTime = keyFramedTimeRange - remider;
                DbgConsole.WriteLine( "diffTime = {0}", diffTime );

                return Evaluate( (float)( firstKey.Time + diffTime ) );
            }
        }

        /// <summary>
        /// 最後のキー以降の値を評価します。
        /// </summary>
        float EvaluatePostInfinity_( float time, IAnmKeyFrame lastKey )
        {
            float keyFramedTimeRange = GetKeyedAreaLength_( _KeyFrames );

            if(_postInfinityType == AnmCurveInfinityType.Constant ||
                keyFramedTimeRange == 0 )
            {
                return lastKey.ValueAsFloat;
            }
            else
            {
                Debug.Assert(_postInfinityType == AnmCurveInfinityType.Cycle );
                IAnmKeyFrame firstKey = GetFirstKey_( _KeyFrames );
                // Cycle の場合
                // 最終フレームの次のフレームを０フレーム目として扱います。
                float diffTime = time - (float)lastKey.Time;
                Debug.Assert( diffTime >= 0 );

                float remider = diffTime % keyFramedTimeRange;

                diffTime = remider;
                DbgConsole.WriteLine( "diffTime = {0}", diffTime );

                return Evaluate( (float)(firstKey.Time + diffTime) );
            }
        }




#endregion  // ----------------- IAnmCurve 非公開メソッド



        /// <summary>
        /// 対象アトリビュート型
        /// </summary>
        public AttributeType TargetAttrType
        {
            get{return _targetAttr.Type;}
        }

        /// <summary>
        /// アトリビュート名前
        /// </summary>
        public string AttrName
        {
            get{ return _targetAttr.Name; }
        }

        /// <summary>
        /// キーフレーム数
        /// </summary>
        public int NumKeyFrame
        {
            get{ return _IsParameterizedAnim ? 0 : _KeyFrames.Count; }
        }

        public float FrameOfHead
        {
            get
            {
                if (_IsParameterizedAnim)
                {
                    return _parameterizedAnimParameters[0].offset;
                }
                else
                {
                    return _KeyFrames.Count > 0 ? _KeyFrames[0].Time : 0.0f;
                }
            }
        }

        public float FrameOfTail
        {
            get
            {
                if (_IsParameterizedAnim)
                {
                    int index = _parameterizedAnimParameters.Length - 1;
                    return _parameterizedAnimParameters[index].offset + _parameterizedAnimParameters[index].duration;
                }
                else
                {
                    return _KeyFrames.Count > 0 ? _KeyFrames[_KeyFrames.Count - 1].Time : 0.0f;
                }
            }
        }

        public float ValueOfHead
        {
            get
            {
                if (_IsParameterizedAnim)
                {
                    return _parameterizedAnimParameters[0].startValue;
                }
                else
                {
                    return _KeyFrames.Count > 0 ? _KeyFrames[0].ValueAsFloat : 0.0f;
                }
            }
        }

        public float ValueOfTail
        {
            get
            {
                if (_IsParameterizedAnim)
                {
                    return _parameterizedAnimParameters[_parameterizedAnimParameters.Length - 1].targetValue;
                }
                else
                {
                    return _KeyFrames.Count > 0 ? _KeyFrames[_KeyFrames.Count - 1].ValueAsFloat : 0.0f;
                }
            }
        }

        public bool IsParameterizedAnim { get { return _IsParameterizedAnim; } }

        public LECore.Structures.SerializableObject.Lan.ParameterizedAnimParameter[] ParameterizedAnimParameters
        {
            get { return _parameterizedAnimParameters; }
        }

        /// <summary>
        /// 選択キーフレームを取得します。
        /// </summary>
        public IAnmKeyFrame[]  SelectedKeySet
        {
            get{ return _IsParameterizedAnim ? new IAnmKeyFrame[0] : _SelectedKeyFrames.ToArray();}
        }

        /// <summary>
        /// 選択キーの個数を取得します。
        /// </summary>
        public int NumSelectedKey
        {
            get{ return _IsParameterizedAnim ? 0 : _SelectedKeyFrames.Count;}
        }

        /// <summary>
        /// キーフレームが選択されているか調査します。
        /// </summary>
        public bool IsSelectedKey( IAnmKeyFrame key )
        {
           return _IsParameterizedAnim ? false : _SelectedKeyFrames.Contains( key );
        }

        /// <summary>
        /// キーフレーム配列を取得します。
        /// </summary>
        public IAnmKeyFrame[]    IKeyFrameSet
        {
            get
            {
                if (_IsParameterizedAnim)
                {
                    return new IAnmKeyFrame[0];
                }

                if (_KeyFrames != null)
                {
                    return _KeyFrames.ToArray();
                }
                return null;
            }
        }

        /// <summary>
        /// 先頭キーフレーム以前のカーブの計算方法
        /// </summary>
        public AnmCurveInfinityType PreInfinityType
        {
            get { return _IsParameterizedAnim ? AnmCurveInfinityType.Constant : _preInfinityType; }
            set
            {
                if (_preInfinityType != value)
                {
                    _preInfinityType = value;
                    NotifyChangeEvent_();
                }
            }
        }

        /// <summary>
        /// 最終キーフレーム以前のカーブの計算方法
        /// </summary>
        public AnmCurveInfinityType PostInfinityType
        {
            get { return _IsParameterizedAnim ? AnmCurveInfinityType.Constant : _postInfinityType; }
            set
            {
                if (_postInfinityType != value)
                {
                    _postInfinityType = value;
                    NotifyChangeEvent_();
                }
            }
        }


        /// <summary>
        /// カーブ上での、指定キーフレームの前のキーフレームを取得する。
        /// </summary>
        public IAnmKeyFrame    GetPrevKeyFrame( IAnmKeyFrame key )
        {
            if (_IsParameterizedAnim)
            {
                return null;
            }

            IAnmKeyFrame prevKey = null;

            int index = _KeyFrames.IndexOf( key );
            if( index != -1 && index > 0 )
            {
                prevKey = _KeyFrames[index-1] as IAnmKeyFrame;
                Debug.Assert( prevKey != null );
            }
            return prevKey;
        }

        /// <summary>
        /// カーブ上での、指定時間の直前のキーフレームを取得する。
        /// </summary>
        public IAnmKeyFrame GetPrevKeyFrame( float time )
        {
            if (_IsParameterizedAnim)
            {
                return null;
            }

            List<IAnmKeyFrame> revList = new List<IAnmKeyFrame>(_KeyFrames);
            revList.Reverse();
            foreach (IAnmKeyFrame keyFrame in revList)
            {
                if (keyFrame.Time < time)
                {
                    return keyFrame;
                }
            }

            return null;
        }

        /// <summary>
        /// カーブ上での、指定キーフレームの次のキーフレームを取得する。
        /// </summary>
        public IAnmKeyFrame    GetNextKeyFrame( IAnmKeyFrame key )
        {
            if (_IsParameterizedAnim)
            {
                return null;
            }

            IAnmKeyFrame nextKey = null;
            int index = _KeyFrames.IndexOf( key );
            if( index !=  -1 && index+1 < _KeyFrames.Count )
            {
                nextKey = _KeyFrames[index+1] as IAnmKeyFrame;
                Debug.Assert( nextKey != null );
            }
            return nextKey;
        }

        /// <summary>
        /// カーブ上での、指定時間の直後のキーフレームを取得する。
        /// </summary>
        public IAnmKeyFrame GetNextKeyFrame(float time)
        {
            if (_IsParameterizedAnim)
            {
                return null;
            }

            foreach (IAnmKeyFrame keyFrame in _KeyFrames)
            {
                if (keyFrame.Time > time)
                {
                    return keyFrame;
                }
            }

            return null;
        }

        /// <summary>
        /// 指定番号のキーフレームを取得します。
        /// AnimCurveを直接操作するクラスが利用するメソッド。
        /// ラッパー経由ではないので、Undoなどの処理がおこなわれません。
        /// システム外部非公開関数です。
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public AnmKeyFrame GetKeyFrameByIndex_Direct(int index)
        {
            if (_IsParameterizedAnim)
            {
                return null;
            }

            return _KeyFrames[index] as AnmKeyFrame;
        }

       /// <summary>
       /// キーがアニメーションカーブに含まれるか調査します。
       /// </summary>
       /// <param name="keyFrame"></param>
       /// <returns></returns>
        public bool Contains( IAnmKeyFrame keyFrame )
        {
            if (_IsParameterizedAnim)
            {
                return false;
            }

            return Contains( keyFrame as AnmKeyFrame );
        }

        /// <summary>
        /// 規定の補間種類：キーフレーム生成時に参照されます。
        /// </summary>
        public InterporationType DefaultInterpType
        {
            get { return _targetAttr.DefaultInterpType; }
            set
            {
                if(_targetAttr.DefaultInterpType != value &&
                    AnmAttribute.CheckInterpTypeValid( TargetAttrType, value ) )
                {
                    _targetAttr.DefaultInterpType = value;
                }
            }
        }

        /// <summary>
        /// 補間種類が固定されているか？
        /// 補間種類を限定したいアトリビュートで設定します。
        ///
        /// アトリビュートとバインドする際に決定されます。
        /// アトリビュートのコンストラクタにオプションで指定してください。
        /// </summary>
        public bool LockInterpType
        {
            get { return _targetAttr.LockInterpType; }
        }

        #endregion

        public AnmCurve Clone(KeyCloneType key_clone)
        {
            AnmCurve obj = new AnmCurve();

            obj.BindTargetKeepInterpType(this._targetAttr, this.LockInterpType, this._targetAttr.UseCustomTimeSource);

            obj.CurveMax = this.CurveMax;
            obj.CurveMin = this.CurveMin;

            obj._KeyFrames = new List<IAnmKeyFrame>();
            foreach (AnmKeyFrame keyFrame in this._KeyFrames)
            {
                switch (key_clone)
                {
                    case KeyCloneType.Copy:
                        obj._KeyFrames.Add(keyFrame);
                        break;
                    case KeyCloneType.Clone:
                        obj._KeyFrames.Add(keyFrame.Clone() as AnmKeyFrame);
                        break;
                    case KeyCloneType.None:
                        break;
                }
            }

            obj._KeyShouldBeSorted = this._KeyShouldBeSorted;
            obj._SelectedKeyFrames = new List<IAnmKeyFrame>(); // 空のリスト
            obj.ViewScale = this.ViewScale;

            obj._preInfinityType = this._preInfinityType;
            obj._postInfinityType = this._postInfinityType;

            obj._IsParameterizedAnim = this._IsParameterizedAnim;

            return obj;
        }

        /// <summary>
        /// アニメーションカーブ変更イベントを変更対象のアトリビュートに伝えます。
        /// キーフレームからのみ呼び出されます。
        /// </summary>
        void NotifyChangeEvent_()
        {
            this._targetAttr.OnAnmCurveModified();
        }

        /// <summary>
        /// アニメーションカーブ追加イベント通知。
        /// </summary>
        void NotifyAddRemoveEvent_()
        {
            this._targetAttr.OnAnmCurveKeyAddRemove();
        }

        /// <summary>
        /// キーフレームの表示スケールを設定します。
        /// </summary>
        public float ViewScale
        {
            get
            {
                if (this._targetAttr != null)
                {
                    return this._targetAttr.ViewScale;
                }
                return 1.0f;
            }

            set
            {
                if (this._targetAttr != null)
                {
                    this._targetAttr.ViewScale = value;
                }
            }
        }



    }


}
