﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;

namespace LECore.Structures
{
    using System.Collections.Generic;
    using Core;
    using LECoreInterface;

    /// <summary>
    /// 領域ペインです。
    /// </summary>
    internal class Capture :
        LEDataNode,
        ICapture,
        IPaneExParamater,
        IDrawable,
        IDisposable
    {
        public const string CaptureNodeName = "Capture";

        // ペイン本体への参照
        readonly Pane   _ownerPane = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Capture( Pane ownerPane )
            :base( ownerPane, CaptureNodeName )
        {
            Debug.Assert( ownerPane != null );
            _ownerPane = ownerPane;

            // サイズ・回転・スケール・透明度・表示非表示は、仕様によりアニメーション不可能とします。
            _ownerPane.PaneAttrRef.SizeAttr.IsActiveAttribute = false;
            _ownerPane.PaneAttrRef.RotAngAttr.IsActiveAttribute = false;
            _ownerPane.PaneAttrRef.ScaleAttr.IsActiveAttribute = false;
            _ownerPane.PaneAttrRef.TransparencyAttr.IsActiveAttribute = false;
            _ownerPane.PaneAttrRef.VisibleAttr.IsActiveAttribute = false;

            _ownerPane.BindPaneExData( this );
        }

        /// <summary>
        /// リソースを開放します。
        /// </summary>
        public void Dispose()
        {

        }

        #region ICapture

        CaptureTexture FindTargetCaptureTexture()
        {
            if (_ownerPane == null ||
                _ownerPane.OwnerSubScene == null)
            {
                return null;
            }

            return PaneHelper.GetCaptureTextureFromPane(_ownerPane) as CaptureTexture;
        }

        /// <summary>
        /// フレームバッファキャプチャを行うかどうか
        /// </summary>
        public bool FrameBufferCaptureEnabled
        {
            get { return FindTargetCaptureTexture().FrameBufferCaptureEnabled; }
        }

        /// <summary>
        /// フレームバッファキャプチャの際に初回だけキャプチャするかどうか
        /// </summary>
        public bool CaptureOnlyFirstFrame
        {
            get { return FindTargetCaptureTexture().CaptureOnlyFirstFrame; }
        }

        /// <summary>
        /// キャプチャ画像のクリアカラー
        /// </summary>
        public FloatColor ClearColor
        {
            get { return FindTargetCaptureTexture().ClearColor; }
        }

        /// <summary>
        /// テクスチャフォーマットの変更
        /// </summary>
        public TexImagePixelFmt Format
        {
            get { return FindTargetCaptureTexture().Format; }
        }

        /// <summary>
        /// テクスチャ解像度のスケール
        /// </summary>
        public float TextureScale
        {
            get { return FindTargetCaptureTexture().TextureScale; }
        }

        /// <summary>
        /// 自分自身から作成されたテクスチャ
        /// </summary>
        public ICaptureTexture OwnTexture
        {
            get { return FindTargetCaptureTexture(); }
        }

        #endregion

        #region IPaneExParamater
        /// <summary>
        /// 自身を持っているペインの参照です。
        /// </summary>
        public IPane OwnerPane
        {
            get{ return _ownerPane;}
        }

        /// <summary>
        /// 自身を持っているペインの参照です。
        /// </summary>
        public void          UpdateMatarialName   ()
        {
            // Do Nothing
        }

        /// <summary>
        /// シーン登録時の初期化処理
        /// </summary>
        public void OnJoinSceneInitialize()
        {
            // 何もしません。
        }

        /// <summary>
        /// 編集対象に設定される直前の初期化処理
        /// </summary>
        public void FirstTimeInitialize()
        {
            // 何もしません。
        }

        /// <summary>
        /// UV座標を列挙します。
        /// </summary>
        public IEnumerable<TexCoord4> TexCoords
        {
            get { yield break; }
        }

        /// <summary>
        /// 警告文字列を取得します。UIの各所で表示します。
        /// </summary>
        public string WarningMsg
        {
            get { return string.Empty; }
        }

        /// <summary>
        /// 境界を書くかどうか
        /// </summary>
        public bool PaneBoundDrawEnabled
        {
            get { return true; }
        }
        #endregion IPaneExParamater


        #region IDrawable
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            if ((renderer.OptionFlag & IRenderOptionFlag.DrawOutline) != 0)
            {
                return;
            }

            if (option.CheckActive(DrawableOptionFlag.IgnoreCapturePaneDrawing))
            {
                option.TerminateDrawTraverse = true;
                return;
            }

            System.Drawing.Font sysFont = LayoutEditorCore.GuiFont;
            renderer.DbgDrawString( "Capture_Pane", sysFont, 0, 0 );

            Color  oldColor = renderer.Color;

            renderer.Color = option.SystemOpeYellow;
            renderer.FillRectangle(0, 0, 0, OwnerPane.RenderingWidth, OwnerPane.RenderingHeight);
            renderer.Color = oldColor;
        }
        #endregion IDrawable

    }
}
