﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Lyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;

    /// <summary>
    /// XMLシリアライズ可能なクラスへの変換。
    /// (基本的なデータ構造に関する部分)
    /// </summary>
    public partial class RlytConverter
    {
        #region File => App
        /// <summary>
        /// 背景・スクリーン情報を構築します。
        /// </summary>
        void Convert_ScreenSetting_( FileFmt.ScreenSetting srcSetting, ISubScene subScene )
        {
            try
            {
                // 背景・スクリーン情報を構築します。
                if( srcSetting != null )
                {
                    OutLog_( "LECORE_RLYTLOAD_RLYTSETTINGS" );
                    if( srcSetting.backGround != null )
                    {
                        ScreenSettingBackGround srcBackGround = srcSetting.backGround;

                        subScene.BackGround.BackGroundColor =
                            Convert_RGBAColor_( srcBackGround.color ).AsColor;

                        if( srcBackGround.imageFile != null && srcBackGround.imageFile != string.Empty )
                        {
                            string absolutePath = GetAbsolutePath_( srcBackGround.imageFile );
                            subScene.BackGround.BackgroundImage =
                                TextureImageFactory.BuildTextureImageFromTga( absolutePath );
                        }
                    }

                    // グリッド関連
                    if( srcSetting.grid != null )
                    {
                        ScreenSettingGrid srcGrid = srcSetting.grid;
                        subScene.BackGround.GridColor = Convert_RGBAColor_( srcGrid.color ).AsColor;
                        subScene.BackGround.GridSize = srcGrid.thickLineInterval;
                        subScene.BackGround.GridDivisionNum = srcGrid.thinDivisionNum;
                        subScene.BackGround.GridVisible = srcGrid.visible;
                        subScene.BackGround.GridSnapEnalbed = srcGrid.moveMethod != GridMoveMethod.None;
                    }

                    // レイアウトサイズ
                    subScene.BackGround.ScreenSize =
                        new FVec2( srcSetting.layoutSize.x, srcSetting.layoutSize.y );
                }
            }
            catch
            {
                string msg = LECoreStringResMgr.Get( "LECORE_SYS_ERROR_LOAD_LAYOUTSETTING" );
                ReportErrorMsg_( msg );
            }
        }

        #endregion

        #region App => File
        /// <summary>
        /// スクリーン設定を変換します。
        /// </summary>
        FileFmt.ScreenSetting Convert_SOScreenSetting_( ISubScene scene )
        {

            AppData.BackGround backGround = scene.BackGround;
            FileFmt.ScreenSetting screenSetting = new ScreenSetting();

            //-------------- backGround 関連
            screenSetting.backGround = new ScreenSettingBackGround();
            screenSetting.backGround.color =
                Convert_Color4_( new RGBAColor( backGround.BackGroundColor ) );

            // 背景画像が存在する場合は...
            if( scene.BackGround.BackgroundImage != null )
            {
                screenSetting.backGround.imageFile =
                    GetRelativePath_( backGround.BackgroundImage.FilePath );
            }

            //-------------- layoutSize 関連
            screenSetting.layoutSize = new Vec2();
            screenSetting.layoutSize.x = backGround.ScreenSize.X;
            screenSetting.layoutSize.y = backGround.ScreenSize.Y;

            screenSetting.grid = new ScreenSettingGrid();
            screenSetting.grid.color = Convert_Color4_( new RGBAColor( backGround.GridColor ) );
            screenSetting.grid.thickLineInterval = backGround.GridSize;
            screenSetting.grid.thinDivisionNum = backGround.GridDivisionNum;
            screenSetting.grid.visible = backGround.GridVisible;
            screenSetting.grid.moveMethod = ( backGround.GridSnapEnalbed ) ? GridMoveMethod.Grid : GridMoveMethod.None;


            //-------------- 座標系関連
            screenSetting.origin = Convert_ScreenCoordinate_( LayoutEditorCore.Config.ScreenCoordinate );

            return screenSetting;
        }

        #endregion
    }

}
