﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;

namespace LECore.Save_Load
{
    using Structures;
    using Structures.Core;
    using Structures.SerializableObject.Lan;
    using static RlanHelper;
    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lan;

    class RludConverter
    {
        #region ------------- Rlud(読み込み)

        /// <summary>
        /// ターゲット名、ターゲットIDから、アニメーションアトリビュートを取得します。
        /// </summary>
        static AnmAttribute GetTargetAnmAttribute_(AppData.Pane dstPane, string targetName, byte targetId)
        {
            UserDataElement element = (UserDataElement)dstPane.UserDataHolder.UserDataElementSet.FirstOrDefault(e => e.Name == targetName);
            if (element == null || element.IAnmAttribute == null)
            {
                Debug.Assert(false);
                return null;
            }

            if (element.IAnmAttribute.NumSubAttribute == 0)
            {
                Debug.Assert(targetId == 0);
                return (AnmAttribute)element.IAnmAttribute;
            }

            return (AnmAttribute)element.IAnmAttribute.FindSubAttributeByIdx(targetId);
        }

        /// <summary>
        /// シリアライズデータ => アプリケーションデータ
        /// </summary>
        static public void LoadRludToExtUserData( object dstObject, FileFmt.AnimContent srcAnimContent, string tag)
        {
            Pane dstPane = dstObject as Pane;
            if( dstPane == null )
            {
                return;
            }

            if (dstPane.PaneName != srcAnimContent.name)
            {
                return;
            }

            foreach (FileFmt.AnimTarget srcAnimTarget in srcAnimContent.Items)
            {
                AnmAttribute dstAttr = GetTargetAnmAttribute_(dstPane, srcAnimContent.extUserDataTargetName, srcAnimTarget.id);
                Debug.Assert(dstAttr != null);

                RlanHelper.SetHermiteKeyFrame_(dstAttr, srcAnimTarget, tag);
            }
        }

        #endregion ------------- Rlud

        #region ------------- Rlud(保存)
        /// <summary>
        /// AppData.IPane => FileFmt.AnimContent[]
        /// </summary>
        static public FileFmt.AnimContent[] SaveExtUserDataAnim(AppData.IPane srcPane, string tag)
        {
            return ConvertToAnimContents<IUserDataElement, AnimExtUserDataTarget>(
                tag,
                ParamaterKind.Animation_ExtUserData,
                srcPane,
                x => x.IUserDataHolder.UserDataElementSet.Select(y => new AnimContentSource<IUserDataElement>(srcPane.PaneName, y.Name, y)),
                x =>
                {
                    switch (x.UserDataKind)
                    {
                        case UserDataKind.AnmInt:
                        case UserDataKind.AnmFloat:
                            return new[] { new AnimTargetSource(x.IAnmAttribute, AnimTargetType.TranslateX) };
                        case UserDataKind.AnmIntVec2:
                        case UserDataKind.AnmFloatVec2:
                        case UserDataKind.AnmFloatVec3:
                        case UserDataKind.AnmByteRGBA4:
                            return Enumerable.Range(0, x.IAnmAttribute.NumSubAttribute).Select(y => new AnimTargetSource(x.IAnmAttribute.FindSubAttributeByIdx(y), AnimTargetType.TranslateX, (byte)y));
                        case UserDataKind.AnmByteRGBA4Degamma:
                            {
                                var types = new[] {
                                    AnimTargetType.ExUserDataDegammaByteColor_r,
                                    AnimTargetType.ExUserDataDegammaByteColor_g,
                                    AnimTargetType.ExUserDataDegammaByteColor_b,
                                    AnimTargetType.ExUserDataDegammaByteColor_a
                                };
                                return Enumerable.Range(0, x.IAnmAttribute.NumSubAttribute).Select(y => new AnimTargetSource(x.IAnmAttribute.FindSubAttributeByIdx(y), types[y], (byte)y));
                            }
                        default:
                            return Enumerable.Empty<AnimTargetSource>();
                    }
                }

            ).ToArray();
        }
        #endregion ------------- Rlud
    }
}
