﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;

namespace LECore.Save_Load
{
    using Structures;
    using System.Drawing;
    using System.Linq;
    using Structures.Core;
    using Structures.LECoreInterface;
    using Structures.SerializableObject.Lan;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lan;

    using LECore.Manipulator;


    /// <summary>
    /// RlanConverter の概要の説明です。
    /// </summary>
    public class RlanConverter
    {
        #region        フィールド

        /// <summary>
        /// メッセージ報告クラス
        /// </summary>
        readonly LEMsgReporter _msgReporter = null;
        ParamaterKind _animationFileType = ParamaterKind.Animation_PaneSRT;
        static readonly string _VersionString = "1.0.0";

        #endregion     フィールド

        #region プロパティ

        // TODOT
        /// <summary>
        /// キー領域外のカーブを焼き付けるか？
        /// </summary>
        private bool FixInfinityCurve
        {
            get { return this.SaveOptions.BakeAnimationInfinityCurve; }
        }

        /// <summary>
        /// 出力オプション
        /// </summary>
        public ExportOption SaveOptions
        {
            get;
            set;
        } = new ExportOption();

        #endregion プロパティ

        /// <summary>
        /// ログ出力します。
        /// </summary>
        void OutLog_( string msgID, params object[] paramaters )
        {
            if( _msgReporter != null )
            {
                string title = LECoreStringResMgr.Get( msgID, paramaters );
                _msgReporter.OutLog( title, string.Empty );
            }
        }

        #region FileFmt => AppData

        /// <summary>
        /// FileFmt.Document から ParamaterKind に対応する RLAN を取得します。
        /// </summary>
        FileFmt.LAN
            GetRlan_( FileFmt.Document srcDocument, ParamaterKind animationFileType )
        {
            if( srcDocument.body.lan != null )
            {
                foreach( FileFmt.LAN srcRlan in srcDocument.body.lan )
                {
                    if( srcRlan.animType == RlanHelper.ParamaterKindToAnimationType( animationFileType ) )
                    {
                        return srcRlan;
                    }
                }
            }

            // 発見できませんでした。
            return null;
        }

        /// <summary>
        /// FileFmt.Document から ParamaterKind に対応する RLAN を取得します。
        /// アニメーション分割管理モードの場合に呼び出されます。
        /// </summary>
        FileFmt.LAN
            GetRlan_(AnimTag animTag, ParamaterKind animationFileType)
        {
            if (animTag.lan != null)
            {
                foreach (FileFmt.LAN srcRlan in animTag.lan)
                {
                    if (srcRlan.animType == RlanHelper.ParamaterKindToAnimationType(animationFileType))
                    {
                        return srcRlan;
                    }
                }
            }

            // 発見できませんでした。
            return null;
        }

        #endregion FileFmt => AppData

        #region AppData => FileFmt


        /// <summary>
        /// AppData.SubScene => FileFmt.RLVI
        /// </summary>
        FileFmt.LAN
            Convert_AppDataToRLAN_(AppData.ISubScene srcSubScene)
        {
            Debug.Assert(srcSubScene != null);

            FileFmt.LAN rlan = new LAN();

            rlan.startFrame = srcSubScene.AnimStartTime;
            rlan.endFrame = srcSubScene.AnimEndTime;
            rlan.convertStartFrame = srcSubScene.AnimPlayStartTime;
            rlan.convertEndFrame = srcSubScene.AnimPlayEndTime;

            IAnimFrameSectionSet animFrameSectionSet = srcSubScene?.IAnimFrameSectionSet;
            if (animFrameSectionSet != null)
            {
                // 「ビューア転送時に再生範囲を自動的に計算する」
                if (SaveOptions.AutoAdjustPlayRange)
                {
                    if (animFrameSectionSet.IAnimFrameSectionSet != null)
                    {
                        int sectionMargin = 10; // 結合の仕様で、区間と区間のあいだに10フレームの隙間が作られます
                        rlan.convertStartFrame = 0;
                        rlan.convertEndFrame = 0;
                        foreach (IAnimFrameSection section in srcSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet)
                        {
                            rlan.convertEndFrame += section.EndFrame + sectionMargin;
                        }
                        rlan.convertEndFrame -= sectionMargin;  // 余分な範囲を切り取り

                        if (rlan.convertEndFrame <= 0)
                        {
                            rlan.convertEndFrame = 100;
                        }
                    }
                }
                // 「選択された区間をアニメーション再生範囲に設定」
                else if (SaveOptions.UseTargetFrameSectionRangeAsPlayRange)
                {
                    if (animFrameSectionSet.TargetIAnimFrameSection != null)
                    {
                        var frames = CalcFrame(srcSubScene);
                        rlan.convertStartFrame = frames.Item1;
                        rlan.convertEndFrame = frames.Item2;
                    }
                }
            }

            rlan.animContent = null;
            rlan.animLoop = FileFmt.AnimLoopType.Loop;

            // 長さゼロが指定されている場合は、Waitアニメーションとして保存する
            if (rlan.convertStartFrame == rlan.convertEndFrame || rlan.startFrame == rlan.endFrame)
            {
                rlan.animLoop = AnimLoopType.Wait;
            }

            return rlan;
        }

        /// <summary>
        /// ビューアに転送する区間タグの開始終了フレームを算出します。
        /// </summary>
        private Tuple<int, int> CalcFrame(AppData.ISubScene srcSubScene)
        {
            if (!srcSubScene.IsAnimEditSeparateMode())
            {
                return new Tuple<int, int>(
                    srcSubScene.IAnimFrameSectionSet.TargetIAnimFrameSection.StartFrame,
                    srcSubScene.IAnimFrameSectionSet.TargetIAnimFrameSection.EndFrame);
            }

            var rangeList = SubSceneHelper.CalcSectionMaxRange(srcSubScene.IPaneArray, srcSubScene.IAnimFrameSectionSet);
            var marginList = SubSceneHelper.CalcSectionMaxMargin(srcSubScene.IPaneArray, srcSubScene.IAnimFrameSectionSet);
            var sectionParam = SubSceneHelper.CalcSectionParam(srcSubScene.IAnimFrameSectionSet, rangeList, marginList);

            return sectionParam[srcSubScene.IAnimFrameSectionSet.TargetIAnimFrameSection.Name];
        }

        /// <summary>
        /// アニメーションタグ情報を変換します。
        /// </summary>
        FileFmt.AnimTag[] ConvertAppFrameSectionDataToTagData_(
            IEnumerable<IAnimFrameSection> animFrameSections,
            Dictionary<string, List<LAN>> rlanList)
        {
            List<AnimTag> animTagSet = new List<AnimTag>();
            foreach (IAnimFrameSection srcAfs in animFrameSections)
            {
                AnimTag dstTag = new AnimTag();

                dstTag.name = srcAfs.Name;
                dstTag.userData = Convert_PaneUserDataEx_(srcAfs.UserDataHolder.UserDataElementSet);
                dstTag.lan = (rlanList != null) ? rlanList[srcAfs.Name]?.ToArray() : new List<LAN>().ToArray();
                dstTag.comment = srcAfs.Comment;
                dstTag.startFrame = srcAfs.StartFrame;
                dstTag.endFrame = srcAfs.EndFrame;

                dstTag.outputEnabled = srcAfs.BinarySettings.Active;
                dstTag.fileName = srcAfs.BinarySettings.FileName;
                dstTag.animLoop = (AnimLoopType)srcAfs.BinarySettings.LoopType;
                dstTag.group = Convert_AppDataToGroupRef_( srcAfs );
                dstTag.descendingBind = srcAfs.BindAnimationRecursively;

                // 出力アニメーション種類フラグを設定します。
                ParamaterKind animKind = srcAfs.BinarySettings.TargetAnimKind;
                dstTag.outputPaneSRT = ( animKind & ParamaterKind.Animation_PaneSRT ) != 0;
                dstTag.outputVertexColor = ( animKind & ParamaterKind.Animation_VertexColors ) != 0;
                dstTag.outputVisibility = ( animKind & ParamaterKind.Animation_Visivility ) != 0;
                dstTag.outputMaterialColor = (animKind & (ParamaterKind.Animation_MaterialColors | ParamaterKind.Animation_FontShadowColor) ) != 0;
                dstTag.outputTexturePattern = ( animKind & ParamaterKind.Animation_TexturePattern ) != 0;
                dstTag.outputTextureSRT = (animKind & ParamaterKind.Animation_TextureSRT ) != 0;
                dstTag.outputIndTextureSRT = (animKind & ParamaterKind.Animation_IndirectTextureSRT) != 0;
                dstTag.outputAlphaTest = (animKind & ParamaterKind.Animation_AlphaTest) != 0;
                dstTag.outputPerCharacterTransform = (animKind & ParamaterKind.Animation_PerCharacterTransform) != 0;
                dstTag.outputWindow = (animKind & ParamaterKind.Animation_Window) != 0;
                dstTag.outputExtUserData = (animKind & ParamaterKind.Animation_ExtUserData) != 0;
                dstTag.outputMaskTextureSRT = (animKind & ParamaterKind.Animation_MaskTextureSRT) != 0;
                dstTag.outputDropShadow = (animKind & ParamaterKind.Animation_DropShadow) != 0;
                dstTag.outputProceduralShape = (animKind & ParamaterKind.Animation_ProceduralShape) != 0;

                animTagSet.Add( dstTag );
            }
            return animTagSet.ToArray();
        }

        private object[] Convert_PaneUserDataEx_(IUserDataElement[] userDataElement)
        {
            return RlytConverter.Convert_PaneUserDataEx_<UserDataFloatList, UserDataIntList, UserDataNone, UserDataString, UserDataElementType>(userDataElement);
        }

        /// <summary>
        /// アニメーション共有情報を更新します。
        /// </summary>
        FileFmt.AnimShare[] ConvertAppAnimShareInfo_( AnimShareInfomationManager asMgr )
        {
            List<AnimShare> dstAnimShareSet = new List<AnimShare>();
            foreach( IAnimShareInfomationChunk asic in asMgr.IAnimShareInfomationChunkSet )
            {
                if( asic.IAnimShareInfomationSet.Length > 0 )
                {
                    AnimShare dstAnimShare = new AnimShare();

                    List<AnimShareInfo> dstAnimShareInfoSet = new List<AnimShareInfo>();
                    foreach( IAnimShareInfomation srcAsi in asic.IAnimShareInfomationSet )
                    {
                        AnimShareInfo dstAsi = new AnimShareInfo();

                        dstAsi.srcPaneName = srcAsi.SrcPaneName;
                        dstAsi.targetGroupName = srcAsi.TargetGroupName;
                        dstAsi.comment = srcAsi.Comment;

                        dstAnimShareInfoSet.Add( dstAsi );
                    }

                    dstAnimShare.targetTagName = null; // 現状は必ずnullが設定されます。
                    dstAnimShare.animShareInfo = dstAnimShareInfoSet.ToArray();

                    dstAnimShareSet.Add( dstAnimShare );
                }
            }

            return dstAnimShareSet.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.GroupRef[] Convert_AppDataToGroupRef_( IAnimFrameSection srcAfs )
        {
            ILEGroup[] srcGroupSet = srcAfs.TargetGroup;
            GroupRef[]			dstGroupRef = new GroupRef[srcGroupSet.Length];
            for( int i = 0 ; i < srcGroupSet.Length ; i++ )
            {
                dstGroupRef[i] = new GroupRef();
                dstGroupRef[i].name = srcGroupSet[i].GrouprName;
            }

            return dstGroupRef;
        }

        #endregion AppData => FileFmt

        /// <summary>
        /// アニメーション共有情報をシーンに読み込みます。
        /// </summary>
        void SetAnimShareInfoToScene_(
            AppData.ISubScene dstScene,
            AnimShare[] srcAnimShareSet )
        {
            if( srcAnimShareSet == null )
            {
                return;
            }

            Debug.Assert( srcAnimShareSet.Length == 1 );

            var dstAnimShareMgr = dstScene.IAnimShareInfomationManager as AnimShareInfomationManager;
            foreach( AnimShare srcAnimShare in srcAnimShareSet )
            {
                //---------------------------------------------------
                // 注意：現状は必ず1つのチャンクが存在しています。
                // そのデータに対して登録を行います。
                var dstAnimChunk = dstAnimShareMgr.IAnimShareInfomationChunkSet.ElementAt(0) as AnimShareInfomationChunk;
                foreach( AnimShareInfo srcAnimShareInfo in srcAnimShare.animShareInfo )
                {
                    IPane pane = SubSceneHelper.FindPaneByName( dstScene, srcAnimShareInfo.srcPaneName );
                    ILEGroup group = dstScene.ILEGroupMgr.FindGroupByName( srcAnimShareInfo.targetGroupName );

                    IAnimShareInfomation asi = dstAnimChunk.Register( pane, group, srcAnimShareInfo.comment );

                    if( pane == null || group == null )
                    {
                        dstAnimChunk.MarkAsInvalid( asi, srcAnimShareInfo.srcPaneName, srcAnimShareInfo.targetGroupName );
                    }
                }
            }
        }

        /// <summary>
        /// LのバージョンがRと等しいかより古いことを判定する。
        /// </summary>
        /// <returns></returns>
        bool IsSameOrOlderThan(string versionL, string versionR)
        {
            if (versionL == versionR)
            {
                return true;
            }

            string[] verNumsL = versionL.Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries);
            string[] verNumsR = versionR.Split(new char[] { '.' }, StringSplitOptions.RemoveEmptyEntries);

            for (int i = 0; i < verNumsL.Length && i < verNumsR.Length; i++)
            {
                int verNumL = int.Parse(verNumsL[i]);
                int verNumR = int.Parse(verNumsR[i]);

                if (verNumL == verNumR)
                {
                    continue;
                }

                return (verNumL < verNumR);
            }

            // 比較できる範囲ではバージョンが一致した。
            // この場合はバージョン数の多いほうを古いと判断しておく。(1.2 と 1.2.3 なら 1.2 が古い)
            return verNumsL.Length < verNumsR.Length;
        }

        /// <summary>
        /// 中間ファイルのフォーマットバージョン更新処理を行います。
        /// 新しいツールで仕様変更を行い、古いバージョンのデータを整合性を保ったまま読み込む必要が生じた場合
        /// ここで対処してください。
        /// </summary>
        /// <param name="srcDocument"></param>
        private void VersionUpdate_(FileFmt.Document srcDocument)
        {
            // 今のところなにも処理はありません。
        }

        /// <summary>
        /// アニメーションをシーンに読み込みます。
        /// </summary>
        public void SetDataToScene(
            AppData.ISubScene dstScene,
            FileFmt.Document srcDocument,
            ParamaterKind animationFileType,
            IImporterOption importerOption)
        {
            // バージョン固有のアップデート処理を施す
            // 1.5.1 より古いデータなら、、、
            VersionUpdate_(srcDocument);

            // アニメーション区間タグの読み込み
            try
            {
                dstScene.BeginMassiveModify();
                if (srcDocument != null && srcDocument.body != null)
                {
                    bool isSuccess = SetTagDataToScene(dstScene, srcDocument.body.animTag, srcDocument, null, importerOption);
                    if (!isSuccess)
                    {
                        // 失敗した場合、以降のアニメーション読み込み処理を行いません。
                        return;
                    }
                }
            }
            finally
            {
                dstScene.EndMassiveModify();
            }

            // アニメーション区間フレームを読み込みます。
            //
            foreach( ParamaterKind animKind in ParamaterKindHelper.AnimationKindSet )
            {
                // 指定されていれば...
                if( ( animationFileType & animKind ) != 0 )
                {
                    // 指定された animationFileType にふさわしい、rlan を 取得します。
                    FileFmt.LAN srcRlan = GetRlan_( srcDocument, animKind );
                    if( srcRlan != null && srcRlan.animContent.Length != 0 )
                    {
                        // 一度でも、読み込み試行をおこなえば終了します。
                        (dstScene as SubScene).SetTimeRange(
                            srcRlan.startFrame,
                            srcRlan.endFrame,
                            srcRlan.convertStartFrame,
                            srcRlan.convertEndFrame);
                        break;
                    }
                }
            }

            dstScene.BeginMassiveModify();
            // 全てのアニメーションについて...
            foreach( ParamaterKind animKind in ParamaterKindHelper.AnimationKindSet )
            {
                // 指定されていなければ中断
                if( ( animationFileType & animKind ) == 0 )
                {
                    continue;
                }

                // 指定された animationFileType にふさわしい、rlan を 取得します。
                FileFmt.LAN srcRlan = GetRlan_( srcDocument, animKind );
                if( srcRlan == null || srcRlan.animContent.Length == 0 )
                {
                    continue;
                }

                OutLog_( "LECORE_RLANLOAD_LOAD_ANIM" );

                // 読み込み関数を実行
                foreach( RlanHelper.AnimLoadFunction loader in RlanHelper.GetAnimLoadFunctionSet( animKind ) )
                {
                    RlanHelper.DoLoad(loader, dstScene.IPaneArray, srcRlan.animContent, AnimCurvesParam.__NoSelected__);
                }
            }
            dstScene.EndMassiveModify();

            // アニメーション共有情報の読み込み
            dstScene.BeginMassiveModify();
            if( srcDocument != null && srcDocument.body != null )
            {
                SetAnimShareInfoToScene_( dstScene, srcDocument.body.animShare );
            }
            dstScene.EndMassiveModify();

        }

        /// <summary>
        /// タグ情報をファイルフォーマットに書き出します。
        /// </summary>
        public FileFmt.AnimTag[] SetTagDataToFileFmt(
            IEnumerable<IAnimFrameSection> animFrameSections)
        {
            return ConvertAppFrameSectionDataToTagData_(animFrameSections, null);
        }

        /// <summary>
        ///
        /// </summary>
        public void AddFrameSection( ISubScene subScene, FileFmt.AnimTag[] animTags, List<string> ngTagReports, CreateAnimFrameSectionHandler creator, SetAnimFrameSectionParamHandler setter, CreateSectionHandler sectionCreator)
        {
            subScene.BeginMassiveModify();
            SetTagDataToScene( subScene, animTags, null, ngTagReports, creator, setter, sectionCreator, IImporterOption.IsAdditionalImport);
            subScene.EndMassiveModify();
        }

        /// <summary>
        /// タグ情報をシーンに読み込みます。
        /// </summary>
        public delegate IAnimFrameSection CreateAnimFrameSectionHandler(string name, string comment, int startFrame, int endFrame, ISubScene subScene);
        public delegate void SetAnimFrameSectionParamHandler(IAnimFrameSection frameSection, ILEGroup[] groups, bool descendingBind);
        public delegate void CreateSectionHandler(string animTagName);

        /// <summary>
        ///
        /// </summary>
        internal bool SetTagDataToScene(AppData.ISubScene subScene, AnimTag[] animTags, FileFmt.Document srcDocument, List<string> ngTagReports, IImporterOption importerOption)
        {
            AnimFrameSectionSet frameSectionSet = subScene.IAnimFrameSectionSet as AnimFrameSectionSet;

            CreateAnimFrameSectionHandler createNewAnimTag_ = delegate(string name, string comment, int startFrame, int endFrame, ISubScene s)
            {
                return frameSectionSet.AddNewFrameSection(name, comment, startFrame, endFrame);
            };

            SetAnimFrameSectionParamHandler registerNewAnimTag_ = delegate(IAnimFrameSection frameSection, ILEGroup[] groups, bool descendingBind)
            {
                ((AnimFrameSection)frameSection).SetTargetGroup(groups, descendingBind);
            };

            CreateSectionHandler createSection_ = delegate (string animTagName)
            {
                SubSceneHelper.AddAnimSectionDirect(subScene, animTagName);
            };

            return SetTagDataToScene(subScene, animTags, srcDocument, ngTagReports, createNewAnimTag_, registerNewAnimTag_, createSection_, importerOption);
        }

        /// <summary>
        /// シーンにタグ情報を設定します。
        /// </summary>
        public bool SetTagDataToScene(
            AppData.ISubScene dstScene,
            AnimTag[] animTagSet,
            FileFmt.Document srcDocument,
            List<string> ngTagReportStrSet,
            CreateAnimFrameSectionHandler creator,
            SetAnimFrameSectionParamHandler setter,
            CreateSectionHandler sectionCreator,
            IImporterOption importerOption)
        {
            // アニメーション管理モードを設定します
            // 区間タグインポートの場合はsrcDocumentがnullになります
            if (srcDocument != null)
            {
                // 追加読み込み時は、同じモードでしか読み込みを行わない。
                if (importerOption.HasFlag(IImporterOption.IsAdditionalImport) &&
                    dstScene.IsAnimEditSeparateMode() != srcDocument.body.separateMode)
                {
                    LayoutEditorCore.MsgReporter.ReportError(
                        LECoreStringResMgr.Get("LECORE_DLG_TITLE_LOAD"),
                        LECoreStringResMgr.Get("LAYOUT_ERROR_ANIMTAG_SEPARATEMODE_DIFFERENT",
                        dstScene.IsAnimEditSeparateMode(), srcDocument.body.separateMode));

                    return false;
                }

                (dstScene as SubScene).ChangeAnimEditMode(srcDocument.body.separateMode);
            }

            // dstSubSceneが既に区間タグを持っている場合、それらに対応するAnmCurveを生成します
            foreach (string tag in SubSceneHelper.GetTags(dstScene))
            {
                sectionCreator(tag);
            }

            if (animTagSet != null)
            {
                OutLog_("LECORE_RLANLOAD_LOAD_ANIMTAG");

                foreach (AnimTag animTag in animTagSet)
                {
                    //-----------------------------------------------
                    AnimFrameSectionSet dstSectionSet = dstScene.IAnimFrameSectionSet as AnimFrameSectionSet;
                    Debug.Assert(dstSectionSet != null);

                    AnimFrameSection frameSection = creator(
                        animTag.name,
                        SerializableConverterUtil.ReplaceLineFeedCode(animTag.comment),
                        animTag.startFrame,
                        animTag.endFrame,
                        dstScene) as AnimFrameSection;

                    if (frameSection == null)
                    {
                        continue;
                    }

                    // 区間タグごとにAnmCurveを生成
                    // dstScene.IsAnimEditSeparateがfalseの場合AnmCurveは生成されない
                    sectionCreator(animTag.name);

                    //-----------------------------------------------
                    if (animTag.group != null && animTag.group.Length > 0)
                    {
                        // 対象グループを読み込みます。
                        List<string> ngGroupNameSet = new List<string>();
                        {
                            List<ILEGroup> groupSet = new List<ILEGroup>();
                            foreach (GroupRef groupRef in animTag.group)
                            {
                                ILEGroup group = dstScene.ILEGroupMgr.FindGroupByName(groupRef.name);
                                if (group == null)
                                {
                                    ngGroupNameSet.Add(groupRef.name);
                                    continue;
                                }

                                if (!groupSet.Contains(group))
                                {
                                    groupSet.Add(group);
                                }
                            }

                            setter(frameSection, groupSet.ToArray(), animTag.descendingBind);
                        }


                        // 入力できなかったデータがある場合はレポートします。
                        if (ngTagReportStrSet != null && ngGroupNameSet.Count > 0)
                        {
                            string ngTagReportStr = string.Format("*** [ {0} ] *** ", animTag.name) + Environment.NewLine;
                            foreach (string ngGroupName in ngGroupNameSet)
                            {
                                ngTagReportStr += "    " + ngGroupName + Environment.NewLine;
                            }
                            ngTagReportStrSet.Add(ngTagReportStr);
                        }
                    }

                    //-----------------------------------------------
                    // アニメーションがある場合は読み込みます
                    // Import時は animTag.lan はNULLのはずなのでこの処理は通らない想定
                    if (animTag.lan != null)
                    {
                        // TimeRangeの読み込み
                        foreach (ParamaterKind animKind in ParamaterKindHelper.AnimationKindSet)
                        {
                            // 指定されていれば...
                            if ((ParamaterKind.Animation_All & animKind) != 0)
                            {
                                // 指定された animationFileType にふさわしい、rlan を 取得します。
                                FileFmt.LAN srcRlan = GetRlan_(animTag, animKind);
                                if (srcRlan != null && srcRlan.animContent.Length != 0)
                                {
                                    // 一度でも、読み込み試行をおこなえば終了します。
                                    (dstScene as SubScene).SetTimeRange(
                                        srcRlan.startFrame,
                                        srcRlan.endFrame,
                                        srcRlan.convertStartFrame,
                                        srcRlan.convertEndFrame);
                                    break;
                                }
                            }
                        }

                        // アニメーションの読み込み


                        // 全てのアニメーションについて...
                        foreach (ParamaterKind animKind in ParamaterKindHelper.AnimationKindSet)
                        {
                            // 指定されていなければ中断
                            if ((ParamaterKind.Animation_All & animKind) == 0)
                            {
                                continue;
                            }

                            // 指定された animationFileType にふさわしい、rlan を 取得します。
                            FileFmt.LAN srcRlan = GetRlan_(animTag, animKind);
                            if (srcRlan == null || srcRlan.animContent.Length == 0)
                            {
                                continue;
                            }

                            OutLog_("LECORE_RLANLOAD_LOAD_ANIM");

                            // 読み込み関数を実行
                            foreach (RlanHelper.AnimLoadFunction loader in RlanHelper.GetAnimLoadFunctionSet(animKind))
                            {
                                RlanHelper.DoLoad(loader, dstScene.IPaneArray, srcRlan.animContent, frameSection.Name);
                            }
                        }
                    }

                    //-----------------------------------------------
                    dstSectionSet.SetOutBinayActive(frameSection, animTag.outputEnabled);
                    dstSectionSet.SetOutBinayFileName(frameSection, animTag.fileName);
                    dstSectionSet.SetOutBinayLoopType(frameSection, (AnimationLoopType)animTag.animLoop);

                    ParamaterKind animParamKind = RlanHelper.ConvertAnimTagFlagsToParamaterKind(animTag);
                    dstSectionSet.SetOutBinayParamaterKind(frameSection, animParamKind);

                    // 拡張ユーザデータをアプリ形式に変換
                    foreach (UserDataElement ud in GetUserDataElementSetImpl_(animTag.userData))
                    {
                        (frameSection.UserDataHolder as UserDataHolder).AddUserDataElement(ud.Name, ud.UserDataKind, ud.Overwrite, ud.Value);
                    }

                    // 参照設定を行ないます
                    SubSceneHelper.MakeReferenceCurves(dstScene);
                }

                if (ngTagReportStrSet != null && ngTagReportStrSet.Count > 0)
                {
                    // =========== グループが発見できず、設定されなかったタグ区間 ===========
                    ngTagReportStrSet.Insert(0,
                        LECoreStringResMgr.Get("LECORE_ANIMTAG_WARNING_GROUPNOTFOUND") + Environment.NewLine);
                }
            }

            return true;
        }

        internal static UserDataElement[] GetUserDataElementSetImpl_(object[] userData)
        {
            return RlytConverter.GetUserDataElementSetImpl_<UserDataFloatList, UserDataIntList, UserDataNone, UserDataString, UserDataElementType>(userData);
        }

        /// <summary>
        /// アニメーションを書き出します。
        /// </summary>
        public FileFmt.Document Convert_SODocument(
            ISubScene scene,
            ParamaterKind kind )
        {
            FileFmt.Document result;

            if (scene.IsAnimEditSeparateMode())
            {
                result = Convert_SODocument_Multi_(scene, kind);
            }
            else
            {
                result = Convert_SODocument_Single_(scene, kind);
            }

            return result;
        }

        /// <summary>
        /// アニメーションを書き出します。
        /// アニメーションの一括管理モード時に呼び出されます。
        /// </summary>
        private FileFmt.Document Convert_SODocument_Single_(
            ISubScene scene,
            ParamaterKind kind)
        {
            // ------------ RLAN を書き出します。
            // 全ての種類のアニメーションから...
            List<LAN> rlanSet = new List<LAN>();
            foreach (ParamaterKind paramKind in ParamaterKindHelper.AnimationKindSet)
            {
                _animationFileType = paramKind;

                // 指定された種類のアニメーションのみを出力します。
                if ((paramKind & kind) != 0)
                {
                    // AnimContent を抽出します。
                    // 抽出されれば、RLANを生成して登録します。
                    FileFmt.AnimContent[] contentSet = RlanHelper.SaveAllPaneAnim(paramKind, scene, AnimCurvesParam.__NoSelected__);
                    if (contentSet.Length != 0)
                    {
                        LAN rlan = Convert_AppDataToRLAN_(scene);

                        rlan.animType = RlanHelper.ParamaterKindToAnimationType(paramKind);
                        rlan.animContent = contentSet;

                        // Infinity領域のキーフレーム焼付け処理
                        if (FixInfinityCurve)
                        {
                            Rlan.BakedInfinityKeyBuilder.BuildBakedKey(rlan.animContent, rlan.convertStartFrame, rlan.convertEndFrame);
                        }

                        rlanSet.Add(rlan);
                    }
                }
            }

            // ------------ FileFmt.Document を書き出します。
            FileFmt.Document result = new Document();
            result.head = Convert_head_(scene.DocumentHeader as AppData.DocumentHeader);
            result.body = new DocumentBody();


            // result.versionについては、
            // 明示的な設定を行わず、スキーマに設定されている、
            // 既定の値を利用します。

            // DocumentBody について初期化します。
            result.body.lan = rlanSet.ToArray();
            result.body.animTag = ConvertAppFrameSectionDataToTagData_(scene.IAnimFrameSectionSet.IAnimFrameSectionSet, null);
            result.body.animShare = ConvertAppAnimShareInfo_(scene.IAnimShareInfomationManager as AnimShareInfomationManager);
            result.body.separateMode = scene.IsAnimEditSeparateMode();

            return result;
        }

        /// <summary>
        /// アニメーションを書き出します。
        /// アニメーションの分割管理モード時に呼び出されます。
        /// </summary>
        private FileFmt.Document Convert_SODocument_Multi_(
            ISubScene scene,
            ParamaterKind kind)
        {
            // ------------ RLAN を書き出します。

            // 区間タグごとに抽出します
            Dictionary<string, List<LAN>> rlanList = new Dictionary<string, List<LAN>>();
            Dictionary<string, LAN> texturePatternLanMap = new Dictionary<string, LAN>();
            foreach (IAnimFrameSection section in scene.IAnimFrameSectionSet.IAnimFrameSectionSet)
            {
                // 全ての種類のアニメーションから...
                List<LAN> rlanSet = new List<LAN>();
                foreach (ParamaterKind paramKind in ParamaterKindHelper.AnimationKindSet)
                {
                    _animationFileType = paramKind;

                    // 指定された種類のアニメーションのみを出力します。
                    if ((paramKind & kind) != 0)
                    {
                        // AnimContent を抽出します。
                        // 抽出されれば、RLANを生成して登録します。
                        FileFmt.AnimContent[] contentSet = RlanHelper.SaveAllPaneAnim(paramKind, scene, section.Name);
                        if (contentSet.Length != 0)
                        {
                            bool bAdd = true;
                            LAN rlan = Convert_AppDataToRLAN_(scene);

                            rlan.animType = RlanHelper.ParamaterKindToAnimationType(paramKind);
                            rlan.animContent = contentSet;

                            // Infinity領域のキーフレーム焼付け処理
                            if (FixInfinityCurve)
                            {
                                Rlan.BakedInfinityKeyBuilder.BuildBakedKey(rlan.animContent, rlan.convertStartFrame, rlan.convertEndFrame);
                            }

                            // 文字毎アニメーションSRTを処理する...
                            if (rlan.animType == AnimationType.PerCharacterTransformCurve)
                            {
                                if (scene.IAnimFrameSectionSet.IAnimFrameSectionSet.First() != section)
                                {
                                    bAdd = false;
                                }
                            }

                            if (bAdd)
                            {
                                rlanSet.Add(rlan);
                            }
                        }
                    }
                }

                rlanList.Add(section.Name, rlanSet);
            }

            // テクスチャアニメーション用のLANを生成
            LAN texturePatternLan = Convert_AppDataToRLAN_(scene);
            texturePatternLan.animType = RlanHelper.ParamaterKindToAnimationType(ParamaterKind.Animation_TexturePattern);
            texturePatternLan.animContent = RlanHelper.SaveAllPaneAnim(ParamaterKind.Animation_TexturePattern, scene, AnimCurvesParam.__NoSelected__);
            texturePatternLan.startFrame = 0;
            texturePatternLan.endFrame = 0;
            texturePatternLan.convertStartFrame = 0;
            texturePatternLan.convertEndFrame = 0;

            // ------------ FileFmt.Document を書き出します。
            FileFmt.Document result = new Document();
            result.head = Convert_head_(scene.DocumentHeader as AppData.DocumentHeader);
            result.body = new DocumentBody();


            // result.versionについては、
            // 明示的な設定を行わず、スキーマに設定されている、
            // 既定の値を利用します。

            // DocumentBody について初期化します。
            result.body.lan = texturePatternLan.animContent.Any() ? new [] {texturePatternLan} : new LAN[0];
            result.body.animTag = ConvertAppFrameSectionDataToTagData_(scene.IAnimFrameSectionSet.IAnimFrameSectionSet, rlanList);
            result.body.animShare = ConvertAppAnimShareInfo_(scene.IAnimShareInfomationManager as AnimShareInfomationManager);
            result.body.separateMode = SubSceneHelper.IsAnimEditSeparateMode(scene);

            return result;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RlanConverter( LEMsgReporter msgReporter )
        {
            _msgReporter = msgReporter;
        }


        /// <summary>
        /// TODO:
        /// 名前空間が異なるだけで、全く同じデータ構造に対して
        /// 全く同じコードを複製している。
        ///
        /// 名前空間が異なるので、文法上は完全に別の型であるので、
        /// 同一のコードで処理が行えない。
        ///
        /// 何とか、共通化できないものか...
        ///
        /// </summary>
        #region FileFmt.Head ヘッダ情報関連


        /// <summary>
        /// FileFmt.HeadCreate へと変換します。
        /// </summary>
        FileFmt.HeadCreate Convert_headCreate_(LECore.Structures.SerializableObject.Lyt.HeadCreate srcData )
        {
            FileFmt.HeadCreate result = new FileFmt.HeadCreate();

            result.date = srcData != null ? srcData.date : DateTime.Now;

            result.host = srcData != null ? srcData.host : Environment.MachineName;
            result.user = srcData != null ? srcData.user : Environment.UserName;

            // TODO:
            result.source = "";

            return result;
        }

        /// <summary>
        /// FileFmt.HeadGenerator へと変換します。
        /// </summary>
        FileFmt.HeadGenerator Convert_headGenerator_(LECore.Structures.SerializableObject.Lyt.HeadGenerator srcData )
        {
            FileFmt.HeadGenerator rusult = new FileFmt.HeadGenerator();

            rusult.name = "";
            rusult.version = _VersionString;

            return rusult;
        }

        /// <summary>
        /// FileFmt.Head へと変換します。
        /// </summary>
        FileFmt.Head Convert_head_(AppData.DocumentHeader srcData )
        {
            FileFmt.Head result = new FileFmt.Head();

            result.comment = "";
            result.title = "";
            result.create = Convert_headCreate_(srcData.Create);
            result.generator = Convert_headGenerator_(srcData.Generator);

            return result;
        }

        #endregion FileFmt.Head ヘッダ情報関連
    }
}
