﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Diagnostics;
using System.Xml.Serialization;
using System.Linq;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.SerializableObject.Lan;


    /// <summary>
    /// rlyt(レイアウト情報)を出力するクラスです。
    /// </summary>
    internal abstract class BaseSingleExporter : IExporter
    {

        /// <summary>
        /// ログ出力などを受け取るクラスを設定します。
        /// </summary>
        LEMsgReporter        _msgReporter = null;

        /// <summary>
        /// ログ出力などを受け取るクラスを設定します。
        /// </summary>
        public LEMsgReporter    MsgReporter
        {
            set{ _msgReporter = value;}
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public BaseSingleExporter()
        {
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public virtual void Initialize( object initParam )
        {
            // 標準では、何もしません。
        }

        #region BaseSingleExporter abstract メンバ
        /// <summary>
        /// シリアル化可能オブジェクトの型
        /// </summary>
        /// <returns></returns>
        public abstract Type   GetSelializableObjType();

        /// <summary>
        /// シリアル化可能オブジェクトに変換します。
        /// </summary>
        public abstract object ConvertToSerializable_( SubScene targetScene, string outPath, LEMsgReporter msgReporter );

        /// <summary>
        /// 出力データをバリデートします
        /// </summary>
        protected virtual bool Validate_(SubScene targetScene, LEMsgReporter msgReporter)
        {
            return true;
        }

        /// <summary>
        /// ファイル保存をしてもよいかユーザに確認します。
        /// </summary>
        protected virtual bool ComfirmSaving_(SubScene targetScene, object serializableDocument, string outPath, LEMsgReporter msgReporter)
        {
            return true;
        }

        #endregion BaseSingleExporter abstract メンバ

        #region IExporter メンバ
        /// <summary>
        /// 出力ファイルの拡張子を取得します(短縮)。
        /// </summary>
        public abstract string[] GetFileExtensionShort();

        /// <summary>
        /// 出力ファイルの拡張子を取得します(長い)。
        /// </summary>
        public abstract string[] GetFileExtensionLong();

        /// <summary>
        /// FileIO イベントを抑止する
        /// </summary>
        public virtual bool StopFileIOEvent { get { return false; } }

        /// <summary>
        /// 出力するものが何もない場合は false を返すと、一切の処理がスキップされます。
        /// </summary>
        public virtual bool CheckNothingToExport(ISubScene scene)
        {
            return false;
        }

        /// <summary>
        /// 保存オプションです。
        /// </summary>
        public ExportOption ExportOption { get; set; } = new ExportOption();

        /// <summary>
        /// エラータイトル
        /// </summary>
        protected virtual string GetErrorMesageTitle_()
        {
            return "ERROR";
        }

        /// <summary>
        /// エラー報告
        /// </summary>
        protected void ReportError_(string message)
        {
            _msgReporter.ReportError(GetErrorMesageTitle_(), message);
        }

        /// <summary>
        /// 出力ファイルパスを調整する。
        /// </summary>
        public virtual string TrimFilePath(string originalOutPath)
        {
            return Path.ChangeExtension(originalOutPath, GetFileExtensionShort().First());
        }

        /// <summary>
        /// ファイル出力をします。
        /// </summary>
        public bool DoSave( string outPath, SubScene targetScene )
        {
            try
            {
                _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_DLG_TITLE_BEFORESAVE_VALIDATION"));
                Validate_(targetScene, _msgReporter);
            }
            finally
            {
                _msgReporter.EndPacking();
            }

            // コンバートクラスを使用して、シーンからアニメーション情報を格納する
            // シリアル化可能クラスへと変換します。
            // 空の場合は何も出力しません。
            object serializableDocument = ConvertToSerializable_(targetScene, outPath, _msgReporter);
            if( serializableDocument == null )
            {
                return true;
            }

            // 特殊なデータに関して、本当に保存してもよいかユーザに確認をします。
            if(!ComfirmSaving_(targetScene, serializableDocument, outPath, _msgReporter))
            {
                return false;
            }

            // XML シリアライズをおこない、ファイル書き出しをします。
            StreamWriter		    writer		    = null;
            XmlSerializer		 xmlSerializer	    = null;
            try
            {
                // ApplicationSettings 型の
                // XmlSerializer を作成します。
                string[]      extStrSet  = GetFileExtensionShort();
                Debug.Assert( extStrSet.Length == 1 );

                xmlSerializer  = XmlSerializerCache.GetXmlSerializer( GetSelializableObjType() );
                writer          = new StreamWriter( outPath , false );
                // ApplicationSettings クラスのこのインスタンスを
                // 構成ファイルにシリアル化します。
                xmlSerializer.Serialize( writer, serializableDocument );

                // 末尾に改行を追加する
                writer.WriteLine();
            }
            catch(Exception ex)
            {
                if( ex is UnauthorizedAccessException )
                {
                    // 書き込み例外メッセージを表示します。
                    if( _msgReporter != null )
                    {
                        ReportError_(LECoreStringResMgr.Get("LECORE_SYS_ERROR_WRITE_EXCEPTION", outPath));
                    }
                }
                //
                // かならず、例外を投げなおすこと！
                // 現状は、エラーを外部に伝達する方法が、それ以外にありません。
                //
                throw ex;
            }
            finally
            {
                // FileStream が開いている場合は閉じます。
                if(writer != null)
                {
                    writer.Close();
                }
            }

            // ファイルアクセス日時を更新します。
            File.SetLastWriteTime( outPath, DateTime.Now );

            return true;
        }

        #endregion IExporter メンバ
    }
}


