﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Core.Command;
    using LECore.Structures.SerializableObject.Lyt;
    using Structures.Core;
    using System;
    using WindowCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.ILEWindow>;

    /// <summary>
    /// TextBoxManipulator の概要の説明です。
    /// </summary>
    public class WindowManipulator : BaseManipulator, ITexCoordmanipulator
    {
        // 操作対象
        LEWindow _target = null;
        WindowCommandFactory _WindowCommandFactory;
        AnmAttributeManipulator _anmAttributeManipulator = new AnmAttributeManipulator();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public WindowManipulator()
        {
            _WindowCommandFactory = new WindowCommandFactory
                ( delegate( ILEWindow src ){ return new WindowMemento( src ); } );
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="targetTextBox"></param>
        public void BindTarget( ILEWindow target )
        {
            _target = target as LEWindow;
            Debug.Assert( _target != null );
        }

        /// <summary>
        /// 操作対象を取得します。
        /// </summary>
        public ILEWindow  ILEWindow
        {
            get{ return _target;}
        }

        /// <summary>
        /// ウインドウ種類を設定します。
        /// </summary>
        public WindowKind WindowKind
        {
            set
            {
                if (_target.WindowKind != value)
                {
                    _WindowCommandFactory.Modify(_target, delegate()
                    {
                        _target.WindowKind = value;

                        // ユーザ操作からの変更時は、明示的にサイズ調整も実行します。
                        _target.OnWindowKindChangeUpdate();
                        _target.AdjustPaneSize();
                    });
                }
            }
        }

        /// <summary>
        /// 頂点カラーをペイン全体に適用するかを指定します。
        /// </summary>
        public bool UseVtxColorForAllWindow
        {
            set
            {
                if (_target.UseVtxColorForAllWindow != value)
                {
                    _WindowCommandFactory.Modify(_target, delegate()
                    {
                        _target.UseVtxColorForAllWindow = value;
                    });
                }
            }
        }

        /// <summary>
        /// 使用するテクスチャ数を指定します。
        /// </summary>
        public int NumTexture
        {
            set
            {
                if( _target.NumTexture != value )
                {
                    _WindowCommandFactory.Modify( _target, delegate()
                    {
                        _target.NumTexture = value;
                    });
                }
            }
        }

        /// <summary>
        /// テクスチャ座標数を指定します。
        /// </summary>
        public int NumContentsTexCoord
        {
            set
            {
                if( _target.NumContentsTexCoord != value )
                {
                    _WindowCommandFactory.Modify( _target, delegate()
                    {
                        _target.NumContentsTexCoord = value;
                    } );
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public ITexCoordHolder TexCoordOwner { get { return _target; } }

        /// <summary>
        ///
        /// </summary>
        public void SetTexCoord(TexCoord4 newTexCoord, int coord4Idx)
        {
            this.SetContentsTexCoord4(coord4Idx, newTexCoord);
        }

        /// <summary>
        ///
        /// </summary>
        public int NumTexCoord
        {
            set { this.NumContentsTexCoord = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public RGBAColor VtxCol_LT
        {
            set
            {
                if (_target.VtxCol_LT != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.VtxCol_LTAnmAttr, value);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public RGBAColor VtxCol_RT
        {
            set
            {
                if (_target.VtxCol_RT != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.VtxCol_RTAnmAttr, value);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public RGBAColor VtxCol_LB
        {
            set
            {
                if (_target.VtxCol_LB != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.VtxCol_LBAnmAttr, value);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public RGBAColor VtxCol_RB
        {
            set
            {
                if (_target.VtxCol_RB != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.VtxCol_RBAnmAttr, value);
                }
            }
        }

        /// <summary>
        /// 画像の反転・回転設定を設定します。
        /// </summary>
        public void SetMaterialFlipState(
                                         LEWindowPartsID id,
                                         LEWindowFrameFlipRotate state
                                         )
        {
            if( _target.GetMaterialFlipRotateState( id ) != state )
            {
                _WindowCommandFactory.Modify( _target, delegate()
                {
                    _target.SetMaterialFlipState( id, state );
                });
            }
        }

        /// <summary>
        /// コンテント拡大量を設定します。
        /// </summary>
        public void SetContentsInflation( float top, float bottom, float left, float right )
        {
            ILEWindowContentsInflation newValue = new ILEWindowContentsInflation(  top, bottom, left, right );
            if( !_target.ContentsInflation.IsSame( newValue ) )
            {
                _WindowCommandFactory.Modify( _target, delegate()
                {
                    _target.ContentsInflation = newValue;
                });
            }
        }

        /// <summary>
        /// フレームサイズを設定します。
        /// </summary>
        public void SetFrameSize(float top, float bottom, float left, float right)
        {
            LEWindowFrameSize newValue = new LEWindowFrameSize(top, bottom, left, right, true);
            if (!_target.FrameSize.IsSame(newValue))
            {
                Scene.CurrentSubScene.BeginMassiveModify();

                ManipulatorUtility.ModifyPropWithCreatingCmd(
                    () => _target.IsFrameSizeSpecified,
                    x => _target.IsFrameSizeSpecified = x,
                    true
                    );
                ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.FrameSizeAnmAttr.TopAnmAttr, top);
                ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.FrameSizeAnmAttr.BottomAnmAttr, bottom);
                ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.FrameSizeAnmAttr.LeftAnmAttr, left);
                ManipulatorUtility.ModifyAttrWithCreatingCmd(_target.FrameSizeAnmAttr.RightAnmAttr, right);

                Scene.CurrentSubScene.EndMassiveModify();

            }
        }

        /// <summary>
        /// フレームサイズをリセットします。
        /// </summary>
        public void ResetCustomFrameSize()
        {
            ISubScene parentSubScene = (_target as IPaneExParamater).OwnerPane.OwnerSubScene;
            parentSubScene.BeginMassiveModify();

            // Undo 時に基本値を復元する
            // Redo 時は不要
            var createSetBaseValueCommand = (Action<AnmAttribute>)(attr =>
            {
                object val;
                attr.GetBaseValue(out val);
                bool undo = true;
                ((SubScene)parentSubScene).CommandFactory.SetActionCommand(
                    (Action)(() =>
                    {
                        if (undo)
                        {
                            attr.SetBaseValue(val);
                        }
                        undo = !undo;
                    }));
            });

            createSetBaseValueCommand(_target.FrameSizeAnmAttr.TopAnmAttr);
            createSetBaseValueCommand(_target.FrameSizeAnmAttr.BottomAnmAttr);
            createSetBaseValueCommand(_target.FrameSizeAnmAttr.LeftAnmAttr);
            createSetBaseValueCommand(_target.FrameSizeAnmAttr.RightAnmAttr);

            _WindowCommandFactory.Modify(_target, delegate ()
            {
                _anmAttributeManipulator.BindTarget(_target.FrameSizeAnmAttr);
                _anmAttributeManipulator.RemoveAllKeyRecursively();

                // isSpecified が false なので代入後の値は自動調整されたものになる
                _target.FrameSize = new LEWindowFrameSize(
                    _target.FrameSize.Top, _target.FrameSize.Bottom, _target.FrameSize.Left, _target.FrameSize.Right, false);
                _target.OnWindowKindChangeUpdate();
            });

            parentSubScene.EndMassiveModify();
        }

        /// <summary>
        /// コンテンツ領域のテクスチャ座標を設定します。
        /// </summary>
        /// <param name="texCoordIdx"></param>
        /// <param name="texcoord4"></param>
        public void SetContentsTexCoord4( int texCoordIdx, TexCoord4 texCoord4 )
        {
            if( !TexCoord4.CheckSame( _target.GetContentsTexCoord4( texCoordIdx ), texCoord4 ) )
            {
                _WindowCommandFactory.Modify( _target, delegate()
                {
                    _target.SetContentsTexCood4( texCoordIdx, texCoord4 );
                });
            }
        }

        /// <summary>
        /// 詳細なマテリアルを使用するかどうかのフラグを設定します。
        /// </summary>
        public void SetUseDetailedMaterialFlag( LEWindowPartsID id, bool flag )
        {
            if( _target.GetUseDetailedMaterialFlag( id ) != flag )
            {
                _WindowCommandFactory.Modify( _target, delegate()
                {
                    _target.SetUseDetailedMaterialFlag( id, flag );
                } );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool UseLTMaterial
        {
            set
            {
                if( _target.UseLTMaterial != value )
                {
                    _WindowCommandFactory.Modify( _target, delegate()
                    {
                        _target.UseLTMaterial = value;
                    } );
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool NotDrawContent
        {
            set
            {
                if (_target.NotDrawContent != value)
                {
                    _WindowCommandFactory.Modify(_target, delegate()
                    {
                        _target.NotDrawContent = value;
                    });
                }
            }
        }
    }
}
