﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.Linq;
using System.Windows.Forms;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.Core.Command;
    using System.IO;
    using LECore.Structures.LECoreInterface;
    using System.Threading;
    using Util;

    /// <summary>
    /// SubSceneManipulator の概要の説明です。
    /// </summary>
    public partial class SubSceneManipulator :
        BaseManipulator
    {
        public const string DefaultNullPaneName             = "N_null_00";
        public const string DefaultPicturePaneName          = "P_pict_00";
        public const string DefaultTextBoxPaneName          = "T_text_00";
        public const string DefaultWindowPaneName           = "W_window_00";
        public const string DefaultCapturePaneName          = "C_capture_00";
        public const string DefaultBoundingPaneName         = "B_bounding_00";
        public const string DefaultAlignmentPaneName        = "A_alignment_00";
        public const string DefaultScissorPaneName          = "S_scissor_00";
        public const string DefaultAnimationSharePaneName   = "A_anmShare_00";
        public const string DefaultGroupName                = "G_group_00";
        public const string DefaultPartsPaneName            = "L_parts_00";
        public const string DefaultPicturePaneSuffixName    = "_00";

        public static readonly Size DefaultInitialPaneSize = new Size(30, 40);

        public const bool ValueRouwndDownEnable = true;
        public const bool ValueRouwndDownDisable = false;

        private const int _sectionMargin = 10;

        SubScene _target = null;
        PaneManipulator _paneMnp = new PaneManipulator();
        TextureMgrManipulator _textureMgrMnp = new TextureMgrManipulator();
        AnimFrameSectionSetManipulator _animFrameSectionSetMnp = null;
        AnmKeyFrameManipulator _animKeyFrameManipulator = null;

        #region -------------------- private --------------------

        bool IsPaneSizeDefaultSize_(SizeF paneSize)
        {
            return paneSize.Width == DefaultInitialPaneSize.Width && paneSize.Height == DefaultInitialPaneSize.Height;
        }

        /// <summary>
        /// ペインを追加し、正しいペイン名に補正を行う。
        /// </summary>
        void AddOnePane_(Pane newPane)
        {
            _target.AddPane(newPane);
            newPane.PaneName = _target.GetUniqueNewPaneName(newPane, newPane.PaneName, newPane.PaneName);
        }

        /// <summary>
        /// シーンにNullペインを追加します。
        /// </summary>
        Pane AddNullPane_(string newPaneName)
        {
            Pane newPane = new Pane();
            var initialPaneSize = LayoutEditorCore.DefaultInitialPaneSize;
            newPane.Size = new FVec3(initialPaneSize.Width, initialPaneSize.Height, newPane.Size.Z);

            newPane.PaneName = newPaneName;

            AddOnePane_(newPane);

            _CommandFactory.MakeSceneAddPaneCmd(_target, newPane);

            return newPane;
        }

        #endregion -------------------- private --------------------

        /// <summary>
        /// 操作対象のサブシーンを取得します。
        /// </summary>
        public ISubScene ISubScene
        {
            get{ return _target;}
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="target"></param>
        public SubSceneManipulator()
        {
            _animFrameSectionSetMnp = new AnimFrameSectionSetManipulator(this);
            _animKeyFrameManipulator = new AnmKeyFrameManipulator();
        }

        /// <summary>
        /// 操作対象を関連付けます
        /// </summary>
        /// <param name="target"></param>
        public void BindTarget( ISubScene target )
        {
            _target = target as SubScene;
        }

        /// <summary>
        /// 未使用フォントを削除
        /// </summary>
        public string[] DeleteUnusedFont()
        {
            // 未使用リストを取得
            ILEFont[] unusedFontSet = SubSceneHelper.GetUnuseFontSet( _target );

            // プロジェクト設定に記述されたフォントは削除しない。
            unusedFontSet = unusedFontSet.Where((font) => font.IsProjectFont() == false).ToArray();

            // フォントパスに変換
            string[] deletedFontPath = unusedFontSet.Select((font)=> font.FontPath).ToArray();

            // フォントを削除
            FontManager fontManager = _target.ILEFontManager as FontManager;
            foreach( LEFont font in unusedFontSet )
            {
                fontManager.RemoveFont( font );
            }

            return deletedFontPath;
        }

        #region Undo/Redo
        /// <summary>
        /// アンドゥ
        /// </summary>
        public void Undo()
        {
            _target.BeginMassiveModify();
            _target.Undo();
            _target.EndMassiveModify();
        }

        /// <summary>
        /// リドゥ
        /// </summary>
        public void Redo()
        {
            _target.BeginMassiveModify();
            _target.Redo();
            _target.EndMassiveModify();
        }

        /// <summary>
        /// Undoコマンド状態をリセットします。
        /// </summary>
        public void ResetUndoState()
        {
            _target.ResetUndoState();
        }
        #endregion Undo/Redo

        #region ペインの新規登録

        /// <summary>
        /// ペイン追加処理を行います。
        /// </summary>
        /// <param name="newPaneName"> </param>
        /// <param name="nullPaneCreation">元となる null ペインを作る処理。基本的に、AddNullPane_ が使われます。</param>
        /// <param name="paneExtentionSetup">IPaneExParamater の部分を初期化する処理。ペイン種類ごとに違う処理が実行されます。</param>
        /// <returns>追加するペインの実体です。</returns>
        private IPane AddPaneImpl_(string newPaneName, Func<string, Pane> nullPaneCreation, Action<Pane> paneExtentionSetup)
        {
            Debug.Assert(_target != null);

            try
            {
                _target.BeginMassiveModify();

                Debug.Assert(nullPaneCreation != null);
                Pane pane = nullPaneCreation.Invoke(newPaneName);

                paneExtentionSetup?.Invoke(pane);

                // paneExtentionSetup で値が書き換わっているかもしれないので基本値を更新する
                StoreCurrentValueToBaseValue_(pane);

                // 分割モードでは、タグの追加が必要
                if (_target.IsAnimEditSeparate)
                {
                    AddAllAnimTagsToAttribute_(pane);
                }

                // キャプチャテクスチャのもとになる可能性のあるペインは、作成時に既存のリンク切れのキャプチャテクスチャのソースになれないかチェックします。
                PaneHelper.ReconnectInvalidCaptureTexture(pane);

                // 追加したペインを選択状態にする
                ResetSelectedSet();
                SelectPanesByPaneRef(pane);

                return pane;
            }
            finally
            {
                _target.EndMassiveModify();
            }
        }

        /// <summary>
        /// 新しいヌルペインをシーンに追加します。
        /// </summary>
        /// <param name="newPaneName"></param>
        /// <returns></returns>
        public IPane AddNullPane( string newPaneName )
        {
            return AddPaneImpl_(newPaneName, AddNullPane_, null);
        }

        /// <summary>
        /// 新しいTextBoxペインをシーンに追加します。
        /// </summary>
        public IPane AddTextBoxPane
            (
            string    newPaneName,
            ILEFont   font,
            string    contentsText
            )
        {
            return AddPaneImpl_(newPaneName, AddNullPane_, (textBoxPane) =>
            {
                // TextBox へと変更します。
                // 文字変更イベント（TextboxContentsChanged）を確実に発生させるため、一旦空文字で生成して、ContentsText を指定しています。
                TextBox textBox = new TextBox(textBoxPane, string.Empty, font.FontName);
                textBox.FontSize = new FVec2(font.Width, font.Height);
                textBox.FontSizeOriginal = textBox.FontSize;
                textBox.ContentsText = contentsText;

                // サイズの調整のためにフォントの更新を待つ
                for (int i = 0; i < 1000; i++)
                {
                    if (!font.IsWaitingUpdate)
                    {
                        DbgConsole.WriteLine($"Waited {i * 10} ms");
                        break;
                    }

                    Thread.Sleep(10);
                }

                // ペインサイズを調整します。
                textBox.AdjustPaneSize();
            });
        }

        /// <summary>
        /// 新しいPictureペインをシーンに追加します。
        /// </summary>
        public IPane AddPicturePane
            (
            string    newPaneName,
            string    textureName,
            bool      useTextureNameAsPicturePaneName
            )
        {
            if(useTextureNameAsPicturePaneName)
            {
                newPaneName = textureName;
            }

            return AddPaneImpl_(newPaneName, AddNullPane_, (pane) =>
            {
                // 画像が発見できれば
                ITextureImage texImg = _target.ITextureMgr.FindITextureImageByName(textureName);
                if (texImg != null)
                {
                    // pictureペイン へと変更します。
                    Picture picture = new Picture(pane);

                    // マテリアルにテクスチャを設定
                    var resourceType = MaterialTexMapHelper.GetAttrTextureResourceType(texImg.SourceType);
                    picture.Material.RegisterMatTexture(texImg.Name, 0, resourceType);

                    // ペインサイズをテクスチャと同サイズに設定します。
                    pane.Width = texImg.Size.X;
                    pane.Height = texImg.Size.Y;
                }
            });
        }

        /// <summary>
        /// テクスチャを使わずにピクチャペインをシーンに追加します。
        /// </summary>
        public IPane AddPicturePaneWithoutTexture(string newPaneName)
        {
            return AddPaneImpl_(newPaneName, AddNullPane_, (pane) =>
            {
                if(IsPaneSizeDefaultSize_(pane.Size.FVec2.AsSizeF))
                {
                    const float defaultWidth = 128.0f;
                    const float defaultHeight = 128.0f;

                    pane.Width = defaultWidth;
                    pane.Height = defaultHeight;
                }

                // ピクチャペインへと変更します。
                Picture window = new Picture(pane);
            });
        }

        /// <summary>
        /// ウインドウペインをシーンに追加します。
        /// </summary>
        /// <param name="newPaneName"></param>
        public IPane AddWindowPane
            (
             string newPaneName
            )
        {
            return AddPaneImpl_(newPaneName, AddNullPane_, (pane) =>
            {
                if (IsPaneSizeDefaultSize_(pane.Size.FVec2.AsSizeF))
                {
                    pane.Width *= 2.0F;
                    pane.Height *= 2.0F;
                }

                if (pane != null)
                {
                    // windowペイン へと変更します。
                    LEWindow window = new LEWindow(pane);
                }
            });
        }

        /// <summary>
        /// キャプチャペインをシーンに追加します。
        /// </summary>
        /// <param name="newPaneName"></param>
        public IPane AddCapturePane
            (
            string newPaneName
            )
        {
            return AddPaneImpl_(newPaneName, AddNullPane_, (pane) =>
            {
                // これ以前の処理がどのようなものであっても
                // CreateICaptureTexture 実行前に基本値が設定されているようにする。
                StoreCurrentValueToBaseValue_(pane);

                TextureMgrManipulator mnp = new TextureMgrManipulator();

                mnp.BindTarget(_target.ITextureMgr);
                mnp.RegisterCaptureTexture(pane, pane.PaneName, false);

                // Capture へと変更します。
                Capture capture = new Capture(pane);
            });
        }

        /// <summary>
        /// 境界ペインをシーンに追加します。
        /// </summary>
        /// <param name="newPaneName"></param>
        public IPane AddBoudingPane
            (
            string newPaneName
            )
        {
            return AddPaneImpl_(newPaneName, AddNullPane_, (pane) =>
            {
                Bounding bounding = new Bounding(pane);
            });
        }

        /// <summary>
        /// 整列ペインをシーンに追加します。
        /// </summary>
        public IPane AddAlignmentPane(string newPaneName)
        {
            return AddPaneImpl_(newPaneName, AddNullPane_, (pane) =>
            {
                Alignment bounding = new Alignment(pane);
            });
        }

        /// <summary>
        /// シザーペインをシーンに追加します。
        /// </summary>
        public IPane AddScissorPane(string newPaneName)
        {
            return AddPaneImpl_(newPaneName, AddNullPane_, (pane) =>
            {
                Scissor bounding = new Scissor(pane);
            });
        }

        /// <summary>
        /// 部品ペインをシーンに追加します。
        /// </summary>
        public IPane AddPartsPane(
             string newPaneName,
            string partsLayoutName,
            ISubScene partsSubScene)
        {
            return AddPartsPane_(newPaneName, partsLayoutName, partsSubScene, AddNullPane_);
        }

        /// <summary>
        /// 部品ペインをシーンに追加します。Undoを処理しません。
        /// </summary>
        public IPane AddPartsPaneWithoutUndo(
             string newPaneName,
            string partsLayoutName,
            ISubScene partsSubScene)
        {
            Func<string, Pane> nullPaneCreationFunc = delegate(string name)
            {
                Pane newPane = new Pane();
                newPane.PaneName = newPaneName;

                AddOnePane_(newPane);

                return newPane;
            };

            return AddPartsPane_(newPaneName, partsLayoutName, partsSubScene, nullPaneCreationFunc);
        }

        /// <summary>
        /// 部品ペインをシーンに追加します。
        /// </summary>
        private IPane AddPartsPane_(
             string newPaneName,
            string partsLayoutName,
            ISubScene partsSubScene,
            Func<string,Pane> nullPaneCreationFunc)
        {
            return AddPaneImpl_(newPaneName, nullPaneCreationFunc, (pane) =>
            {
                // 部品ペイン へと変更します。
                PartsLayout partsLayout = new PartsLayout(pane, partsSubScene, partsLayoutName);

                List<ILEFont> fonts = new List<ILEFont>();
                List<ITextureImage> textures = new List<ITextureImage>();
                foreach (var propertySource in partsLayout.RawPartsPropaerties)
                {
                    if (propertySource.Paramater == null || propertySource.Paramater.OwnerPane == null)
                    {
                        continue;
                    }

                    IPane overridePane = propertySource.Paramater.OwnerPane;

                    var referenceFonts = partsSubScene.ILEFontManager.ILEFontSet.Where(
                        (font) => overridePane.UsesFont(font) && !fonts.Contains(font));
                    fonts.AddRange(referenceFonts);

                    var referencedTextures = partsSubScene.ITextureMgr.ITextureImageSet.Where(
                        (img) => overridePane.UsesTextureImage(img.Name) && !textures.Contains(img));
                    textures.AddRange(referencedTextures);
                }

                //  部品ペインが依存するフォントやテクスチャを登録する。
                {
                    foreach (var font in fonts)
                    {
                        if ((_target.ILEFontManager as FontManager).FindFontByName(font.FontName) == null)
                        {
                            (_target.ILEFontManager as FontManager).CreateILEFont(font.FontName, font.FontPath, font.FontSettings);
                        }
                    }

                    (_target.ITextureMgr as TextureMgr).BeginUpdate();
                    foreach (var texture in textures)
                    {
                        _target.ITextureMgr.SyncTextureImageNoEvent(texture);
                    }
                    (_target.ITextureMgr as TextureMgr).EndUpdate();
                }

                // 登録時の初期化を明示的に実行しておきます。
                partsLayout.OnJoinSceneInitialize();
            });

        }

        /// <summary>
        /// 現在の値を BaseValue として取り込みます。
        /// </summary>
        private void StoreCurrentValueToBaseValue_(IPane pane)
        {
            _target.BeginMassiveModify();

            pane.StoreCurrentValueToBaseValue();

            _target.EndMassiveModify();
        }

        /// <summary>
        /// IAnmAttribute に対して、適切な タグを追加 します。
        /// </summary>
        private void AddAllAnimTagsToAttribute_(IPane pane)
        {
            if (pane == null)
            {
                return;
            }

            if (!_target.IsAnimEditSeparate)
            {
                Debug.Assert(false);
                return;
            }

            // 分割モードの場合のみタグとの関連付けを行います
            foreach (IAnimFrameSection section in _target.IAnimFrameSectionSet.IAnimFrameSectionSet)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    curveAttr.AddAnmTag(section.Name);
                }
            }
        }

        #endregion ペインの新規登録

        #region 選択セット変更に関するメソッド群
        /// <summary>
        /// 選択セットの変更開始
        ///
        /// 複数回の選択セットの変更が起こることが、わかっていて、
        /// 不必要な更新イベントの発生を抑制したいときに使用します。
        /// EndSelectSetChange()と対になって使用されます。
        /// </summary>
        public void BeginSelectSetChange()
        {
            _target.SelectedPaneSet.BeginMassiveModify();
        }

        /// <summary>
        /// 選択セットの変更終了
        /// 更新イベントの発生を抑制したいときに使用します。
        /// </summary>
        public void EndSelectSetChange()
        {
            _target.SelectedPaneSet.EndMassiveModify();
        }

        /// <summary>
        /// ペイン参照でペインを選択します。
        /// </summary>
        /// <param name="pane"></param>
        public void SelectPanesByPaneName(string paneName)
        {
            var pane = _target.FindPaneByName(paneName);
            if(pane == null)
            {
                return;
            }

            this.SelectPanesByPaneRef(pane);
        }

        /// <summary>
        /// ペイン参照でペインを選択します。
        /// </summary>
        /// <param name="pane"></param>
        public void SelectPanesByPaneRef( IPane pane )
        {
            _target.SelectedPaneSet.BeginMassiveModify();
            _target.SelectedPaneSet.AddPane( pane as Pane );

            // 同一階層に属するペインの重複選択を削除します。
            // 自分の属する親子階層の最上位親ペインを取得します。
            // 親ペインが選択されていないなら選択します。
            // 子ペイン自身は選択されません。
            // _target.SelectedPaneSet.RemoveFamilyPane();
            _target.SelectedPaneSet.EndMassiveModify();
        }

        /// <summary>
        /// ペイン参照でオーダーに従ってペインを選択します。
        /// </summary>
        /// <param name="paneSet">操作対象のペイン配列</param>
        /// <param name="order">オーダー</param>
        public void SelectedChangeByOrder(IPane[] paneSet, bool[] order)
        {
            BeginSelectSetChange();
            ResetSelectedSet();
            for (int i = 0; i < order.Count(); i++)
            {
                if (order.ElementAt(i))
                {
                    SelectPanesByPaneRef(paneSet.ElementAt(i));
                }
            }
            EndSelectSetChange();
        }

        /// <summary>
        /// 指定ペインの選択状態を反転します。
        /// </summary>
        /// <param name="pane"></param>
        public void InvertSelection( IPane pane )
        {
            if( !_target.SelectedPaneSet.Contains( pane as Pane ) )
            {
                SelectPanesByPaneRef( pane );
            }
            else
            {
                _target.SelectedPaneSet.RemovePane( pane as Pane );
            }
        }

        /// <summary>
        /// 選択セットをリセットします
        /// </summary>
        public void ResetSelectedSet()
        {
            _target.SelectedPaneSet.Reset();
        }



        /// <summary>
        /// 選択ペインセットに位置オフセットを加算します。
        /// </summary>
        /// <param name="trans"></param>
        public void AddOffsetToSelectedSet(
            PointF difference,
            bool bTransAffectsChildren,
            bool bRouwndDown )
        {
            // 子供に影響を与える動作モードの場合には、
            // 選択セットのうちの親のみに操作をします。
            IPane[] paneSet = ( bTransAffectsChildren ) ?
                _target.SelectedPaneSet.PaneParentArray :
                _target.SelectedPaneSet.PaneArray;

            if( paneSet.Length != 0 )
            {
                int numDecimalX = LEMath.CalcDecimals( difference.X );
                int numDecimalY = LEMath.CalcDecimals( difference.Y );


                bool oldFlag = _paneMnp.TransAffectsChildren;
                _paneMnp.TransAffectsChildren = bTransAffectsChildren;

                for( int i = 0;i < paneSet.Length; i++ )
                {
                    _paneMnp.BindTarget( paneSet[i] );
                    _paneMnp.TranslateInWorld( new FVec3( difference ) );

                    // 必要であれば、小数点以下を丸めます。
                    if (bRouwndDown)
                    {
                        FVec3 posAligned = new FVec3(
                            LEMath.ToRoundDownDecimals(paneSet[i].X, numDecimalX),
                            LEMath.ToRoundDownDecimals(paneSet[i].Y, numDecimalY),
                            paneSet[i].Z);

                        _paneMnp.Trans = posAligned;
                    }
                }

                _paneMnp.TransAffectsChildren = oldFlag;
            }
        }

        /// <summary>
        /// 選択ペインをコピーします。
        /// </summary>
        public void CopySelectedPanes()
        {
            IPane[] selectedSet = _target.SelectedPaneSet.IPaneArray;

            ClipBoardPane  clipBoardPane =
                new ClipBoardPane( selectedSet,
                PaneSetDupulicator.Option.CopyAnimation |
                PaneSetDupulicator.Option.CopyHierarchy |
                PaneSetDupulicator.Option.ConvertTransToWorld );

            LEClipboard.Instance.Copy( clipBoardPane );
        }

        /// <summary>
        /// 選択ペインを子ペインを含めてコピーします。
        /// </summary>
        public IEnumerable<bool> CopySelectedPanesRecursive()
        {
            List<IPane> panes = new List<IPane>();
            var selectedArray = _target.SelectedPaneSet.IPaneArray;
            foreach (IPane pane in selectedArray)
            {
                var hierarchy = GenericUtil.GetNodeRecursively(pane, (p) => p.Children.OfType<IPane>());
                panes.AddRange(hierarchy);
            }

            var paneSet = panes.Distinct().ToArray();
            ClipBoardPane clipBoardPane =
                new ClipBoardPane(paneSet,
                PaneSetDupulicator.Option.CopyAnimation |
                PaneSetDupulicator.Option.CopyHierarchy |
                PaneSetDupulicator.Option.ConvertTransToWorld);

            LEClipboard.Instance.Copy(clipBoardPane);

            // 選択状態を返します
            List<bool> ret = new List<bool>();
            foreach (IPane pane in paneSet)
            {
                ret.Add(pane.IsSelected);
            }

            return ret;
        }

        /// <summary>
        /// ペイン貼り付け時のキャプチャテクスチャ追加、更新処理。
        /// </summary>
        /// <param name="pane">新たに作成されたペインのインスタンス</param>
        /// <param name="oldName">コピー前のペインの名前</param>
        /// <param name="clipboardPane">クリップボード情報</param>
        void AddPastedCaptureTexture_(Pane pane, string oldName, ClipBoardPane clipboardPane)
        {
            bool addNewTexture = false;

            // マスク用のキャプチャテクスチャを作成します。
            if (pane.IMask.IsMaskEnabled)
            {
                MaskManipulator maskMpn = new MaskManipulator();
                maskMpn.BindTarget(pane.IMask);
                maskMpn.CreateCaptureTextureForMask(pane, pane.OwnerSubScene.ITextureMgr);
            }

            // ドロップシャドウ効果用のキャプチャテクスチャを作成します。
            if (pane.IDropShadow.IsDropShadowEtcEnabled)
            {
                DropShadowManipulator dropShadowMpn = new DropShadowManipulator();
                dropShadowMpn.BindTarget(pane.IDropShadow);
                dropShadowMpn.CreateCaptureTextureForDropShadow(pane, pane.OwnerSubScene.ITextureMgr);
            }

            // キャプチャテクスチャがコピーされたら、コピー先のキャプチャテクスチャを新たに追加する。
            foreach (var res in clipboardPane.CaptureTextureResourceSet)
            {
                if (res.Name == oldName)
                {
                    TextureMgrManipulator mnp = new TextureMgrManipulator();

                    mnp.BindTarget(_target.ITextureMgr);
                    CaptureTexture dst = mnp.RegisterCaptureTexture(pane, pane.PaneName, res.IsHidingFromList) as CaptureTexture;

                    dst.FrameBufferCaptureEnabled = res.FrameBufferCaptureEnabled;
                    dst.CaptureOnlyFirstFrame = res.CaptureOnlyFirstFrame;
                    dst.ClearColor = res.ClearColor;
                    dst.Format = res.Format;
                    dst.TextureScale = res.TextureScale;
                    dst.Usage = res.Usage;
                    addNewTexture = true;
                    break;
                }
            }

            if (addNewTexture)
            {
                // 古い名前で自己参照しているマテリアルの参照名を更新する。
                IRevHWMaterial[] revMats = PaneHelper.GetRevHWMatFromPane(pane);

                foreach (var mat in revMats)
                {
                    int texMapIndex = 0;
                    foreach (var itexMap in mat.IMaterialTexMapSet)
                    {
                        MaterialTexMap texMap = itexMap as MaterialTexMap;

                        if (texMap.TexImgName == oldName)
                        {
                            var mnp = new RevMaterialManipulator();
                            mnp.BindTarget(mat);

                            mnp.RenameMatReferencedTexture(pane.PaneName, texMapIndex, texMap.ResourceType);
                        }
                        ++texMapIndex;
                    }
                }
            }
        }

        /// <summary>
        /// ペースト先に存在しないキャプチャテクスチャを不正なテクスチャとして登録します。
        /// </summary>
        /// <param name="pane"></param>
        void AddInvalidCaptureTexture_(IPane pane)
        {
            if (pane.IMask.IsMaskEnabled)
            {
                IMaterialTexMap texMap = pane.IMask.MaskTexMap;
                if (texMap.ResourceType != Structures.Nsrif.Attributes.AttrTextureResourceType.LocalFile &&
                    TextureMgrHelper.FindCaptureTextureByName(_target.ITextureMgr, texMap.TexImgName) == null)
                {
                    TextureMgrManipulator mnp = new TextureMgrManipulator();

                    mnp.BindTarget(_target.ITextureMgr);
                    mnp.RegisterCaptureTexture(null, texMap.TexImgName, false);
                }
            }

            IRevHWMaterial[] revMats = PaneHelper.GetRevHWMatFromPane(pane);

            foreach (var mat in revMats)
            {
                foreach (MaterialTexMap texMap in mat.IMaterialTexMapSet.EnumerateCaptureRelatedTexMap())
                {
                    if (TextureMgrHelper.FindCaptureTextureByName(_target.ITextureMgr, texMap.TexImgName) == null)
                    {
                        TextureMgrManipulator mnp = new TextureMgrManipulator();

                        mnp.BindTarget(_target.ITextureMgr);
                        mnp.RegisterCaptureTexture(null, texMap.TexImgName, false);
                    }
                }
            }
        }

        /// <summary>
        /// クリップボードの内容(コピーされたペイン)をシーンに貼り付けます。
        /// </summary>
        public void PastePanes( bool bPasteAnimation, Func<IPane,bool> condition)
        {
            _target.BeginMassiveModify();

            // 選択を解除します。
            _target.SelectedPaneSet.Reset();

            object pasteResult = LEClipboard.Instance.Paste( typeof( ClipBoardPane ) );
            if( pasteResult != null )
            {
                ClipBoardPane  clipBoardPane = pasteResult as ClipBoardPane;

                // テクスチャリソースを読み込みます。
                if( clipBoardPane.TextureResourcePathSet.Length > 0 )
                {
                    _textureMgrMnp.BindTarget( ISubScene.ITextureMgr );
                    _textureMgrMnp.RegisterITextureImageSetFromFile( clipBoardPane.TextureResourcePathSet );

                    // フォーマットを設定
                    foreach (var texImg in ISubScene.ITextureMgr.ITextureImageSet.OfType<TextureImage>())
                    {
                        if (texImg.PixelFmtIsFixed)
                        {
                            continue;
                        }

                        TexImagePixelFmt fmt;
                        if (clipBoardPane.TextureFormatMap.TryGetValue(texImg.FilePath, out fmt))
                        {
                            texImg.PixelFmt = fmt;
                        }
                    }
                }

                // フォントリソースを読み込みます。
                foreach( ClipBoardPane.FontResource fontRes in clipBoardPane.FontResourcePathSet )
                {
                    ILEFont font = ISubScene.ILEFontManager.CreateILEFont(fontRes.FontName, fontRes.FontPath, null);
                    Debug.Assert( font != null );
                }

                IPane[] newPanes = clipBoardPane.PaneSet;
                Debug.Assert( newPanes != null );

                foreach (Pane newPane in newPanes)
                {
                    // 条件を満たさないものは登録をキャンセルします。
                    if (condition != null && !condition(newPane))
                    {
                        continue;
                    }

                    // キャプチャテクスチャの情報を引いてくるために古い名前を一時的に保存します。
                    string oldName = newPane.PaneName;

                    // 新しい名前を取得します。
                    AddOnePane_(newPane);

                    SelectPanesByPaneRef(newPane);

                    _CommandFactory.MakeSceneAddPaneCmd(_target, newPane);

                    // アニメーション貼り付けが有効でなければ、
                    // アニメーションを削除します。
                    if (!bPasteAnimation)
                    {
                        IAnmAttribute[] anmAttributeSet = PaneAnimationHelper.GetAllActiveAnmAttribute(newPane);
                        foreach (IAnmAttribute anmAttribute in anmAttributeSet)
                        {
                            AnmAttributeHelper.RemoveKeyFrame(anmAttribute);
                        }
                    }

                    // ペーストされたペインがリンク切れのキャプチャテクスチャのソースになれないかチェックする
                    PaneHelper.ReconnectInvalidCaptureTexture(newPane);
                    // ペーストされたペインから作成されるテクスチャを登録する。
                    AddPastedCaptureTexture_(newPane, oldName, clipBoardPane);
                    // ペーストしたペインで参照している、貼り付け先シーンに存在しないキャプチャテクスチャを不正なテクスチャとして登録する。
                    AddInvalidCaptureTexture_(newPane);

                    // パーツペイン内で使用されているキャプチャテクスチャの参照情報を更新します。
                    // パーツペインを含むサブシーン全体の階層構造が確定した後に処理する必要があります。
                    if (newPane.PaneKind == PaneKind.Parts)
                    {
                        SubSceneHelper.ResolvePartsLayoutCaptureTextureReference(newPane);
                    }

                    // カーブを再評価する
                    PaneAnimationHelper.EvaluateAnimationForce(newPane, GlobalTime.Inst.Time);
                }
            }
           _target.EndMassiveModify();
        }

        /// <summary>
        /// 選択ペインの削除
        /// </summary>
        public void DeleteSelectedPanes()
        {
            _target.BeginMassiveModify();

            IPane[] selectedSet = _target.SelectedPaneSet.SelectedPaneArrayWithChildren;

            // 選択解除
            _target.SelectedPaneSet.Reset();

            // 階層構造を一旦リセットします。
            foreach( Pane pane in selectedSet )
            {
                HierarchyManipulator.ResetHierarchy( pane );
            }

            // グループメンバ設定をリセットします。
            GroupMgrManipulator    groupMgrMnp = new GroupMgrManipulator();
            groupMgrMnp.BindTarget( _target.ILEGroupMgr );
            groupMgrMnp.BeginEditGroup();

            foreach( Pane pane in selectedSet )
            {
                groupMgrMnp.RemoveMemberFromAll( pane );
            }

            // 区間タグの対象グループを整理
            AnimFrameSectionSetHelper.RemoveInvalidTargetGroupAll(_target.IAnimFrameSectionSet, _target.ILEGroupMgr);

            // キャプチャテクスチャが参照しているペインが削除される場合は、テクスチャからペインへの参照を削除する。
            // さらに誰にも使用されなくなっていたらキャプチャテクスチャ自体を削除します。
            SubSceneHelper.TryToDeleteCaptureTexture(_target, selectedSet);

            // すべてのペインを削除します。
            foreach ( Pane pane in selectedSet )
            {
                _target.RemovePain( pane );
                _CommandFactory.MakeSceneRemovePaneCmd( _target, pane );
            }

            groupMgrMnp.EndEditGroup();
            _target.EndMassiveModify();
        }


        #endregion 選択セット変更に関するメソッド群
    }

    /// <summary>
    /// サブシーンの"元に戻す"、"やり直し"を行うヘルパークラス
    /// </summary>
    public class SubSceneUndoRedoHelper
    {
        /// <summary>
        ///
        /// </summary>
        public static Control GetFocusedControl( Control baseControl)
        {
            if( baseControl.Focused != false )
            {
                return baseControl;
            }

            foreach( Control childControl in baseControl.Controls )
            {
                Control control = null;
                if(( control = GetFocusedControl( childControl)) != null )
                {
                    return control;
                }
            }

            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private static bool DoTextEditableControl( Control control)
        {
            System.Windows.Forms.TextBox textBox = control as System.Windows.Forms.TextBox;
            NumericUpDown numericUpDown = control as NumericUpDown;

            if( textBox       != null ||
                numericUpDown != null )
            {
                return true;
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public static bool CanUndoRedo( Control targetControl, Keys keyData)
        {
            if((keyData == (Keys.Z | Keys.Control)) ||
                (keyData == (Keys.Z | Keys.Control | Keys.Shift)) )
            {
                Control control = GetFocusedControl( targetControl);
                if( DoTextEditableControl( control) == false )
                {
                    return true;
                }
            }

            return false;
        }
    }
}
