﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Core.Command;
    using LECore.Structures.Nsrif.Attributes;

    using MaskCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IMask>;

    public class MaskManipulator : BaseManipulator
    {
        Mask _targetMask = null;
        MaskCommandFactory _commandFactory;

        public MaskManipulator()
        {
            _commandFactory = new MaskCommandFactory
                (
                delegate (IMask mementoSrc) { return new MaskMemento(mementoSrc); }
                );
        }

        public void BindTarget(IMask targetMask)
        {
            _targetMask = targetMask as Mask;
            System.Diagnostics.Debug.Assert(_targetMask != null);
        }

        public IMask IMask
        {
            get
            {
                return _targetMask;
            }
        }

        public bool IsMaskEnabled
        {
            set
            {
                if (_targetMask.IsMaskEnabled != value)
                {
                    _commandFactory.Modify(_targetMask, delegate ()
                    {
                        _targetMask.IsMaskEnabled = value;
                    });
                }
            }
        }

        public bool IsMaskWithPane
        {
            set
            {
                if (_targetMask.IsMaskWithPane != value)
                {
                    _commandFactory.Modify(_targetMask, delegate ()
                    {
                        _targetMask.IsMaskWithPane = value;
                    });
                }
            }
        }

        public AttrTexWrap WrapS
        {
            set
            {
                if (_targetMask.WrapS != value)
                {
                    _commandFactory.Modify(_targetMask, delegate ()
                    {
                        _targetMask.WrapS = value;
                    });
                }
            }
        }

        public AttrTexWrap WrapT
        {
            set
            {
                if (_targetMask.WrapT != value)
                {
                    _commandFactory.Modify(_targetMask, delegate ()
                    {
                        _targetMask.WrapT = value;
                    });
                }
            }
        }

        public AttrTexFilterMag MagFilter
        {
            set
            {
                if (_targetMask.MagFilter != value)
                {
                    _commandFactory.Modify(_targetMask, delegate ()
                    {
                        _targetMask.MagFilter = value;
                    });
                }
            }
        }

        public AttrTexFilterMin MinFilter
        {
            set
            {
                if (_targetMask.MinFilter != value)
                {
                    _commandFactory.Modify(_targetMask, delegate ()
                    {
                        _targetMask.MinFilter = value;
                    });
                }
            }
        }

        public FVec2 MtxTrans
        {
            get
            {
                return _targetMask.MtxTransAttr.GetAsFVec2();
            }
            set
            {
                if (_targetMask.MtxTrans != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_targetMask.MtxTransAttr, value);
                }
            }
        }

        public FVec2 MtxScale
        {
            get
            {
                return _targetMask.MtxScaleAttr.GetAsFVec2();
            }
            set
            {
                if (_targetMask.MtxScale != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_targetMask.MtxScaleAttr, value);
                }
            }
        }

        public float MtxRotate
        {
            get
            {
                return _targetMask.MtxRotateAttr.GetAsFloat();
            }
            set
            {
                if (_targetMask.MtxRotate != value)
                {
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(_targetMask.MtxRotateAttr, value);
                }
            }
        }

        /// <summary>
        /// マスク内部で保持しているマテリアルにテクスチャを登録します。
        /// </summary>
        public void RegisterInternalMaterialTexture(string texName, AttrTextureResourceType resType)
        {
            // _MaterialRVLCommandFactoryではなく、
            // _MaterialCommandFactoryを利用して、
            // コマンドを生成します。内部で作成される、
            // オブジェクトのスナップショットの型が異なり、それによって
            // 保存されるデータの種類が変わってきます。
            _commandFactory.Modify(_targetMask, delegate ()
            {
                if (resType == AttrTextureResourceType.LocalCaptured)
                {
                    IPane ownerPane = _targetMask.OwnerPane;
                    if (texName == PaneHelper.FramebufferCaptureDummyTextureName)
                    {
                        TextureMgr mgr = ownerPane.OwnerSubScene.ITextureMgr as TextureMgr;
                        ITextureImage tex = PaneHelper.GetCaptureTextureFromPane(ownerPane);
                        if (tex == null)
                        {
                            TextureMgrManipulator mnp = new TextureMgrManipulator();

                            mnp.BindTarget(mgr);
                            CaptureTexture newTex = mnp.RegisterCaptureTexture(ownerPane, ownerPane.PaneName, false) as CaptureTexture;
                        }

                        (_targetMask.InternalMaterial as Material).RegisterMatTexture(PaneHelper.GetFullyQualifiedCaptureTextureName(ownerPane), 0, resType);
                    }
                    else
                    {
                        string prefix = PaneHelper.GetCaptureTexturePrefix(ownerPane);
                        (_targetMask.InternalMaterial as Material).RegisterMatTexture(PaneHelper.MakeCaptureTextureName(prefix, texName), 0, resType);
                    }
                }
                else
                {
                    (_targetMask.InternalMaterial as Material).RegisterMatTexture(texName, 0, resType);
                }
            });
        }

        /// <summary>
        /// マスク内部で保持しているマテリアルテクスチャを消去します。
        /// </summary>
        public void RemoveInternalMaterialTexture()
        {
            _commandFactory.Modify(_targetMask, delegate ()
            {
                (_targetMask.InternalMaterial as Material).RemoveMatTexture(0);
            });
        }

        /// <summary>
        /// マスク用のキャプチャテクスチャを作成します。
        /// </summary>
        public void CreateCaptureTextureForMask(IPane pane, ITextureMgr textureMgr)
        {
            _targetMask.CreateCaptureTextureForMask(pane, textureMgr);
        }
    }
}
