﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;

namespace LECore.Manipulator
{
    using Structures.Core.Command;
    using Structures.Core;
    using Structures;

    /// <summary>
    /// Manipulator 共通処理を提供する 静的メソッドを集めたクラス。
    /// </summary>
    internal class ManipulatorUtility :
        BaseManipulator
    {
        static public bool CanModifyBase()
        {
            var subScene = Scene.CurrentSubScene;
            return !subScene.IsAnimEditSeparate ||
                   (subScene.IsAnimEditSeparate && subScene.IAnimFrameSectionSet?.TargetIAnimFrameSection == null);
        }
        /// <summary>
        /// アトリビュート変更コマンドを生成します。
        /// </summary>
        static public void ModifyAttrWithCreatingCmdAsFloat(AnmAttribute attribute, float newValue)
        {
            ManipulatorUtility.ModifyAttrWithCreatingCmd(attribute, newValue, CanModifyBase(), true);
        }

        /// <summary>
        /// アトリビュート変更コマンドを生成します。
        /// </summary>
        static public void ModifyAttrWithCreatingCmd(AnmAttribute attribute, object newValue)
        {
            ManipulatorUtility.ModifyAttrWithCreatingCmd(attribute, newValue, CanModifyBase(), false);
        }

        /// <summary>
        /// アトリビュート変更コマンドを生成します。
        /// </summary>
        static public void ModifyAttrWithCreatingCmd( AnmAttribute attribute, object newValue, bool modifyBase)
        {
            ModifyAttrWithCreatingCmd(attribute, newValue, modifyBase, false);
        }

        /// <summary>
        /// アトリビュート変更コマンドを生成します。
        /// </summary>
        static private void ModifyAttrWithCreatingCmd(AnmAttribute attribute, object newValue, bool modifyBase, bool setAsFloat)
        {
            Scene.CurrentSubScene.BeginMassiveModify();
            object beforeVal = null;
            object afterVal = null;
            object beforeBaseVal = null;
            object afterBaseVal = null;

            attribute.GetValue(out beforeVal);
            attribute.GetBaseValue(out beforeBaseVal);

            if (setAsFloat)
            {
                attribute.SetAsFloat((float)newValue);
            }
            else
            {
                attribute.SetValue(newValue);
            }

            if (modifyBase)
            {
                attribute.SetBaseValueRecursive();
            }

            attribute.GetValue(out afterVal);
            attribute.GetBaseValue(out afterBaseVal);
            Scene.CurrentSubScene.EndMassiveModify();
            // コマンドの生成
            _CommandFactory.MakeAttrModCmd(attribute, beforeVal, afterVal, beforeBaseVal, afterBaseVal);
        }

        /// <summary>
        /// プロパティ変更コマンドを生成します。
        /// </summary>
        /// <param name="attribute"></param>
        /// <param name="newValue"></param>
        static public void ModifyPropWithCreatingCmd( IValueAccessor accessor, object newValue )
        {
            object beforeVal = null;
            object afterVal  = null;

            accessor.GetValue( out beforeVal );
            accessor.SetValue( newValue );
            accessor.GetValue( out afterVal );

            // コマンドの生成
            _CommandFactory.MakePropModCmd( accessor, beforeVal, afterVal );
        }

        static public void ModifyPropWithCreatingCmd<T>(Func<T> getter, Action<T> setter, T value)
        {
            var accessor = new ValueAccessor<T>(getter, setter);
            ModifyPropWithCreatingCmd(accessor, value);
        }

        private class ValueAccessor<T> : IValueAccessor
        {
            Func<T> _getter;
            Action<T> _setter;
            public ValueAccessor(Func<T> getter, Action<T> setter)
            {
                _getter = getter;
                _setter = setter;
            }

            public void GetValue(out object value)
            {
                value = _getter();
            }

            public void SetValue(object value)
            {
                _setter((T)value);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    internal interface IValueAccessor
    {
        void GetValue( out object value);
        void SetValue( object value);
    }
}
