﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Save_Load;
    using LECore.Structures.Core.Command;
    using LECore.Structures.LECoreInterface;

    using RlytConverter = LECore.Save_Load.RlytConverter;
    using AlignmentCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IAlignment>;

    /// <summary>
    /// TextBoxManipulator の概要の説明です。
    /// </summary>
    public class AlignmentManipulator : BaseManipulator
    {
        /// <summary>
        ///
        /// </summary>
        static readonly RlytConverter _RlytConverter =
        new RlytConverter( string.Empty, LECore.LayoutEditorCore.MsgReporter );

        // 操作対象
        Alignment _target = null;

        /// <summary>
        /// Undoコマンド生成クラス。
        /// </summary>
        AlignmentCommandFactory _commandFactory;

        /// <summary>
        /// 変更コマンドを登録しつつ、パラメータを更新します。
        /// 任意のValueChangeFunctionでパラメータを更新してください。
        /// </summary>
        void ValueChangeWithCommandRegister_(AlignmentCommandFactory.InstanceModifyFunction valueChangeFunction )
        {
            _commandFactory.Modify(_target, valueChangeFunction );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AlignmentManipulator()
        {
            _commandFactory = new AlignmentCommandFactory(delegate ( IAlignment mementoSrc ){return new AlignmentMemento( mementoSrc );});
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="targetCapture"></param>
        public void BindTarget( IAlignment target)
        {
            _target = target as Alignment;
            Debug.Assert(_target != null );
        }

        /// <summary>
        /// 操作対象
        /// </summary>
        public IAlignment IAlignment
        {
            get{ return _target;}
        }

        /// <summary>
        /// 水平方向の並び方
        /// </summary>
        public HorizontalLocation Horizontal
        {
            set
            {
                if (_target.Horizontal != value)
                {
                    _commandFactory.Modify(_target, () =>
                    {
                        _target.Horizontal = value;
                    });
                }
            }
        }

        public VerticalLocation Vertical
        {
            set
            {
                if (_target.Vertical != value)
                {
                    _commandFactory.Modify(_target, () =>
                    {
                        _target.Vertical = value;
                    });
                }
            }
        }

        /// <summary>
        /// 水平マージン
        /// </summary>
        public float DefaultMargin
        {
            set
            {
                if (_target.DefaultHorizontalMargin != value)
                {
                    _commandFactory.Modify(_target, () =>
                    {
                        _target.DefaultHorizontalMargin = value;
                    });
                }
            }
        }

        /// <summary>
        /// 垂直マージン
        /// </summary>
        public float DefaultVerticalMargin
        {
            set
            {
                if (_target.DefaultVerticalMargin != value)
                {
                    _commandFactory.Modify(_target, () =>
                    {
                        _target.DefaultVerticalMargin = value;
                    });
                }
            }
        }

        /// <summary>
        /// 端（水平）のペインを拡張するかどうか
        /// </summary>
        public bool IsExtendEdgeEnabled
        {
            set
            {
                if(_target.IsExtendEdgeHorizontalEnabled != value)
                {
                    _commandFactory.Modify(_target, () =>
                    {
                        _target.IsExtendEdgeHorizontalEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// 端（垂直）のペインを拡張するかどうか
        /// </summary>
        public bool IsExtendEdgeVerticalEnabled
        {
            set
            {
                if(_target.IsExtendEdgeVerticalEnabled != value)
                {
                    _commandFactory.Modify(_target, () =>
                    {
                        _target.IsExtendEdgeVerticalEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// 整列を実施する
        /// </summary>
        public void DoAlign()
        {
            PaneManipulator mnp = new PaneManipulator();

            _target.OwnerPane.OwnerSubScene.BeginMassiveModify();
            {
                var modifyAction = new Alignment.ModifyAction
                {
                    transModifyAction = (pane, transValue) => { mnp.BindTarget(pane); mnp.Trans = transValue; },
                    widthModifyAction = (pane, width) => { mnp.BindTarget(pane); mnp.Width = width; },
                    heightModifyAction = (pane, height) => { mnp.BindTarget(pane); mnp.Height = height; },
                };

                Alignment.DoAlign(_target, modifyAction);
            }
            _target.OwnerPane.OwnerSubScene.EndMassiveModify();
        }

        /// <summary>
        /// 整列（水平）の設定を有効にするかどうか
        /// </summary>
        public bool IsAlignmentHorizontalEnabled
        {
            set
            {
                if(_target.IsAlignmentHorizontalEnabled != value)
                {
                    _commandFactory.Modify(_target, () =>
                    {
                        _target.IsAlignmentHorizontalEnabled = value;
                        _target.IsAlignmentVerticalEnabled = !value;
                    });
                }
            }
        }

        /// <summary>
        /// 整列（垂直）の設定を有効にするかどうか
        /// </summary>
        public bool IsAlignmentVerticalEnabled
        {
            set
            {
                if(_target.IsAlignmentVerticalEnabled != value)
                {
                    _commandFactory.Modify(_target, () =>
                    {
                        _target.IsAlignmentHorizontalEnabled = !value;
                        _target.IsAlignmentVerticalEnabled = value;
                    });
                }
            }
        }
    }
}
