﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Diagnostics;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.IO;
using System.Threading;
using System.Linq;
using System.Text;



namespace LECore
{
    using Structures;
    using Util;
    using Structures.Core;
    using Save_Load;

    using SerializableFmt = LECore.Structures.SerializableObject;
    using System.Runtime.InteropServices;
    using System.Reflection;
    using System.Collections.Generic;
    using LECore.Manipulator;

    /// <summary>
    /// アプリケーションを表すクラスです。
    /// </summary>
    public class LayoutEditorCore
    {
        //--------------------------------------------------------------------
        #region 定数
        const string _CoreSystemName = "NW LayoutEditor Core";
        static readonly string _CoreSystemVersion;
        #endregion 定数

        [Flags]
        public enum LoadOption
        {
            /// <summary>
            /// 無し
            /// </summary>
            None,
            /// <summary>
            /// Rlanの読み込みを試行します。
            /// </summary>
            TryToOpenRlan,
            /// <summary>
            /// 派生レイアウトのベースとして読み込みます。
            /// </summary>
            LoadAsDerivativeBase,
        }

        //--------------------------------------------------------------------

        /// <summary>
        /// 入力前イベント
        /// </summary>
        static public event Func<string, bool> BeforeImport = null;

        /// <summary>
        /// 閉じる後イベント
        /// </summary>
        static public event Func<string, bool> AfterClose = null;

        /// <summary>
        /// 出力前イベント引数
        /// </summary>
        public class BeforeExportEventArgs : EventArgs
        {
            public BeforeExportEventArgs(string path, ExportOption option)
            {
                FilePath = path;
                ExportOption = option;
            }

            /// <summary>
            /// 出力ファイルパス
            /// </summary>
            public string FilePath { get; set; }

            /// <summary>
            /// 出力オプション
            /// </summary>
            public ExportOption ExportOption { get; }

            /// <summary>
            /// 出力前処理の取り消しイベント
            /// </summary>
            public event EventHandler RetriveChanges;

            /// <summary>
            /// 出力前処理の取り消し
            /// </summary>
            public void ExecuteRetriveChanges()
            {
                RetriveChanges?.Invoke(this, new EventArgs());
            }
        }

        /// <summary>
        /// 出力前イベント
        /// </summary>
        static public event EventHandler<BeforeExportEventArgs> BeforeExport = null;
        /// <summary>
        /// 出力後イベント
        /// </summary>
        static public event Func<string, bool> AfterExport = null;


        public class ExportThumbnailEventArgs : EventArgs
        {
            public ExportThumbnailEventArgs(ISubScene subScene, string subScenePath, string subSceneOriginalPath)
            {
                SubScene = subScene;
                SubScenePath = subScenePath;
                SubSceneOriginalPath = subSceneOriginalPath;
            }

            public ISubScene SubScene { get; set; }
            public string SubScenePath { get; set; }
            public string SubSceneOriginalPath { get; set; }
            public bool Result { get; set; }
            public string Error { get; set; }
        }

        static public event EventHandler<ExportThumbnailEventArgs> ExportThumbnail = null;

        /// <summary>
        /// 削除前イベント
        /// </summary>
        static public event Func<string, bool> BeforeDelete = null;
        /// <summary>
        /// 削除後イベント
        /// </summary>
        static public event Func<string, bool> AfterDelete = null;

        /// <summary>
        /// 最近使ったファイルに追加イベントの引数
        /// </summary>
        public class AddToRecentDocsEventArgs
        {
            public string filePath { get; set; }
        }

        /// <summary>
        /// 最近使ったファイルに追加イベント
        /// </summary>
        static public event EventHandler<AddToRecentDocsEventArgs> AddToRecentDocs = null;

        //--------------------------------------------------------------------
        #region フィールド
        static System.Drawing.Font _guiFont = null;
        static AppConfig _config = null;
        static LEMsgReporter _msgReporter = new LEMsgReporter();
        static readonly IPlatformDetail _PlatformDetail;

        //--------------------------------------------------------------------
        // ファイル入出力用シリアライザ
        //--------------------------------------------------------------------
        static IImporter _RlanImporter = null;
        static IImporter _RlytImporter = null;

        static IExporter _RlanExporter = null;
        static IExporter _RlytExporter = null;

        #endregion フィールド

        //--------------------------------------------------------------------
        #region プロパティ

        /// <summary>
        /// フォントデータ
        /// </summary>
        public static System.Drawing.Font GuiFont
        {
            get { return _guiFont; }
        }

        /// <summary>
        /// コンフィグデータ。
        /// </summary>
        public static AppConfig Config
        {
            get { return _config; }
        }

        /// <summary>
        /// アプリケーション名
        /// </summary>
        public static string AppName
        {
            get { return _CoreSystemName; }
        }

        /// <summary>
        /// バージョン文字列を取得します。
        /// </summary>
        public static string Version
        {
            get { return _CoreSystemVersion; }
        }

        /// <summary>
        /// シーン実体を取得します。
        /// </summary>
        static public IScene Scene
        {
            get { return LECore.Structures.Scene.Instance; }
        }

        /// <summary>
        /// メッセージ報告クラスを取得します。
        /// </summary>
        static public LEMsgReporter MsgReporter
        {
            get { return _msgReporter; }
        }

        /// <summary>
        /// テクスチャ詳細
        /// </summary>
        public static IPlatformDetail PlatformDetail
        {
            get { return _PlatformDetail; }
        }

        /// <summary>
        /// フルアクセス可能な、シーンの実体(外部非公開)
        ///
        /// Importer Exporter にこの情報をどのように公開するか？
        /// 決めかねているので、Importer Exporter モジュールを
        /// LECore外部に分離できずにいます。
        /// </summary>
        static SubScene _CoreSubScene
        {
            get { return LECore.Structures.Scene.CurrentSubScene; }
        }

        public static bool AstcTextureEnabled
        {
            get; set;
        }

        /// <summary>
        /// デフォルトテクスチャフォーマット設定（通常は、AppSettings によって上書きされます。）
        /// </summary>
        public static Structures.LECoreInterface.DefaultTextureFormat DefaultTextureFormat { get; set; } = new Structures.LECoreInterface.DefaultTextureFormat();

        /// <summary>
        /// 拡張タグを解釈するかどうかの設定（通常は、AppSettings によって上書きされます。）
        /// </summary>
        public static bool ExtendedTagEnabled { get; set; } = false;

        /// <summary>
        /// 動画テクスチャの機能を有効化するかどうかの設定（通常は、AppSettings によって上書きされます。）
        /// </summary>
        public static bool MovieTextureEnabled { get; set; } = false;

        /// <summary>
        /// 動画テクスチャの出力ディレクトリ（通常は、AppSettings によって上書きされます。）
        /// </summary>
        public static string MovieTextureOutputDirectory { get; set; } = null;

        /// <summary>
        /// フォントが切り替わったときにサイズスケールを優先するかどうかの設定（通常は、AppSettings によって上書きされます。）
        /// </summary>
        public static bool KeepingSizeScaleEnabled { get; set; } = false;

        /// <summary>
        /// デフォルトのペインサイズ
        /// </summary>
        public static Size DefaultInitialPaneSize { get; set; } = SubSceneManipulator.DefaultInitialPaneSize;
        #endregion

        //--------------------------------------------------------------------
        #region 初期化
        /// <summary>
        /// クリップボードの初期化を行います。
        /// </summary>
        static void InitLEClipBoard_()
        {
            // 変換関数を登録し、コピー/ペーストに対応します。
            LEClipboard.Instance.RegisterConvertionFuncSet(new LEClipboardPaneSetConvertFunc());
            LEClipboard.Instance.RegisterConvertionFuncSet(new LEClipboardAnmAttributeSetConvertFunc());
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        static LayoutEditorCore()
        {
            // バージョン文字列
            {
                Assembly assembly = Assembly.GetExecutingAssembly();
                FileVersionInfo info = FileVersionInfo.GetVersionInfo(assembly.Location);

                _CoreSystemVersion = string.Format("{0}.{1}.{2}.{3}",
                    info.ProductMajorPart, info.ProductMinorPart, info.ProductBuildPart, info.ProductPrivatePart);
            }

            // アプリケーション・コンフィグを初期化します。
            _config = new AppConfig();

            // アプリケーションで使用するフォントを初期化します。
            _guiFont = new System.Drawing.Font("Tahoma", 8.25F, FontStyle.Regular, GraphicsUnit.Point, 0);

            // DLL を列挙して、プラットフォーム固有情報を取得します。
            {
                Type type = EnumrateInterfaceTypesFromDLL( (dllFileName) => dllFileName.StartsWith("LECore") && dllFileName != "LECore" && dllFileName != "LECoreTest", typeof(LECore.Structures.IPlatformDetail)).FirstOrDefault();
                if(type != null)
                {
                    _PlatformDetail = Activator.CreateInstance(type) as IPlatformDetail;
                }
            }

            // クリップボードの初期化を行います。
            InitLEClipBoard_();

            // 文字列リソースマネージャの初期化。
            LECoreStringResMgr.Initialize();
        }

        /// <summary>
        ///
        /// </summary>
        static public IEnumerable<Type> EnumrateInterfaceTypesFromDLL(Predicate<string> passPredicate, Type interfaceType)
        {
            foreach (var dllFilePath in Directory.GetFiles(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "*.dll"))
            {
                string dllFileName = Path.GetFileNameWithoutExtension(dllFilePath);

                if (!passPredicate(dllFileName))
                {
                    continue;
                }

                Assembly platformAssembly = Assembly.LoadFrom(dllFilePath);
                if (platformAssembly != null)
                {
                    var types = platformAssembly.GetTypes().Where(p => interfaceType.IsAssignableFrom(p));

                    foreach (var type in types)
                    {
                        yield return type;
                    }
                }
            }
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        static public void Initialize()
        {
        }

        /// <summary>
        /// 現在のシーンをリセットします。
        /// </summary>
        static public void ResetCurrentScene()
        {
            _CoreSubScene.ResetScene();
        }

        /// <summary>
        /// OpenGLコンテキストの初期化です。
        /// </summary>
        public static bool InitializeOpenGLContext(IntPtr hdc)
        {
            return FontFactory.InitializeOpenGLContext(hdc);
        }

        /// <summary>
        /// OpenGLコンテキストの破棄です。
        /// </summary>
        public static void FinalizeOpenGLContext()
        {
            FontFactory.FinalizeOpenGLContext();
        }

        public static void SetTextureCacheSizeForScalableFont(Size size)
        {
            FontFactory.SetTextureCacheSizeForScalableFont(size);
        }

        public static void SetForceConvertBffntFromSameNameFfnt(bool value)
        {
            FontFactory.ForceConvertBffntFromSameNameFfnt = value;
        }

        public static void SetFloatColorWriteFlag(bool value)
        {
            RlytConverter.FloatColorWrite = value;
        }

        #endregion 初期化

        //--------------------------------------------------------------------
        #region ファイルへのセーブ/ロード

        /// <summary>
        /// ファイル保存をします。
        /// </summary>
        static public bool FileSave(string outPath, Func<string, bool> fileSaveFunction)
        {
            ExportOption exportOption = new ExportOption() {
                UseBaseValue = true,
                Frame = GlobalTime.Inst.Time
            };
            var args = new BeforeExportEventArgs(outPath, exportOption);
            try
            {
                BeforeExport?.Invoke(null, args);

                if (!fileSaveFunction(outPath))
                {
                    return false;
                }
            }
            finally
            {
                args.ExecuteRetriveChanges();
            }

            if (AfterExport != null)
            {
                if (!AfterExport(outPath))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// ファイル削除をします。
        /// </summary>
        static public bool FileDelete(string outPath, Func<string, bool> fileDeleteFunction)
        {
            if (BeforeDelete != null)
            {
                if (!BeforeDelete(outPath))
                {
                    return false;
                }
            }

            if (!fileDeleteFunction(outPath))
            {
                return false;
            }

            if (AfterDelete != null)
            {
                if (!AfterDelete(outPath))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 最近使ったファイルへの追加イベントを呼び出します。
        /// </summary>
        static public void OnAddToRecentDocs(string path)
        {
            AddToRecentDocs?.Invoke(null, new AddToRecentDocsEventArgs() { filePath = path });
        }

        //----------------------------------
        #region インポータ・エクスポータの取得
        /// <summary>
        /// エクスポータを取得します。
        /// </summary>
        static public IEnumerable<IFileInfoHolder> GetExporter(ImportExportFileKind kind, ExportOption saveOption)
        {
            if (_RlanExporter == null) { _RlanExporter = new RlanExporter(); }
            if (_RlytExporter == null) { _RlytExporter = new RlytExporter(); }

            switch (kind)
            {
                case ImportExportFileKind.All:
                    _RlytExporter.ExportOption = saveOption;
                    _RlanExporter.ExportOption = saveOption;
                    yield return _RlanExporter;
                    yield return _RlytExporter;
                    yield break;
                case ImportExportFileKind.Rlan:
                    _RlanExporter.ExportOption = saveOption;
                    yield return _RlanExporter;
                    yield break;
                case ImportExportFileKind.Rlyt:
                    _RlytExporter.ExportOption = saveOption;
                    yield return _RlytExporter;
                    yield break;
                default:
                    Debug.Assert(false);
                    yield break;
            }
        }

        /// <summary>
        /// 種類を指定して、インポータを取得します。
        /// </summary>
        static public IEnumerable<IImporter> GetImporter(ImportExportFileKind kind)
        {
            if (_RlanImporter == null) { _RlanImporter = new RlanImporter(); }
            if (_RlytImporter == null) { _RlytImporter = new RlytImporter(); }

            switch (kind)
            {
                case ImportExportFileKind.All:
                    yield return _RlytImporter;
                    yield return _RlanImporter;
                    yield break;
                case ImportExportFileKind.Rlan:
                    yield return _RlanImporter;
                    yield break;
                case ImportExportFileKind.Rlyt:
                    yield return _RlytImporter;
                    yield break;
                default:
                    yield break;
            }
        }

        /// <summary>
        /// ファイルパスからインポータを取得します。
        /// </summary>
        static public IEnumerable<IImporter> GetImporter(string filename)
        {
            IEnumerable<IImporter> importers = null;
            switch (Path.GetExtension(filename))
            {
                case AppConstants.LayoutFileExt: importers = GetImporter(ImportExportFileKind.Rlyt); break;
                case AppConstants.AnimationFileExt: importers = GetImporter(ImportExportFileKind.Rlan); break;
                default: importers = null; break;
            }

            if (importers != null)
            {
                foreach (var importer in importers)
                {
                    yield return importer;
                }
            }
        }

        #endregion インポータ・エクスポータの取得

        /// <summary>
        /// シーンを保存します。
        /// </summary>
        static public bool ExportToFileAllWithoutEventNotifications(ISubScene subScene, string outPath, ExportOption options)
        {
            try
            {
                LECore.Structures.Scene.Instance.ForceDisableEventNotifications = true;
                LayoutEditorCore.MsgReporter.ForceStopErrorReport = true;
                return ExportToFileAll(subScene, outPath, options);
            }
            finally
            {
                LECore.Structures.Scene.Instance.ForceDisableEventNotifications = false;
                LayoutEditorCore.MsgReporter.ForceStopErrorReport = false;
            }
        }

        /// <summary>
        /// シーンを保存します。
        /// </summary>
        static public bool ExportToFileAll(ISubScene subScene, string outPath, ExportOption options)
        {
            bool result = true;
            try
            {
                subScene.BeginMassiveModify();
                _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_DLG_TITLE_SAVE"));

                var exporters = LayoutEditorCore.GetExporter(ImportExportFileKind.All, options);//
                foreach (var exporter in exporters)
                {
                    result &= ExportToFile(subScene, exporter, outPath);
                }
            }
            finally
            {
                _msgReporter.EndPacking(true);
                subScene.EndMassiveModify();
            }

            return result;
        }

        /// <summary>
        /// ファイルへのエクスポート
        ///
        /// この処理は、将来、LECore モジュールから、
        /// LayoutEditor( App )モジュールへ移行される予定です。
        /// </summary>
        static public bool ExportToFile(ISubScene subScene, IFileInfoHolder fileInfoHolder, string outPath)
        {
            if (SubSceneHelper.IsPartsReloading)
            {
                _msgReporter.OutLog("", LECoreStringResMgr.Get("WARNING_CACEL_SAVE_RELOADING"));
                return true;
            }

            // まず最初に、エクスポートする必要があるか判断する。
            IExporter exporter = fileInfoHolder as IExporter;
            if (exporter.CheckNothingToExport(subScene))
            {
                return true;
            }

            BeforeExportEventArgs args = null;
            try
            {
                string exportFilePath = exporter.TrimFilePath(outPath);

                subScene.BeginMassiveModify();
                _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_DLG_TITLE_SAVE") + string.Format(" [{0}]", exportFilePath));


                args = new BeforeExportEventArgs(exportFilePath, exporter.ExportOption);

                if (BeforeExport != null && !exporter.StopFileIOEvent)
                {
                    BeforeExport(null, args);
                }

                // ファイル名が不正かチェックします。
                string outFileName = Path.GetFileNameWithoutExtension(exportFilePath);
                if (!SaveLoadHelper.IsValidStringForFileName(outFileName))
                {
                    _msgReporter.ReportError(
                        LECoreStringResMgr.Get("LECORE_DLG_TITLE_SAVE"),
                        LECoreStringResMgr.Get("LECORE_SYS_ERROR_NGFILENAME"), outFileName);
                    return false;
                }

                if (!Path.IsPathRooted(exportFilePath))
                {
                    Debug.Assert(false);
                    return false;
                }

                if (Path.GetExtension(exportFilePath).ToLower() == AppConstants.LayoutFileExt && exporter.ExportOption.CreateThumbnail)
                {
                    var thumbnailArgs = new ExportThumbnailEventArgs(subScene, exportFilePath, exporter.ExportOption.OriginalPath);
                    ExportThumbnail?.Invoke(null, thumbnailArgs);
                    if (!string.IsNullOrEmpty(thumbnailArgs.Error))
                    {
                        _msgReporter.ReportError(
                            LECoreStringResMgr.Get("LECORE_DLG_TITLE_SAVE"),
                            thumbnailArgs.Error);
                    }
                }

                bool result = ExecuteExport_(exporter as IExporter, exportFilePath, subScene as SubScene);
                if (AfterExport != null && !exporter.StopFileIOEvent)
                {
                    if (!AfterExport(exportFilePath))
                    {
                        return false;
                    }
                }

                if(result)
                {
                    // 最近使ったファイルに追加(ただし、一時ファイルは除く)
                    if(!TemporaryFileUtil.IsInTempDirectory(outPath))
                    {
                        LECore.LayoutEditorCore.OnAddToRecentDocs(exportFilePath);
                    }
                }

                return result;
            }
            finally
            {
                if (args != null)
                {
                    args.ExecuteRetriveChanges();
                }

                _msgReporter.EndPacking();
                subScene.EndMassiveModify();
            }
        }

        /// <summary>
        /// エクスポート前の設定を行ないます。
        /// </summary>
        static public void PrepareExport(BeforeExportEventArgs args)
        {
            if (Path.GetExtension(args.FilePath) == AppConstants.LayoutFileExt)
            {
                ApplyExportOption_(args);
            }
        }

        /// <summary>
        /// 出力オプションを適用します。
        /// </summary>
        static void ApplyExportOption_(BeforeExportEventArgs args)
        {
            if (args.ExportOption.UseBaseValue)
            {
                // 元に戻す操作を設定します。
                SetRetriveChangesAction_(args);

                foreach (var subScene in LayoutEditorCore.Scene.ISubSceneSet)
                {
                    if (!subScene.IsAnimEditSeparateMode())
                    {
                        // 時間0で評価します
                        GlobalTime.Inst.SetTimeForcibly(0);
                    }
                    else
                    {
                        SubSceneHelper.ResetValueAsBaseValue(subScene); // すべてのアトリビュートをbaseValueにリセットします
                    }
                }
            }
            else
            {
                switch (args.ExportOption.AfterExport)
                {
                    case ExportOption.AfterExportOption.RetriveValues:
                        {
                            // 元に戻す操作を設定します。
                            SetRetriveChangesAction_(args);
                        }
                        break;
                    case ExportOption.AfterExportOption.ResetFrame:
                        {
                            var frame = GlobalTime.Inst.Time;
                            args.RetriveChanges += (s, e) =>
                            {
                                GlobalTime.Inst.SetTimeForcibly(frame);
                            };
                        }
                        break;
                }
                GlobalTime.Inst.SetTimeForcibly(args.ExportOption.Frame);
            }
        }

        /// <summary>
        /// 出力前処理の取り消しイベントを設定します。
        /// </summary>
        static private void SetRetriveChangesAction_(BeforeExportEventArgs args)
        {
            var actions = (from subScene in LayoutEditorCore.Scene.ISubSceneSet
                           from pane in subScene.IPaneArray.OfType<IAnmAttribute>()
                           from attr in pane.EnumCurveAttribute()
                           select AnmAttributeHelper.GetResetValueAction(attr)).Where(x => x != null)
             .Reverse().ToArray();

            args.RetriveChanges += (s, e) =>
            {
                GlobalTime.Inst.SetTimeForcibly(args.ExportOption.Frame); // 現在時間に戻します

                var tmp = GlobalTime.Inst.SettingTimeForcibly;
                GlobalTime.Inst.SettingTimeForcibly = true;
                foreach (var action in actions)
                {
                    action();
                };
                GlobalTime.Inst.SettingTimeForcibly = tmp;
            };
        }

        /// <summary>
        /// 出力パスを指定してエクスポータを実行します。
        /// </summary>
        static private bool ExecuteExport_(IExporter exporter, string outPath, SubScene scene)
        {
            if (!Path.IsPathRooted(outPath))
            {
                Debug.Assert(false);
                return false;
            }

            try
            {
                exporter.MsgReporter = MsgReporter;

                return exporter.DoSave(outPath, scene);
            }
            catch (Exception e)
            {
                MsgReporter.ReportError(outPath, "!!! UnkownException !!!\n\r\n\r{0}\n\r\n\r{1}", e.Message, e.StackTrace);

                DbgConsole.WriteLine("Error in Exporting {0} file. ", exporter.GetFileExtensionShort());
                DbgConsole.WriteLine("file path = {0}", outPath);
                DbgConsole.WriteLine(e.ToString());
            }
            return false;
        }

        /// <summary>
        /// ファイルを入力します。
        /// </summary>
        static public bool ImportFromFileAll(ISubScene iSubScene, string filename, bool isAdditionalImport)
        {
            bool result = true;
            try
            {
                iSubScene.BeginMassiveModify();
                _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_DLG_TITLE_LOAD"));

                var importers = GetImporter(ImportExportFileKind.All);
                foreach (var importer in importers)
                {
                    if (isAdditionalImport) { importer.SetOptionFlag(IImporterOption.IsAdditionalImport); }

                    result &= ImportFromFile(iSubScene, importer, filename);
                }
            }
            finally
            {
                _msgReporter.EndPacking();
                iSubScene.EndMassiveModify();
            }

            return result;
        }

        /// <summary>
        /// ファイルを入力します。
        /// </summary>
        /// <param name="iSubScene"></param>
        /// <param name="filename"></param>
        /// <returns></returns>
        static public bool ImportFromFile
        (
            ISubScene iSubScene,
            IImporter importer,
            string filePath)
        {
            string importFilePath = importer.TrimFilePath(filePath);
            if (!File.Exists(importFilePath))
            {
                return false;
            }

            try
            {
                _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_DLG_TITLE_LOAD"));

                // 読み込み前イベント
                if (BeforeImport != null && !importer.StopFileIOEvent)
                {
                    if (!BeforeImport(importFilePath))
                    {
                        return false;
                    }
                }

                // ファイルを開きます
                SubScene subScene = iSubScene as SubScene;
                bool result = ExcuteImport_(importer, importFilePath, subScene);

                return result;
            }
            finally
            {
                _msgReporter.EndPacking();
            }
        }

        /// <summary>
        /// インポータを実行します。
        /// </summary>
        static private bool ExcuteImport_(IImporter importer, string importPath, SubScene scene)
        {
            try
            {
                importer.MsgReporter = MsgReporter;

                try
                {
                    _msgReporter.BeginPacking(string.Format("Loading ... [{0}]", importPath));
                    return importer.DoLoad(importPath, scene);
                }
                finally
                {
                    _msgReporter.EndPacking();
                }
            }
            catch (ApplicationException e)
            {
                string title = LECoreStringResMgr.Get("LECORE_DLG_TITLE_LOAD");
                string message = LECoreStringResMgr.Get("LECORE_RLYTLOAD_ERROR");
                message = String.Format(message, Path.GetFileName(importPath));
                _msgReporter.ReportError(title, "{0}\r\n{1}", message, e.ToString());
                return false;
            }
            catch (Exception e)
            {
                // ファイル読み込み中に深刻なエラーが発生しました。ファイル[{0}]が読み込めませんでした。
                string title = LECoreStringResMgr.Get("LECORE_DLG_TITLE_LOAD");
                string message = LECoreStringResMgr.Get("LECORE_SYS_ERROR_FATAL_FILELOAD");
                _msgReporter.ReportError(title, message + "\n\n{1}", Path.GetFileName(importPath), e.ToString());
                return false;
            }
        }

        /// <summary>
        /// 派生部品として初期化します。
        /// </summary>
        static public void InitializeAsDelivativeParts(ISubScene subScene, ISubScene basePartsSubScene, string baseSubSceneName)
        {
            (subScene.IPartsSettings as PartsSettings).InitializeAsDelivativeParts(basePartsSubScene, baseSubSceneName);
        }

        #endregion ファイルへのセーブ/ロード


        /// <summary>
        /// シーンを読み込む
        /// </summary>
        static public ISubScene CreateSubSceneWithoutEventNotifications(string filePath, LECore.LayoutEditorCore.LoadOption option)
        {
            try
            {
                LECore.Structures.Scene.Instance.ForceDisableEventNotifications = true;
                LayoutEditorCore.MsgReporter.ForceStopErrorReport = true;
                return CreateSubScene(filePath, option);
            }
            finally
            {
                LECore.Structures.Scene.Instance.ForceDisableEventNotifications = false;
                LayoutEditorCore.MsgReporter.ForceStopErrorReport = false;
            }
        }

        /// <summary>
        /// レイアウトファイルを作る
        /// </summary>
        static public LayoutDocument CreateAsNewFile()
        {
            var sceneManipulator = new SceneManipulator();
            sceneManipulator.BindTarget(LayoutEditorCore.Scene);

            // 時間を0に再設定します
            GlobalTime.Inst.SetTimeForcibly(0);

            LECore.Structures.ISubScene newScene = sceneManipulator.CreateSubScne();

            var layoutDocument = new LayoutDocument(newScene, LayoutEditorCore.MakeLECoreDocument(AppConstants.InitialFileName), DateTime.Now);
            sceneManipulator.CurrentISubScene = newScene;

            return layoutDocument;
        }

        /// <summary>
        /// レイアウトファイルを読み込む
        /// </summary>
        static public LayoutDocument LoadLayoutFile(string filePath, LECore.LayoutEditorCore.LoadOption option)
        {
            return CreateSubSceneFromFile_(filePath, option, filePath);

        }

        /// <summary>
        /// レイアウトファイルを読み込む
        /// </summary>
        static public LayoutDocument LoadLayoutFileAsNewFile(string filePath, LECore.LayoutEditorCore.LoadOption option)
        {
            return CreateSubSceneFromFile_(filePath, option, AppConstants.InitialFileName);
        }

        /// <summary>
        /// シーンを読み込む
        /// </summary>
        static private LayoutDocument CreateSubSceneFromFile_(string filePath, LECore.LayoutEditorCore.LoadOption option, string documentPath)
        {
            var sceneManipulator = new SceneManipulator();
            sceneManipulator.BindTarget(LayoutEditorCore.Scene);

            ISubScene newSubscene = null;
            ISubScene oldSubscene = LayoutEditorCore.Scene.CurrentISubScene;


            // 時間を0に再設定します
            GlobalTime.Inst.SetTimeForcibly(0);

            LayoutDocument newDocument = null;
            try
            {
                // CreateSubScene の処理中に LayoutEditorCore.Scene.CurrentISubScene に変更履歴がつまれるなどの変更を防ぐため
                // CurrentISubScene を一時的に null に設定する。
                sceneManipulator.CurrentISubScene = null;

                newSubscene = LayoutEditorCore.CreateSubScene(filePath, option);
                if(newSubscene != null)
                {
                    // キャプチャテクスチャの名前解決処理にはパーツの親子関係が完全に構築されている必要があるため
                    // ボディレイアウトの読み込みが完了しすべての情報が確定タイミングでキャプチャテクスチャの修飾名解決処理を行う。
                    foreach (var pane in newSubscene.IPaneArray)
                    {
                        if (pane.PaneKind == Structures.PaneKind.Parts)
                        {
                            SubSceneHelper.ResolvePartsLayoutCaptureTextureReference(pane);
                        }
                    }
                }
            }
            finally
            {
                if (newSubscene != null)
                {
                    newDocument = new LayoutDocument(newSubscene, LayoutEditorCore.MakeLECoreDocument(documentPath), DateTime.Now);

                    // カレントに設定する
                    {
                        sceneManipulator.CurrentISubScene = newSubscene;
                    }

                    // 最近使ったファイルに登録する。
                    LECore.LayoutEditorCore.OnAddToRecentDocs(filePath);
                }
                else
                {
                    sceneManipulator.CurrentISubScene = oldSubscene;
                }
            }

            return newDocument;
        }

        /// <summary>
        /// LECoreDocument を作る
        /// </summary>
        static internal LECoreDocument MakeLECoreDocument(string filePath)
        {
            var newLECoreDocument = new LECoreDocument(filePath);

            DbgConsole.WriteLine("    DOC_Load----[{0}]", newLECoreDocument.FilePath);

            return newLECoreDocument;
        }

        /// <summary>
        /// 利用しなくなった、LECoreDocument のアンロードを試行する。
        /// </summary>
        /// <returns>成否。false なら ファイルを閉じる処理はキャンセルする。</returns>
        static internal bool NotifyUnloadLECoreDocument(LECoreDocument document)
        {
            DbgConsole.WriteLine("    DOC_UnLoad----[{0}]", document.FilePath);

            if (AfterClose != null)
            {
                if (!AfterClose(document.FilePath))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// シーンを読み込む
        /// 作り変える
        /// 　シーンを作る。
        /// 　ファイルを次々と読み、シーンに適用する。
        /// </summary>
        static public ISubScene CreateSubScene(string filePath, LoadOption option)
        {
            LECore.Structures.ISubScene scene = null;
            try
            {
                _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_DLG_TITLE_LOAD") + string.Format(" [{0}]", filePath));

                // 部品が依存するファイルがあれば事前に読み込んでおきます。
                //
                foreach (string dependsFileName in PartsSubSceneHelper.GetDependsFiles(filePath))
                {
                    LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded(dependsFileName);
                }

                //---------- シーンを読み込みます
                string rlytPath = GetRlytFilePath(filePath);
                if (rlytPath != string.Empty)
                {
                    //---------- rlytとrlanを読み込みます。
                    // レイアウトファイル以外は処理しない。
                    SubScene newScene = null;
                    if (Path.GetExtension(filePath) == AppConstants.LayoutFileExt)
                    {
                        // サブシーンを追加します。
                        newScene = (Scene as Scene).CreateSubScene() as SubScene;

                        ////// 基底部品について...
                        string basePartsFileName = PartsSubSceneHelper.GetBasePartsFile(filePath);
                        if (!string.IsNullOrEmpty(basePartsFileName))
                        {
                            // まず基底部品をツール内に読んでおく
                            LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded(basePartsFileName);
                            var basePartsSubScene = Scene.PartsSubScenes.FirstOrDefault((pss) => Path.GetFileName(pss.FilePath) == basePartsFileName);

                            // 派生部品として設定します。
                            if (basePartsSubScene != null)
                            {
                                LayoutEditorCore.InitializeAsDelivativeParts(newScene, basePartsSubScene.SubScene, basePartsFileName);
                            }
                            else
                            {
                                _msgReporter.ReportError(
                                    LECoreStringResMgr.Get("LECORE_DERIVPARTS_INIT", Path.GetFileName(filePath)),
                                    LECoreStringResMgr.Get("LECORE_DERIVPARTS_INIT_MSG", Path.GetFileName(basePartsFileName)));

                                // サブシーンは不正なため、取り除きます
                                (Scene as Scene).RemoveSubScene(newScene);
                                return null;
                            }
                        }

                        // サブシーンにファイルをロードします。
                        IImporter importer = LayoutEditorCore.GetImporter(filePath).FirstOrDefault();
                        if (importer != null)
                        {
                            if (!LayoutEditorCore.ImportFromFile(newScene, importer, filePath))
                            {
                                // サブシーンは不正なため、取り除きます
                                (Scene as Scene).RemoveSubScene(newScene);
                                newScene = null;
                            }
                        }

                        // 指定があれば、アニメーションの読み込みを試みます
                        if (newScene != null && (option & LoadOption.TryToOpenRlan) != 0)
                        {
                            var animImporters = LayoutEditorCore.GetImporter(LECore.Save_Load.ImportExportFileKind.Rlan);
                            var animImporter = animImporters.FirstOrDefault();
                            if (animImporter != null)
                            {
                                animImporter.SetOptionFlag(IImporterOption.DoNotReportFailure);

                                LayoutEditorCore.ImportFromFile(newScene, animImporter, filePath);

                                // 現在の時間で評価。
                                // 一旦無効化します。
                                // 関連：http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-72544
                                //GlobalTime.Inst.SetTimeForcibly(GlobalTime.Inst.Time);
                            }
                        }

                        // 派生基底レイアウトに設定します。
                        if ((option & LoadOption.LoadAsDerivativeBase) != 0)
                        {
                            LayoutEditorCore.InitializeAsDelivativeParts(newScene, newScene, Path.GetFileName(filePath));
                        }
                    }

                    if (newScene != null)
                    {
                        // 部品設定の参照関係リセット
                        foreach (ControlSettings ctrlSetting in newScene.IControlSettings)
                        {
                            ctrlSetting.ResetReference();
                        }
                    }

                    // 不正なアニメーションがないかチェックし、ユーザに通知します。
                    RepoertInactiveAnimations(newScene);

                    // 不正な詳細コンバイナがないかチェックし、ユーザに通知します。
                    RepoertDetailedCombiner(newScene);

                    scene = newScene;
                }

            }
            finally
            {
                _msgReporter.EndPacking();
            }

            return scene;
        }

        private static string GetRlytFilePath(string filePath)
        {
            //---------- 読み込むrlytの名前を決定します。
            // 拡張子がrlytなら
            string rlytPath = string.Empty;
            if (string.Compare(Path.GetExtension(filePath), AppConstants.LayoutFileExt) == 0)
            {
                // そのまま使用
                rlytPath = filePath;
            }
            else
            {
                // 拡張子がrlanなら
                if (string.Compare(Path.GetExtension(filePath), AppConstants.AnimationFileExt) == 0)
                {
                    // 同名のrlytがあるか検索する、あれば採用する。
                    rlytPath = Path.GetDirectoryName(filePath) + "/" +
                                Path.GetFileNameWithoutExtension(filePath) + AppConstants.LayoutFileExt;

                    if (!File.Exists(rlytPath))
                    {
                        // アニメーション単体での読み込みは行えないことをユーザに通知します。
                        rlytPath = string.Empty;
                        MessageBox.Show(LECoreStringResMgr.Get("SYSTEM_LOAD_CANT_FIND_RLYT"), AppConstants.AppName);
                    }
                }
            }
            return rlytPath;
        }

        /// <summary>
        /// 使われなくなったアニメーションをユーザに報告する。
        /// </summary>
        static public void RepoertInactiveAnimations(ISubScene subScene)
        {
            if (subScene == null)
            {
                return;
            }

            List<IPane> invalidPanes = new List<IPane>();
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.GatherInactiveAnimations().Count() > 0)
                {
                    invalidPanes.Add(pane);
                }
            }

            if (invalidPanes.Count > 0)
            {
                // 勝手に削除してしまう。
                subScene.BeginMassiveModify();
                foreach (IPane pane in subScene.IPaneArray)
                {
                    List<IAnmAttribute> paneAttrs = new List<IAnmAttribute>(pane.GatherInactiveAnimations());
                    if (paneAttrs.Count() <= 0)
                    {
                        continue;
                    }

                    AnmAttributeManipulator attrMnp = new AnmAttributeManipulator();
                    foreach (var paneAttr in paneAttrs)
                    {
                        attrMnp.BindTarget(paneAttr);
                        attrMnp.RemoveAllKeyRecursively();
                    }
                }

                subScene.EndMassiveModify();

                // 削除した旨をユーザに通知する。
                _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_REPORT_INACTIVEANIM_TITLE"));
                _msgReporter.ReportError("", LECoreStringResMgr.Get("LECORE_REPORT_INACTIVEANIM_MSG"));
                foreach (var invalidPane in invalidPanes)
                {
                    _msgReporter.ReportError("", " * pane=[{0}]", invalidPane.PaneName);
                }
                _msgReporter.EndPacking();
            }
        }

        /// <summary>
        /// 詳細コンバイナのステージ数が不正な場合にユーザーに報告する。
        /// </summary>
        /// <param name="subScene"></param>
        static public void RepoertDetailedCombiner(ISubScene subScene)
        {
            if (subScene == null)
            {
                return;
            }

            List<IPane> invalidPanes = new List<IPane>();
            foreach (IPane srcPane in subScene.IPaneArray)
            {
                if (srcPane.PaneKind == PaneKind.Picture)
                {
                    IRevHWMaterialHolder revPicture = srcPane.IPicture as IRevHWMaterialHolder;
                    IRevHWMaterial appMat = revPicture.IRevHWMaterial[0];

                    if (appMat.LowLevelCombinerSettingsEnabled && appMat.ITevData.NumStages <= 0)
                    {
                        invalidPanes.Add(srcPane);
                    }
                }
                else if (srcPane.PaneKind == PaneKind.Window)
                {
                    ILEWindow srcWindow = srcPane.ILEWindow;
                    IRevHWMaterialHolder srcRevWindow = srcWindow as IRevHWMaterialHolder;
                    IRevHWMaterial appMat = srcRevWindow.IRevHWMaterial[srcPane.ILEWindow.PartsIDToInt(LEWindowPartsID.Content)];

                    if (appMat.LowLevelCombinerSettingsEnabled && appMat.ITevData.NumStages <= 0)
                    {
                        invalidPanes.Add(srcPane);
                    }

                    // フレーム領域のマテリアル
                    for (int i = 0; i < srcWindow.NumTexture - 1; i++)
                    {
                        int frameIndex = i + 1;
                        LEWindowPartsID partsID = srcWindow.IntToPartsID(frameIndex);
                        IRevHWMaterial appMatFrame = srcRevWindow.IRevHWMaterial[frameIndex];

                        if (appMatFrame.LowLevelCombinerSettingsEnabled && appMatFrame.ITevData.NumStages <= 0)
                        {
                            invalidPanes.Add(srcPane);
                        }
                    }
                }
            }

            if (invalidPanes.Count > 0)
            {
                // ユーザに通知する。
                _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_REPORT_DETAILEDCOMBINER_TITLE"));
                _msgReporter.ReportError("", LECoreStringResMgr.Get("LECORE_REPORT_DETAILEDCOMBINERSTAGE_MSG"));
                foreach (var invalidPane in invalidPanes)
                {
                    _msgReporter.ReportError("", " * pane=[{0}]", invalidPane.PaneName);
                }
                _msgReporter.EndPacking();
            }
        }
    }

    /// <summary>
    /// ファイル入出力ユーザスクリプトを処理するクラス。
    /// </summary>
    public static class FileIOUserScriptHandler
    {
        static string _PreSaveCommandPath { get; set; }
        static string _PostSaveCommandPath { get; set; }
        static string _PreDeleteCommandPath { get; set; }
        static string _PostDeleteCommandPath { get; set; }

        static string _PreLoadCommandPath { get; set; }
        static string _PostCloseCommandPath { get; set; }

        static int _TimeOutMilliSec { get; set; }
        static LEMsgReporter _MessageReporter { get; set; }

        static public bool StopUserScriptCall { get; set; }

        public static void Init(
            string preSave,
            string postSave,
            string preDel,
            string postDel,
            string preLoad,
            string postClose,
            int    timeOutMilliSec,
            LEMsgReporter messageReporter)
        {
            _PreSaveCommandPath = preSave;
            _PostSaveCommandPath = postSave;
            _PreDeleteCommandPath = preDel;
            _PostDeleteCommandPath = postDel;

            _PreLoadCommandPath = preLoad;
            _PostCloseCommandPath = postClose;

            _TimeOutMilliSec = timeOutMilliSec;

            _MessageReporter = messageReporter;

            LayoutEditorCore.BeforeExport += new EventHandler<LayoutEditorCore.BeforeExportEventArgs>((sender, args) => CallFileIOUserScriptImpl_(new string[] { args.FilePath }, _PreSaveCommandPath, LECoreStringResMgr.Get("USERCOMMAND_PRESAVE")));
            LayoutEditorCore.AfterExport += new Func<string, bool>((filePath) => CallFileIOUserScriptImpl_(new string[] { filePath }, _PostSaveCommandPath, LECoreStringResMgr.Get("USERCOMMAND_POSTSAVE")));

            LayoutEditorCore.BeforeDelete += new Func<string, bool>((filePath) => CallFileIOUserScriptImpl_(new string[] { filePath }, _PreDeleteCommandPath, LECoreStringResMgr.Get("USERCOMMAND_PREDELETE")));
            LayoutEditorCore.AfterDelete += new Func<string, bool>((filePath) => CallFileIOUserScriptImpl_(new string[] { filePath }, _PostDeleteCommandPath, LECoreStringResMgr.Get("USERCOMMAND_POSTDELETE")));

            LayoutEditorCore.BeforeImport += new Func<string, bool>((filePath) => CallFileIOUserScriptImpl_(new string[] { filePath }, _PreLoadCommandPath, LECoreStringResMgr.Get("USERCOMMAND_PRELOAD")));
            LayoutEditorCore.AfterClose += new Func<string, bool>((filePath) => CallFileIOUserScriptImpl_(new string[] { filePath }, _PostCloseCommandPath, LECoreStringResMgr.Get("USERCOMMAND_POSTCLOSE")));

            // ファイルが蓄積し続けない様に、テンポラリフォルダを作り直す。
            {
                string tempDir = GetTempFilePath_();
                if (Directory.Exists(tempDir))
                {
                    try
                    {
                        Directory.Delete(tempDir, true);
                    }
                    catch
                    {
                        DbgConsole.WriteLine("FileIOUserScriptHandler : Failed to Delete tempDir.");
                    }
                }
                TryToCreateTempFileDirectory_(tempDir);
            }
        }

        /// <summary>
        ///
        /// </summary>
        static void TryToCreateTempFileDirectory_(string tempDirPath)
        {
            if (!Directory.Exists(tempDirPath))
            {
                Directory.CreateDirectory(tempDirPath);
            }
        }

        /// <summary>
        ///
        /// </summary>
        static string GetTempFilePath_()
        {
            return Path.Combine(Path.GetTempPath(), "LayoutEditorUserScript");
        }

        /// <summary>
        ///
        /// </summary>
        static string GetUniqueTempFilePath_()
        {
            string path = Path.Combine(GetTempFilePath_(), Path.ChangeExtension(Path.GetRandomFileName(),".txt"));
            while (File.Exists(path))
            {
                path = Path.Combine(GetTempFilePath_(), Path.ChangeExtension(Path.GetRandomFileName(), ".txt"));
            }

            return path;
        }

        /// <summary>
        ///
        /// </summary>
        static bool CallFileIOUserScriptImpl_(string[] filePathSet, string userCommandPath, string notificationTitle)
        {
            if (StopUserScriptCall)
            {
                return true;
            }

            string tempFilePath = GetUniqueTempFilePath_();
            string errorMsg = string.Empty;
            try
            {
                // スクリプト指定がなければ終了
                if (string.IsNullOrEmpty(userCommandPath))
                {
                    return true;
                }

                userCommandPath = SceneHelper.GetAbsolutePathFromPartsRootBasedPath(LayoutEditorCore.Scene, userCommandPath);

                // 指定されたスクリプトが存在しない。
                if (!File.Exists(userCommandPath))
                {
                    // 通知して中断
                    errorMsg = LECoreStringResMgr.Get("USERCOMMAND_CANTFIND", userCommandPath);
                    return false;
                }

                // ファイルリストを作る
                string content = string.Join(Environment.NewLine, filePathSet);
                try
                {
                    TryToCreateTempFileDirectory_(Path.GetDirectoryName(tempFilePath));

                    File.WriteAllText(tempFilePath, content, new UTF8Encoding(false));
                }
                catch (System.Exception ex)
                {
                    errorMsg = LECoreStringResMgr.Get("USERCOMMAND_CANTCREATEFILELIST", ex.ToString());
                    return false;
                }

                // ユーザーコマンドを実行する
                Process proc = new Process();
                try
                {
                    proc.StartInfo.FileName = userCommandPath;
                    proc.StartInfo.Arguments = tempFilePath;
                    proc.StartInfo.UseShellExecute = false;
                    proc.StartInfo.CreateNoWindow = true;
                    proc.StartInfo.RedirectStandardOutput = true;
                    proc.StartInfo.RedirectStandardError = true;
                    proc.Start();

                    proc.ErrorDataReceived += (s, a) =>
                    {
                        if (!string.IsNullOrEmpty(a.Data))
                        {
                            errorMsg += a.Data + Environment.NewLine;
                        }
                    };

                    proc.BeginErrorReadLine();

                    bool finished = proc.WaitForExit(_TimeOutMilliSec);
                    if (!finished)
                    {
                        // タイムアウトした
                        errorMsg = LECoreStringResMgr.Get("USERCOMMAND_TIMEOUT", _TimeOutMilliSec);
                        return false;
                    }
                }
                catch (Exception e)
                {
                    if (proc != null)
                    {
                        try
                        {
                            proc.Kill();
                        }
                        catch
                        {
                            DbgConsole.WriteLine("FileIOUserScriptHandler : Failed to Kill process");
                        }
                    }

                    errorMsg = LECoreStringResMgr.Get("USERCOMMAND_EXCEPTION") + Environment.NewLine + e.ToString();
                    return false;
                }

                if (proc.ExitCode != 0)
                {
                    errorMsg = LECoreStringResMgr.Get("USERCOMMAND_ERROR") + Environment.NewLine + errorMsg;
                    return false;
                }
                else
                {
                    // 成功したら、エラーは一切報告しない。
                    errorMsg = string.Empty;
                }

                // 処理成功！
                return true;
            }
            finally
            {
                // エラーがあれば通知
                if (!string.IsNullOrEmpty(errorMsg))
                {
                    _MessageReporter.ReportError(notificationTitle, errorMsg, 0);
                }

                // 一時ファイルを削除
                if (File.Exists(tempFilePath))
                {
                    try
                    {
                        File.Delete(tempFilePath);
                    }
                    catch (System.Exception fileDeleteEx)
                    {
                        _MessageReporter.ReportError(notificationTitle, fileDeleteEx.ToString(), 0);
                    }
                }
            }
        }
    }

    /// <summary>
    /// 外部DLLからExternした関数郡
    /// </summary>
    internal class DllClass
    {
        [DllImport("user32.dll")]
        public static extern IntPtr GetDC(IntPtr hWnd);
        [DllImport("user32.dll")]
        public static extern IntPtr ReleaseDC(IntPtr hDc);
    }
}
