﻿/*--------------------------------------------------------------------------------*
Copyright (C)Nintendo All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*--------------------------------------------------------------------------------*/

#pragma once

using namespace System;
using namespace System::Collections::Generic;

nn::gfx::util::PrimitiveShapeFormat vertexFormat = static_cast< nn::gfx::util::PrimitiveShapeFormat >(nn::gfx::util::PrimitiveShapeFormat_Pos | nn::gfx::util::PrimitiveShapeFormat_Uv);

namespace Gfx {

    public ref struct PrimitiveShapeVertex
    {
        float   x;
        float   y;
        float   z;
        float   u;
        float   v;
    };

    public ref class PrimitiveShape
    {
    public:
        //--------------------------------------------------------------------------
        //! @brief  PrimitiveShape で作成される頂点データのストライドを取得します。
        //!
        //! @return PrimitiveShape で作成される 1 頂点のサイズをバイト単位のサイズで返します。
        //--------------------------------------------------------------------------
        static int GetVertexStride()
        {
            //  pos : x, y, z
            //  uv  : u, v
            return sizeof(float) * 3 + sizeof(float) * 2;
        }

        //--------------------------------------------------------------------------
        //! @brief  立方体プリミティブを作成します。
        //!
        //! @param[out] vertexBuffer    頂点バッファ。
        //! @param[out] indexBuffer     インデックスバッファ。
        //--------------------------------------------------------------------------
        static void CreateCube([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer);

        //--------------------------------------------------------------------------
        //! @brief  球プリミティブを作成します。
        //!
        //! @param[out] vertexBuffer    頂点バッファ。
        //! @param[out] indexBuffer     インデックスバッファ。
        //! @param[in]  sliceCount      横方向の分割数。
        //! @param[in]  stackCount      縦方向の分割数。
        //--------------------------------------------------------------------------
        static void CreateSphere([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, int sliceCount, int stackCount);

        //--------------------------------------------------------------------------
        //! @brief  半球プリミティブを作成します。
        //!
        //! @param[out] vertexBuffer    頂点バッファ。
        //! @param[out] indexBuffer     インデックスバッファ。
        //! @param[in]  sliceCount      横方向の分割数。
        //--------------------------------------------------------------------------
        static void CreateHemiSphere([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, int sliceCount);

        //--------------------------------------------------------------------------
        //! @brief  円柱プリミティブを作成します。
        //!
        //! @param[out] vertexBuffer    頂点バッファ。
        //! @param[out] indexBuffer     インデックスバッファ。
        //! @param[in]  sliceCount      円周の分割数。
        //--------------------------------------------------------------------------
        static void CreateCylinder([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, int sliceCount);

        //--------------------------------------------------------------------------
        //! @brief  円プリミティブを作成します。
        //!
        //! @param[out] vertexBuffer    頂点バッファ。
        //! @param[out] indexBuffer     インデックスバッファ。
        //! @param[in]  sliceCount      円周方向の分割数。
        //--------------------------------------------------------------------------
        static void CreateCircle([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, int sliceCount);

        //--------------------------------------------------------------------------
        //! @brief  角丸四角形プリミティブを作成します。
        //!
        //! @param[out] vertexBuffer    頂点バッファ。
        //! @param[out] indexBuffer     インデックスバッファ。
        //! @param[in]  width           矩形の幅
        //! @param[in]  height          矩形の高さ
        //! @param[in]  cornerRadiusW   横方向角丸の半径
        //! @param[in]  cornerRadiusH   縦方向角丸の半径
        //! @param[in]  sliceCount      角丸の分割数
        //--------------------------------------------------------------------------
        static void CreateRoundRect([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, float width, float height, float cornerRadiusW, float cornerRadiusH, int sliceCount);
    private:
        //--------------------------------------------------------------------------
        //! @brief  プリミティブ生成の共通処理。
        //!
        //! @param[out] vertexBuffer    頂点バッファ。
        //! @param[out] indexBuffer     インデックスバッファ。
        //! @param[in]  pShape          生成するシェイプデータへのポインタ。
        //! @param[in]  vertexCount     シェイプデータが作成する頂点数。
        //! @param[in]  indexCount      シェイプデータが作成するインデックス数。
        //--------------------------------------------------------------------------
        static void CreateShapeImpl([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, nn::gfx::util::PrimitiveShape* pShape);

        //--------------------------------------------------------------------------
        //! @brief  テクスチャ座標の計算処理。
        //!
        //! @param[in]  vertexX 頂点の X 座標。
        //! @param[in]  halfWidth   全体の幅の半分。
        //!
        //! @return テクスチャの U 座標。
        //--------------------------------------------------------------------------
        static float CalculateTexMapU(float vertexX, float halfWidth)
        {
            return 0.5f + (vertexX / halfWidth * 0.5f);
        }

        //--------------------------------------------------------------------------
        //! @brief  テクスチャ座標の計算処理。
        //!
        //! @param[in]  vertexY 頂点の Y 座標。
        //! @param[in]  halfHeight   全体の高さの半分。
        //!
        //! @return テクスチャの V 座標。
        //--------------------------------------------------------------------------
        static float CalculateTexMapV(float vertexY, float halfHeight)
        {
            return 1.0f - (0.5f + (vertexY / halfHeight * 0.5f));
        }
    };
}
