﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

using namespace System;
using namespace System::Runtime::InteropServices;


#include "TextWriter.h"
#include "WideTextWriter.h"


#include "NW4RFont.h"

#include "TagProcessor.h"
#include "WideTagProcessor.h"

#include "NW4RGlyph.h"




namespace NW4R
{
namespace Font
{

    // 描画を担当するクラスのインタフェース
    // NW4RTextWriter が 描画処理を外部のモジュールにコールバックする際に必要となります。
    // NW4RTextWriter は コンストラクタで本インタフェースを
    // 実装するクラスを引数として要求します。
    public interface class INW4RTextRenderer
    {
    public:
        virtual void DrawChar( NW4RGlyph^ glyph, float x, float y, int index ) = 0;
    };


    // 文字列を描画するクラス。
    // 基本的には、NW4R の TextWriter のラッパーである。
    public ref class NW4RTextWriter : public IDisposable
    {
    private:

       // 文字列描画クラス（アンマネージド）
        // UnManaged::TextWriter*       m_pTextWriter;
       UnManaged::WideTextWriter*   m_pTextWriter;

        // タグ処理を行うクラス（アンマネージド）
        // UnManaged::TagProcessor*     m_pTagProcessor;
       UnManaged::WideTagProcessor*     m_pTagProcessor;
        // 描画に使用するフォント（マネージド）
        NW4RResFont^                m_pFont;
        // 描画処理を委譲するモジュール
        INW4RTextRenderer^          m_pRenderer;


    public:

    	// 文字列描画方法のフラグ
       enum class PositionFlag
       {
           HORIZONTAL_ALIGN_LEFT       = 0x0,
           HORIZONTAL_ALIGN_CENTER     = 0x1,
           HORIZONTAL_ALIGN_RIGHT      = 0x2,
           HORIZONTAL_ALIGN_MASK       = 0x3,

           HORIZONTAL_ORIGIN_LEFT      = 0x00,
           HORIZONTAL_ORIGIN_CENTER    = 0x10,
           HORIZONTAL_ORIGIN_RIGHT     = 0x20,
           HORIZONTAL_ORIGIN_MASK      = 0x30,

           VERTICAL_ORIGIN_TOP         = 0x000,
           VERTICAL_ORIGIN_MIDDLE      = 0x100,
           VERTICAL_ORIGIN_BOTTOM      = 0x200,
           VERTICAL_ORIGIN_MASK        = 0x300,

           VERTICAL_BASE_TOP           = 0x0000,
           VERTICAL_BASE_BASELINE      = 0x1000,
           VERTICAL_BASE_BOTTOM        = 0x2000,
           VERTICAL_BASE_MASK          = 0x3000
       };

        // コンストラクタ
        NW4RTextWriter( INW4RTextRenderer^ pRenderer );

        // デストラクタ
        ~NW4RTextWriter()
        {
            // 外部リソースの解放
            delete m_pTagProcessor;
            delete m_pTextWriter;
        }

        // 描画フラグを設定します。
        void SetDrawFlag( u32 flags )
        {
            m_pTextWriter->SetDrawFlag( flags );
        }


        // フォントの設定
        void SetFont( INWFont^ pFont );

        // フォントの取得
        NW4RResFont^ GetFont();

        // 行間を設定します。
        void SetLineSpace( float space )
        {
           m_pTextWriter->SetLineSpace( space );
        }

        // 行の高さを指定します。
        void SetLineHeight( float height )
        {
           m_pTextWriter->SetLineHeight( height );
        }

        // 文字間を設定します。
        void SetCharSpace( float space )
        {
           m_pTextWriter->SetCharSpace( space );
        }

        // 文字間を取得します。
        float GetCharSpace()
        {
            return m_pTextWriter->GetCharSpace();
        }

        // 文字列の幅を計算します
        float CalcStringWidth
        (
           String^     str,
           int         length
        );

        // 文字列の高さを計算します。
        float CalcStringHeight( String^ str, int length );

        // 文字のスケールを設定します。
        void SetScale( float hScale, float vScale );

        // 文字の大きさを指定します。
        void SetFontSize( float width, float height );

        // カーソル位置を設定します。
        void SetCursor( float x, float y );

        // TagProcessor をリセットします。
        void ResetTagProcessor(bool isExtendedTagEnabled);

        // 文字の描画
        // TODO:複数のAPI
        // 可変長引数は、どのようにあつかうのか？？？
        // とりあえず不要であるので、実装しません。
        float  Print( String^ str, array<float>^ lineOffset );

        // 水平方向のスケールを取得します
        float GetScaleH(){ return m_pTextWriter->GetScaleH(); }
        // 垂直方向のスケールを取得します
        float GetScaleV(){ return m_pTextWriter->GetScaleV(); }

        // 描画フラグを設定します。
        void SetWidthLimit(float limit )
        {
            m_pTextWriter->SetWidthLimit( limit );
        }

        // 自動折り返し機能を無効に設定します。
        void ResetWidthLimit()
        {
            m_pTextWriter->ResetWidthLimit();
        }

        // 右側から描画するかどうかを設定します。
        void SetDrawFromRightToLeftEnabled(bool isDrawFromRightToLeftEnabled)
        {
            m_pTextWriter->SetDrawFromRightToLeftEnabled(isDrawFromRightToLeftEnabled);
        }

    protected:

    };

}
}
