﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// これは メイン DLL ファイルです。
#include "stdafx.h"
#include <malloc.h>
#include <new>

#define NW4R_FROM_TOOL
#include <nw/types.h>
#include "NW4RFont.h"
#include "FontEnvironment.h"

namespace NW4R
{
namespace Font
{
        /*---------------------------------------------------------------------------*
           Name:         NW4RResFont::GetSheetPropaties()

           Description:  各グリフが参照するフォントイメージを表すテクスチャデータの配列を返します。

           Arguments:    なし。

           Returns:      テクスチャデータの配列。
        *---------------------------------------------------------------------------*/
        array<INWFont::SheetPropaties^>^
            NW4RResFont::GetSheetPropaties()
        {
            if(m_pSheetPropSet != nullptr)
            {
                return m_pSheetPropSet;
            }

            // SheetPropaties を格納する配列を初期化しています。
            const UnManaged::FontTextureGlyph* pGlyph = m_pResFont->GetFontTextureGlyph();
            array<INWFont::SheetPropaties^>^ sheetPropSet = gcnew array<INWFont::SheetPropaties^>(pGlyph->sheetNum);

            // 存在するsheet分だけデータを読み取ります。
            for( int i = 0;i < sheetPropSet->Length; i++ )
            {
                // 確実に 64bit に対応できるように、pGlyph->sheetImage を使わず、RgbaSheetImageBase を使います。
                char*      pImage = (char*)m_pResFont->GetRgbaSheetImageBase() + pGlyph->sheetSize * i ;

                sheetPropSet[i] = gcnew INWFont::SheetPropaties;

                sheetPropSet[i]->texWidth    = pGlyph->sheetWidth;
                sheetPropSet[i]->texHeight   = pGlyph->sheetHeight;
                sheetPropSet[i]->imgPtr      = (IntPtr)pImage;
            }

            m_pSheetPropSet = sheetPropSet;

            return m_pSheetPropSet;
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RResFont::GetUnManagedFont()

           Description:  アンマネージドフォントを取得します。
                         NW4RTextWriter から利用されます。

           Arguments:    なし。

           Returns:      テクスチャデータの配列。
        *---------------------------------------------------------------------------*/
        UnManaged::Font*
        NW4RResFont::GetUnManagedFont()
        {
           return m_pResFont;
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RResFont::NW4RResFont()

           Description:  コンストラクタ。

           Arguments:    なし。

           Returns:      なし。
        *---------------------------------------------------------------------------*/
        NW4RResFont::NW4RResFont()
            :m_pSheetPropSet(nullptr)
            , m_pResFont(nullptr)
            , m_intPtrToResourceDataCopy(IntPtr::Zero)
        {
           m_pResFont = nullptr;
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RResFont::~NW4RResFont()

           Description:  デストラクタ。

           Arguments:    なし。

           Returns:      なし。
        *---------------------------------------------------------------------------*/
        NW4RResFont::~NW4RResFont()
        {
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RResFont::InitializeAtTheFirstTimeDraw()

           Description:  初回描画時に必要なら内部状態を初期化します。
        *---------------------------------------------------------------------------*/
        void NW4RResFont::InitializeAtTheFirstTimeDraw()
        {
            if(!m_pResFont->IsReadyToDraw() && !m_pResFont->IsInvalid())
            {
                try
                {
                    System::IntPtr pNewRGBAImage = mPlatformUtility->BuildNewTextureRGBAImage((System::IntPtr)m_pResFont->GetFINF(), (System::IntPtr)m_pResFont->GetFilePtrOffsetBase());
                    m_pResFont->InitializeRGBAImage((void*)pNewRGBAImage);
                }
                catch(System::Exception^ e)
                {
                    // 初期化を何度も試行しないように、NULL 以外の適当な値を設定しておきます。
                    m_pResFont->InitializeRGBAImage((void*)UnManaged::detail::ResFontBase::INVALID_IMAGE_PTR);
                    throw e;
                }
            }
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RResFont::SetResource( Byte[] )

           Description:  ファイルロード結果バイト配列から、データを設定します。

           Arguments:    ファイルロード結果へのポインタ。

           Returns:      初期化に成功したら、true。
        *---------------------------------------------------------------------------*/
        // ファイルロード結果バイト配列から、データを設定します。
        bool NW4RResFont::SetResource(array<Byte>^ byteArray, ITextureConvertUtility^ platformDetail, bool isEndianSwapEnabled, Object^ paramater)
        {
           NW_UNUSED_VARIABLE(paramater);
           try
           {
               // DLL を選択する。
               mPlatformUtility = platformDetail;

               // アンマネージド側にコピーを生成し、そのコピーを使用します。
               m_intPtrToResourceDataCopy = Marshal::AllocCoTaskMem( byteArray->Length );
               Marshal::Copy( byteArray , 0, m_intPtrToResourceDataCopy, byteArray->Length );
               void* pBrfnt = static_cast<void*>(m_intPtrToResourceDataCopy);

               // TODO : 種類を判定して適切な実態クラスを生成する。
               bool bResult = false;
               if( m_pResFont == NULL )
               {
                    UnManaged::ResFont*  pNewFont = new UnManaged::ResFont;

                    bResult = pNewFont->SetResource( pBrfnt, isEndianSwapEnabled);

                    m_pResFont = pNewFont;
               }

               return bResult;
           }catch( std::bad_alloc e ){
               (void)e;
               if (m_pResFont != NULL)
               {
                   delete m_pResFont;
                   m_pResFont = NULL;
               }

               if (m_intPtrToResourceDataCopy != IntPtr::Zero)
               {
                   Marshal::FreeCoTaskMem(m_intPtrToResourceDataCopy);
                   m_intPtrToResourceDataCopy = IntPtr::Zero;
               }


               return false;
           }
        }

        bool NW4RResFont::SetResources( array<array<Byte>^>^ byteArrays, ITextureConvertUtility^ platformDetail, bool isEndianSwapEnabled, Object^ paramater )
        {
            return SetResource(byteArrays[0], platformDetail, isEndianSwapEnabled, paramater);
        }

        /*---------------------------------------------------------------------------*
        *---------------------------------------------------------------------------*/
        int NW4RResFont::ValidateBinary(array<Byte>^ byteArray, System::UInt32 correctResourceVersion)
        {
            try
                {
                    int headerSize = sizeof(UnManaged::BinaryFileHeader);

                    // アンマネージド側にコピーを生成し、そのコピーを使用します。
                    IntPtr copyData = Marshal::AllocCoTaskMem( headerSize );
                    Marshal::Copy( byteArray , 0, copyData, headerSize );
                    void* pBrfnt = static_cast<void*>( copyData);

                    UnManaged::BinaryFileHeader* fileHeader =
                        reinterpret_cast<UnManaged::BinaryFileHeader*>( pBrfnt);

                    UnManaged::BinaryFileValidateCode code;
                    code = UnManaged::ValidateBinaryFile(
                        fileHeader,
                        UnManaged::BINFILE_SIG_FONT,
                        UnManaged::BINFILE_SIG_FONTA,
                        (u32)correctResourceVersion,
                        2);

                    Marshal::FreeCoTaskMem( copyData);
                    return code;
                }

            catch( std::bad_alloc e ){
                (void)e;
                return UnManaged::UnknownInvalid;
            }
        }

        /*---------------------------------------------------------------------------*
           Name:         NW4RResFont::GetPlatform()

           Description:  フォントファイル中の bntx に記述されているプラットフォームの情報を取得します。

           Returns:      フォントファイル中の bntx に記述されているプラットフォームの情報。
           　　　　　　　取得に失敗したら nullptr を返す。
        *---------------------------------------------------------------------------*/
        System::String^ NW4RResFont::GetPlatform()
        {
            return mPlatformUtility->GetPlatform((System::IntPtr)m_pResFont->GetFINF(), (System::IntPtr)m_pResFont->GetFilePtrOffsetBase());
        }



        /*---------------------------------------------------------------------------*
           Name:         NW4RResFont::SetResource( Byte[] )

           Description:  オブジェクトからフォントリソースを切り離します。

           Arguments:    なし。

           Returns:      なし。
        *---------------------------------------------------------------------------*/
        // リソース解除
        void NW4RResFont::RemoveResource()
        {
           // ResFontか調査して、開放処理。
           UnManaged::ResFont* pResFont = dynamic_cast<UnManaged::ResFont*>( m_pResFont );
           if( pResFont != NULL )
           {
                pResFont->RemoveResource();
           }

           // リソースを開放。
           if( m_intPtrToResourceDataCopy != IntPtr::Zero )
           {
                // アンマネージド側にコピーを破棄します。
                Marshal::FreeCoTaskMem( m_intPtrToResourceDataCopy );
                m_intPtrToResourceDataCopy = IntPtr::Zero;
           }

           // アンマネージフォントインスタンス自身を解放
           delete( m_pResFont );
           m_pResFont = nullptr;
        }

       int NW4RResFont::GetHeight()
       {
           return m_pResFont->GetHeight();
       }

       int NW4RResFont::GetWidth()
       {
           return m_pResFont->GetWidth();
       }

       /*---------------------------------------------------------------------------*
           Description:  カーニングを持っているか
        *---------------------------------------------------------------------------*/
       bool NW4RResFont::HasKerning()
       {
           return m_pResFont->HasKerning();
       }

        /*---------------------------------------------------------------------------*
           Description:  パック形式か取得します。
        *---------------------------------------------------------------------------*/
       bool NW4RResFont::IsPackedFormatFont()
       {
           return m_pResFont->GetFontType() == 2;
       }

       /*---------------------------------------------------------------------------*
           Description:  特殊フチ形式が設定可能かを取得します。
        *---------------------------------------------------------------------------*/
       bool NW4RResFont::IsAvailableBorder()
       {
           return m_pResFont->GetFontType() == 2;
       }

       /*---------------------------------------------------------------------------*
           Description:  SRGBフェッチが必要な形式か取得します。
        *---------------------------------------------------------------------------*/
       bool NW4RResFont::IsNeededToSrgbFetch()
       {
           return mPlatformUtility->IsNeededToSrgbFetchFormat((System::IntPtr)(UnManaged::FontTextureGlyph*)m_pResFont->GetFontTextureGlyph());
       }

       /*---------------------------------------------------------------------------*
           Description:  準備完了か取得します。
        *---------------------------------------------------------------------------*/
       bool NW4RResFont::IsReadyToDraw()
       {
           return m_pResFont != NULL && m_pResFont->IsReadyToDraw();
       }

       /*---------------------------------------------------------------------------*
           Name:         NW4RResFont::GetEncoding()

           Description:  フォントのエンコーディング種類を取得します。

           Arguments:    なし。

           Returns:      フォントのエンコーディング種類。
        *---------------------------------------------------------------------------*/
       NW4RResFont::FontEncoding   NW4RResFont::GetEncoding()
       {
           switch( m_pResFont->GetEncoding() )
           {
                case NW4R::Font::UnManaged::FONT_ENCODING_UTF8: return NW4RResFont::FontEncoding::FONT_ENCODING_UTF8;
                case NW4R::Font::UnManaged::FONT_ENCODING_UTF16: return NW4RResFont::FontEncoding::FONT_ENCODING_UTF16;
                case NW4R::Font::UnManaged::FONT_ENCODING_SJIS: return NW4RResFont::FontEncoding::FONT_ENCODING_SJIS;
                case NW4R::Font::UnManaged::FONT_ENCODING_CP1252: return NW4RResFont::FontEncoding::FONT_ENCODING_CP1252;
                case NW4R::Font::UnManaged::NUM_OF_FONT_ENCODING: return NW4RResFont::FontEncoding::NUM_OF_FONT_ENCODING;
                default: return NW4RResFont::FontEncoding::NUM_OF_FONT_ENCODING;
           }
       }

       // 使用文字コード範囲のセットを設定します。
       void NW4RResFont::SetCharCodeRange(array<u32>^ first, array<u32>^ last)
       {
           pin_ptr<u32> pFirst = &first[0];
           pin_ptr<u32> pLast = &last[0];
           m_pResFont->SetCharCodeRange(first->Length, pFirst, pLast);
       }
}
}
