﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW4R_FROM_TOOL
#include <windows.h>

#include <nw/types.h>
#include "nw4r/misc.h"
#include "CharWriter.h"

namespace NW4R
{
namespace Font
{
namespace UnManaged
{




/* ------------------------------------------------------------------------
        クラス変数定義
   ------------------------------------------------------------------------ */

CharWriter::LoadingTexture CharWriter::mLoadingTexture;



/* =======================================================================
        public
   ======================================================================== */

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::CharWriter()

  Description:  コンストラクタ。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
CharWriter::CharWriter()
: mAlpha(Color::ALPHA_MAX),
  mIsWidthFixed(false),
  mFixedWidth(0),
  mFont(NULL)
{
    mLoadingTexture.Reset();            // mLoadingTexture
    ResetColorMapping();                // mColorMapping
    SetGradationMode(GRADMODE_NONE);    // mTextColor, mVertexColor
    SetTextColor(Color::WHITE);         // mTextColor, mVertexColor
    SetScale(1, 1);                     // mScale
//u SetCursor(0, 0);                    // mCursorPos
    SetCursor(0, 0, 0);                 // mCursorPos
    EnableLinearFilter(true, true);     // mFilter

    // 登録されているタスクをすべて解放します。(追加処理)
    ClearRenderingTask();
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::~CharWriter()

  Description:  デストラクタ。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
CharWriter::~CharWriter()
{
}



/* ------------------------------------------------------------------------
        フォント
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetFont( const Font& )

  Description:  文字描画に用いるフォントを設定します。

  Arguments:    font:   新しいフォント。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetFont( const Font& font )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_REFERENCE_ASSERT( font );
    mFont = &font;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetFont()

  Description:  文字描画に用いているフォントを取得します。

  Arguments:    なし。

  Returns:      文字描画に用いているフォントへのポインタ。
                フォントが設定されていなければ NULL を返します。
 *---------------------------------------------------------------------------*/
const Font*
CharWriter::GetFont() const
{
    NW4R_POINTER_ASSERT( this );
    return mFont;
}



/* ------------------------------------------------------------------------
        描画準備
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetupGX()

  Description:  グラフィックスエンジンを文字描画用に設定します。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetupGX()
{
    NW4R_POINTER_ASSERT( this );

    ResetTextureCache();

    if( (mColorMapping.min == DEFAULT_COLOR_MAPPING_MIN)
        && (mColorMapping.max == DEFAULT_COLOR_MAPPING_MAX) )
    {
        SetupGXWithColorMapping(mColorMapping.min, mColorMapping.max);
    }
    else
    {
        if( mFont )
        {
            GXTexFmt format = mFont->GetTextureFormat();

            switch( format )
            {
            case GX_TF_I4:
            case GX_TF_I8:
                SetupGXForI();
                break;

            case GX_TF_IA4:
            case GX_TF_IA8:
                SetupGXDefault();
                break;

            case GX_TF_RGB565:
            case GX_TF_RGB5A3:
            case GX_TF_RGBA8:
                SetupGXForRGBA();
                break;

            default:
                SetupGXDefault();
                break;
            }
        }
        else
        {
            SetupGXDefault();
        }
    }
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetColorMapping( Color, Color )

  Description:  描画する文字のテクスチャカラーの線形変換を設定します。

  Arguments:    min:    最小値のカラーの変換後のカラーを指定します。
                max:    最大値のカラーの変換後のカラーを指定します。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetColorMapping(
    Color min,
    Color max
)
{
    NW4R_POINTER_ASSERT( this );
    mColorMapping.min = min;
    mColorMapping.max = max;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetColorMappingMin()

  Description:  最小値のテクスチャカラーの線形変換後のカラーを取得します。

  Arguments:    なし。

  Returns:      最小値のテクスチャカラーの線形変換後のカラー。
 *---------------------------------------------------------------------------*/
const Color
CharWriter::GetColorMappingMin() const
{
    NW4R_POINTER_ASSERT( this );
    return mColorMapping.min;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetColorMappingMax()

  Description:  最大値のテクスチャカラーの線形変換後のカラーを取得します。

  Arguments:    なし。

  Returns:      最大値のテクスチャカラーの線形変換後のカラー。
 *---------------------------------------------------------------------------*/
const Color
CharWriter::GetColorMappingMax() const
{
    NW4R_POINTER_ASSERT( this );
    return mColorMapping.max;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::ResetColorMapping()

  Description:  文字描画時のテクスチャカラーの線形変換を変換なしにします。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::ResetColorMapping()
{
    NW4R_POINTER_ASSERT( this );
    SetColorMapping(DEFAULT_COLOR_MAPPING_MIN, DEFAULT_COLOR_MAPPING_MAX);
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::ResetTextureCache()

  Description:  グリフテクスチャシートのロードキャッシュをリセットします。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::ResetTextureCache()
{
    NW4R_POINTER_ASSERT( this );
    mLoadingTexture.Reset();
}



/* ------------------------------------------------------------------------
        文字色
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetAlpha( u8 )

  Description:  文字描画時の追加不透明度を設定します。

  Arguments:    alpha:  新しい不透明度。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetAlpha( u8 alpha )
{
    NW4R_POINTER_ASSERT( this );
    mAlpha = alpha;
    UpdateVertexColor();
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetAlpha()

  Description:  文字描画時の追加不透明度を取得します。

  Arguments:    なし。

  Returns:      文字描画時の不透明度。
 *---------------------------------------------------------------------------*/
u8
CharWriter::GetAlpha() const
{
    NW4R_POINTER_ASSERT( this );
    return mAlpha;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetGradationMode( GradationMode )

  Description:  文字描画時のグラデーションモードを指定します。

  Arguments:    mode:   新しいグラデーションモード

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetGradationMode( GradationMode mode )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_MINMAX_ASSERT( mode, GRADMODE_NONE, NUM_OF_GRADMODE - 1 );
    mTextColor.gradationMode = mode;
    UpdateVertexColor();
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetGradationMode()

  Description:  文字描画時のグラデーションモードを取得します。

  Arguments:    なし。

  Returns:      文字描画時のグラデーションモード。
 *---------------------------------------------------------------------------*/
//typename CharWriter::GradationMode
CharWriter::GradationMode
CharWriter::GetGradationMode() const
{
    NW4R_POINTER_ASSERT( this );
    return mTextColor.gradationMode;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetTextColor( Color )

  Description:  文字描画時の追加色を指定します。

  Arguments:    color:  新しい追加色。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetTextColor( Color color )
{
    NW4R_POINTER_ASSERT( this );
    mTextColor.start = color;
    UpdateVertexColor();
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetTextColor( Color, Color )

  Description:  文字描画時のグラデーションカラーを指定します。

  Arguments:    start:  グラデーション開始色。
                end:    グラデーション終了色。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetTextColor(
    Color start,
    Color end
)
{
    NW4R_POINTER_ASSERT( this );
    mTextColor.start = start;
    mTextColor.end   = end;
    UpdateVertexColor();
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetTextColor()

  Description:  文字描画時の追加色を取得します。

  Arguments:    なし。

  Returns:      文字描画時の追加色。
 *---------------------------------------------------------------------------*/
const Color
CharWriter::GetTextColor() const
{
    NW4R_POINTER_ASSERT( this );
    return mTextColor.start;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetGradationStartColor()

  Description:  文字描画時のグラデーション開始色を取得します。

  Arguments:    なし。

  Returns:      文字描画時のグラデーション開始色。
 *---------------------------------------------------------------------------*/
const Color
CharWriter::GetGradationStartColor() const
{
    NW4R_POINTER_ASSERT( this );
    return mTextColor.start;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetGradationEndColor()

  Description:  文字描画時のグラデーション終了色。

  Arguments:    なし。

  Returns:      文字描画時のグラデーション終了色。
 *---------------------------------------------------------------------------*/
const Color
CharWriter::GetGradationEndColor() const
{
    NW4R_POINTER_ASSERT( this );
    return mTextColor.end;
}



/* ------------------------------------------------------------------------
        文字サイズ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetScale( float, float )

  Description:  文字描画時の拡大率を設定します。
                水平方向と垂直方向で異なる値を指定します。

  Arguments:    hScale: 水平方向拡大率。
                vScale: 垂直方向拡大率。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetScale(
    float hScale,
    float vScale
)
{
    NW4R_POINTER_ASSERT( this );
    mScale.x = hScale;
    mScale.y = vScale;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetScale( float )

  Description:  文字描画時の拡大率を設定します。
                水平方向と垂直方向で同じ値を指定します。

  Arguments:    scale:  拡大率。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetScale( float scale )
{
    NW4R_POINTER_ASSERT( this );
    mScale.x = scale;
    mScale.y = scale;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetScaleH()

  Description:  文字描画時の水平方向拡大率を取得します。

  Arguments:    なし。

  Returns:      文字描画時の水平方向拡大率。
 *---------------------------------------------------------------------------*/
float
CharWriter::GetScaleH() const
{
    NW4R_POINTER_ASSERT( this );
    return mScale.x;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetScaleV()

  Description:  文字描画時の垂直方向拡大率を取得します。

  Arguments:    なし。

  Returns:      文字描画時の垂直方向拡大率。
 *---------------------------------------------------------------------------*/
float
CharWriter::GetScaleV() const
{
    NW4R_POINTER_ASSERT( this );
    return mScale.y;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetFontSize( float, float )

  Description:  文字描画時の拡大率を拡大後のサイズで指定します。
                縦横別々に指定します。

  Arguments:    width:  拡大後のセルの幅。
                height: 拡大後のセルの高さ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetFontSize(
    float width,
    float height
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFont );
    NW4R_MIN_ASSERT( mFont->GetWidth(),  1 );
    NW4R_MIN_ASSERT( mFont->GetHeight(), 1 );
    SetScale(
        width  / mFont->GetWidth(),
        height / mFont->GetHeight()
    );
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetFontSize( float )

  Description:  文字描画時の拡大率を拡大後のサイズで指定します。
                文字の高さで指定します。

  Arguments:    height: 拡大後のセルの高さ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetFontSize( float height )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFont );
    NW4R_MIN_ASSERT( mFont->GetHeight(), 1 );
    const f32 scale = height / mFont->GetHeight();
    SetScale( scale );
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetFontWidth()

  Description:  文字描画時のセルの幅を取得します。

  Arguments:    なし。

  Returns:      文字描画時のセルの幅。
 *---------------------------------------------------------------------------*/
float
CharWriter::GetFontWidth() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFont );
    return mFont->GetWidth() * mScale.x;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetFontHeight()

  Description:  文字描画時のセルの高さを取得します。

  Arguments:    なし。

  Returns:      文字描画時のセルの高さ。
 *---------------------------------------------------------------------------*/
float
CharWriter::GetFontHeight() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFont );
    return mFont->GetHeight() * mScale.y;
}


/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetFontAscent()

  Description:  文字描画時のアセントを取得します。

  Arguments:    なし。

  Returns:      文字描画時のアセント。
 *---------------------------------------------------------------------------*/
f32
CharWriter::GetFontAscent() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFont );
    return mFont->GetAscent() * mScale.y;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetFontDescent()

  Description:  文字描画時のディセントを取得します。

  Arguments:    なし。

  Returns:      文字描画時のディセント。
 *---------------------------------------------------------------------------*/
f32
CharWriter::GetFontDescent() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFont );
    return mFont->GetDescent() * mScale.y;
}


/* ------------------------------------------------------------------------
        テクスチャフィルタ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::EnableLinearFilter( bool, bool )

  Description:  文字描画時にテクスチャの線形補間を行うかどうかを指定します。

  Arguments:    atSmall:    文字が縮小表示されるときに線形補間を行うなら
                            true を指定します。
                atLarge:    文字が拡大表示されるときに線形補間を行うなら
                            true を指定します。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::EnableLinearFilter(
    bool atSmall,
    bool atLarge
)
{
    NW4R_POINTER_ASSERT( this );
    mFilter.atSmall = atSmall ? GX_LINEAR: GX_NEAR;
    mFilter.atLarge = atLarge ? GX_LINEAR: GX_NEAR;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::IsLinearFilterEnableAtSmall()

  Description:  文字の縮小描画時にテクスチャの線形補間が行われるかどうかを
                取得します。

  Arguments:    なし。

  Returns:      線形補間が行われるなら true。
 *---------------------------------------------------------------------------*/
bool
CharWriter::IsLinearFilterEnableAtSmall() const
{
    NW4R_POINTER_ASSERT( this );
    return (mFilter.atSmall == GX_LINEAR);
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::IsLinearFilterEnableAtLarge()

  Description:  文字の拡大描画時にテクスチャの線形補間が行われるかどうかを
                取得します。

  Arguments:    なし。

  Returns:      線形補間が行われるなら true。
 *---------------------------------------------------------------------------*/
bool
CharWriter::IsLinearFilterEnableAtLarge() const
{
    NW4R_POINTER_ASSERT( this );
    return (mFilter.atLarge == GX_LINEAR);
}



/* ------------------------------------------------------------------------
        等幅
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::IsWidthFixed()

  Description:  文字描画時に強制等幅描画が行われるかどうかを取得します。

  Arguments:    なし。

  Returns:      強制等幅描画が行われるなら true。
 *---------------------------------------------------------------------------*/
bool
CharWriter::IsWidthFixed() const
{
    NW4R_POINTER_ASSERT( this );
    return mIsWidthFixed;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::EnableFixedWidth( bool )

  Description:  文字描画時に強制等幅描画を行うかどうかを指定します。

  Arguments:    isFixed:    強制等幅描画を行うなら true。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::EnableFixedWidth(bool isFixed)
{
    mIsWidthFixed = isFixed;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetFixedWidth()

  Description:  強制等幅描画時の文字幅を取得します。

  Arguments:    なし。

  Returns:      強制等幅描画時の文字幅。
 *---------------------------------------------------------------------------*/
float
CharWriter::GetFixedWidth() const
{
    NW4R_POINTER_ASSERT( this );
    return mFixedWidth;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetFixedWidth( float )

  Description:  強制等幅描画時の文字幅を設定します。

  Arguments:    width:  新しい文字幅。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetFixedWidth(float width)
{
    NW4R_POINTER_ASSERT( this );
    mFixedWidth = width;
}




/* ------------------------------------------------------------------------
        文字描画
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::Print( CharCode )

  Description:  文字を描画します。

  Arguments:    code:   描画する文字の文字コード。

  Returns:      描画した文字の幅。
 *---------------------------------------------------------------------------*/
float
CharWriter::Print( CharCode code )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFont );
    NW4R_ASSERT( code != Font::INVALID_CHARACTER_CODE );
    Glyph glyph;
    float width;
    float left;

    mFont->GetGlyph(&glyph, code);

    Glyph::GlyphCharWidths &widths = glyph.widths;

    if( mIsWidthFixed )
    {
        float margin = (mFixedWidth - widths.charWidth * mScale.x) / 2;

        width   = mFixedWidth;
        left    = margin + widths.left * mScale.x;
    }
    else
    {
        width   = widths.charWidth * mScale.x;
        left    = widths.left      * mScale.x;
    }

    PrintGlyph(mCursorPos.x + left, mCursorPos.y, mCursorPos.z, glyph);

    mCursorPos.x += width;

    return width;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::Print( const Glyph& )

  Description:  グリフを描画します。

  Arguments:    glyph:  グリフデータ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::Print( const Glyph& glyph )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_REFERENCE_ASSERT( glyph );

    PrintGlyph( mCursorPos.x, mCursorPos.y, mCursorPos.z, glyph );

    mCursorPos.x += glyph.widths.glyphWidth * mScale.x;
}


/* ------------------------------------------------------------------------
        カーソル
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetCursor( float, float )

  Description:  カーソルを移動します。

  Arguments:    x:      新しいカーソルの座標 x
                y:      新しいカーソルの座標 y

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetCursor(
    float x,
    float y
)
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.x = x;
    mCursorPos.y = y;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetCursor( float, float, float )

  Description:  カーソルを移動します。

  Arguments:    x:      新しいカーソルの座標 x
                y:      新しいカーソルの座標 y
                z:      新しいカーソルの座標 z

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetCursor(
    float x,
    float y,
    float z
)
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.x = x;
    mCursorPos.y = y;
    mCursorPos.z = z;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::MoveCursor( float, float )

  Description:  カーソルの正方向に進めます。

  Arguments:    dx: カーソルの移動量
                dx: カーソルの移動量

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::MoveCursor(
    float dx,
    float dy
)
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.x += dx;
    mCursorPos.y += dy;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::MoveCursor( float, float, float )

  Description:  カーソルの正方向に進めます。

  Arguments:    dx: カーソルの移動量
                dx: カーソルの移動量
                dz: カーソルの移動量

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::MoveCursor(
    float dx,
    float dy,
    float dz
)
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.x += dx;
    mCursorPos.y += dy;
    mCursorPos.z += dz;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetCursorX( float )

  Description:  カーソルのX座標を設定します。

  Arguments:    x:  新しいカーソルのX座標。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetCursorX( float x )
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.x = x;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetCursorY( float )

  Description:  カーソルのY座標を設定します。

  Arguments:    y:  新しいカーソルのY座標。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetCursorY( float y )
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.y = y;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetCursorZ( float )

  Description:  カーソルのZ座標を設定します。

  Arguments:    z:  新しいカーソルのZ座標。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetCursorZ( float z )
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.z = z;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::MoveCursorX( float )

  Description:  カーソルのX座標を正方向に進めます。

  Arguments:    dx: カーソルの移動量

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::MoveCursorX( float dx )
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.x += dx;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::MoveCursorY( float )

  Description:  カーソルのY座標を正方向に進めます。

  Arguments:    dy: カーソルの移動量

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::MoveCursorY( float dy )
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.y += dy;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::MoveCursorZ( float )

  Description:  カーソルのZ座標を正方向に進めます。

  Arguments:    dz: カーソルの移動量

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::MoveCursorZ( float dz )
{
    NW4R_POINTER_ASSERT( this );
    mCursorPos.z += dz;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetCursorX()

  Description:  カーソルのX座標を取得します。

  Arguments:    なし。

  Returns:      カーソルのX座標。
 *---------------------------------------------------------------------------*/
float
CharWriter::GetCursorX() const
{
    NW4R_POINTER_ASSERT( this );
    return mCursorPos.x;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetCursorY()

  Description:  カーソルのY座標を取得します。

  Arguments:    なし。

  Returns:      カーソルのY座標。
 *---------------------------------------------------------------------------*/
float
CharWriter::GetCursorY() const
{
    NW4R_POINTER_ASSERT( this );
    return mCursorPos.y;
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetCursorZ()

  Description:  カーソルのZ座標を取得します。

  Arguments:    なし。

  Returns:      カーソルのY座標。
 *---------------------------------------------------------------------------*/
float
CharWriter::GetCursorZ() const
{
    NW4R_POINTER_ASSERT( this );
    return mCursorPos.z;
}


/*---------------------------------------------------------------------------*
  Name:         CharWriter::GetTaskArray()

  Description:  文字描画タスク配列を取得します。

  Arguments:    なし。

  Returns:      文字描画タスク配列
 *---------------------------------------------------------------------------*/
const CharWriter::RenderingTaskPtrArray&
CharWriter::GetTaskArray() const
{
   return mTaskArray_;
}


/*---------------------------------------------------------------------------*
  Name:         CharWriter::ClearRenderingTask()

  Description:  文字描画タスク配列をクリアします。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::ClearRenderingTask()
{
   RenderingTaskPtrArray::iterator      it;

   for( it = mTaskArray_.begin(); it != mTaskArray_.end(); it++ )
   {
       delete (*it);
   }

   mTaskArray_.clear();
}



/* =======================================================================
        private
   ======================================================================== */


/*---------------------------------------------------------------------------*
  Name:         CharWriter::PrintGlyph( float, float, const Glyph& )

  Description:  グリフを指定位置に描画します。
                カーソル移動は行いません。

  Arguments:    x:      描画位置のX座標。
                y:      描画位置のY座標。
                glyph:  描画するグリフデータ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::PrintGlyph(
    float           x,
    float           y,
    float           z,
    const Glyph&    glyph
)
{
    //NW4R_POINTER_ASSERT( this );
    //NW4R_REFERENCE_ASSERT( glyph );
    //NW4R_MIN_ASSERT( glyph.texWidth, 1 );
    //NW4R_MIN_ASSERT( glyph.texHeight, 1 );
    //const float posLeft   = x;
    //const float posTop    = y;
    //const float posRight  = posLeft + glyph.widths.glyphWidth * mScale.x;
    //const float posBottom = posTop  + glyph.height            * mScale.y;
    //const float posZ      = 0;
    //const u16   texLeft   = static_cast<u16>( glyph.cellX
    //                        * TEXTURE_COODINATE_ONE / glyph.texWidth );
    //const u16   texTop    = static_cast<u16>( glyph.cellY
    //                        * TEXTURE_COODINATE_ONE / glyph.texHeight );
    //const u16   texRight  = static_cast<u16>( (glyph.cellX + glyph.widths.glyphWidth)
    //                        * TEXTURE_COODINATE_ONE / glyph.texWidth );
    //const u16   texBottom = static_cast<u16>( (glyph.cellY + glyph.height)
    //                        * TEXTURE_COODINATE_ONE / glyph.texHeight );

    //LoadTexture(glyph, GX_TEXMAP0);

    //GXBegin( GX_QUADS, GX_VTXFMT0, 4 );
    //{
    //    GXPosition3f32( posLeft , posTop,    posZ );
    //    GXColor1u32( mVertexColor.lu );
    //    GXTexCoord2u16( texLeft, texTop );

    //    GXPosition3f32( posRight, posTop,    posZ );
    //    GXColor1u32( mVertexColor.ru );
    //    GXTexCoord2u16( texRight, texTop );

    //    GXPosition3f32( posRight, posBottom, posZ );
    //    GXColor1u32( mVertexColor.rd );
    //    GXTexCoord2u16( texRight, texBottom );

    //    GXPosition3f32( posLeft,  posBottom, posZ );
    //    GXColor1u32( mVertexColor.ld );
    //    GXTexCoord2u16( texLeft, texBottom );
    //}
    //GXEnd();

    //
    // 注意：処理が大きく変更されています。
    // ランタイム版では、実際にGX関数群を使用して、HWに描画処理を発行しています。
    // .NET版では、具体的な描画はマネージドコード側に委譲したいので、
    // ここでは、描画に必要なパラメータをタスクバッファに格納するだけとなっています。
    //

    // タスクを生成して、ベクタに登録します。
    CharRenderingTask* pNewTask = new CharRenderingTask;

    pNewTask->x                 = x;
    pNewTask->y                 = y;
    pNewTask->z                 = z;
    // pNewTask->glyph             = glyph;
    memcpy( &pNewTask->glyph, &glyph, sizeof( Glyph ) );

    mTaskArray_.push_back( pNewTask );
}





/* ------------------------------------------------------------------------
        内部処理
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::LoadTexture( const Glyph&, GXTexMapID )

  Description:  グリフデータの持つテクスチャを指定スロットにロードします。

  Arguments:    glyph:  ロードすべきテクスチャ情報を持つグリフデータ。
                slot:   ロード先のスロット。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::LoadTexture(
    const Glyph&    glyph,
    GXTexMapID      slot
)
{
    NW_UNUSED_VARIABLE(glyph);
    NW_UNUSED_VARIABLE(slot);
    //NW4R_POINTER_ASSERT( this );
    //NW4R_REFERENCE_ASSERT( glyph );
    //LoadingTexture loadInfo;

    //loadInfo.slot = slot;
    //loadInfo.texture = glyph.pTexture;
    //loadInfo.filter = mFilter;

    //if( loadInfo != mLoadingTexture )
    //{
    //    GXTexObj tobj;

    //    GXInitTexObj(
    //        &tobj,
    //        const_cast<void*>(glyph.pTexture),
    //        glyph.texWidth,
    //        glyph.texHeight,
    //        glyph.texFormat,
    //        GX_CLAMP,
    //        GX_CLAMP,
    //        GX_FALSE
    //    );
    //    GXInitTexObjLOD(
    //        &tobj,
    //        mFilter.atSmall,
    //        mFilter.atLarge,
    //        0, 0,
    //        0,
    //        GX_DISABLE,
    //        GX_DISABLE,
    //        GX_ANISO_1
    //    );
    //    GXLoadTexObj(&tobj, slot);

    //    mLoadingTexture = loadInfo;
    //}

    //
    // 注意：処理が大きく変更されています。
    //
    // .NET版では、何も行いません。
    //
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::UpdateVertexColor()

  Description:  テクスチャを張る四角形ポリゴンの頂点カラーを更新します。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::UpdateVertexColor()
{
    NW4R_POINTER_ASSERT( this );
    mVertexColor.lu = mTextColor.start;
    mVertexColor.ru = (mTextColor.gradationMode != GRADMODE_H) ? mTextColor.start: mTextColor.end;
    mVertexColor.ld = (mTextColor.gradationMode != GRADMODE_V) ? mTextColor.start: mTextColor.end;
    mVertexColor.rd = (mTextColor.gradationMode == GRADMODE_NONE) ? mTextColor.start: mTextColor.end;

    mVertexColor.lu.a = static_cast<u8>(mVertexColor.lu.a * mAlpha / Color::ALPHA_MAX);
    mVertexColor.ru.a = static_cast<u8>(mVertexColor.ru.a * mAlpha / Color::ALPHA_MAX);
    mVertexColor.ld.a = static_cast<u8>(mVertexColor.ld.a * mAlpha / Color::ALPHA_MAX);
    mVertexColor.rd.a = static_cast<u8>(mVertexColor.rd.a * mAlpha / Color::ALPHA_MAX);
}



/* ------------------------------------------------------------------------
        static
   ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------
        描画準備
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetupVertexFormat()

  Description:  頂点フォーマットを文字描画用に設定します。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetupVertexFormat()
{
    //GXSetVtxAttrFmt(GX_VTXFMT0, GX_VA_POS , GX_POS_XYZ , GX_F32  ,  0);
    //GXSetVtxAttrFmt(GX_VTXFMT0, GX_VA_CLR0, GX_CLR_RGBA, GX_RGBA8,  0);
    //GXSetVtxAttrFmt(GX_VTXFMT0, GX_VA_TEX0, GX_TEX_ST  , GX_U16  ,  TEXTURE_COODINATE_FRACTION_BITS);

    //GXClearVtxDesc();
    //GXSetVtxDesc(GX_VA_POS , GX_DIRECT);
    //GXSetVtxDesc(GX_VA_CLR0, GX_DIRECT);
    //GXSetVtxDesc(GX_VA_TEX0, GX_DIRECT);

    //
    // 注意：処理が大きく変更されています。
    //
    // .NET版では、何も行いません。
    //
}



/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetupGXDefault()

  Description:  グラフィックスエンジンを文字描画用のデフォルト設定にします。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetupGXDefault()
{
    ////---- TEV 1段
    //GXSetNumTexGens(1);
    //GXSetNumChans(1);
    //GXSetNumTevStages(1);

    //GXSetChanCtrl(GX_COLOR0A0, GX_DISABLE, GX_SRC_REG, GX_SRC_VTX, GX_LIGHT_NULL, GX_DF_NONE, GX_AF_NONE);

    ////---- TEV 1 段目 MODULATE C=CtCv, A=AtAv
    ////---- Ct=テクスチャカラー Cv=頂点カラー At=テクスチャアルファ Av=頂点アルファ
    //GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR0A0);
    //GXSetTevOp(GX_TEVSTAGE0, GX_MODULATE);

    ////---- フレームバッファ: ソースアルファでブレンド
    //GXSetBlendMode(GX_BM_BLEND, GX_BL_SRCALPHA, GX_BL_INVSRCALPHA, GX_LO_SET);

    //SetupVertexFormat();

    //
    // 注意：処理が大きく変更されています。
    //
    // .NET版では、何も行いません。
    //
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetupGXWithColorMapping( Color, Color )

  Description:  グラフィックスエンジンをテクスチャカラーの線形変換ありの
                文字描画用に設定します。

  Arguments:    min:    最小値のカラーの変換後のカラーを指定します。
                max:    最大値のカラーの変換後のカラーを指定します。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetupGXWithColorMapping(Color min, Color max)
{
    ////---- TEV 2段
    //GXSetNumTexGens(1);
    //GXSetNumChans(1);
    //GXSetNumTevStages(2);

    //GXSetChanCtrl(GX_COLOR0A0, GX_DISABLE, GX_SRC_REG, GX_SRC_VTX, GX_LIGHT_NULL, GX_DF_NONE, GX_AF_NONE);

    ////---- TEV 1 段目 C=min+Ct(max-min) A=min+At(max-min)
    ////---- Ct=テクスチャカラー At=テクスチャアルファ
    ////---- C=線形変換後のテクスチャカラー A=線形変換後のテクスチャアルファ
    //GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR_NULL);
    //GXSetTevColor(GX_TEVREG0, min);
    //GXSetTevColor(GX_TEVREG1, max);
    //GXSetTevColorIn(GX_TEVSTAGE0, GX_CC_C0, GX_CC_C1, GX_CC_TEXC, GX_CC_ZERO);
    //GXSetTevAlphaIn(GX_TEVSTAGE0, GX_CA_A0, GX_CA_A1, GX_CA_TEXA, GX_CA_ZERO);
    //GXSetTevColorOp(GX_TEVSTAGE0, GX_TEV_ADD, GX_TB_ZERO, GX_CS_SCALE_1, GX_TRUE, GX_TEVPREV);
    //GXSetTevAlphaOp(GX_TEVSTAGE0, GX_TEV_ADD, GX_TB_ZERO, GX_CS_SCALE_1, GX_TRUE, GX_TEVPREV);

    ////---- TEV 2 段目 MODULATE C=C1Cv A=A1Av
    ////---- C1=1段目出力カラー Cv=頂点カラー A1=1段目出力アルファ Av=頂点アルファ
    ////----
    //GXSetTevOrder(GX_TEVSTAGE1, GX_TEXCOORD_NULL, GX_TEXMAP_NULL, GX_COLOR0A0);
    //GXSetTevColorIn(GX_TEVSTAGE1, GX_CC_ZERO, GX_CC_CPREV, GX_CC_RASC, GX_CC_ZERO);
    //GXSetTevAlphaIn(GX_TEVSTAGE1, GX_CA_ZERO, GX_CA_APREV, GX_CA_RASA, GX_CA_ZERO);
    //GXSetTevColorOp(GX_TEVSTAGE1, GX_TEV_ADD, GX_TB_ZERO, GX_CS_SCALE_1, GX_TRUE, GX_TEVPREV);
    //GXSetTevAlphaOp(GX_TEVSTAGE1, GX_TEV_ADD, GX_TB_ZERO, GX_CS_SCALE_1, GX_TRUE, GX_TEVPREV);

    ////---- フレームバッファ: ソースアルファでブレンド
    //GXSetBlendMode(GX_BM_BLEND, GX_BL_SRCALPHA, GX_BL_INVSRCALPHA, GX_LO_SET);

    //SetupVertexFormat();

    //
    // 注意：処理が大きく変更されています。
    //
    // .NET版では、何も行いません。
    //
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetupGXForI()

  Description:  グラフィックエンジンをI4 or I8フォントの描画用に設定します。
                フォントのインテンシティはαのみに影響を与えます。
                カラーはラスタライズカラーがそのまま出力されます。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetupGXForI()
{
    ////---- TEV 1段
    //GXSetNumTexGens(1);
    //GXSetNumChans(1);
    //GXSetNumTevStages(1);

    //GXSetChanCtrl(GX_COLOR0A0, GX_DISABLE, GX_SRC_REG, GX_SRC_VTX, GX_LIGHT_NULL, GX_DF_NONE, GX_AF_NONE);

    ////---- TEV 1 段目 C=Cv, A=AtAv
    ////---- Cv=頂点カラー At=テクスチャアルファ Av=頂点アルファ
    //GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR0A0);
    //
    //GXSetTevColorIn(GX_TEVSTAGE0, GX_CC_ZERO, GX_CC_ZERO, GX_CC_ZERO, GX_CC_RASC);
    //GXSetTevAlphaIn(GX_TEVSTAGE0, GX_CA_A0, GX_CA_A1, GX_CA_TEXA, GX_CA_ZERO);
    //GXSetTevColorOp(GX_TEVSTAGE0, GX_TEV_ADD, GX_TB_ZERO, GX_CS_SCALE_1, GX_TRUE, GX_TEVPREV);
    //GXSetTevAlphaOp(GX_TEVSTAGE0, GX_TEV_ADD, GX_TB_ZERO, GX_CS_SCALE_1, GX_TRUE, GX_TEVPREV);

    ////---- フレームバッファ: ソースアルファでブレンド
    //GXSetBlendMode(GX_BM_BLEND, GX_BL_SRCALPHA, GX_BL_INVSRCALPHA, GX_LO_SET);

    //SetupVertexFormat();

    //
    // 注意：処理が大きく変更されています。
    //
    // .NET版では、何も行いません。
    //
}

/*---------------------------------------------------------------------------*
  Name:         CharWriter::SetupGXForRGBA()

  Description:  グラフィックエンジンをRGBAフォントの描画用に設定します。
                現在はデフォルトと同じ設定を行います。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CharWriter::SetupGXForRGBA()
{
    SetupGXDefault();
}



} /* UnManaged */
} /* namespace ut */
} /* namespace nw4r */
