﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_MISC_H_
#define NW4R_MISC_H_

#include <nw4r/config.h>
#include <climits>
#include <cstddef>

//--------------------------------------------------------------------------------
//  たいていの型を警告無く u32 にキャストする
// クラスメンバなどは無理です
//

namespace nw4r
{
#ifdef NW4R_COMPILER_WITH_W64
typedef __w64 unsigned long IntPtr;
typedef __w64 signed long PtrDiff;
#else
typedef std::size_t IntPtr;
typedef std::ptrdiff_t PtrDiff;
#endif
}

// 名前が衝突するので定義しない。
//namespace NW4R
//{
//    typedef nw4r::IntPtr IntPtr;
//    typedef nw4r::PtrDiff PtrDiff;
//}

#ifndef NW4R_ANY_TO_PTR_VALUE
#define NW4R_ANY_TO_PTR_VALUE(ptr) reinterpret_cast<nw4r::IntPtr>(reinterpret_cast<const void*>(ptr))
#endif



//--------------------------------------------------------------------------------
//  C++用マクロ
//
#ifdef __cplusplus
#define NW4R_IFDEFCPP extern "C" {
#define NW4R_ENDIFCPP }
#else
#define NW4R_IFDEFCPP
#define NW4R_ENDIFCPP
#endif


#ifndef NW4R_FROM_TOOL
    #include <nw4r/db/assert.h>
#endif


//--------------------------------------------------------------------------------
//  コンパイル時チェック
//
#define NW4R_COMPILER_ASSERT(expr) \
    extern void nw4r_compiler_assert ## __LINE__ ( char is[(expr) ? +1 : -1] )


//--------------------------------------------------------------------------------
//  Assert message
//
#ifndef NW4R_ASSERTMSG
#ifdef  NW4R_FROM_TOOL
#define NW4R_ASSERTMSG  true ? 0:
#else
#define NW4R_ASSERTMSG  NW4R_DB_ASSERTMSG
#endif
#endif

//--------------------------------------------------------------------------------
//  Assert
//
#ifndef NW4R_ASSERT
#define NW4R_ASSERT(exp) \
            NW4R_ASSERTMSG( (exp), "NW4C:Failed assertion " #exp )
#endif

//--------------------------------------------------------------------------------
//  NULL Assert
//
#ifndef NW4R_NULL_ASSERT
#define NW4R_NULL_ASSERT(exp) \
            NW4R_ASSERTMSG( (exp) != NULL, "NW4C:Pointer must not be NULL ("#exp")" )
#endif

//--------------------------------------------------------------------------------
//  Min Assert
//
#ifndef NW4R_MIN_ASSERT
#define NW4R_MIN_ASSERT(exp, min) \
            NW4R_ASSERTMSG( (exp) >= (min), #exp " is out of bounds(%d)\n%d <= "#exp" not satisfied.", static_cast<int>(exp), static_cast<int>(min) )
#endif

//--------------------------------------------------------------------------------
//  Max Assert
//
#ifndef NW4R_MAX_ASSERT
#define NW4R_MAX_ASSERT(exp, max) \
            NW4R_ASSERTMSG( (exp) <= (max), #exp " is out of bounds(%d)\n"#exp" <= %d not satisfied.", static_cast<int>(exp), static_cast<int>(max) )
#endif

//--------------------------------------------------------------------------------
//  Min Max Assert
//
#ifndef NW4R_MINMAX_ASSERT
#define NW4R_MINMAX_ASSERT(exp, min, max) \
            NW4R_ASSERTMSG( (exp) >= (min) && (exp) <= (max), #exp " is out of bounds(%d)\n%d <= "#exp" <= %d not satisfied.", static_cast<int>(exp), static_cast<int>(min), static_cast<int>(max))
#endif

//--------------------------------------------------------------------------------
//  Min Max Less Than Assert
//
#ifndef NW4R_MINMAXLT_ASSERT
#define NW4R_MINMAXLT_ASSERT(exp, min, max) \
            NW4R_ASSERTMSG( (exp) >= (min) && (exp) < (max), #exp " is out of bounds(%d)\n%d <= "#exp" < %d not satisfied.", static_cast<int>(exp), static_cast<int>(min), static_cast<int>(max))
#endif

//--------------------------------------------------------------------------------
//  Min Assert for floating point value
//
#ifndef NW4R_FMIN_ASSERT
#define NW4R_FMIN_ASSERT(exp, min) \
            NW4R_ASSERTMSG( (exp) >= (min), #exp " is out of bounds(%f)\n%f <= "#exp" not satisfied.", static_cast<double>(exp), static_cast<double>(min) )
#endif

//--------------------------------------------------------------------------------
//  Max Assert for floating point value
//
#ifndef NW4R_FMAX_ASSERT
#define NW4R_FMAX_ASSERT(exp, max) \
            NW4R_ASSERTMSG( (exp) <= (max), #exp " is out of bounds(%f)\n"#exp" <= %f not satisfied.", static_cast<double>(exp), static_cast<double>(max) )
#endif

//--------------------------------------------------------------------------------
//  Min Max Assert for floating point value
//
#ifndef NW4R_FMINMAX_ASSERT
#define NW4R_FMINMAX_ASSERT(exp, min, max) \
            NW4R_ASSERTMSG( (exp) >= (min) && (exp) <= (max), #exp " is out of bounds(%f)\n%f <= "#exp" <= %f not satisfied.", static_cast<double>(exp), static_cast<double>(min), static_cast<double>(max))
#endif

//--------------------------------------------------------------------------------
//  Fatal error
//
#ifndef NW4R_FATAL_ERROR
#ifdef  NW4R_FROM_TOOL
#define NW4R_FATAL_ERROR    true ? 0:
#else
#define NW4R_FATAL_ERROR(...) \
            NW4R_ASSERTMSG( false, "NW4R:Fatal Error\n"__VA_ARGS__ )
#endif
#endif

//--------------------------------------------------------------------------------
//  Internal error
//
#ifndef NW4R_INTERNAL_ERROR
#ifdef  NW4R_FROM_TOOL
#define NW4R_INTERNAL_ERROR true ? 0:
#else
#define NW4R_INTERNAL_ERROR(...) \
            NW4R_ASSERTMSG( false, "NW4R Internal error\n" __VA_ARGS__ )
#endif
#endif


//--------------------------------------------------------------------------------
//  Cache Alignment error
//
#ifndef NW4R_CACHE_ALIGN_ASSERT
#if defined( NW4R_PLATFORM_GC )
#define NW4R_CACHE_ALIGN_ASSERT     NW4R_ALIGN32_ASSERT
#else
#define NW4R_CACHE_ALIGN_ASSERT     NW4R_ALIGN32_ASSERT
#endif
#endif


//--------------------------------------------------------------------------------
//  Alignment error
//
#ifndef NW4R_ALIGN_ASSERT
#define NW4R_ALIGN_ASSERT(exp,align) \
             NW4R_ASSERTMSG( (NW4R_ANY_TO_PTR_VALUE(exp) & ((align) - 1)) == 0, "NW4R:Alignment Error(0x%x)\n"#exp" must be aligned to %d bytes boundary.", exp, align )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(128 bytes)
//
#ifndef NW4R_ALIGN128_ASSERT
#define NW4R_ALIGN128_ASSERT(exp) \
             NW4R_ASSERTMSG( (NW4R_ANY_TO_PTR_VALUE(exp) & 127) == 0, "NW4R:Alignment Error(0x%x)\n"#exp" must be aligned to 128 bytes boundary.", exp )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(32 bytes)
//
#ifndef NW4R_ALIGN32_ASSERT
#define NW4R_ALIGN32_ASSERT(exp) \
             NW4R_ASSERTMSG( (NW4R_ANY_TO_PTR_VALUE(exp) & 31) == 0, "NW4R:Alignment Error(0x%x)\n"#exp" must be aligned to 32 bytes boundary.", exp )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(4 bytes)
//
#ifndef NW4R_ALIGN4_ASSERT
#define NW4R_ALIGN4_ASSERT(exp) \
            NW4R_ASSERTMSG( (NW4R_ANY_TO_PTR_VALUE(exp) & 3) == 0, "NW4R:Alignment Error(0x%x)\n"#exp" must be aligned to 4 bytes boundary.", (exp) )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(2 bytes)
//
#ifndef NW4R_ALIGN2_ASSERT
#define NW4R_ALIGN2_ASSERT(exp) \
            NW4R_ASSERTMSG( (NW4R_ANY_TO_PTR_VALUE(exp) & 1) == 0, "NW4R:Alignment Error(0x%x)\n"#exp" must be aligned to 2 bytes boundary.", (exp) )
#endif

//--------------------------------------------------------------------------------
//  buffer size error
//
#ifndef NW4R_BUFFERSIZE_ASSERT
#ifdef  NW4R_PLATFORM_RVL
#define NW4R_BUFFERSIZE_ASSERT(exp) \
            NW4R_ASSERTMSG( (exp) <= 64 * 1024 * 1024, "NW4R:Buffer Size Error\n" #exp "(=%d) over maximum physical memory size.", (exp))
#else
#define NW4R_BUFFERSIZE_ASSERT(exp) \
            NW4R_ASSERTMSG( (exp) <= 24 * 1024 * 1024, "NW4R:Buffer Size Error\n" #exp "(=%d) over maximum physical memory size.", (exp))
#endif
#endif

//--------------------------------------------------------------------------------
//  Invalid pointer error
//
#ifndef NW4R_POINTER_ASSERT
#ifdef  NW4R_PLATFORM_RVL
#define NW4R_POINTER_ASSERT(exp)                                                                                            \
            NW4R_ASSERTMSG(                                                                                                 \
                   ((NW4R_ANY_TO_PTR_VALUE(exp) & 0xFF000000) == 0x80000000)    /* 8000_0000～80FF_FFFF MEM1 Cached   */    \
                || ((NW4R_ANY_TO_PTR_VALUE(exp) & 0xFF800000) == 0x81000000)    /* 8100_0000～817F_FFFF MEM1 Cached   */    \
                || ((NW4R_ANY_TO_PTR_VALUE(exp) & 0xF8000000) == 0x90000000)    /* 9000_0000～97FF_FFFF MEM2 Cached   */    \
                || ((NW4R_ANY_TO_PTR_VALUE(exp) & 0xFF000000) == 0xC0000000)    /* C000_0000～C0FF_FFFF MEM1 Uncached */    \
                || ((NW4R_ANY_TO_PTR_VALUE(exp) & 0xFF800000) == 0xC1000000)    /* C100_0000～C17F_FFFF MEM1 Uncached */    \
                || ((NW4R_ANY_TO_PTR_VALUE(exp) & 0xF8000000) == 0xD0000000)    /* D000_0000～D7FF_FFFF MEM2 Uncached */    \
                || ((NW4R_ANY_TO_PTR_VALUE(exp) & 0xFFFFC000) == 0xE0000000),   /* E000_0000～E000_3FFF Locked Cache  */    \
                "NW4R:Pointer Error\n"#exp"(=%p) is not valid pointer.", (exp) )
#else
#define NW4R_POINTER_ASSERT(exp)                                                                \
            NW4R_ASSERTMSG(                                                                     \
                   ((NW4R_ANY_TO_PTR_VALUE(exp) & 0xF0000000) == 0x80000000)    /* Cached   */  \
                || ((NW4R_ANY_TO_PTR_VALUE(exp) & 0xF0000000) == 0xC0000000),   /* Uncached */  \
                "NW4R:Pointer Error\n"#exp"(=%p) is not valid pointer.", (exp) )
#endif
#endif

#ifndef NW4R_REFERENCE_ASSERT
#define NW4R_REFERENCE_ASSERT(ref) NW4R_POINTER_ASSERT(&ref)
#endif

//--------------------------------------------------------------------------------
//  Range error
//

#ifndef NW4R_U8_RANGE_ASSERT
#define NW4R_U8_RANGE_ASSERT(exp) \
            NW4R_MINMAX_ASSERT(exp, 0, UCHAR_MAX)
#endif

#ifndef NW4R_S8_RANGE_ASSERT
#define NW4R_S8_RANGE_ASSERT(exp) \
            NW4R_MINMAX_ASSERT(exp, SCHAR_MIN, SCHAR_MAX)
#endif

#ifndef NW4R_U16_RANGE_ASSERT
#define NW4R_U16_RANGE_ASSERT(exp) \
            NW4R_MINMAX_ASSERT(exp, 0, USHRT_MAX)
#endif

#ifndef NW4R_S16_RANGE_ASSERT
#define NW4R_S16_RANGE_ASSERT(exp) \
            NW4R_MINMAX_ASSERT(exp, SHRT_MIN, SHRT_MAX)
#endif

#ifndef NW4R_U32_RANGE_ASSERT
#define NW4R_U32_RANGE_ASSERT(exp) \
            NW4R_MINMAX_ASSERT(exp, 0, UINT_MAX)
#endif

#ifndef NW4R_S32_RANGE_ASSERT
#define NW4R_S32_RANGE_ASSERT(exp) \
            NW4R_MINMAX_ASSERT(exp, INT_MIN, INT_MAX)
#endif

//--------------------------------------------------------------------------------
//  Floating point value error
//  VC がC99に準拠していないため、isfinite, isnan を利用できません。
#ifdef  NW4R_FROM_TOOL

#ifndef NW4R_FLOAT_ASSERT
#define NW4R_FLOAT_ASSERT(exp)
#endif

#else

#ifndef NW4R_FLOAT_ASSERT
#define NW4R_FLOAT_ASSERT(exp) \
            NW4R_ASSERTMSG((isfinite(exp) && !isnan(exp)), "NW4R:Floating Point Value Error(%f)\n"#exp" is infinite or nan.", exp)
#endif

#endif
//--------------------------------------------------------------------------------
//  Warning
//

#ifndef NW4R_WARNING
#ifdef  NW4R_FROM_TOOL
#define NW4R_WARNING    true ? 0:
#else
#define NW4R_WARNING    NW4R_DB_EXP_WARNING
#endif
#endif

//--------------------------------------------------------------------------------
//  Log
//
#ifndef NW4R_LOG
#ifdef  NW4R_FROM_TOOL
#define NW4R_LOG    true ? 0:
#else
#define NW4R_LOG    NW4R_DB_LOG
#endif
#endif

/* NW4R_MISC_H__ */
#endif
