﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_UT_CHARWRITER_H_
#define NW4R_UT_CHARWRITER_H_

#include <cstddef>
#include <cstdio>
#include <dolphin/types.h>
#include "Color.h"
#include "Font.h"
//#include "nw4r/math/types.h"
#include <vector>




//
// ラインタイム側では、math モジュールの一部分だけを利用していました。
// math モジュールはVC7では、ビルド不能であったため、
// .NET 版Fontライブラリでは、利用箇所だけを切り取って利用することとしました。
//
namespace NW4R
{
namespace Font
{
namespace math
{


struct _VEC2
{
    f32 x;
    f32 y;
};

struct _VEC3
{
    f32 x;
    f32 y;
    f32 z;
};

struct VEC2 : public _VEC2
{
public:
    typedef VEC2 self_type;
    typedef f32  value_type;
public:
    VEC2() {}
    VEC2(const f32* p) { x = p[0]; y = p[1]; }
    VEC2(f32 fx, f32 fy) { x = fx; y = fy; }

    operator f32*() { return &x; }
    operator const f32*() const { return &x; }

    self_type& operator+=(const self_type& rhs) { x += rhs.x; y += rhs.y; return *this; }
    self_type& operator-=(const self_type& rhs) { x -= rhs.x; y -= rhs.y; return *this; }
    self_type& operator*=(f32 f) { x *= f; y *= f; return *this; }
    self_type& operator/=(f32 f) { f32 r = 1.f / f; x *= r; y *= r; return *this; }

    self_type operator+() const { return *this; }
    self_type operator-() const { return self_type(-x, -y); }

    self_type operator+(const self_type& rhs) const { return self_type(x + rhs.x, y + rhs.y); }
    self_type operator-(const self_type& rhs) const { return self_type(x - rhs.x, y - rhs.y); }
    self_type operator*(f32 f) const { return self_type(f * x, f * y); }
    self_type operator/(f32 f) const { f32 r = 1.f / f; return self_type(r * x, r * y); }
    bool operator==(const self_type& rhs) const { return x == rhs.x && y == rhs.y; }
    bool operator!=(const self_type& rhs) const { return x != rhs.x || y != rhs.y; }

    // void Report(bool bNewline = true, const char* name = NULL) const;
};


/* ------------------------------------------------------------------------
    VEC3
   ------------------------------------------------------------------------ */
struct VEC3 : public _VEC3
{
public:
    typedef VEC3 self_type;
    typedef f32  value_type;
public:
    /*
    VEC3() {}
    VEC3(const f32* p) { x = p[0]; y = p[1]; z = p[2]; }
    VEC3(f32 fx, f32 fy, f32 fz) { x = fx; y = fy; z = fz; }
    VEC3(const _VEC3& v) { x = v.x; y = v.y; z = v.z; }
    VEC3(const Vec& v) { x = v.x; y = v.y; z = v.z; }

    operator f32*() { return &x; }
    operator const f32*() const { return &x; }
#if 1
    operator Vec*() { return (Vec*)&x; }
    operator const Vec*() const { return (const Vec*)&x; }
#endif

    self_type& operator+=(const self_type& rhs);
    self_type& operator-=(const self_type& rhs);
    self_type& operator*=(f32 f);
    self_type& operator/=(f32 f) { return operator*=(1.f / f); }

    self_type operator+() const { return *this; }
    self_type operator-() const { return self_type(-x, -y, -z); }

    // ２項演算子は戻り値最適化が行われるかどうかで最適な実装が異なる
    self_type operator+(const self_type& rhs) const;
    self_type operator-(const self_type& rhs) const;
    self_type operator*(f32 f) const;
    self_type operator/(f32 f) const { f32 r = 1.f / f; return operator*(r); }

    bool operator==(const self_type& rhs) const { return x == rhs.x && y == rhs.y && z == rhs.z; }
    bool operator!=(const self_type& rhs) const { return x != rhs.x || y != rhs.y || z != rhs.z; }

    void Report(bool bNewline = true, const char* name = NULL) const;
    */
};

}


}
}

namespace NW4R
{
namespace Font
{
namespace UnManaged
{


//-----------------------------------------------------------------------------
// 文字描画タスク(.NET版で追加になりました。)
struct CharRenderingTask
{
public:
    char                 c;
    float                x;
    float                y;
    float                z;
    Glyph                glyph;
};

class CharWriter
{
public:
    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    enum GradationMode
    {
        GRADMODE_NONE,                          // グラデーションなし
        GRADMODE_H,                             // 水平方向グラデーション
        GRADMODE_V,                             // 垂直方向グラデーション
        NUM_OF_GRADMODE
    };

    // 文字描画タスク セット(.NET版で追加になりました。)
    typedef std::vector<CharRenderingTask*>       RenderingTaskPtrArray;

    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const u32 DEFAULT_COLOR_MAPPING_MIN = 0x00000000UL;
    static const u32 DEFAULT_COLOR_MAPPING_MAX = 0xFFFFFFFFUL;


    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- コンストラクタ/デストラクタ
                        CharWriter();
                        ~CharWriter();

    //---- フォント
    void                SetFont( const Font& font );
    const Font*         GetFont() const;


    //---- 描画準備
    void                SetupGX();
    void                ResetTextureCache();

    void                SetColorMapping(
                            Color           min,
                            Color           max );

    const Color         GetColorMappingMin() const;
    const Color         GetColorMappingMax() const;
    void                ResetColorMapping();

    //---- 文字色
    void                SetAlpha( u8 alpha );
    u8                  GetAlpha() const;
    void                SetGradationMode( GradationMode mode );
    GradationMode       GetGradationMode() const;
    void                SetTextColor( Color color );

    void                SetTextColor(
                            Color           start,
                            Color           end );

    const Color         GetTextColor() const;
    const Color         GetGradationStartColor() const;
    const Color         GetGradationEndColor() const;

    //---- 文字サイズ
    void                SetScale(
                            float           hScale,
                            float           vScale );

    void                SetScale( float hvScale );
    float               GetScaleH() const;
    float               GetScaleV() const;

    void                SetFontSize(
                            float           width,
                            float           height );

    void                SetFontSize( float height );
    float               GetFontWidth() const;
    float               GetFontHeight() const;
    float               GetFontAscent() const;
    float               GetFontDescent() const;

    //---- テクスチャフィルタ
    void                EnableLinearFilter(
                            bool            atSmall,
                            bool            atLarge );
    bool                IsLinearFilterEnableAtSmall() const;
    bool                IsLinearFilterEnableAtLarge() const;

    //---- 等幅
    bool                IsWidthFixed() const;
    void                EnableFixedWidth(bool isFixed);
    float               GetFixedWidth() const;
    void                SetFixedWidth(float width);

    //---- 文字描画
    float               Print( CharCode code );
    void                Print( const Glyph& glyph );

    //---- カーソル
    void                SetCursor(
                            float x,
                            float y );
    void                SetCursor(
                            float x,
                            float y,
                            float z );

    void                MoveCursor(
                            float dx,
                            float dy );
    void                MoveCursor(
                            float dx,
                            float dy,
                            float dz );
    void                MoveCurosr(float dx, float dy ) { MoveCursor(dx, dy); }

    void                SetCursorX( float x );
    void                SetCursorY( float y );
    void                SetCursorZ( float z );
    void                MoveCursorX( float dx );
    void                MoveCursorY( float dy );
    void                MoveCursorZ( float dz );
    float                 GetCursorX() const;
    float                 GetCursorY() const;
    float                 GetCursorZ() const;


     //---- 描画タスク操作関係
     const RenderingTaskPtrArray&      GetTaskArray() const;
     void ClearRenderingTask();


private:



    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    typedef math::VEC2      CharScale;
    typedef math::VEC3      CursorPos;

    struct ColorMapping
    {
        Color               min;
        Color               max;
    };

    struct TextColor
    {
        Color               start;
        Color               end;
        GradationMode       gradationMode;
    };

    struct VertexColor
    {
        Color               lu;
        Color               ru;
        Color               ld;
        Color               rd;
    };

    class TextureFilter
    {
    public:
        GXTexFilter         atSmall;
        GXTexFilter         atLarge;

        bool                operator !=(const TextureFilter& rhs) const
        {
            return (atSmall != rhs.atSmall)
                || (atLarge != rhs.atLarge);
        }
    };

    class LoadingTexture
    {
    public:
        GXTexMapID          slot;
        const void*         texture;
        TextureFilter       filter;

        bool                operator !=(const LoadingTexture& rhs) const
        {
            return (slot != rhs.slot)
                || (texture != rhs.texture)
                || (filter != rhs.filter);
        }
        void                Reset()
        {
            slot    = GX_TEXMAP_NULL;
            texture = NULL;
        }
    };


    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const int TEXTURE_COODINATE_FRACTION_BITS = 15;
    static const u32 TEXTURE_COODINATE_ONE = (1 << TEXTURE_COODINATE_FRACTION_BITS);


    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    static LoadingTexture       mLoadingTexture;        // ロード済みテクスチャ

    ColorMapping                mColorMapping;          // カラーマッピング
    VertexColor                 mVertexColor;           // 頂点カラー
    TextColor                   mTextColor;             // 文字色
    CharScale                   mScale;                 // 文字拡大率
    CursorPos                   mCursorPos;             // カーソル位置
    TextureFilter               mFilter;                // テクスチャ補間
    u8                          padding_[2];            //
    u8                          mAlpha;                 // 不透明度
    bool                        mIsWidthFixed;          // 等幅描画するかどうか
    float                       mFixedWidth;            // 等幅時の幅
    const Font*                 mFont;                  // フォント

    RenderingTaskPtrArray       mTaskArray_;            // 描画タスクが格納されるベクタ(.NET版で追加)

    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- 描画準備
    static void         SetupGXDefault();

    static void         SetupGXWithColorMapping(
                            Color                   min,
                            Color                   max
                        );

    static void         SetupGXForRGBA();
    static void         SetupGXForI();
    static void         SetupVertexFormat();

    void                PrintGlyph(
                            float           x,
                            float           y,
                            float           z,
                            const Glyph&    glyph);

    //---- 内部処理関数
    void                UpdateVertexColor();

    void                LoadTexture(
                            const Glyph&            glyph,
                            GXTexMapID              slot );
};




} /* UnManaged */
} /* namespace ut */
} /* namespace nw4r */

#endif //  NW4R_UT_CHARWRITER_H_
