﻿/*--------------------------------------------------------------------------------*
Copyright (C)Nintendo All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*--------------------------------------------------------------------------------*/

#include "font_PairFont.h"
#include <nw/types.h>
#include <algorithm>
namespace nn {
namespace font {
    namespace font = NW4R::Font::UnManaged;
    /* ------------------------------------------------------------------------
    コンストラクタ/デストラクタ
    ------------------------------------------------------------------------ */

    PairFont::PairFont()
        : m_First(NULL),
        m_Second(NULL),
        m_AlternateWithPrimary(true)
    {
    }

    PairFont::PairFont(Font* primary, Font* secondary)
        : m_First(NULL),
        m_Second(NULL),
        m_AlternateWithPrimary(true)
    {
        SetFont(primary, secondary);
    }


    PairFont::~PairFont()
    {
    }


    /* ------------------------------------------------------------------------
    フォントアクセサ
    ------------------------------------------------------------------------ */

    void
        PairFont::SetFont(Font* primary, Font* secondary)
    {
        //NN_SDK_ASSERT_NOT_NULL(primary);
        //NN_SDK_ASSERT_NOT_NULL(secondary);
        //NN_SDK_ASSERT(primary->GetCharacterCode() == secondary->GetCharacterCode());
        m_First = primary;
        m_Second = secondary;

        // 古い基底クラスの仕様と合わせるために必要。
        InitReaderFunc(GetEncoding());
    }


    /* ------------------------------------------------------------------------
    フォント全体情報アクセサ
    ------------------------------------------------------------------------ */

    int
        PairFont::GetWidth() const
    {
        return std::max(m_First->GetWidth(), m_Second->GetWidth());
    }

    int
        PairFont::GetHeight() const
    {
        return std::max(m_First->GetHeight(), m_Second->GetHeight());
    }

    int
        PairFont::GetAscent() const
    {
        return m_First->GetHeight() >= m_Second->GetHeight() ?
            m_First->GetAscent() :
            m_Second->GetAscent();
    }

    int
        PairFont::GetDescent() const
    {
        return m_First->GetHeight() >= m_Second->GetHeight() ?
            m_First->GetDescent() :
            m_Second->GetDescent();
    }

    int
        PairFont::GetBaselinePos() const
    {
        return m_First->GetHeight() >= m_Second->GetHeight() ?
            m_First->GetBaselinePos() :
            m_Second->GetBaselinePos();
    }

    int
        PairFont::GetCellHeight() const
    {
        return m_First->GetHeight() >= m_Second->GetHeight() ?
            m_First->GetCellHeight() :
            m_Second->GetCellHeight();
    }

    int
        PairFont::GetCellWidth() const
    {
        return m_First->GetWidth() >= m_Second->GetWidth() ?
            m_First->GetCellWidth() :
            m_Second->GetCellWidth();
    }

    int
        PairFont::GetMaxCharWidth() const
    {
        return std::max(m_First->GetMaxCharWidth(), m_Second->GetMaxCharWidth());
    }

    font::Font::Type
        PairFont::GetType() const
    {
        return font::Font::Type::TYPE_PAIR;
    }

    GXTexFmt
        PairFont::GetTextureFormat() const
    {
        // テクスチャフォーマットは一致していなければならない
        //NN_SDK_ASSERT(m_First->GetTextureFormat() == m_Second->GetTextureFormat());

        return m_First->GetTextureFormat();
    }

    int
        PairFont::GetLineFeed() const
    {
        return std::max(m_First->GetLineFeed(), m_Second->GetLineFeed());
    }

    const font::CharWidths
        PairFont::GetDefaultCharWidths() const
    {
        return m_First->GetWidth() >= m_Second->GetWidth() ?
            m_First->GetDefaultCharWidths() :
            m_Second->GetDefaultCharWidths();
    }

    void
        PairFont::SetDefaultCharWidths(const font::CharWidths& widths)
    {
        m_First->SetDefaultCharWidths(widths);
        m_Second->SetDefaultCharWidths(widths);
    }

    bool
        PairFont::SetAlternateChar(font::CharCode c)
    {
        bool bPrimary = m_First->SetAlternateChar(c);
        bool bSecondary = m_Second->SetAlternateChar(c);
        if (bPrimary)
        {
            m_AlternateWithPrimary = true;
            return true;
        }
        else if (bSecondary)
        {
            m_AlternateWithPrimary = false;
            return true;
        }
        else
        {
            return false;
        }
    }

    void
        PairFont::SetLineFeed(int linefeed)
    {
        //NN_SDK_ASSERT_RANGE(linefeed, 0, USHRT_MAX);

        return m_First->GetHeight() >= m_Second->GetHeight() ?
            m_First->SetLineFeed(linefeed) :
            m_Second->SetLineFeed(linefeed);
    }



    /* ------------------------------------------------------------------------
    文字単体情報アクセサ
    ------------------------------------------------------------------------ */

    int
        PairFont::GetCharWidth(font::CharCode c) const
    {
        if (m_First->HasGlyph(c))
        {
            return m_First->GetCharWidth(c);
        }
        else if (m_Second->HasGlyph(c))
        {
            return m_Second->GetCharWidth(c);
        }
        else if (m_AlternateWithPrimary)
        {
            return m_First->GetCharWidth(c);
        }
        else
        {
            return m_Second->GetCharWidth(c);
        }
    }

    const font::CharWidths
        PairFont::GetCharWidths(font::CharCode c) const
    {
        if (m_First->HasGlyph(c))
        {
            return m_First->GetCharWidths(c);
        }
        else if (m_Second->HasGlyph(c))
        {
            return m_Second->GetCharWidths(c);
        }
        else if (m_AlternateWithPrimary)
        {
            return m_First->GetCharWidths(c);
        }
        else
        {
            return m_Second->GetCharWidths(c);
        }
    }

    void
        PairFont::GetGlyph(
            font::Glyph*     glyph,
            font::CharCode   c
        ) const
    {
        if (m_First->HasGlyph(c))
        {
            m_First->GetGlyph(glyph, c);
            glyph->baselineDifference += static_cast<int8_t>(m_First->GetBaselinePos() - GetBaselinePos());
        }
        else if (m_Second->HasGlyph(c))
        {
            m_Second->GetGlyph(glyph, c);
            glyph->baselineDifference += static_cast<int8_t>(m_Second->GetBaselinePos() - GetBaselinePos());

            glyph->sheetNo += m_First->GetSheetNum();
        }
        else if (m_AlternateWithPrimary)
        {
            m_First->GetGlyph(glyph, c);
            glyph->baselineDifference += static_cast<int8_t>(m_First->GetBaselinePos() - GetBaselinePos());
        }
        else
        {
            m_Second->GetGlyph(glyph, c);
            glyph->baselineDifference += static_cast<int8_t>(m_Second->GetBaselinePos() - GetBaselinePos());

            glyph->sheetNo += m_First->GetSheetNum();
        }
    }

    bool
        PairFont::HasGlyph(font::CharCode c) const
    {
        return m_First->HasGlyph(c) || m_Second->HasGlyph(c);
    }

    int
        PairFont::GetKerning(font::CharCode c0, font::CharCode c1) const
    {
        if (m_First->HasGlyph(c0) && m_First->HasGlyph(c1))
        {
            return m_First->GetKerning(c0, c1);
        }

        if (m_Second->HasGlyph(c0) && m_Second->HasGlyph(c1))
        {
            return m_Second->GetKerning(c0, c1);
        }

        return 0;
    }

    bool PairFont::HasKerning() const
    {
        // UI 表示にしか使われていないようなので、この条件でよい。
        return m_First->HasKerning() || m_Second->HasKerning();
    }

    font::FontType PairFont::GetFontType() const
    {
        // IsPackedFormatFont からしか呼ばれないのでここにはこないはず。
        return m_First->GetFontType();
    }

    font::FontEncoding PairFont::GetEncoding() const
    {
        // First と Second で同じでなければいけない。
        // 常に UTF-16 のはず。
        return m_First->GetEncoding();
    }


    // 追加：.NET モジュールでテクスチャの検索を行う際に利用します。
    int PairFont::GetSheetNum() const
    {
        return m_First->GetSheetNum() + m_Second->GetSheetNum();
    }
}
}
