﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Data;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace NW4R.ColorPicker
{
    /// <summary>
    /// SVグラデーションピッカーコントロールクラス。
    /// </summary>
    public class SaturationValueBar : System.Windows.Forms.Control
    {
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        private Bitmap	m_imageBMP		= null;
        private bool	m_isMouseDown	= false;
        private bool	m_changePaint	= false;
        private Point	m_currentPoint;
        private Color	m_color			= Color.FromArgb( 0, 0, 0 );

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SaturationValueBar()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            //	ちらつきの防止用
            this.SetStyle( ControlStyles.UserPaint | ControlStyles.AllPaintingInWmPaint | ControlStyles.DoubleBuffer, true );
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            components = new System.ComponentModel.Container();
        }
        #endregion

        #region イベント

        /// <summary>
        /// カラー選択されたときに発行される
        /// </summary>
        internal delegate void ColorSelectedHandler( object sender, ColorChangedEventArgs e );
        internal event ColorSelectedHandler ColorSelected;

        #endregion

        #region オーバーライド

        /// <summary>
        /// OnMouseDownのオーバーライド（色選択イベント発行）
        /// </summary>
        /// <param name="e"></param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if( e.Button != MouseButtons.Left )
                return;
            m_isMouseDown	= true;
            m_currentPoint	= new Point( e.X , e.Y );
            if( ColorSelected != null )
                ColorSelected( this, new ColorChangedEventArgs( GetSelectColor( m_currentPoint ), false ) );
        }

        /// <summary>
        /// OnMouseUpのオーバーライド
        /// </summary>
        /// <param name="e"></param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            if( e.Button != MouseButtons.Left )
                return;
            if( ColorSelected != null )
                ColorSelected( this, new ColorChangedEventArgs( GetSelectColor( m_currentPoint ), true ) );
            m_isMouseDown	= false;
        }

        /// <summary>
        /// OnMouseMoveのオーバーライド（色選択イベント発行）
        /// </summary>
        /// <param name="e"></param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove( e );
            if( !m_isMouseDown )
                return;
            if( e.Button != MouseButtons.Left )
                return;

            int	x	= e.X, y	= e.Y;
            if( x < 0 )
                x	= 0;
            else if( x > (this.Size.Width-1) )
                x	= (this.Size.Width-1);
            if( y < 0 )
                y	= 0;
            else if( y > (this.Size.Height-1) )
                y	= (this.Size.Height-1);
            m_currentPoint	= new Point( x, y );
            if( ColorSelected != null )
                ColorSelected( this, new ColorChangedEventArgs( GetSelectColor( m_currentPoint ), false ) );
        }

        /// <summary>
        /// OnPaintのオーバーライド（ビットマップ表示）
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPaint(PaintEventArgs e)
        {
            if( this.DesignMode )
                return;

            if( this.Enabled )
            {
                Graphics g = e.Graphics;
                g.DrawImage( m_imageBMP, this.DisplayRectangle );
                using( Pen blackPen   = new Pen( Color.Black, 3 ) )
                {
                    g.DrawRectangle( blackPen, m_currentPoint.X - 6, m_currentPoint.Y - 6, 12, 12 );
                    g.DrawRectangle( Pens.White, m_currentPoint.X - 6, m_currentPoint.Y - 6, 12, 12 );
                }
                m_changePaint	= false;
            }
            else
                base.OnPaint(e);
        }

        /// <summary>
        /// デザインモード時は親を再描画
        /// </summary>
        /// <param name="e"></param>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (this.DesignMode)
            {
                if (this.Parent != null)
                {
                    this.Parent.Invalidate();
                }
            }
            base.OnSizeChanged(e);
        }

        #endregion

        #region プロパティ

        /// <summary>
        /// 選択色
        /// </summary>
        public Color Color
        {
            get	{	return m_color;		}
            set	{	m_color	= value;	}
        }

        #endregion

        /// <summary>
        /// 表示Bitmapの変更
        /// </summary>
        /// <param name="color">カラー</param>
        /// <param name="point">選択ポイント</param>
        /// <param name="resetPoint">選択している場所をリセットするか</param>
        private void UpdateCurrentColorBitmap( Color color, Point point, bool resetPoint )
        {
            //	マウスダウン中は変更しない
            if( m_isMouseDown )
                return;

            if( resetPoint )
                m_currentPoint = point;

            Int32	w	= this.ClientSize.Width, h	= this.ClientSize.Height, x, y, col;

            //	BMPをラインで書き換え
            m_imageBMP		= new Bitmap( w, h );
            BitmapData	bitmapData	= m_imageBMP.LockBits( this.DisplayRectangle, ImageLockMode.ReadWrite, PixelFormat.Format32bppRgb);
            IntPtr	pData	= bitmapData.Scan0;

            double	redIndex	= (double)m_color.R / (h-1);
            double	greenIndex	= (double)m_color.G / (h-1);
            double	blueIndex	= (double)m_color.B / (h-1);

            double	rR		= (double)m_color.R;
            double	rG		= (double)m_color.G;
            double	rB		= (double)m_color.B;

            int		total	= 0;
            for ( y = 0; y < (h-1); y ++ )
            {
                double	lR	= 255 - (y * 255 / (h-1));
                double	lG	= lR;
                double	lB	= lR;
                double	oR	= ( rR - lR ) / (w-1);
                double	oG	= ( rG - lG ) / (w-1);
                double	oB	= ( rB - lB ) / (w-1);
                total	= y * bitmapData.Stride;
                for( x = 0; x < w; x ++, total += 4 )
                {
                    col	= ((int)Math.Round( lR ) << 16 ) | ((int)Math.Round( lG ) << 8 ) | (int)Math.Round( lB );
                    Marshal.WriteInt32( pData, total, col );
                    lR	+= oR;
                    lG	+= oG;
                    lB	+= oB;
                }
                rR	-= redIndex;
                rG	-= greenIndex;
                rB	-= blueIndex;
            }
            total	= y * bitmapData.Stride;
            for( x = 0; x < w; x ++, total += 4 )
            {
                Marshal.WriteInt32( pData, total, (int)0 );
            }

            m_imageBMP.UnlockBits( bitmapData );
        }
        /// <summary>
        /// 現在選択色の取得
        /// </summary>
        /// <param name="p">カラーを取得する位置</param>
        /// <returns>選択色</returns>
        private Color GetSelectColor(Point p)
        {
            HSV	hsb		= HSV.FromRGB( new RGB( m_color ) );
            int	h		= hsb.H;
            int	v		= (int)Math.Round( 255 - p.Y * 255d / (this.Size.Height-1) );
            int	s		= (int)Math.Round( p.X * 255d / (this.Size.Width-1) );
            if( s == 0 )
            {
                return Color.FromArgb( v, v, v );
            }
            int	dif		= ( h * 6 ) % 360;
            int	num1	= (int)(v * (255 - s) / 255.0f);
            int	num2	= (int)(v * (255 - s * dif / 360.0f) / 255.0f);
            int	num3	= (int)(v * (255 - s * (360 - dif) / 360.0f) / 255.0f);

            switch( h / 60 )
            {
                case 0:
                    return Color.FromArgb( v, num3, num1 );
                case 1:
                    return Color.FromArgb( num2, v, num1 );
                case 2:
                    return Color.FromArgb( num1, v, num3 );
                case 3:
                    return Color.FromArgb( num1, num2, v );
                case 4:
                    return Color.FromArgb( num3, num1, v );
                default:
                    return Color.FromArgb( v, num1, num2 );
            }
        }

        /// <summary>
        /// 現在のポイントから選択色を返す
        /// </summary>
        /// <returns>選択色</returns>
        public Color GetSelectPointColor()
        {
            return GetSelectColor( m_currentPoint );
        }

        /// <summary>
        /// 表示カラーを更新する
        /// </summary>
        /// <param name="color">選択色のHSV(書き換えにHを使用)</param>
        /// <param name="resetPoint">選択位置を変更するか？</param>
        /// <returns>表示領域に変更があったか否か</returns>
        public bool UpdateColor( HSV color, bool resetPoint )
        {
            bool	result	= false;
            Color	col	= RGB.FromHSV( new HSV( color.H, 100, 100 ) ).GetColor();
            Point	pp	= new Point( (int)Math.Round( color.S * (this.Size.Width-1) / 100d ), (int)Math.Round( (100 - color.V) * (this.Size.Height-1) / 100d ) );
            //	領域全域を変更
            if( !m_color.Equals( col ) )
            {
                result	= true;
                m_color	= col;

                // 非表示状態でも更新します。
                // if( this.Visible )
                //	this.UpdateCurrentColorBitmap( col, pp, resetPoint );
                this.UpdateCurrentColorBitmap( col, pp, resetPoint );

                if( !m_changePaint )
                    this.Invalidate();
                m_changePaint	= true;
            }
            //	選択している場所のみ変更
            else if( resetPoint && !m_currentPoint.Equals( pp ) )
            {
                result	= true;
                m_currentPoint	= pp;
                if( !m_changePaint )
                    this.Invalidate();
                m_changePaint	= true;
            }
            return result;
        }

        /// <summary>
        /// 表示Bitmapの作成
        /// </summary>
        /// <param name="color">HSV(書き換えにHを使用)</param>
        public void	CreateColorBitmap( HSV color )
        {
            Color	col	= RGB.FromHSV( new HSV( color.H, 100, 100 ) ).GetColor();
            Point	pp	= new Point( (int)Math.Round( color.S * (this.Size.Width-1) / 100d ), (int)Math.Round( (100 - color.V) * (this.Size.Height-1) / 100d ) );
            m_color	= col;
            this.UpdateCurrentColorBitmap( col, pp, true );
            m_changePaint	= true;
        }
    }
}
