﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace NW4R.ColorPicker
{
    /// <summary>
    /// 16進カラーボックスクラス。
    /// </summary>
    public sealed class HexColorBox : UserControl
    {
        private System.Windows.Forms.Label m_label;
        private System.Windows.Forms.TextBox m_txtValue;
        private Color m_color = Color.Black;

        /// <summary>
        /// 必要なデザイナー変数です。
        /// </summary>
        private System.ComponentModel.IContainer components = null;

        public HexColorBox()
        {
            InitializeComponent();
        }

        /// <summary>
        /// 使用中のリソースをすべてクリーンアップします。
        /// </summary>
        /// <param name="disposing">マネージ リソースを破棄する場合は true を指定し、その他の場合は false を指定します。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing && (components != null))
            {
                components.Dispose();
            }
            base.Dispose(disposing);
        }

        #region コンポーネント デザイナーで生成されたコード

        /// <summary>
        /// デザイナー サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディターで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.m_label = new System.Windows.Forms.Label();
            this.m_txtValue = new System.Windows.Forms.TextBox();
            this.SuspendLayout();
            //
            // m_label
            //
            this.m_label.Location = new System.Drawing.Point(0, 4);
            this.m_label.Name = "m_label";
            this.m_label.Size = new System.Drawing.Size(16, 19);
            this.m_label.TabIndex = 0;
            this.m_label.Text = "#";
            this.m_label.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            //
            // m_txtValue
            //
            this.m_txtValue.AcceptsReturn = true;
            this.m_txtValue.ImeMode = System.Windows.Forms.ImeMode.Disable;
            this.m_txtValue.Location = new System.Drawing.Point(16, 4);
            this.m_txtValue.MaxLength = 7;
            this.m_txtValue.Multiline = true;
            this.m_txtValue.Name = "m_txtValue";
            this.m_txtValue.Size = new System.Drawing.Size(48, 19);
            this.m_txtValue.TabIndex = 1;
            this.m_txtValue.Text = "0";
            this.m_txtValue.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.m_txtValue.KeyDown += new System.Windows.Forms.KeyEventHandler(this.TextBox_KeyDown);
            this.m_txtValue.KeyPress += new System.Windows.Forms.KeyPressEventHandler(this.TextBox_KeyPress);
            this.m_txtValue.Leave += new System.EventHandler(this.TextBox_Leave);
            //
            // HexColorBox
            //
            this.AutoScaleDimensions = new System.Drawing.SizeF(6F, 12F);
            this.AutoScaleMode = System.Windows.Forms.AutoScaleMode.Font;
            this.Controls.Add(this.m_label);
            this.Controls.Add(this.m_txtValue);
            this.Name = "HexColorBox";
            this.Size = new System.Drawing.Size(64, 24);
            this.ResumeLayout(false);
            this.PerformLayout();

        }

        #endregion

        #region テキストボックスのキーやフォーカスのイベント処理
        /// <summary>
        /// 16進英数字のみ入力許可
        /// </summary>
        private void TextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            if ((e.KeyChar < '0' || e.KeyChar > '9') &&
                (e.KeyChar < 'A' || e.KeyChar > 'F') &&
                (e.KeyChar < 'a' || e.KeyChar > 'f') &&
                (e.KeyChar != '\b'))
            {
                e.Handled = true;
            }
        }

        /// <summary>
        /// リターンキーなら入力確定
        /// </summary>
        private void TextBox_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter || e.KeyCode == Keys.Return)
            {
                TextBox_Leave(sender, EventArgs.Empty);
                m_txtValue.SelectAll();
            }
            else if (e.KeyData == (Keys.Control | Keys.C))
            {
                m_txtValue.Copy();
            }
            else if (e.KeyData == (Keys.Control | Keys.V))
            {
                m_txtValue.Paste();
            }
        }

        /// <summary>
        /// 数値の確定処理
        /// </summary>
        private void TextBox_Leave(object sender, EventArgs e)
        {
            // 6桁未満の場合は先頭に0を付与する
            const int cMaxLength = 6;
            if (m_txtValue.TextLength < cMaxLength)
            {
                string zero = String.Empty;
                int limit = cMaxLength - m_txtValue.TextLength;
                for (int n = 0; n < limit; n++)
                {
                    zero += "0";
                }
                m_txtValue.Text = zero + m_txtValue.Text;
            }

            // 入力値チェック
            bool bError = false;
            if (CheckInputParam(m_txtValue.Text) == false)
            {
                // 不正な値が入力された場合は現在値を設定する
                m_txtValue.Text = ConverColorToHtmlString(m_color);
                bError = true;
            }

            // 色を確定する
            String htmlColor = String.Format("#{0}", m_txtValue.Text);
            m_color = ColorTranslator.FromHtml(htmlColor);

            // RGB値をUIに反映させる
            ColorChangedEventArgs arg = new ColorChangedEventArgs(m_color, true);
            ColorChanged(sender, arg);

            if (bError)
            {
                // エラーダイアログを表示する
                if (HexColorError != null)
                {
                    HexColorError(sender, null);
                }
            }
        }

        #endregion

        #region イベント

        /// <summary>
        /// 16進カラーのエラーイベントを処理するメソッドを表します。
        /// </summary>
        public delegate void HexColorErrorEventHandler(object sender);

        /// <summary>
        /// カラーが変更されたときに発生します。
        /// </summary>
        public event ColorChangedEventHandler ColorChanged;

        /// <summary>
        /// 16進カラーのエラーが検出されたときに発生します。
        /// </summary>
        public event EventHandler HexColorError;

        #endregion

        #region プロパティ
        /// <summary>
        /// 色情報。
        /// </summary>
        public Color Color
        {
            get { return m_color; }
            set
            {
                m_color = value;
                m_txtValue.Text = ConverColorToHtmlString(m_color);
            }
        }

        #endregion

        #region ローカル関数
        /// <summary>
        /// RGBを16進文字列に変換します。
        /// </summary>
        private string ConverColorToHtmlString(Color color)
        {
            return String.Format("{0:X2}{1:X2}{2:X2}", color.R, color.G, color.B);
        }

        /// <summary>
        /// 入力チェックを行ないます。
        /// </summary>
        private bool CheckInputParam(string str)
        {
            uint uint_param;
            if (uint.TryParse(str, NumberStyles.AllowHexSpecifier, CultureInfo.CurrentCulture, out uint_param))
            {
                // 0x000000～0xFFFFFFまでを有効範囲とする
                if ((uint)0x000000 <= uint_param && uint_param <= (uint)0xFFFFFF)
                {
                    return true;
                }
            }

            return false;
        }

        #endregion

    }
}
