﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Data;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace NW4R.ColorPicker
{
    /// <summary>
    /// カラースライダーの基底クラス。
    /// </summary>
    public class ColorSlider : Control
    {
        private int		m_currentPoint	= 0;
        private int		m_currentValue	= 0;
        private int		m_max	= 255;
        private int		m_changeCount	= 0;
        /// <summary>スライダー表示用Bitmap</summary>
        protected Bitmap	m_imageBMP	= null;
        /// <summary>選択保存カラー</summary>
        protected Color m_color = Color.Black;
        /// <summary>マウスダウンフラグ</summary>
        protected bool	m_isMouseDown	= false;
        /// <summary>描画フラグ</summary>
        protected bool	m_changePaint	= false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ColorSlider()
        {
            // ちらつきの防止用
            this.DoubleBuffered = true;
        }

        #region イベント

        internal delegate void ColorSelectedHandler( object sender, ColorChangedEventArgs e );
        internal event ColorSelectedHandler ColorSelected;

        #endregion

        #region オーバーライド

        /// <summary>
        /// OnMouseDownのオーバーライド（色選択イベント発行）
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (e.Button != MouseButtons.Left) { return; }

            m_isMouseDown = true;
            SliderValue = GetMousePoint(e);
            if (ColorSelected != null)
            {
                ColorSelected(this, new ColorChangedEventArgs(GetPointColor(SliderValue), false));
            }
            m_changeCount = 0;
        }

        /// <summary>
        /// OnMouseUpのオーバーライド
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            //if (e.Button != MouseButtons.Left) { return; }

            m_isMouseDown = false;
            if (ColorSelected != null)
            {
                ColorSelected(this, new ColorChangedEventArgs(GetPointColor(SliderValue), true));
            }
        }

        /// <summary>
        /// OnMouseMoveのオーバーライド（色選択イベント発行）
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);
            if (!m_isMouseDown)
            {
                return;
            }
            if (e.Button != MouseButtons.Left)
            {
                return;
            }

            int num = GetMousePoint(e);
            if (num < 0)
            {
                num = 0;
            }
            else if (num > GetBarSize() - 1)
            {
                num = GetBarSize() - 1;
            }
            if ((++m_changeCount) > 0)
            {
                m_changeCount = 0;
            }
            SliderValue = num;
            if (m_changeCount == 0 && ColorSelected != null)
            {
                ColorSelected(this, new ColorChangedEventArgs(GetPointColor(SliderValue), false));
            }
        }

        /// <summary>
        /// OnPaintのオーバーライド（ビットマップ表示）
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            if (this.DesignMode) { return; }

            if (this.Enabled && m_imageBMP != null)
            {
                Graphics g = e.Graphics;
                // 補間モード変更
                InterpolationMode im = g.InterpolationMode;
                g.InterpolationMode = InterpolationMode.NearestNeighbor;

                g.DrawImage(m_imageBMP, this.DisplayRectangle, -0.5f, -0.5f, m_imageBMP.Width, m_imageBMP.Height, GraphicsUnit.Pixel);
                // 補間モード復帰
                g.InterpolationMode = im;

                m_changePaint = false;
            }
            else
            {
                base.OnPaint(e);
            }
        }

        /// <summary>
        /// ColorValue計算用の幅を返す
        /// </summary>
        protected virtual int GetBarSize()
        {
            return this.ClientSize.Width;
        }

        /// <summary>
        /// デザインモード時は親を再描画
        /// </summary>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (this.DesignMode)
            {
                if (this.Parent != null)
                {
                    this.Parent.Invalidate();
                }
            }
            base.OnSizeChanged(e);
        }

        #endregion

        #region プロパティ
        /// <summary>
        /// RGBAのいずれか？
        /// </summary>
        public int MaxValue
        {
            get { return m_max; }
            set { m_max = value; }
        }

        /// <summary>
        /// カラー値（整数）
        /// </summary>
        public int ColorValue
        {
            get { return m_currentValue; }
            set
            {
                m_currentValue = value;
                m_currentPoint = (int)Math.Round((double)value * (GetBarSize() - 1) / m_max);
            }
        }

        /// <summary>
        /// スライダー値（横向きのスライダーならX座標、縦向きのスライダーならY座標）
        /// </summary>
        protected int SliderValue
        {
            get { return m_currentPoint; }
            set
            {
                m_currentPoint = value;
                m_currentValue = (int)Math.Round((double)value * m_max / (GetBarSize() - 1));
            }
        }

        #endregion

        /// <summary>
        /// 指定位置色の取得
        /// </summary>
        public virtual Color GetPointColor(int p)
        {
            return Color.Empty;
        }

        /// <summary>
        /// マウスの位置を返す
        /// </summary>
        protected virtual int GetMousePoint(MouseEventArgs e)
        {
            return e.X;
        }

        /// <summary>
        /// 保存されている選択カラー
        /// </summary>
        public Color GetSelectColor()
        {
            return m_color;
        }
    }
}
