﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;

    public class CmdLine
    {
        private readonly OptionMap options = new OptionMap();
        private readonly StringList paramList = new StringList();

        public CmdLine()
        {
        }

        public enum Option
        {
            None,
            LTOne,
            One,
        }

        public string this[string name]
        {
            get
            {
                StringList value;
                if (this.options.TryGetValue(name, out value))
                {
                    Debug.Assert(value.Count > 0);
                    // 最後に設定された値を使う
                    return value[value.Count - 1];
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        public bool Exists(string name)
        {
            return this.options.ContainsKey(name);
        }

        public bool GetFlag(string name)
        {
            if(this.options.ContainsKey(name))
            {
                bool result;
                return bool.TryParse(this[name], out result) ? result : false;
            }

            return false;
        }

        public StringList FindOptionValueOf(string name)
        {
            StringList value;
            if (this.options.TryGetValue(name, out value))
            {
                return value;
            }

            return null;
        }

        public int Number(string name, int def)
        {
            StringList value;

            if (this.options.TryGetValue(name, out value))
            {
                return int.Parse(value[0]);
            }
            else
            {
                return def;
            }
        }

        public void SetOptionValue(string optionName, string value)
        {
            StringList valueList;
            if (this.options.TryGetValue(optionName, out valueList))
            {
                valueList.Clear();
                valueList.Add(value);
            }
            else
            {
                valueList = new StringList();
                valueList.Add(value);
                this.options.Add(optionName, valueList);
            }
        }

        public int GetNum()
        {
            return this.options.Count;
        }

        private bool IsNUL(char c)
        {
            return c == '\0';
        }

        private bool IsWhiteSpace(char c)
        {
            return c <= ' ';
        }

        private int SkipWhiteSpace(string str, int pos)
        {
            while (pos < str.Length && this.IsWhiteSpace(str[pos]))
            {
                pos++;
            }

            return pos;
        }

        public void AnalyzeAdditionalOption(string cmdline, OptionDef[] options, int num)
        {
            Analyze(cmdline, options, num);
        }

        public void Analyze(string cmdline, OptionDef[] options, int num)
        {
            int pos;

            for (pos = 0; pos < cmdline.Length;)
            {
                if (cmdline[pos] == '-')
                {
                    pos++;
                    string option = this.GetToken(cmdline, pos);

                    if (option.Length == 0)
                    {
                        throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_ILLEGAL_HYPHON);
                    }

                    pos += option.Length;

                    OptionDef pod;
                    var isFindOpt = this.FindOptionDef(option, options, num, out pod);

                    if (!isFindOpt)
                    {
                        throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_UNKNONW_OPTION, option);
                    }

                    switch (pod.Type)
                    {
                        case Option.None:
                            {
                                Rpt._RPT1("option 0 {0}\n", option);
                                var temp = this.options[option];
                            }

                            break;

                        case Option.LTOne:
                        case Option.One:
                            {
                                pos = this.SkipWhiteSpace(cmdline, pos);
                                string param = this.GetToken(cmdline, pos);

                                if ((param.Length > 0) && (param[0] != '-'))
                                {
                                    this.options[option].Add(param);
                                    pos += param.Length;
                                }
                                else if (pod.Type == Option.LTOne)
                                {
                                    var temp = this.options[option];
                                }
                                else
                                {
                                    throw GlCm.ErrMsg(ErrorType.CmdLine, Strings.IDS_ERR_LACK_PARAMETER);
                                }

                                Rpt._RPT2("option 1 {0} => {1}\n", option, param);
                            }

                            break;

                        default:
                            throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_UNKNOWN_OPTION_TYPE, pod.Type);
                    }

                    AnalizedAnOption?.Invoke(this, new AnalizedAnOptionEventArgs(option));
                }
                else if (this.IsWhiteSpace(cmdline[pos]))
                {
                    pos = this.SkipWhiteSpace(cmdline, pos);
                }
                else
                {
                    string token = this.GetToken(cmdline, pos);

                    if (token.Length == 0)
                    {
                        throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_TYPE0);
                    }

                    pos += token.Length;

                    this.paramList.Add(token);
                }


            }
        }

        public class AnalizedAnOptionEventArgs : EventArgs
        {
            public string Option { get; set; }

            public AnalizedAnOptionEventArgs(string option)
            {
                Option = option;
            }
        }

        public event EventHandler<AnalizedAnOptionEventArgs> AnalizedAnOption;

        private bool FindOptionDef(string token, OptionDef[] options, int num, out OptionDef value)
        {
            for (int i = 0; i < num; ++i)
            {
                if (options[i].Name == token)
                {
                    value = options[i];
                    return true;
                }
            }

            value = default(OptionDef);
            return false;
        }

        private string GetToken(string str, int pos)
        {
            int ed;
            bool isQuoted = false;

            if (str[pos] == '"')
            {
                pos++;
                isQuoted = true;
            }

            for (ed = pos; ed < str.Length; ++ed)
            {
                if (!isQuoted && this.IsWhiteSpace(str[ed]))
                {
                    break;
                }
                else if (isQuoted && str[ed] == '"')
                {
                    break;
                }
            }

            return str.Substring(pos, ed - pos);
        }

        public struct OptionDef
        {
            public readonly string Name;
            public readonly Option Type;
            private readonly Func<CmdLine, StringList, string> _WarningFunc;

            public OptionDef(string name, Option type)
                : this(name, type, null)
            {
            }

            public OptionDef(string name, Option type, Func<CmdLine, StringList, string> warningFunc)
            {
                this.Name = name;
                this.Type = type;
                this._WarningFunc = warningFunc;
            }

            public string GetWarningMesasge(CmdLine options, StringList values)
            {
                return this._WarningFunc != null ? this._WarningFunc(options, values) : string.Empty;
            }
        }

        protected class OptionMap : Dictionary<string, StringList>
        {
            public new StringList this[string key]
            {
                get
                {
                    StringList value;
                    if (!this.TryGetValue(key, out value))
                    {
                        value = new StringList();
                        this.Add(key, value);
                    }

                    return value;
                }
            }
        }
    }
}
