﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Runtime.InteropServices;
    using NintendoWare.Font.Runtime;
    using NW4F.LayoutBinaryConverter;
    using TexConv = NW4F.LayoutBinaryConverter.TexConv;

    public class GenericFontWriter : NitroFontWriter
    {
        private bool _convertBCFormatToRGBA8 = false;
        private GenericTextureWriter _genericTextureWriter = null;

        protected override GlyphImageFormat ConvertGlyphImageFormat_(GlyphImageFormat gif)
        {
            // CTR lyt2 のPCプレビュー用途で、Cafe用フォントを出力する場合。
            if (_convertBCFormatToRGBA8)
            {
                // BC系圧縮フォーマットを使うと、絵が壊れてしまうため
                // 問題の回避として、実機で BC系圧縮フォーマットに変換される可能性があるフォーマットはRGBA8へと変換します。
                switch (gif)
                {
                    case GlyphImageFormat.RGB565:
                    case GlyphImageFormat.RGB8:
                    case GlyphImageFormat.RGB5A1:
                    case GlyphImageFormat.RGB5A3:
                    case GlyphImageFormat.RGBA4:
                    case GlyphImageFormat.RGBA8:
                        return GlyphImageFormat.RGBA8;
                    default: return gif;
                }
            }

            return gif;
        }

        /// <summary>
        /// ガンマ補正が必要かどうか
        /// </summary>
        private bool IsDegamma(GlyphImageFormat format)
        {
            // 明度・アルファ値を表すフォーマットは、ガンマ補正の必要はない
            // カラーフォントで利用される、Srgbフォーマットのテクスチャは実行時に処理されるのでやはり補正は不要。
            return false;
        }

        /// <summary>
        /// SRGB テクスチャフォーマットを利用するかどうか
        /// </summary>
        private bool IsSrgbFormat(GlyphImageFormat format)
        {
            if(format == GlyphImageFormat.RGB565_SRGB || format == GlyphImageFormat.RGB5A3_SRGB || format == GlyphImageFormat.RGBA8_SRGB ||
               format == GlyphImageFormat.RGBA8_BC7_SRGB)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 実機テクスチャ列に変換します。
        /// </summary>
        protected override List<byte> ConvertToNativeImage_(RgbImage sheet_plane, GlyphImageFormat gif)
        {
            // すべてのシートを一枚一枚変換しながらListに結果を詰めていきます。
            bool needToInvertColortValue = NeedToInvertColortValue(gif);
            int sheetW = this.TglpInfo.SheetWidth;
            int sheetH = this.TglpInfo.SheetHeight;

            List<TexConv.SrcImage> srcImages = new List<TexConv.SrcImage>();
            for (int sheetIdx = 0; sheetIdx < this.TglpInfo.SheetNum; sheetIdx++)
            {
                int offsetY = sheetH * sheetIdx;

                TexConv.SrcImage image = new TexConv.SrcImage();
                image.name = sheetIdx.ToString();
                image.lyColor = new TexConv.Layer(TexConv.LayerType.COLOR_RGB24, sheetW, sheetH);
                image.lyAlpha = new TexConv.Layer(TexConv.LayerType.ALPHA_A8, sheetW, sheetH);
                image.dstFormat = GlyphImageFormatToTexConvTexelFormat(gif);
                image.isIndirectTexture = false;
                image.isDegamma = IsDegamma(gif);
                image.isSrgbFormat = IsSrgbFormat(gif);
                image.swizzlePattern = 0;

                for (int y = 0; y < sheetH; y++)
                {
                    // テクスチャ座標系が反転しているため、画像を逆順にコピーします。
                    int dstReverseY = sheetH - 1 - y;
                    int srcY = y + offsetY;
                    for (int x = 0; x < sheetW; x++)
                    {
                        var pixcel = sheet_plane.GetPixel(x, srcY);

                        // 色の反転が必要な形式
                        if (needToInvertColortValue)
                        {
                            pixcel.R = (byte)(255 - pixcel.R);
                            pixcel.G = (byte)(255 - pixcel.G);
                            pixcel.B = (byte)(255 - pixcel.B);
                        }

                        // シェーダ側で特殊な解釈をする形式。
                        if (gif == GlyphImageFormat.LA4Packed ||
                            gif == GlyphImageFormat.LA4PackedNoCompress)
                        {
                            byte packedValue = 0;
                            if (pixcel.A == 255)
                            {
                                // 128 - 255 にルミナンス値として書き込む。アルファ値は 1 と解釈されます。
                                packedValue = (byte)((pixcel.R >> 1) + 128);
                            }
                            else
                            {
                                // 0 - 127 にアルファ値として書き込む。ルミナンスは 0 と解釈されます。
                                packedValue = (byte)(pixcel.A >> 1);
                            }

                            pixcel.R = packedValue;
                            pixcel.G = packedValue;
                            pixcel.B = packedValue;
                            pixcel.A = packedValue;
                        }
                        else if (gif == GlyphImageFormat.A4 || gif == GlyphImageFormat.A8)
                        {
                            // 実機テクスチャ変換モジュールは、A の情報だけが採用されるので、R チャンネルの情報をAに設定します。
                            // ImageFontReader.PixelFormatNormalization で A チャンネルの情報は1ビット相当に丸められているため、採用しません。
                            pixcel.A = pixcel.R;
                        }

                        image.lyColor.SetValue(x, dstReverseY, pixcel.R, pixcel.G, pixcel.B);
                        image.lyAlpha.SetValue(x, dstReverseY, pixcel.A, 0, 0);
                    }
                }

                srcImages.Add(image);
            }

            List<byte> convertResultTotal = new List<byte>();

            TexConv.ConvertedImage convertedImage = _genericTextureWriter.ConvertSrcImageToNative2DArrayFormat(srcImages.ToArray(), ConverterEnvironment.TileMode, ConverterEnvironment.UseBntx);
            if (convertedImage != null)
            {
                convertResultTotal.AddRange(convertedImage.image);
            }

            return convertResultTotal;
        }

        /// <summary>
        /// 実機テクスチャのフォーマットを取得します。
        /// </summary>
        protected override ushort GlyphImageFormatToTextureFormat(GlyphImageFormat gif)
        {
            // 現状ではRGBA8以外は圧縮フォーマットに変換する
            ushort result = 0;
            switch (gif)
            {
                case GlyphImageFormat.A4:
                case GlyphImageFormat.A8: result = (ushort)TextureFormatGeneric.BC4; break;
                case GlyphImageFormat.LA4:
                case GlyphImageFormat.LA8: result = (ushort)TextureFormatGeneric.BC5; break;
                case GlyphImageFormat.LA4Packed: result = (ushort)TextureFormatGeneric.BC4; break;
                case GlyphImageFormat.LA4PackedNoCompress: result = (ushort)TextureFormatGeneric.A8; break;
                case GlyphImageFormat.LA4PackedPassThrough: result = (ushort)TextureFormatGeneric.BC4; break;
                case GlyphImageFormat.LA4PackedNoCompressPassThrough: result = (ushort)TextureFormatGeneric.A8; break;
                case GlyphImageFormat.RGB565:

                case GlyphImageFormat.RGB8: result = (ushort)TextureFormatGeneric.BC1; break;
                case GlyphImageFormat.RGB5A1:
                case GlyphImageFormat.RGB5A3:
                case GlyphImageFormat.RGBA4: result = (ushort)TextureFormatGeneric.BC3; break;
                case GlyphImageFormat.RGBA8: result = (ushort)TextureFormatGeneric.RGBA8; break;

                case GlyphImageFormat.RGB565_SRGB: result = (ushort)TextureFormatGeneric.BC1_SRGB; break;
                case GlyphImageFormat.RGB5A3_SRGB: result = (ushort)TextureFormatGeneric.BC3_SRGB; break;
                case GlyphImageFormat.RGBA8_SRGB: result = (ushort)TextureFormatGeneric.RGBA8_SRGB; break;

                case GlyphImageFormat.RGBA8_BC7: result = (ushort)TextureFormatGeneric.BC7; break;
                case GlyphImageFormat.RGBA8_BC7_SRGB: result = (ushort)TextureFormatGeneric.BC7_SRGB; break;

                default: throw new InvalidOperationException("unsupported format. " + gif);
            }

            // プラットフォームに応じてリニアフォーマットを示すフラグを設定します。
            // とりあえずNXのみ何もしない

            if (string.Compare(ConverterEnvironment.TileMode, "NX", StringComparison.OrdinalIgnoreCase) == 0)
            {
                ;
            }
            else
            {
                result |= (ushort)NitroFontWriter.GlyphImageFormatFlag.LinearFormat;
            }

            return result;
        }

        /// <summary>
        /// テクスチャ変換向け実機テクスチャのフォーマットを取得します。
        /// </summary>
        protected override TexConv::TexelFormat GlyphImageFormatToTexConvTexelFormat(GlyphImageFormat gif)
        {
            // 現状ではRGBA8以外は圧縮フォーマットに変換する
            switch (gif)
            {
                case GlyphImageFormat.A4:
                case GlyphImageFormat.A8: return TexConv::TexelFormat.BC4A;
                case GlyphImageFormat.LA4:
                case GlyphImageFormat.LA8: return TexConv::TexelFormat.BC5;
                case GlyphImageFormat.LA4Packed: return TexConv::TexelFormat.BC4A;
                case GlyphImageFormat.LA4PackedPassThrough: return TexConv::TexelFormat.BC4A;
                case GlyphImageFormat.LA4PackedNoCompress: return TexConv::TexelFormat.A8;
                case GlyphImageFormat.LA4PackedNoCompressPassThrough: return TexConv::TexelFormat.A8;
                case GlyphImageFormat.RGB565:
                case GlyphImageFormat.RGB565_SRGB:
                case GlyphImageFormat.RGB8: return TexConv::TexelFormat.BC1;
                case GlyphImageFormat.RGB5A1:
                case GlyphImageFormat.RGB5A3:
                case GlyphImageFormat.RGB5A3_SRGB:
                case GlyphImageFormat.RGBA4: return TexConv::TexelFormat.BC3;
                case GlyphImageFormat.RGBA8:
                case GlyphImageFormat.RGBA8_SRGB: return TexConv::TexelFormat.RGBA8;
                case GlyphImageFormat.RGBA8_BC7: return TexConv::TexelFormat.BC7;
                case GlyphImageFormat.RGBA8_BC7_SRGB: return TexConv::TexelFormat.BC7;
                default: throw new InvalidOperationException("unsupported format. " + gif);
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public GenericFontWriter(
            string file,
            string groupsFile,
            GlyphDataType type,
            uint alter,
            int linefeed,
            int width,
            int left,
            int right,
            CharEncoding encoding,
            int sheetWidth,
            int sheetHeight,
            int sheetPixels,
            bool isOutputTargetLittleEndian,
            bool isOutKerningData,
            bool convertBCFormatToRGBA8 = false)
            : base(file, groupsFile, type, alter, linefeed, width, left, right, encoding, sheetWidth, sheetHeight, sheetPixels, isOutputTargetLittleEndian, isOutKerningData)
        {
            _convertBCFormatToRGBA8 = convertBCFormatToRGBA8;
            _genericTextureWriter = new GenericTextureWriter();

            string execPath = Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
            string dllPath = GetGenericEncoderDllLoadPath_(execPath);
            if(_genericTextureWriter.InitializeEncoder(dllPath) == false)
            {
                ProgressControl.Error(Strings.ErrorGenericTextureEncoderCantInitialized);
                _genericTextureWriter = null;
            }

            _genericTextureWriter.SetLogSilent(ConverterEnvironment.IsLogSilent);
            _genericTextureWriter.SetGpuEncodingEnabled(ConverterEnvironment.IsGpuEncodingEnabled);
        }

        /// <summary>
        /// Generic用 エンコーダー用 の DLL パスを取得する。
        /// </summary>
        static string GetGenericEncoderDllLoadPath_(string execPath)
        {
            // 正式な位置
            string dllPath = execPath + "\\..\\GraphicsTools";
            if (Directory.Exists(dllPath))
            {
                return dllPath;
            }

            // 開発時の出力フォルダ位置
            // Encoder.vcxproj の Debug ビルドのビルド後イベントで下記の場所に 必要な dll 一式がコピーされます。
            string dllPathForDebug = execPath + "\\..\\..\\..\\..\\..\\..\\..\\..\\..\\Tools\\Graphics\\GraphicsTools";
            if (Directory.Exists(dllPathForDebug))
            {
                return dllPathForDebug;
            }

            // 開発時の出力フォルダ位置
            // Encoder.vcxproj の Debug ビルドのビルド後イベントで下記の場所に 必要な dll 一式がコピーされます。
            dllPathForDebug = execPath + "\\..\\..\\..\\..\\..\\..\\Win32\\Tools\\Graphics\\GraphicsTools\\TextureConverter\\Debug";
            if (Directory.Exists(dllPathForDebug))
            {
                return dllPathForDebug;
            }

            ProgressControl.Error(Strings.ErrorGenericTextureEncoderCantFound, dllPath, dllPathForDebug);

            return string.Empty;
        }
    }
}
