﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using System.Diagnostics.CodeAnalysis;
    using System.IO;
    using System.Runtime.InteropServices;

    public class TgaFile
    {
        private const string TgaSignature = "TRUEVISION-XFILE.\0";

        private const string TgaFontcvtrId = "fontcvtr output.";

        private string fileName;

        public TgaFile(string fname)
        {
            this.fileName = fname;
        }

        protected enum ColorMap
        {
            None,
            Exists,
        }

        [Flags]
        protected enum ImageType
        {
            FlagRunLengthCompressed = (1 << 3),
            None = 0,
            ColorMap = 1,
            TrueColor = 2,
            BlackAndWhite = 3,
            RunLengthColorMap = (FlagRunLengthCompressed | ColorMap),
            RunLengthTrueColor = (FlagRunLengthCompressed | TrueColor),
            RunLengthBlackAndWhite = (FlagRunLengthCompressed | BlackAndWhite)
        }

        [Flags]
        protected enum ImageDescFormat
        {
            AlphaMask = (0xF << 0),
            RightToLeft = (0x1 << 4),
            TopToBottom = (0x1 << 5),
            UnusedMask = (0x3 << 6)
        }

        [Flags]
        protected enum RunLengthPacketFormat
        {
            RunLengthFlag = 0x80,
            NumDataMask = 0x7F
        }

        public ImageBase Load()
        {
            ImageBase image;
            TgaFileHeader tfh;

            using (var binaryFile = BinaryFile.Open(this.fileName, FileMode.Open, FileAccess.Read))
            {
                var br = new ByteOrderBinaryReader(binaryFile, true);
                this.LoadHeader(br, out tfh);
                this.ValidateFormat(tfh);
                this.ValidateFuture(tfh);

                var isIndex = tfh.IsIndex;

                if (isIndex)
                {
                    IndexImage p = new IndexImage();
                    this.LoadColorTable(br, p, tfh);
                    image = p;
                }
                else
                {
                    image = new RgbImage();
                }

                // read body
                {
                    int bytePerPixel = (tfh.PixelDepth + 7) / 8;
                    int rawImageDataSize = tfh.ImageWidth * tfh.ImageHeight * bytePerPixel;
                    var rawImage = new byte[rawImageDataSize];
                    var normalOrderRawImage = new byte[rawImageDataSize];

                    if (tfh.IsRunLengthCompressed)
                    {
                        var fileSize = (int)binaryFile.Length;
                        var bmpImage = new byte[fileSize];
                        br.Read(bmpImage, 0, fileSize);

                        this.DecompressRunLength(
                            rawImage,
                            bmpImage,
                            tfh.ImageWidth,
                            tfh.ImageHeight,
                            bytePerPixel);
                    }
                    else
                    {
                        br.Read(rawImage, 0, rawImageDataSize);
                    }

                    ReorderImageData(
                        normalOrderRawImage,
                        rawImage,
                        tfh.ImageWidth,
                        tfh.ImageHeight,
                        bytePerPixel,
                        (tfh.ImageDesc & (int)ImageDescFormat.RightToLeft) != 0,
                        (tfh.ImageDesc & (int)ImageDescFormat.TopToBottom) == 0);

                    image.Create(tfh.ImageWidth, tfh.ImageHeight, tfh.PixelDepth);
                    image.Paste(normalOrderRawImage, 1);

                    if ((isIndex && (tfh.ColorMapEntrySize == 32))
                        || (!isIndex && tfh.PixelDepth == 32))
                    {
                        image.EnableAlpha();
                    }
                }
            }

            return image;
        }

        public void Save(ImageBase image)
        {
            Debug.Assert(image.IsValid);
            byte[] bmpImageRL;
            IntColor[] colorPalette = null;
            IndexImage indexImage;
            TgaFileHeader fh = new TgaFileHeader();
            TgaFileFooter ff = new TgaFileFooter();
            int numColor = 0;
            int imageSize = 0;

            if ((image.Width > 65535) || (image.Height > 65535))
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_TGA_OVER_SIZE, image.Width, image.Height);
            }

            indexImage = image as IndexImage;

            if (indexImage != null)
            {
                numColor = indexImage.GetColorTableEntryNum();
                colorPalette = new IntColor[numColor];
            }

            // prepaire info
            {
                // tga file header
                {
                    // UNNES memset(&fh, 0, sizeof(fh));
                    fh.IdLength = (byte)TgaFontcvtrId.Length;
                    fh.ColorMapType = (byte)(numColor > 0 ? ColorMap.Exists : ColorMap.None);
                    fh.ImageType = (byte)(numColor > 0 ? ImageType.RunLengthColorMap : ImageType.RunLengthTrueColor);
                    fh.FirstEntryIndex = 0;
                    fh.ColorMapLength = (ushort)numColor;
                    fh.ColorMapEntrySize = (byte)(numColor > 0 ? 32 : 0); // 32 bit per color
                    fh.XOrigin = 0;
                    fh.YOrigin = 0;
                    fh.ImageWidth = (ushort)image.Width;
                    fh.ImageHeight = (ushort)image.Height;
                    fh.PixelDepth = (byte)image.Bpp;
                    fh.ImageDesc = (byte)(image.IsEnableAlpha ? 8 : 0);
                }

                // color palette
                if (indexImage != null)
                {
                    for (int i = 0; i < numColor; ++i)
                    {
                        colorPalette[i] = indexImage.GetColorTable()[i];
                    }
                }

                // image body
                {
                    int bytePerPixel = (image.Bpp + 7) / 8;
                    int rawImageSize = fh.ImageWidth * fh.ImageHeight * bytePerPixel;
                    byte[] rawImage = new byte[rawImageSize];
                    byte[] reorderRawImage = new byte[rawImageSize];

                    image.Extract(rawImage, 1);
                    ReorderImageData(
                        reorderRawImage,
                        rawImage,
                        fh.ImageWidth,
                        fh.ImageHeight,
                        bytePerPixel,
                        false,
                        true);

                    // RunLength後は最大でも 2 倍にしかならない
                    bmpImageRL = new byte[rawImageSize * 2];
                    imageSize = CompressRunLength(
                        bmpImageRL,
                        reorderRawImage,
                        fh.ImageWidth,
                        fh.ImageHeight,
                        bytePerPixel);
                }

                // tga file footer
                {
                    // UNNES memset(&ff, 0, sizeof(ff));

                    // 0 はそれぞれ非存在を示す。
                    ff.ExtensionAreaOffset = 0;
                    ff.DeveloperDirectoryOffset = 0;

                    CopySignature(TgaSignature, ff.Signature);
                }
            }

            // save
            using (var bw = new ByteOrderBinaryWriter(BinaryFile.Open(this.fileName, FileMode.Create, FileAccess.Write), true))
            {
                // write header
                bw.Write(fh);
                var bufId = new byte[TgaFontcvtrId.Length];
                CopySignature(TgaFontcvtrId, bufId);
                bw.Write(bufId, 0, bufId.Length);

                // write palette
                for (var i = 0; i < numColor; ++i)
                {
                    var col = colorPalette[i];
                    var buf = new byte[4];
                    buf[3] = col.R;
                    buf[2] = col.G;
                    buf[1] = col.B;
                    buf[0] = col.A;

                    bw.Write(buf);
                }

                // write body
                bw.Write(bmpImageRL, 0, imageSize);

                // write footer
                bw.Write(ff);
            }
        }

        public bool HasValidIdentifier()
        {
            TgaFileFooter tff;

            using (var binaryFile = BinaryFile.Open(this.fileName, FileMode.Open, FileAccess.Read))
            {
                var br = new ByteOrderBinaryReader(binaryFile, true);
                this.LoadFooter(br, out tff);
            }

            return this.IsValidFooter(tff);
        }

        public bool IsIndex()
        {
            TgaFileHeader tfh;

            using (var binaryFile = BinaryFile.Open(this.fileName, FileMode.Open, FileAccess.Read))
            {
                var br = new ByteOrderBinaryReader(binaryFile, true);
                this.LoadHeader(br, out tfh);
            }

            this.ValidateFormat(tfh);
            return tfh.IsIndex;
        }

        public bool IsInvalid()
        {
            TgaFileHeader tfh;

            using (var binaryFile = BinaryFile.Open(this.fileName, FileMode.Open, FileAccess.Read))
            {
                var br = new ByteOrderBinaryReader(binaryFile, true);
                this.LoadHeader(br, out tfh);
            }

            try
            {
                this.ValidateFormat(tfh);
            }
            catch (GeneralException)
            {
                return true;
            }

            return false;
        }

        private static int MakeRLPacket(byte[] outBuf, int outPos, byte[] color, int inpPos, int bytePerPixel, int num)
        {
            while (num > 0)
            {
                int numPacketPixel = Math.Min(num, 128);
                int copySize = bytePerPixel;

                outBuf[outPos++] = (byte)((int)RunLengthPacketFormat.RunLengthFlag | (numPacketPixel - 1));
                Array.Copy(color, inpPos, outBuf, outPos, copySize);

                outPos += copySize;
                num -= numPacketPixel;
            }

            return outPos;
        }

        private static int MakeRawPacket(byte[] outBuf, int outPos, byte[] colors, int inpPos, int bytePerPixel, int num)
        {
            while (num > 0)
            {
                int numPacketPixel = Math.Min(num, 128);
                int copySize = bytePerPixel * numPacketPixel;

                outBuf[outPos++] = (byte)(numPacketPixel - 1);
                Array.Copy(colors, inpPos, outBuf, outPos, copySize);

                outPos += copySize;
                num -= numPacketPixel;
            }

            return outPos;
        }

        private static int CompressRunLength(byte[] outBuf, byte[] image, int width, int height, int bytePerPixel)
        {
            /* pOut には pImage の 2 倍のサイズが確保されている */

            int inpPos = 0;  // const BYTE* inPos = pImage;
            int outPos = 0; // BYTE* outPos = pOut;

            for (int h = 0; h < height; ++h)
            {
                int prevColorPos = 0;   // const BYTE* prevColorPos;
                int rawStartPos = 0;    // const BYTE* rawStartPos;
                int length = 1;
                var isRaw = true;

                prevColorPos = inpPos;
                rawStartPos = inpPos;

                inpPos += bytePerPixel;

                // 2ピクセル目から始める
                for (int w = 1; w < width; ++w)
                {
                    var isNextRaw = !MemComp(image, prevColorPos, inpPos, bytePerPixel);

                    // raw と run length が切り替わった
                    if (isNextRaw != isRaw)
                    {
                        if (isRaw)
                        {
                            // raw -> run length
                            // 最後の一つは same なので length - 1
                            outPos = MakeRawPacket(outBuf, outPos, image, rawStartPos, bytePerPixel, length - 1);
                            length = 1;
                        }
                        else
                        {
                            // run length -> raw
                            outPos = MakeRLPacket(outBuf, outPos, image, prevColorPos, bytePerPixel, length);
                            length = 0;
                            rawStartPos = inpPos;
                        }

                        isRaw = isNextRaw;
                    }

                    length++;
                    prevColorPos = inpPos;
                    inpPos += bytePerPixel;
                }

                if (isRaw)
                {
                    outPos = MakeRawPacket(outBuf, outPos, image, rawStartPos, bytePerPixel, length);
                }
                else
                {
                    outPos = MakeRLPacket(outBuf, outPos, image, prevColorPos, bytePerPixel, length);
                }
            }

            return outPos;
        }

        private static void ReorderImageData(byte[] outBuf, byte[] image, int width, int height, int bytePerPixel, bool isHReverse, bool isVReverse)
        {
            int lineDataSize = width * bytePerPixel;
            int outPos = 0;

            for (int h = 0; h < height; ++h)
            {
                int inpPos;

                if (isVReverse)
                {
                    inpPos = lineDataSize * (height - h - 1);
                }
                else
                {
                    inpPos = lineDataSize * h;
                }

                if (isHReverse)
                {
                    inpPos += lineDataSize;
                    for (int w = 0; w < width; --w)
                    {
                        inpPos -= bytePerPixel;
                        Array.Copy(image, inpPos, outBuf, outPos, bytePerPixel);
                        outPos += bytePerPixel;
                    }
                }
                else
                {
                    Array.Copy(image, inpPos, outBuf, outPos, lineDataSize);
                    outPos += lineDataSize;
                }
            }
        }

        private static bool SignatureEquals(byte[] b1, string b2)
        {
            if (b1.Length != b2.Length)
            {
                return false;
            }

            for (var i = 0; i < b1.Length; i++)
            {
                if (b1[i] != b2[i])
                {
                    return false;
                }
            }

            return true;
        }

        private static void CopySignature(string sig, byte[] buf)
        {
            Debug.Assert(sig.Length == buf.Length);

            for (var i = 0; i < sig.Length; i++)
            {
                buf[i] = (byte)sig[i];
            }
        }

        private static bool MemComp(byte[] buf, int pos1, int pos2, int size)
        {
            for (int i = 0; i < size; ++i)
            {
                if (buf[pos1++] != buf[pos2++])
                {
                    return false;
                }
            }

            return true;
        }

        private void LoadHeader(ByteOrderBinaryReader br, out TgaFileHeader ptfh)
        {
            br.Seek(0, SeekOrigin.Begin);

            br.Read(out ptfh);
            br.Seek(ptfh.IdLength, SeekOrigin.Current);

            Rpt._RPT1("idLength:          {0}\n", ptfh.IdLength);
            Rpt._RPT1("colorMapType:      {0}\n", ptfh.ColorMapType);
            Rpt._RPT1("imageType:         {0}\n", ptfh.ImageType);
            Rpt._RPT1("firstEntryIndex:   {0}\n", ptfh.FirstEntryIndex);
            Rpt._RPT1("colorMapLength:    {0}\n", ptfh.ColorMapLength);
            Rpt._RPT1("colorMapEntrySize: {0}\n", ptfh.ColorMapEntrySize);
            Rpt._RPT2("origin:            ({0}, {1})\n", ptfh.XOrigin, ptfh.YOrigin);
            Rpt._RPT2("imageSize:         {0} x {1}\n", ptfh.ImageWidth, ptfh.ImageHeight);
            Rpt._RPT1("pixelDepth:        {0}\n", ptfh.PixelDepth);
            Rpt._RPT1("imageDesc:         {0}\n", ptfh.ImageDesc);
        }

        private void LoadFooter(ByteOrderBinaryReader br, out TgaFileFooter ptff)
        {
            var footerSize = Marshal.SizeOf(typeof(TgaFileFooter));
            br.Seek(-footerSize, SeekOrigin.End);
            br.Read(out ptff);

            Rpt._RPT1("extensionAreaOffset:      {0:X8}\n", ptff.ExtensionAreaOffset);
            Rpt._RPT1("developerDirectoryOffset: {0:X8}\n", ptff.DeveloperDirectoryOffset);
            Rpt._RPT0("signature:                ");
            Array.ForEach(ptff.Signature, (ch) => Rpt._RPT1("{0:X2}", ch));
            Rpt._RPT0("\n");
        }

        private void LoadColorTable(ByteOrderBinaryReader br, IndexImage image, TgaFileHeader tfh)
        {
            int bytePerEntry = (tfh.ColorMapEntrySize + 7) / 8;
            int numEntry = tfh.ColorMapLength;
            int colorTableSize = bytePerEntry * numEntry;

            var colorTable = new IntColor[numEntry];
            var colorTableData = new byte[colorTableSize];
            int pos = 0;    // const BYTE* pos = colorTableData;

            //---- ファイルはオープン済み、シーク済み
            br.Read(colorTableData, 0, colorTableSize);

            for (int i = 0; i < numEntry; ++i)
            {
                int r, g, b, a;

                switch (tfh.ColorMapEntrySize)
                {
                case 15:
                    {
                        var c = (colorTableData[pos + 1] << 8) | (colorTableData[pos + 0] << 0);

                        r = (c >> 0) & 0x1F;
                        g = (c >> 5) & 0x1F;
                        b = (c >> 10) & 0x1F;
                        a = 0xFF;
                    }

                    break;

                case 16:
                    {
                        var c = (colorTableData[pos + 1] << 8) | (colorTableData[pos + 0] << 0);

                        r = (c >> 0) & 0x1F;
                        g = (c >> 5) & 0x1F;
                        b = (c >> 10) & 0x1F;
                        a = (c >> 15) != 0 ? 0xFF : 0x00;
                    }

                    break;

                case 24:
                    {
                        r = colorTableData[pos + 2];
                        g = colorTableData[pos + 1];
                        b = colorTableData[pos + 0];
                        a = 0xFF;
                    }

                    break;

                case 32:
                    {
                        r = colorTableData[pos + 3];
                        g = colorTableData[pos + 2];
                        b = colorTableData[pos + 1];
                        a = colorTableData[pos + 0];
                    }

                    break;

                default:
                    Debug.Assert(false);
                    r = g = b = a = 0;
                    break;
                }

                colorTable[i] = GlCm.BMP_RGBA((byte)r, (byte)g, (byte)b, (byte)a);
                pos += bytePerEntry;
            }

            image.SetColorTable(colorTable, numEntry);
        }

        private void ValidateFormat(TgaFileHeader tfh)
        {
            // check color palette
            if (tfh.ImageType == (int)ImageType.ColorMap
                || tfh.ImageType == (int)ImageType.RunLengthColorMap)
            {
                if (tfh.ColorMapType != (int)ColorMap.Exists)
                {
                    throw GlCm.ErrMsg(ErrorType.Tga, Strings.IDS_ERR_REQUIRE_PALETTE);
                }

                int entryIndex = tfh.FirstEntryIndex;
                int numEntry = tfh.ColorMapLength;

                if (numEntry == 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Tga, Strings.IDS_ERR_PALETTE_NO_ENTRY);
                }

                if (entryIndex >= numEntry)
                {
                    throw GlCm.ErrMsg(ErrorType.Tga, Strings.IDS_ERR_PALEETE_OVER_ENTRY);
                }
            }

            // check image size
            if (tfh.ImageWidth == 0 || tfh.ImageHeight == 0)
            {
                throw GlCm.ErrMsg(ErrorType.Tga, Strings.IDS_ERR_ILLEGAL_IMAGE_SIZE, this.fileName, tfh.ImageWidth, tfh.ImageHeight);
            }
        }

        private void ValidateFuture(TgaFileHeader tfh)
        {
            var hasPalette = false;

            // check image type
            if (tfh.ImageType != (int)ImageType.ColorMap
                && tfh.ImageType != (int)ImageType.TrueColor
                && tfh.ImageType != (int)ImageType.RunLengthColorMap
                && tfh.ImageType != (int)ImageType.RunLengthTrueColor)
            {
                throw GlCm.ErrMsg(ErrorType.Tga, Strings.IDS_ERR_TGA_UNSUPPORTED_IMAGE_TYPE, tfh.ImageType);
            }

            // check color palette
            if (tfh.ImageType == (int)ImageType.ColorMap
                || tfh.ImageType == (int)ImageType.RunLengthColorMap)
            {
                if (tfh.ColorMapEntrySize != 24
                    && tfh.ColorMapEntrySize != 32)
                {
                    throw GlCm.ErrMsg(ErrorType.Tga, Strings.IDS_ERR_TGA_UNSUPPORTED_PALETTE_ENTRY, tfh.ColorMapEntrySize);
                }

                hasPalette = true;
            }

            // check bpp
            if (!(hasPalette && tfh.PixelDepth == 8)
                && !(hasPalette && tfh.PixelDepth == 16)
                && !(!hasPalette && tfh.PixelDepth == 24)
                && !(!hasPalette && tfh.PixelDepth == 32))
            {
                throw GlCm.ErrMsg(ErrorType.Tga, Strings.IDS_ERR_TGA_UNSUPPORTED_BPP, tfh.ImageType, tfh.PixelDepth);
            }

            // check image descriptor
            if ((tfh.ImageDesc & (int)ImageDescFormat.UnusedMask) != 0)
            {
                throw GlCm.ErrMsg(ErrorType.Tga, Strings.IDS_ERR_TGA_UNSUPPORTED_IMAGE_DESC, tfh.ImageDesc);
            }

            if ((tfh.ImageDesc & (int)ImageDescFormat.AlphaMask) != 8 && (tfh.ImageDesc & (int)ImageDescFormat.AlphaMask) != 0)
            {
                throw GlCm.ErrMsg(ErrorType.Tga, Strings.IDS_ERR_TGA_UNSUPPORTED_ALPHA, tfh.ImageDesc);
            }
        }

        private bool IsValidFooter(TgaFileFooter tfh)
        {
            if (!SignatureEquals(tfh.Signature, TgaSignature))
            {
                return false;
            }

            return true;
        }

        private void DecompressRunLength(byte[] outBuf, byte[] image, int width, int height, int bytePerPixel)
        {
            int rawImageSize = width * height * bytePerPixel;
            int inpPos = 0;
            int outPos = 0;

            while (outPos < rawImageSize)
            {
                byte header = image[inpPos++];
                int numData = (header & (int)RunLengthPacketFormat.NumDataMask) + 1;

                if ((header & (int)RunLengthPacketFormat.RunLengthFlag) != 0)
                {
                    // run length packet
                    for (int i = 0; i < numData; ++i)
                    {
                        Array.Copy(image, inpPos, outBuf, outPos, bytePerPixel);
                        outPos += bytePerPixel;
                    }

                    inpPos += bytePerPixel;
                }
                else
                {
                    // raw packet
                    int copySize = bytePerPixel * numData;
                    Array.Copy(image, inpPos, outBuf, outPos, copySize);
                    outPos += copySize;
                    inpPos += copySize;
                }
            }
        }

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        [SuppressMessage("Microsoft.StyleCop.CSharp.MaintainabilityRules", "SA1401:FieldsMustBePrivate", Justification = "Binay Image")]
        private class TgaFileHeader
        {
            public byte IdLength;

            public byte ColorMapType;

            public byte ImageType;

            public ushort FirstEntryIndex;

            public ushort ColorMapLength;

            public byte ColorMapEntrySize;

            public ushort XOrigin;

            public ushort YOrigin;

            public ushort ImageWidth;

            public ushort ImageHeight;

            public byte PixelDepth;

            public byte ImageDesc;

            public bool IsIndex
            {
                get { return (this.ImageType & 0x3) == (int)TgaFile.ImageType.ColorMap; }
            }

            public bool IsRunLengthCompressed
            {
                get { return (this.ImageType & (int)TgaFile.ImageType.FlagRunLengthCompressed) != 0; }
            }
        }

        [StructLayout(LayoutKind.Sequential, Pack = 2)]
        [SuppressMessage("Microsoft.StyleCop.CSharp.MaintainabilityRules", "SA1401:FieldsMustBePrivate", Justification = "Binay Image")]
        private class TgaFileFooter
        {
            public const int SignatureSize = 18;

            public uint ExtensionAreaOffset;

            public uint DeveloperDirectoryOffset;

            [MarshalAs(UnmanagedType.ByValArray, SizeConst = SignatureSize)]
            public byte[] Signature = new byte[SignatureSize];
        }
    }
}
