﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using NintendoWare.Font.Runtime;

    /// <summary>
    ///
    /// </summary>
    public abstract class NitroFontReader : FontReader
    {
        private readonly string File;

        /// <summary>
        ///
        /// </summary>
        public NitroFontReader(string file)
        {
            this.File = file;
        }

        /// <summary>
        ///
        /// </summary>
        protected delegate ImageBase.Pixel PixelReader();

        /// <summary>
        ///
        /// </summary>
        public override void ReadFontData(FontData fontData, CharFilter filter)
        {
            G2dFont font = new G2dFont();

            // プログレスバー設定
            ProgressControl.GetInstance().SetStatusString(string.Format(Strings.IDS_STATUS_READ_NITRO, LibFormat.ExtensionFont));
            ProgressControl.GetInstance().ResetProgressBarPos();

            // 入力データ読み取り

            // フォントファイルロード
            Load(this.File, font);
            if (font.IsOldVer())
            {
                ProgressControl.Warning(string.Format(Strings.IDS_WARN_OLD_FORAMT, LibFormat.ExtensionFont));
            }

            // フォント抽出
            this.ExtractFont(fontData, filter, font);
        }

        /// <summary>
        ///
        /// </summary>
        public override void ValidateInput()
        {
            // Font File
            ValidateFontFile(this.File);
        }

        /// <summary>
        ///
        /// </summary>
        private static ImageBase.Pixel ConvertPixelA4(uint p)
        {
            ImageBase.Pixel pixel = new ImageBase.Pixel();
            var value = (byte)GlCm.ExtendBits(p, 4, 8);
            var level = (byte)GlCm.InverseNumber(value, 8);

            pixel.Color = GlCm.GrayScaleToRgb(level);
            pixel.A = value;

            return pixel;
        }

        /// <summary>
        ///
        /// </summary>
        private static void ValidateFontFile(string file)
        {
            if (!IsFileExists(file))
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_FILE_NOT_EXISTS, file);
            }

            G2dFont font = new G2dFont();

            // 試しにフォントを読んでみる
            Load(file, font);
        }

        /// <summary>
        ///
        /// </summary>
        private static void Load(string fileName, G2dFont font)
        {
            using (var binaryFile = BinaryFile.Open(fileName, FileMode.Open, FileAccess.Read))
            {
                var size = binaryFile.Length;

                var nnsReader = new NnsReader(binaryFile);
                var headerOffset = nnsReader.ReadHeader(font);
                font.ValidateHeader();

                while (binaryFile.Position < size)
                {
                    nnsReader.ReadBlock(font, font.GenerateInstance);
                }
            }

            font.Rebuild();
        }

        /// <summary>
        ///
        /// </summary>
        private void ExtractFont(FontData fontData, CharFilter filter, G2dFont font)
        {
            ushort alterIndex;
            CharEncoding enc;

            ProgressControl.GetInstance().SetProgressBarMax(0x110000);

            // フォント情報取得
            var infoBlk = font.GetBlock(RtConsts.BinBlockSigFINF);
            var infoEx = (FontInformationEx)infoBlk.Body;
            var body = (FontInformation)infoEx.Body;

            {

                // alterCharIndex, linefeed
                alterIndex = body.AlterCharIndex;
                fontData.LineHeight = body.Linefeed;

                // width, ascent, descent
                if (!font.IsOldVer())
                {
                    fontData.SetWidth(body.Width);
                    fontData.SetHeight(body.Ascent, body.Height - body.Ascent);
                }

                // defaultWidth, encoding
                if (font.IsOldVer())
                {
                    enc = (CharEncoding)body.DefaultWidth.CharWidth;
                }
                else
                {
                    fontData.DefaultWidth = body.DefaultWidth;
                    enc = (CharEncoding)body.Encoding;
                }
            }

            // グリフ取得
            var bmpGlyphBlk = font.GetBlock(RtConsts.BinBlockSigCGLP);
            var textureGlyphBlk = font.GetBlock(RtConsts.BinBlockSigTGLP);
            Debug.Assert((bmpGlyphBlk != null) != (textureGlyphBlk != null));

            if (bmpGlyphBlk != null)
            {
                this.ExtractBmpFont(fontData, filter, font, (LibFormatFontGlyph)bmpGlyphBlk.Body, enc, alterIndex);
            }
            else
            {
                this.ExtractTextureFont(fontData, filter, font, (GlyphDataType)body.FontType, (FontTextureGlyphEx)textureGlyphBlk.Body, enc, alterIndex);
            }

            /*
                // シートセット取得
                {
                    CG2dFont::SheetSetList list;
                    bool bExists = font.GetSheetSetList(&list);

                    if( bExists )
                    {
                        std::auto_ptr<CGlyphGroups> pgg(new CGlyphGroups);

                        for( uint i = 0; i < list.size(); ++i )
                        {
                            CG2dFont::SheetSet& set = *list[i];
                            std::auto_ptr<CGlyphGroups::Group> g(new CGlyphGroups::Group);

                            g->name = set.name;


                            pgg->AddGroup(g.release());
                        }

                        pFont->SetGlyphGroups(pgg.release());
                    }
                }
            */
        }

        /// <summary>
        ///
        /// </summary>
        private void ExtractTextureFont(
                FontData fontData,
                CharFilter filter,
                G2dFont font,
                GlyphDataType glyphDataType,
                FontTextureGlyphEx fg,
                CharEncoding enc,
                ushort alterIndex)
        {
            RgbImage plane = new RgbImage();
            var isCompressed = (fg.Body.SheetFormat & (int)FontSheetFormat.CompressedFlag) != 0;
            var sheetFormat = (ushort)(fg.Body.SheetFormat & (int)FontSheetFormat.TexFmtMask);

            fontData.OutputFormat = GetGlyphImageFormat(sheetFormat, glyphDataType == GlyphDataType.PackedTexture);
            fontData.BaselinePos = fg.Body.BaselinePos;

            //---- テクスチャデータから CRgbImage へ変換
            {
                int sheetNum = fg.Body.SheetNum;
                int plane_w = fg.Body.SheetWidth;
                int plane_h = fg.Body.SheetHeight * sheetNum;
                byte[] sheet = fg.SheetImage;

                if (isCompressed)
                {
                    // アーカイブフォントは非サポートになりました！
                    Trace.Assert(false);
                }

                plane.Create(plane_w, plane_h, 32);
                plane.Clear(ImageBase.RgbWhite, ImageBase.AlphaTransparent);
                plane.EnableAlpha();
                plane.NumSubImage = sheetNum;

                // 画像変換用に、(Packed 形式かどうかを考慮しない)素のテクスチャ形式を取得して利用します。
                var rawTextureImageFormat = GetGlyphImageFormat(sheetFormat, false);
                this.ConvertFormat(plane, sheet, rawTextureImageFormat, fg.Body);
            }

            //---- CRgbImage からグリフ抽出
            {
                for (int i = 0x0000; i < 0x110000; i++)
                {
                    uint orig_code = (uint)i;
                    ushort idx = font.GetGlyphIndex(orig_code);
                    uint unicode = this.ProcessCode(fontData, filter, enc, alterIndex, orig_code, idx);

                    ProgressControl.GetInstance().StepProgressBar();

                    if (unicode != RtConsts.InvalidCharCode)
                    {
                        this.ExtractTextureGlyph(fontData.GetGlyphList(),fontData.BaselinePos, font, fg.Body, plane, idx, unicode);
                    }
                }
            }
        }

        /// <summary>
        /// CTR用のフォントランタイムで使用するテクスチャフォーマット値から
        /// GlyphImageFormatを取得します。
        /// </summary>
        /// <param name="format">CTR用のフォントランタイムで使用するテクスチャフォーマット値</param>
        /// <returns>GlyphImageFormatを返します。</returns>
        protected abstract GlyphImageFormat GetGlyphImageFormat(ushort format, bool isPackedFormat);

        /// <summary>
        ///
        /// </summary>
        private void ExtractTextureGlyph(
                GlyphList glyphList,
                int? baselinePos,
                G2dFont font,
                FontTextureGlyph fg,
                RgbImage sheetPlane,
                ushort idx,
                uint unicode)
        {
            var widthInfo = font.GetGlyphWidthInfo(idx);
            var glyph = new Glyph();
            var image = new RgbImage();

            // グリフイメージをフォント高xグリフ幅の大きさで取り出し
            {
                var cellsInSheet = fg.SheetRow * fg.SheetLine;
                var sheetNo = idx / cellsInSheet;
                var cellNo = idx % cellsInSheet;
                var cellUnitX = cellNo % fg.SheetRow;
                var cellUnitY = cellNo / fg.SheetRow;
                var cellPixelX = (cellUnitX * (fg.CellWidth + 1)) + 1;
                var cellPixelY = (cellUnitY * (fg.CellHeight + 1)) + 1;
                var sheetYOffset = sheetNo * fg.SheetHeight;
                var extract_x = cellPixelX;
                var extract_y = cellPixelY + sheetYOffset;
                var extract_w = widthInfo.GlyphWidth;
                var extract_h = fg.CellHeight;

                sheetPlane.Extract(image, extract_x, extract_y, extract_w, extract_h);
            }

            IntColor nullColor = sheetPlane.GetPixel(0, 0).IntColor;

            Debug.Assert(!font.IsOldVer());
            int left = widthInfo.Left;
            int right = widthInfo.CharWidth - (left + widthInfo.GlyphWidth);

            glyph.SetGlyphImage(
                image,
                left,
                right,
                baselinePos.Value,
                nullColor);

            glyphList.AddGlyph(glyph, unicode);
        }

        /// <summary>
        ///
        /// </summary>
        private void ExtractBmpFont(
                FontData fontData,
                CharFilter filter,
                G2dFont font,
                LibFormatFontGlyph fg,
                CharEncoding enc,
                ushort alterIndex)
        {
            // グリフ情報取得
            //// pFont->SetBpp(fg.bpp);
            fontData.BaselinePos = fg.BaselinePos;

            // グリフ取得
            {
                for (int i = 0x0000; i < 0x110000; i++)
                {
                    uint orig_code = (uint)i;
                    ushort idx = font.GetGlyphIndex(orig_code);
                    uint unicode = this.ProcessCode(fontData, filter, enc, alterIndex, orig_code, idx);

                    ProgressControl.GetInstance().StepProgressBar();

                    if (unicode != Runtime.RtConsts.InvalidCharCode)
                    {
                        this.ExtractBmpGlyph(fontData, font, fg, idx, unicode);
                    }
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ExtractBmpGlyph(
                FontData fontData,
                G2dFont font,
                LibFormatFontGlyph fg,
                ushort idx,
                uint unicode)
        {
            Debug.Assert(false); // 現在呼び出し禁止
        }

        /// <summary>
        ///
        /// </summary>
        private uint ProcessCode(
                FontData fontData,
                CharFilter filter,
                CharEncoding enc,
                ushort alterIndex,
                uint code,
                ushort idx)
        {
            uint unicode = Runtime.RtConsts.InvalidCharCode;

            // グリフが存在しないものはとばす
            if (idx == Runtime.RtConsts.InvalidGlyphIndex)
            {
                return Runtime.RtConsts.InvalidCharCode;
            }

            // 文字コード変換
            unicode = GlCm.EncodingToUnicode(enc, code);

            //---- Unicode に変換できなかった
            if (unicode == Runtime.RtConsts.InvalidCharCode)
            {
                ProgressControl.Warning(Strings.IDS_WARN_CANT_REPRESETN_IN_UNICODE, code);
                return Runtime.RtConsts.InvalidCharCode;
            }

            // alterIndex を文字コードに置き換え
            if (idx == alterIndex)
            {
                fontData.AlterChar = unicode;
            }

            // フィルタされているものはとばす
            if (filter.IsFiltered(unicode))
            {
                return Runtime.RtConsts.InvalidCharCode;
            }

            return unicode;
        }

        /// <summary>
        ///
        /// </summary>
        protected abstract void ConvertFormat(RgbImage sheet, byte[] image, GlyphImageFormat gif, Runtime.FontTextureGlyph tg);

        /// <summary>
        ///
        /// </summary>
        protected void ConvertFramework(
            RgbImage sheet,
            int bpp,
            PixelReader pc)
        {
            var picker = PixelPickerCreater.Create(sheet, bpp);

            int max = sheet.Width * sheet.Height;

            if (!ConverterEnvironment.PlatformDetails.IsInputFontReversed)
            {
                for (int pi = 0; pi < max; ++pi)
                {
                    // CTR向けはbcfntと同じテクスチャの向きになるようにするため、
                    // 天地が逆転しない状態で格納されています。
                    var pos = picker.GetPixel(pi);
                    sheet.SetPixel(pos.X, pos.Y, pc());
                }
            }
            else
            {
                int oneTextureHeight = sheet.Height / sheet.NumSubImage;

                for (int pi = 0; pi < max; ++pi)
                {
                    var pos = picker.GetPixel(pi);

                    int topOfTextureBoundaryY = (pos.Y / oneTextureHeight) * oneTextureHeight;
                    int yInOneTexture = pos.Y - topOfTextureBoundaryY;

                    // Cafe世代より、テクスチャは天地が逆転した状態で格納されています。
                    // これはたしか、OpenGLのテクスチャフォーマットに配慮した名残りだったと記憶しています。
                    int invY = topOfTextureBoundaryY + (oneTextureHeight - 1 - yInOneTexture);
                    sheet.SetPixel(pos.X, invY, pc());
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected class A4Reader
        {
            private readonly MemoryStream ms;
            private byte state;
            private byte value;

            /// <summary>
            ///
            /// </summary>
            public A4Reader(byte[] image)
            {
                this.ms = new MemoryStream(image);
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelA4_Rvl()
            {
                int bitStart;
                this.state ^= 1;
                if (this.state != 0)
                {
                    this.value = (byte)this.ms.ReadByte();
                    bitStart = 4;
                }
                else
                {
                    bitStart = 0;
                }

                return ConvertPixelA4(GlCm.ExtractBits(this.value, bitStart, 4));
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelA4()
            {
                int bitStart;
                this.state ^= 1;
                if (this.state != 0)
                {
                    this.value = (byte)this.ms.ReadByte();
                    bitStart = 0;
                }
                else
                {
                    bitStart = 4;
                }

                return ConvertPixelA4(GlCm.ExtractBits(this.value, bitStart, 4));
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected class Reader
        {
            private ByteOrderBinaryReader br;

            /// <summary>
            ///
            /// </summary>
            public Reader(byte[] image)
            {
                this.br = new ByteOrderBinaryReader(new MemoryStream(image), ConverterEnvironment.IsInputTargetLittleEndian);
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelA8()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                var value = this.br.ReadByte();
                var level = (byte)GlCm.InverseNumber(value, 8);

                pixel.Color = GlCm.GrayScaleToRgb(level);
                pixel.A = value;

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelAL4()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                byte p = this.br.ReadByte();
                uint i = GlCm.ExtractBits(p, 0, 4);
                uint a = GlCm.ExtractBits(p, 4, 4);
                var level = (byte)GlCm.InverseNumber(GlCm.ExtendBits(i, 4, 8), 8);

                pixel.Color = GlCm.GrayScaleToRgb(level);
                pixel.A = (byte)GlCm.ExtendBits(a, 4, 8);

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelLA4()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                byte p = this.br.ReadByte();
                uint i = GlCm.ExtractBits(p, 4, 4);
                uint a = GlCm.ExtractBits(p, 0, 4);
                var level = (byte)GlCm.InverseNumber(GlCm.ExtendBits(i, 4, 8), 8);

                pixel.Color = GlCm.GrayScaleToRgb(level);
                pixel.A = (byte)GlCm.ExtendBits(a, 4, 8);

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelAL8()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                var p = this.br.ReadUInt16();
                byte level = (byte)GlCm.InverseNumber(GlCm.ExtractBits(p, 0, 8), 8);

                pixel.Color = GlCm.GrayScaleToRgb(level);
                pixel.A = (byte)GlCm.ExtractBits(p, 8, 8);

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelLA8()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                var p = this.br.ReadUInt16();
                byte level = (byte)GlCm.InverseNumber(GlCm.ExtractBits(p, 8, 8), 8);

                pixel.Color = GlCm.GrayScaleToRgb(level);
                pixel.A = (byte)GlCm.ExtractBits(p, 0, 8);

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelRGB565()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                var p = this.br.ReadUInt16();
                uint r = GlCm.ExtractBits(p, 11, 5);
                uint g = GlCm.ExtractBits(p, 5, 6);
                uint b = GlCm.ExtractBits(p, 0, 5);

                pixel.R = (byte)GlCm.ExtendBits(r, 5, 8);
                pixel.G = (byte)GlCm.ExtendBits(g, 6, 8);
                pixel.B = (byte)GlCm.ExtendBits(b, 5, 8);
                pixel.A = ImageBase.AlphaOpacity;

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelA3RGB5()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                var p = this.br.ReadUInt16();

                if (0 != (p & 0x8000))
                {
                    //---- RGB555
                    uint r = GlCm.ExtractBits(p, 5 + 5, 5);
                    uint g = GlCm.ExtractBits(p,     5, 5);
                    uint b = GlCm.ExtractBits(p,     0, 5);

                    pixel.A = ImageBase.AlphaOpacity;
                    pixel.R = (byte)GlCm.ExtendBits(r, 5, 8);
                    pixel.G = (byte)GlCm.ExtendBits(g, 5, 8);
                    pixel.B = (byte)GlCm.ExtendBits(b, 5, 8);
                }
                else
                {
                    //---- RGB4A3
                    uint a = GlCm.ExtractBits(p, 4 + 4 + 4, 3);
                    uint r = GlCm.ExtractBits(p,     4 + 4, 4);
                    uint g = GlCm.ExtractBits(p,         4, 4);
                    uint b = GlCm.ExtractBits(p,         0, 4);

                    pixel.A = (byte)GlCm.ExtendBits(a, 3, 8);
                    pixel.R = (byte)GlCm.ExtendBits(r, 4, 8);
                    pixel.G = (byte)GlCm.ExtendBits(g, 4, 8);
                    pixel.B = (byte)GlCm.ExtendBits(b, 4, 8);
                }

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelRGB5A1()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                var p = this.br.ReadUInt16();
                uint r = GlCm.ExtractBits(p, 5 + 5 + 1, 5);
                uint g = GlCm.ExtractBits(p,     5 + 1, 5);
                uint b = GlCm.ExtractBits(p,         1, 5);
                uint a = GlCm.ExtractBits(p,         0, 1);

                pixel.R = (byte)GlCm.ExtendBits(r, 5, 8);
                pixel.G = (byte)GlCm.ExtendBits(g, 5, 8);
                pixel.B = (byte)GlCm.ExtendBits(b, 5, 8);
                pixel.A = (byte)GlCm.ExtendBits(a, 1, 8);

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelRGBA4()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                var p = this.br.ReadUInt16();
                uint r = GlCm.ExtractBits(p, 4 + 4 + 4, 4);
                uint g = GlCm.ExtractBits(p,     4 + 4, 4);
                uint b = GlCm.ExtractBits(p,         4, 4);
                uint a = GlCm.ExtractBits(p,         0, 4);

                pixel.R = (byte)GlCm.ExtendBits(r, 4, 8);
                pixel.G = (byte)GlCm.ExtendBits(g, 4, 8);
                pixel.B = (byte)GlCm.ExtendBits(b, 4, 8);
                pixel.A = (byte)GlCm.ExtendBits(a, 4, 8);

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelRGB8()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                var p = this.br.Read24Bits();
                pixel.R = (byte)GlCm.ExtractBits(p, 8 + 8, 8);
                pixel.G = (byte)GlCm.ExtractBits(p,     8, 8);
                pixel.B = (byte)GlCm.ExtractBits(p,     0, 8);
                pixel.A = ImageBase.AlphaOpacity;

                return pixel;
            }

            /// <summary>
            ///
            /// </summary>
            public ImageBase.Pixel ReadPixelRGBA8()
            {
                ImageBase.Pixel pixel = new ImageBase.Pixel();
                var p = this.br.ReadUInt32();
                pixel.R = (byte)GlCm.ExtractBits(p, 8 + 8 + 8, 8);
                pixel.G = (byte)GlCm.ExtractBits(p,     8 + 8, 8);
                pixel.B = (byte)GlCm.ExtractBits(p,         8, 8);
                pixel.A = (byte)GlCm.ExtractBits(p,         0, 8);

                return pixel;
            }
        }
    }
}
