﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using System.Linq;
    using s8 = System.SByte;
    using u8 = System.Byte;

    /// <summary>
    ///
    /// </summary>
    public enum GlyphImageFormat
    {
        A4,
        A8,

        LA4,
        LA8,
        LA4Packed,
        LA4PackedPassThrough, // LA4Packed形式の bffnt を読み込んで、LA4Packed固有の画像スキップしてファイル出力する場合に使う。
        LA4PackedNoCompress,
        LA4PackedNoCompressPassThrough, // LA4PackedNoCompress形式の bffnt を読み込んで、LA4PackedNoCompress固有の画像スキップしてファイル出力する場合に使う。

        RGB565,
        RGB5A1,
        RGB5A3,
        RGBA4,

        RGB8,
        RGBA8,

        RGBA8_SRGB, // リニア用 SRGBフェッチ形式
        RGB565_SRGB,　// リニア用 SRGBフェッチ形式
        RGB5A3_SRGB,　// リニア用 SRGBフェッチ形式

        RGBA8_BC7, // リニア用 SRGBフェッチ形式
        RGBA8_BC7_SRGB, // リニア用 SRGBフェッチ形式

        Max
    }

    public struct KerningPair
    {
        // 文字コードが 32bit に対応したカーニングファイルのヘッダ
        public const uint HeaderOf32BitModeKerning = 0x0000ffff;

        public uint First;
        public uint Second;
        public int KernAmount;
    };

    /// <summary>
    ///
    /// </summary>
    public class FontData
    {
        private readonly GlyphList Glyphs = new GlyphList();
        private bool isExplicitWidth;
        private bool isExplicitHeight;

        public int? LineHeight { get; set; }
        public int? Width { get; set; }
        public int? Ascent { get; set; }
        public int? Descent { get; set; }
        public int? BaselinePos { get; set; }
        public int? CellWidth { get; set; }
        public int? CellHeight { get; set; }
        public int? MaxCharWidth { get; set; }
        public Runtime.CharWidths? DefaultWidth { get; set; }
        public uint? AlterChar { get; set; }
        public GlyphImageFormat? OutputFormat { get; set; }
        public IntColor? NullColor { get; set; }
        public KerningPair[] KerningPairs { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FontData()
        {
            this.isExplicitWidth = false;
            this.isExplicitHeight = false;
            this.KerningPairs = null;
        }

        /// <summary>
        ///
        /// </summary>
        public FontData(FontData src)
        {
            this.LineHeight = src.LineHeight;
            this.Width = src.Width;
            this.Ascent = src.Ascent;
            this.Descent = src.Descent;
            this.BaselinePos = src.BaselinePos;
            this.CellWidth = src.CellWidth;
            this.CellHeight = src.CellHeight;
            this.MaxCharWidth = src.MaxCharWidth;
            this.DefaultWidth = src.DefaultWidth;
            this.AlterChar = src.AlterChar;
            this.OutputFormat = src.OutputFormat;
            this.NullColor = src.NullColor;
            this.KerningPairs = src.KerningPairs;
        }

        /// <summary>
        ///
        /// </summary>
        public void SetWidth(int w)
        {
            this.Width = w;
            this.isExplicitWidth = true;
        }

        /// <summary>
        ///
        /// </summary>
        public void SetHeight(int a, int d)
        {
            this.Ascent = a;
            this.Descent = d;
            this.isExplicitHeight = true;
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsWidthExplicit()
        {
            return this.isExplicitWidth;
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsHeightExplicit()
        {
            return this.isExplicitHeight;
        }

        /// <summary>
        ///
        /// </summary>
        public GlyphList GetGlyphList()
        {
            return this.Glyphs;
        }

        /// <summary>
        /// バイナリ化した際に、有効な値なのかどうかチェックする
        /// </summary>
        /// <param name="g"></param>
        /// <returns></returns>
        private string ValidateGlyphSize_(Glyph g)
        {
            // byte  cellWidth
            // byte  cellHeight
            // short baselinePos
            // byte  maxCharWidth

            if (g.Width() > byte.MaxValue)
            {
                return string.Format("Width over 255. val={0} code={1}", g.Width(), g.GetCode().ToString("X4"));
            }

            if (g.Ascent() > byte.MaxValue)
            {
                return string.Format("Ascent over 255. val={0} code={1}", g.Ascent(), g.GetCode().ToString("X4"));
            }

            if (g.Ascent() + g.Descent() > byte.MaxValue)
            {
                return string.Format("Ascent + Descent(Height) over 255. val={0} code={1}", g.Ascent() + g.Descent(), g.GetCode().ToString("X4"));
            }

            if (g.CharFeed() > byte.MaxValue)
            {
                return string.Format("CharFeed over 255. val={0} code={1}", g.CharFeed(), g.GetCode().ToString("X4"));
            }

            return null;
        }

        // グリフリストを走査してフォントに必要な値を計算する
        public void ReflectGlyph()
        {
            // グリフが無い
            if (this.Glyphs.GetNum() == 0)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_NO_GLYPH);
            }

            if (!this.OutputFormat.HasValue)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_OUTPUT_FORMAT_NOT_SPECIFIED);
            }

            // セル幅 セル高 ベースラインを求める
            int maxAscent = int.MinValue;
            int maxDescent = int.MinValue;
            {
                System.Text.StringBuilder errorMsgs = new System.Text.StringBuilder();

                int charWidth = 0;
                int charFeed = 0;

                foreach (var i in this.Glyphs.GetEnum())
                {
                    Glyph g = i;

                    if (g.Height() > 0)
                    {
                        SET_IF_LARGER(ref maxAscent, g.Ascent());
                        SET_IF_LARGER(ref maxDescent, g.Descent());
                    }

                    SET_IF_LARGER(ref charWidth, g.Width());
                    SET_IF_LARGER(ref charFeed, g.CharFeed());

                    // バイナリ化した際に、有効な値なのかどうかチェックする
                    string errorMsg = ValidateGlyphSize_(g);
                    if (errorMsg != null)
                    {
                        errorMsgs.AppendLine(errorMsg);
                    }
                }

                // errorLog に出力があった場合、処理を中断します。
                if (errorMsgs.Length > 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Font, Strings.IDS_ERR_INVALID_GLYPH_SIZE, errorMsgs.ToString());
                }

                if (maxAscent == int.MinValue)
                {
                    maxAscent = 0;
                    maxDescent = 0;
                }

                this.BaselinePos = maxAscent;
                this.CellHeight = maxAscent + maxDescent;
                this.CellWidth = charWidth;
                this.MaxCharWidth = charFeed;

                Rpt._RPT4("a={0} d={1} w={2} f={3}\n", maxAscent, maxDescent, charWidth, charFeed);
            }

            // デフォルト指定
            if (!this.Width.HasValue)
            {
                this.Width = this.MaxCharWidth;
            }

            if (!this.Ascent.HasValue)
            {
                this.Ascent = Math.Max(0, maxAscent);
            }

            if (!this.Descent.HasValue)
            {
                this.Descent = Math.Max(0, maxDescent);
            }

            if (!this.LineHeight.HasValue)
            {
                this.LineHeight = this.Ascent.Value + this.Descent.Value;
            }

            if (!this.DefaultWidth.HasValue)
            {
                Runtime.CharWidths defWidth = new Runtime.CharWidths(0, (u8)this.CellWidth.Value, (u8)this.CellWidth.Value);
                this.DefaultWidth = defWidth;
            }

            if (!this.NullColor.HasValue)
            {
                this.NullColor = GlCm.BMP_RGBA(255, 255, 255, 0);
            }

            // ソート
            this.Glyphs.SortByCode();
        }

        /// <summary>
        ///
        /// </summary>
        private static void SET_IF_LARGER(ref int var, int val)
        {
            if (val > var)
            {
                var = val;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class GlyphImageInfo
    {
        private static readonly GlyphImageInfo[] glyphImageInfos = new GlyphImageInfo[(int)GlyphImageFormat.Max];

        public GlyphImageFormat GlyphImageFormat { get; private set; }
        public int Bpp { get; private set; }
        public bool IsToIntencity { get; private set; }
        public int AlphaBits { get; private set; }
        public bool HasAlpha { get { return this.AlphaBits > 0; } }

        /// <summary>
        ///
        /// </summary>
        public GlyphImageInfo(GlyphImageFormat glyphImageFormat, int bpp, bool isToIntencity, int alphaBits)
        {
            this.GlyphImageFormat = glyphImageFormat;
            this.Bpp = bpp;
            this.IsToIntencity = isToIntencity;
            this.AlphaBits = alphaBits;
        }

        /// <summary>
        ///
        /// </summary>
        public static GlyphImageInfo GetGlyphImageInfo(GlyphImageFormat format)
        {
            var formatDescList = ConverterEnvironment.PlatformDetails.GetImageInputFormatFormatList();
            ImageInputFormatDesc fmtDesc = formatDescList.FirstOrDefault((desc) => desc.KeyGlyphImageFormat == format);

            return (fmtDesc != null) ? new GlyphImageInfo(fmtDesc.GlyphImageFormat, fmtDesc.Bpp, fmtDesc.IsToIntencity, fmtDesc.BppAlpha) : null;
        }
    }
}
