﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Windows;
    using System.Windows.Controls;
    using System.Windows.Interop;
    using NintendoWare.Font.Properties;
    using NintendoWare.Font.Win32;

    public partial class WarningWindow : Window
    {
        private const int DefaultWidth = 400;
        private const int DefaultHeight = 200;

        private Window keepOwner;
        private bool close;

        public WarningWindow(Window owner)
        {
            InitializeComponent();

            this.keepOwner = owner;
            this.keepOwner.IsVisibleChanged += this.OwnerWindow_IsVisibleChanged;
            this.SourceInitialized += this.Window_SourceInitialized;

            var listener = new TextBoxTraceListener(this.tbMessage);
            ProgressControl.AddListener(listener);

            this.Closing += (sender, e) => this.Window_Closing(e, listener);
            this.tbMessage.TextChanged += this.TextBox_TextChanged;
        }

        public bool HasMessage
        {
            get { return this.tbMessage.Text != string.Empty; }
        }

        public new void Show()
        {
            this.Owner = this.keepOwner;
            base.Show();
        }

        public new void Close()
        {
            this.close = true;
            base.Close();
        }

        public void ClearMessage()
        {
            this.tbMessage.Clear();
        }

        private void Window_SourceInitialized(object sender, EventArgs e)
        {
            RECT rect;
            this.LoadWindowRect(this.Owner, out rect);

            var wp = new WINDOWPLACEMENT();
            wp.flags = 0;
            wp.showCmd = ShowWindowCommand.ShowNormal;
            wp.rcNormalPosition = rect;
            User.SetWindowPlacement(new WindowInteropHelper(this).Handle, wp);
        }

        private void Window_Closing(CancelEventArgs e, TraceListener listener)
        {
            // Close()が呼ばれていないときは、ウィンドウを閉じる代わりに隠します。
            if (!this.close)
            {
                e.Cancel = true;
                this.Hide();
                return;
            }

            ProgressControl.RemoveListener(listener);

            this.SaveWindowRect();
        }

        private void LoadWindowRect(Window parent, out RECT rect)
        {
            var wp = (WINDOWPLACEMENT)Settings.DefaultInstance.WarningWindowPlacement;

            if ( wp == null
              || (   wp.rcNormalPosition.right - wp.rcNormalPosition.left < 100
                  || wp.rcNormalPosition.bottom - wp.rcNormalPosition.top < 100))
            {
                User.GetWindowRect(new WindowInteropHelper(parent).Handle, out rect);
                rect.left = rect.right;
                rect.right = rect.left + DefaultWidth;
                rect.bottom = rect.top + DefaultHeight;
                return;
            }

            rect = wp.rcNormalPosition;
        }

        private void SaveWindowRect()
        {
            var wp = new WINDOWPLACEMENT();
            User.GetWindowPlacement(new WindowInteropHelper(this).Handle, wp);
            Settings.DefaultInstance.WarningWindowPlacement = wp;
        }

        private void OwnerWindow_IsVisibleChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            if (!this.IsVisible && this.HasMessage)
            {
                if ((bool)e.NewValue)
                {
                    this.Show();
                }
            }
        }

        private void TextBox_TextChanged(object sender, TextChangedEventArgs e)
        {
            if (!this.IsVisible && this.HasMessage)
            {
                if (this.keepOwner.IsVisible)
                {
                    this.Show();
                }
            }
        }

        private void Window_Loaded(object sender, RoutedEventArgs e)
        {
            string documentUrl = System.IO.Path.Combine(ConverterEnvironment.PlatformDetails.NwToolsRootPath, "../../Documents/Package/contents/Pages/Page_181908940.html").Replace('\\', '/');

            // エラーダイアログにドキュメントへのリンクを貼ります。
            System.Windows.Documents.Paragraph paragraph = new System.Windows.Documents.Paragraph();
            paragraph.Margin = new Thickness(0.0);
            System.Windows.Documents.Hyperlink hyperlink = new System.Windows.Documents.Hyperlink();
            hyperlink.IsEnabled = true;
            hyperlink.Inlines.Add(Strings.IDS_ERR_DOCUMENT_LINK);
            hyperlink.NavigateUri = new Uri(documentUrl);
            hyperlink.RequestNavigate += (send, args) => Process.Start(args.Uri.ToString());
            paragraph.Inlines.Add(Strings.IDS_ERR_DOCUMENT_HEADER + " ");
            paragraph.Inlines.Add(hyperlink);
            paragraph.Inlines.Add(Strings.IDS_ERR_DOCUMENT_FOOTER);
            tbDocument.Document.Blocks.Add(paragraph);
        }
    }
}
