﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Windows.Data;
    using NintendoWare.Font.Properties;
    using NintendoWare.Font.Win32;
    using System.IO;
    using System.Threading;
    using System.Windows;

    /// <summary>
    /// ウィンドウフォント入力設定用ViewModelです。
    /// </summary>
    public class InWinViewModel : InSettingViewModel
    {
        private const int FontNameListHeight = 16;

        private const string FileFilterStr = "Cafe truetype Fonts (*.bfttf)|*.bfttf;*.tga;*.tif"
                                              + "|all files (*.*)|*.*";

        private readonly InWinSettings settings;
        private FontMap fontMap;
        private ICollection<FontInfo> fonts;
        private BppViewModel[] formatTypeValue;
        private WidthTypeViewModel[] widthTypeInfos;
        private IEnumerable<int> fontSizes;
        private ListCollectionView bpps;
        private bool isFontListReady;
        private bool isFontNameChanging;

        public InWinViewModel(InWinSettings settings)
        {
            this.DisplayName = "Windows Font";
            this.settings = settings;
            LoadSysFont_();
        }

        public string FontName
        {
            get
            {
                if (!this.isFontListReady)
                {
                    Debug.WriteLine("FontName property get - isn't ready");
                    return Strings.IDS_NOW_READING;
                }

                Debug.WriteLine("FontName property get - " + this.settings.FontName);
                return this.settings.FontName;
            }

            set
            {
                if (null == value)
                {
                    Debug.WriteLine("FontName property set - null");
                    return;
                }

                if (!this.isFontListReady)
                {
                    Debug.WriteLine("FontName property set - isn't ready - " + value);
                    return;
                }

                if (value == this.settings.FontName)
                {
                    Debug.WriteLine("FontName property set - same");
                    return;
                }

                Debug.WriteLine("FontName property set - " + value);
                this.settings.FontName = value;
                this.UpdateFontName();
            }
        }

        public string FileFilter
        {
            get { return FileFilterStr; }
        }

        public Visibility BfttfFontEnabled
        {
            get { return ConverterEnvironment.PlatformDetails.BfttfEnabled ? Visibility.Visible : Visibility.Hidden; }
        }

        public Visibility ScfontEnabled
        {
            get { return ConverterEnvironment.PlatformDetails.ScfontEnabled ? Visibility.Visible : Visibility.Hidden; }
        }

        public string SystemFilePath
        {
            get
            {
                if (string.IsNullOrEmpty(this.settings.SysFontFilePath))
                {

                    return Strings.SystemFontNotLoaded;
                }
                else if (!File.Exists(this.settings.SysFontFilePath))
                {
                    return string.Format(Strings.SystemFontNotFound, this.settings.SysFontFilePath);
                }

                return this.settings.SysFontFilePath;
            }

            set
            {
                string currentFontName = this.settings.FontName;
                this.settings.FontName = GlCm.SystemFontFaceName;
                this.OnPropertyChanged("FontName");

                if (this.settings.SysFontFilePath != value)
                {
                    // 設定変更された場合は、現在のフォントとして選択する。
                    this.settings.SysFontFilePath = value;
                    this.OnPropertyChanged("SystemFilePath");

                    bool loadResult = LoadSysFont_();
                    if (!loadResult)
                    {
                        // 失敗したので戻す。
                        this.settings.SysFontFilePath = string.Empty;
                        this.OnPropertyChanged("SystemFilePath");
                        this.settings.FontName = currentFontName;
                        this.OnPropertyChanged("FontName");
                    }
                }
            }
        }

        private bool LoadSysFont_()
        {
            if (string.IsNullOrEmpty(this.settings.SysFontFilePath))
            {
                return true;
            }

            if (!File.Exists(this.settings.SysFontFilePath))
            {
                ProgressControl.Error(string.Format(Strings.SystemFontNotFound, this.settings.SysFontFilePath));
                return false;
            }

            // リセットして、フォントリストを再構築します。
            bool result = GlCm.InitSystemFontFromBfttf(this.settings.SysFontFilePath);
            this.Fonts = null;
            if (result)
            {
                return true;
            }
            else
            {
                ProgressControl.Error(string.Format(Strings.SystemFontInvalid, this.settings.SysFontFilePath));
                return false;
            }
        }

        public string FileExtension
        {
            get { return LibFormat.ExtensionScalableFont; }
        }

        public string Bpp
        {
            get
            {
                return this.settings.Bpp;
            }

            set
            {
                if (null == value)
                {
                    Debug.WriteLine("Bpp property set - null");
                    return;
                }

                if (value == this.settings.Bpp)
                {
                    return;
                }

                this.settings.Bpp = value;
                this.OnPropertyChanged("Bpp");
            }
        }

        public int FontSize
        {
            get
            {
                if (this.isFontNameChanging)
                {
                    Debug.WriteLine("FontSize property get - is changing");
                }

                return this.settings.FontSize;
            }

            set
            {
                if (this.isFontNameChanging)
                {
                    Debug.WriteLine("FontSize property set - is changing - " + value);
                    return;
                }

                if (value == this.settings.FontSize)
                {
                    return;
                }

                this.settings.FontSize = value;
                this.OnPropertyChanged("FontSize");
            }
        }

        public int FontWeight
        {
            get
            {
                if (this.isFontNameChanging)
                {
                    Debug.WriteLine("FontWeight property get - is changing");
                }

                return this.settings.FontWeight;
            }

            set
            {
                // 妥当性チェックはここか？
                if (this.isFontNameChanging)
                {
                    Debug.WriteLine("FontWeight property set - is changing - " + value);
                    return;
                }

                if (value == this.settings.FontWeight)
                {
                    return;
                }

                this.settings.FontWeight = value;
                this.OnPropertyChanged("FontWeight");
            }
        }

        public WidthTypeViewModel[] WidthTypes
        {
            get
            {
                if (null == this.widthTypeInfos)
                {
                    this.widthTypeInfos = new WidthTypeViewModel[]
                    {
                        new WidthTypeViewModel(Strings.IDS_INWIN_WIDTHTYPE_GLYPH_ONLY, WidthType.GlyphOnly),
                        new WidthTypeViewModel(Strings.IDS_INWIN_WIDTHTYPE_GLYPH_ONLY_KEEPSP, WidthType.GlyphOnlyKeepSpace),
                        new WidthTypeViewModel(Strings.IDS_INWIN_WIDTHTYPE_INCLUDE_MARGIN, WidthType.IncludeMargin),
                        new WidthTypeViewModel(Strings.IDS_INWIN_WIDTHTYPE_FIXED_WIDTH, WidthType.Fixed),
                    };
                }

                return this.widthTypeInfos;
            }
        }

        public WidthType WidthType
        {
            get
            {
                return this.settings.WidthType;
            }

            set
            {
                if (value == this.settings.WidthType)
                {
                    return;
                }

                this.settings.WidthType = value;
                this.OnPropertyChanged("WidthType");
                this.OnPropertyChanged("IsEnabledFixedValue");
            }
        }

        public bool IsEnabledFixedValue
        {
            get
            {
                return this.WidthType == WidthType.Fixed;
            }
        }

        public int FixedValue
        {
            get
            {
                return this.settings.FixedValue;
            }

            set
            {
                if (value == this.settings.FixedValue)
                {
                    return;
                }

                this.settings.FixedValue = value;
                this.OnPropertyChanged("FixedValue");
            }
        }

        public int AverageWidth
        {
            get
            {
                if (this.isFontNameChanging)
                {
                    Debug.WriteLine("AverageWidth property get - is changing");
                }

                return this.settings.AverageWidth;
            }

            set
            {
                if (this.isFontNameChanging)
                {
                    Debug.WriteLine("AverageWidth property set - is changing - " + value);
                    return;
                }

                if (value == this.settings.AverageWidth)
                {
                    return;
                }

                this.settings.AverageWidth = value;
                this.OnPropertyChanged("AverageWidth");
            }
        }

        public bool EnableAverageWidth
        {
            get
            {
                return this.settings.EnableAverageWidth;
            }

            set
            {
                if (value == this.settings.EnableAverageWidth)
                {
                    return;
                }

                this.settings.EnableAverageWidth = value;
                this.OnPropertyChanged("EnableAverageWidth");
            }
        }

        public bool EnableSoftAntialiasing
        {
            get
            {
                return this.settings.EnableSoftAntialiasing;
            }

            set
            {
                if (value == this.settings.EnableSoftAntialiasing)
                {
                    return;
                }

                this.settings.EnableSoftAntialiasing = value;
                this.OnPropertyChanged("EnableSoftAntialiasing");
            }
        }

        public bool UseScfontKerning
        {
            get
            {
                return this.settings.UseScfontKerning;
            }

            set
            {
                if (value == this.settings.UseScfontKerning)
                {
                    return;
                }

                this.settings.UseScfontKerning = value;
                this.OnPropertyChanged("UseScfontKerning");
            }
        }

        public ICollection<FontInfo> Fonts
        {
            get
            {
                if (null == this.fonts)
                {
                    this.IsReady = false;
                    this.fonts = new List<FontInfo>
                    {
                        new FontInfo(Strings.IDS_NOW_READING, FontType.Vector),
                    };

                    var backgroundWorker = new BackgroundWorker();
                    backgroundWorker.DoWork += this.CollectFont;
                    backgroundWorker.RunWorkerCompleted += (sender, e) => this.FontCollected((FontMap)e.Result);
                    backgroundWorker.RunWorkerAsync();
                }

                return this.fonts;
            }

            private set
            {
                this.fonts = value;
                this.isFontListReady = value != null;

                this.OnPropertyChanged("Fonts");
            }
        }

        public IEnumerable<int> FontSizes
        {
            get
            {
                return this.fontSizes;
            }

            private set
            {
                if (value == this.fontSizes)
                {
                    return;
                }

                this.fontSizes = value;
                this.OnPropertyChanged("FontSizes");
            }
        }

        public ICollectionView Bpps
        {
            get
            {
                if (null == this.bpps)
                {
                    this.bpps = new ListCollectionView(this.FormatTypeValue);
                }

                return this.bpps;
            }
        }

        private BppViewModel[] FormatTypeValue
        {
            get
            {
                if (null == this.formatTypeValue)
                {
                    this.formatTypeValue = CreateBppArray();
                }

                return this.formatTypeValue;
            }
        }

        public override void SaveState()
        {
        }

        public override FontReader GetFontReader()
        {
            if (!this.IsReady)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_NOW_READING);
            }

            BppViewModel bpp = (BppViewModel)this.Bpps.CurrentItem;
            /*
            BppParam bpp = this.FormatTypeValue[0];
            for (int i = 0; i < this.FormatTypeValue.Length; ++i)
            {
                if (this.FormatTypeValue[i].TypeName == this.Bpp)
                {
                    bpp = this.FormatTypeValue[i];
                    break;
                }
            }
             */

            int avgwidth = 0;

            // 平均幅を指定のチェックがある場合
            if (this.EnableAverageWidth)
            {
                avgwidth = this.AverageWidth;
            }

            return
                new WinFontReader(
                    null,
                    this.FontName,
                    this.FontSize,
                    this.FontWeight,
                    avgwidth,
                    bpp.Bpp,
                    bpp.HasAlpha,
                    this.WidthType,
                    this.FixedValue,
                    this.EnableSoftAntialiasing,
                    this.UseScfontKerning,
                    false,
                    null);
        }

        private static List<int> GetBaseFontSizes(string fontName)
        {
            var sizes = new List<int>();

            FONTENUMPROC getFontSizes = delegate(
                ENUMLOGFONTEX elfex,
                NEWTEXTMETRICEX ntmex,
                uint fontType,
                IntPtr param)
            {
                sizes.Add(ntmex.ntmTm.tmHeight);
                return 1;
            };

            var hDC = Gdi.CreateDC("DISPLAY", null, null, IntPtr.Zero);
            LOGFONT lf = new LOGFONT();

            lf.lfCharSet = GdiDef.DEFAULT_CHARSET;
            lf.lfPitchAndFamily = 0;
            lf.lfFaceName = fontName;

            Gdi.EnumFontFamiliesEx(
                hDC,
                ref lf,
                getFontSizes,
                IntPtr.Zero,
                0);

            Gdi.DeleteDC(hDC);

            return sizes;
        }

        /// <summary>
        /// BPPリストを作成します。
        /// </summary>
        /// <returns>BppViewModelの配列を返します。</returns>
        private static BppViewModel[] CreateBppArray()
        {
            // 2諧調形式は動作しておらず、Nitro時代の名残だったので削除しました。
            // LA8Packed は追加しませんでした。BMP入力以外では実質利用されないからです。
            var bpps = new List<BppViewModel>();//
            foreach (var winFontDesc in ConverterEnvironment.PlatformDetails.GetWindowsFontInputFormatList())
            {
                string typeName = winFontDesc.GlyphImageFormat.ToString();
                if (bpps.Find((bpp) => bpp.TypeName == typeName) != null) { continue; }

                bpps.Add(new BppViewModel(winFontDesc.Bpp, winFontDesc.HasAlpha, winFontDesc.Description, typeName));
            }

            return bpps.ToArray();
        }

        private void CollectFont(object sender, DoWorkEventArgs e)
        {
            e.Result = GlCm.GetInstalledFontNames();
        }

        private void FontCollected(FontMap fontMap)
        {
            this.fontMap = fontMap;

#if ORIGINAL_TOOLTIP_CODE
        //---- 文字順序リストをチップ付きコンボボックスに
        FontMap::iterator i;
        UINT maxLen = 0;

        for( i = m_fonts.begin(); i != m_fonts.end(); ++i )
        {
            maxLen = max(maxLen, i->first.length());
        }
        m_pFontNameCombo->Init(m_pTip, maxLen, FONTNAMELIST_HEIGHT, 1);
#endif
            string currentFontName = this.settings.FontName;
            this.Fonts = this.fontMap.Values;

            // 現在の値に一致する項目が無い場合は先頭の内容にする。
            // ある場合は選択を復元する。
            if (!fontMap.ContainsKey(currentFontName))
            {
                this.FontName = this.Fonts.First().FaceName;
            }
            else
            {
                this.FontName = currentFontName;
            }

            this.IsReady = true;
        }

        private void UpdateFontSizeList()
        {
            this.isFontNameChanging = true;

            var sizes = GetBaseFontSizes(this.FontName);
            this.FontSizes = sizes.OrderBy((val) => val).Distinct();

            /*
                bool bRaster = IsRasterFont(fontName);
                if( bRaster )
                {
                    LRESULT iFound;

                    iFound = ::SendMessage(hCmb, CB_FINDSTRING, static_cast<WPARAM>(-1),
                                    reinterpret_cast<LPARAM>(_itot(m_fontSize, buf, 10)) );
                    ::SendMessage(hCmb, CB_SETCURSEL, ((iFound != CB_ERR) ? iFound: 0), 0);
                }
                else
                {
            */
            this.isFontNameChanging = false;
            /*  } */
        }

        private void UpdateBppCombo()
        {
            var fontInfo = this.fontMap[this.FontName];

            // detect raster or vector
            var isRaster = fontInfo.Type != FontType.Vector;

            // 2Bit形式フォントに対するフィルタは現状利用されないので廃止しています。
            // this.Bpps.Filter = isRaster ? this.Bpps.Filter = new Predicate<object>(this.ShowOnly2BitFilter) : null;
            this.Bpps.Filter = null;
        }

        private bool ShowOnly2BitFilter(object obj)
        {
            var product = obj as BppViewModel;
            if (product != null)
            {
                return product.Is2Bit;
            }

            return false;
        }

        private void UpdateFontName()
        {
            Debug.WriteLine("Pre OnPropertyChanged FontName");
            this.OnPropertyChanged("FontName");
            Debug.WriteLine("Aft OnPropertyChanged FontName");
            this.UpdateFontSizeList();
            this.UpdateBppCombo();
        }
    }
}
