﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Globalization;
    using System.Windows.Controls;

    /// <summary>
    /// 整数値の範囲の検証を行うための規則
    /// </summary>
    public class AlternateCharValidationRule : ValidationRule
    {
        /// <summary>
        /// 値の検証チェックを行います。
        /// </summary>
        /// <param name="value">チェックするバインディング ターゲットの値。 </param>
        /// <param name="cultureInfo">この規則で使用するカルチャ。 </param>
        /// <returns>ValidationResult オブジェクト。 </returns>
        public override ValidationResult Validate(object value, CultureInfo cultureInfo)
        {
            var stringValue = value as string;
            if (stringValue == null)
            {
                return new ValidationResult(false, "Illegal type");
            }

            // 文字列が1つの場合は代替文字そのもの
            if (stringValue.Length == 1)
            {
                return ValidationResult.ValidResult;
            }

            // 文字列が空の場合は、とりあえず通す。
            // 将来変更予定。
            if (stringValue.Length == 0)
            {
                return ValidationResult.ValidResult;
            }

            // 4文字以上はエラー
            if (stringValue.Length > 4)
            {
                return CreateResult();
            }

            int intValue = 0;

            // 数値に変換できなくてもエラー
            try
            {
                intValue = Int32.Parse(stringValue, NumberStyles.AllowHexSpecifier);
            }
            catch (FormatException)
            {
                return CreateResult();
            }
            catch (OverflowException)
            {
                return CreateResult();
            }

            return ValidationResult.ValidResult;
        }

        /// <summary>
        /// ValidationResultを生成します。
        /// </summary>
        /// <returns>ValidationResult オブジェクト。 </returns>
        private static ValidationResult CreateResult()
        {
            return new ValidationResult(false, Strings.ErrorAlternateCharFormat);
        }
    }
}
