﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx.h>

#include "detail/util_DebugDrawUtil.h"
#include "detail/util_DrawVfxDebugInfo.h"

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf/perf_Profile.h>


 //------------------------------------------------------------------------------
 // エミッタのGPU処理を計測する為のコールバック関数
 //------------------------------------------------------------------------------
void DrawEmitterProfileCallback( nn::vfx::DrawEmitterProfilerArg& arg ) NN_NOEXCEPT
{
    if ( arg.pEmitterSet && arg.beforeRenderEmitter )
    {
        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU( arg.pCommandBuffer, arg.pEmitterSet->GetName(), 0 );
    }

    if ( arg.pEmitter && !arg.beforeRenderEmitter )
    {
        NN_PERF_SET_COLOR_GPU( nn::util::Color4u8::Blue() );
        NN_PERF_BEGIN_MEASURE_NAME_AND_TAG_GPU( arg.pCommandBuffer, arg.pEmitter->GetName(), 1 );
        arg.pSystem->DrawEmitter( arg.pCommandBuffer, arg.pEmitter, arg.isComputeShaderMode, arg.pUserParam, arg.pDrawParameterArg );
        NN_PERF_END_MEASURE_GPU( arg.pCommandBuffer );
    }

    if ( arg.pEmitterSet && !arg.beforeRenderEmitter )
    {
        NN_PERF_END_MEASURE_GPU( arg.pCommandBuffer );
    }
}

void _DrawEmitterSetInfo( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, const nn::vfx::EmitterSet* pEmitterSet ) NN_NOEXCEPT
{
    const nn::util::Color4u8 captionColor( 79, 129, 189, 200 );
    const nn::util::Color4u8 groundColor( 32, 32, 32, 200 );
    const nn::util::Color4u8 frameColor( 128, 128, 128, 128 );
    const float frameHeight = 20.f;
    const float labelWidth = 140.f;
    const float paramWidth = 140.f;

    const float bufferParamWidth = 180.f;

    float cx = pTextWriter->GetCursorX();
    float cy = pTextWriter->GetCursorY();

    nn::perf::LoadMeterBase* gpuMeter = nn::perf::LoadMeterCenter::GetGpuMeter();

    if ( pEmitterSet == nullptr )
    {
        // Name
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "EmitterSet", cx, cy, labelWidth, frameHeight, 0, captionColor );
        cx += labelWidth;

        // CpuEmitter
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "CPU", cx, cy, paramWidth, frameHeight, 1, captionColor );
        cx += paramWidth;

        // Gpu Emitter
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "GPU", cx, cy, paramWidth, frameHeight, 1, captionColor );
        cx += paramWidth;

        // Compute Emitter
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "GPUSO", cx, cy, paramWidth, frameHeight, 1, captionColor );
        cx += paramWidth;

        // Stripe
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Stripe", cx, cy, paramWidth, frameHeight, 1, captionColor );
        cx += paramWidth;

        // G-Buffer
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Buffer", cx, cy, bufferParamWidth, frameHeight, 1, captionColor );
        cx += bufferParamWidth;

        // GpuCost
        if ( gpuMeter )
        {
            _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "GpuCost", cx, cy, paramWidth, frameHeight, 1, captionColor );
            cx += paramWidth;
        }
    }
    else
    {
        // Name
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, pEmitterSet->GetName(), cx, cy, labelWidth, frameHeight, 0, groundColor );
        cx += labelWidth;

        // CpuEmitter
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "E:", pEmitterSet->GetProcessingCpuEmitterCount(), cx, cy, paramWidth / 2, frameHeight, 2, groundColor );
        cx += paramWidth / 2;

        // Cpu Particle
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "P:", pEmitterSet->GetProcessingCpuParticleCount(), cx, cy, paramWidth / 2, frameHeight, 2, groundColor );
        cx += paramWidth / 2;


        // Gpu Emitter
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "E:", pEmitterSet->GetProcessingGpuEmitterCount(), cx, cy, paramWidth / 2, frameHeight, 2, groundColor );
        cx += paramWidth / 2;

        // Gpu Particle
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "P:", pEmitterSet->GetProcessingGpuParticleCount(), cx, cy, paramWidth / 2, frameHeight, 2, groundColor );
        cx += paramWidth / 2;


        // Compute Emitter
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "E:", pEmitterSet->GetProcessingComputeEmitterCount(), cx, cy, paramWidth / 2, frameHeight, 2, groundColor );
        cx += paramWidth / 2;

        // Compute Particle
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "P:", pEmitterSet->GetProcessingGpuSoParticleCount(), cx, cy, paramWidth / 2, frameHeight, 2, groundColor );
        cx += paramWidth / 2;

        // Stripe
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "S:", pEmitterSet->GetProcessingStripeCount(), cx, cy, paramWidth / 2, frameHeight, 2, groundColor );
        cx += paramWidth / 2;

        // Stripe2
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "S2:", pEmitterSet->GetProcessingSuperStripeCount(), cx, cy, paramWidth / 2, frameHeight, 2, groundColor );
        cx += paramWidth / 2;

        // G-Buffer
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "D:", pEmitterSet->GetAllocatedFromDynamicHeapSize(), cx, cy, bufferParamWidth / 2, frameHeight, 2, groundColor );
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "G:", pEmitterSet->GetAllocatedFromGpuBufferSize(), cx + bufferParamWidth / 2, cy, bufferParamWidth / 2, frameHeight, 2, groundColor );
        cx += bufferParamWidth;

        // Gpu Cost
        if ( gpuMeter )
        {
            float gpuCost = 0.0f;
            int resultNum = gpuMeter->GetLastSectionCount();

            for ( int i = 0; i < resultNum; ++i )
            {
                const nn::perf::LoadMeterBase::Section& section = gpuMeter->GetLastResult( i );
                if ( strcmp( section.name, pEmitterSet->GetName() ) == 0 )
                {
                    gpuCost = ( section.end - section.begin ).ToTimeSpan().GetMicroSeconds() / 1000.f;
                }
            }
            _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, gpuCost, "ms", cx, cy, paramWidth, frameHeight, groundColor );
            cx += paramWidth / 2;
        }
    }
}

void DrawEmitterSetInfo( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::vfx::System* pVfxSystem, float x, float y, int groupID ) NN_NOEXCEPT
{
    pTextWriter->SetCursor( x, y );

    _DrawEmitterSetInfo( pCommandBuffer, pTextWriter, pPrimitiveRenderer, nullptr );
    pTextWriter->SetCursorX( x );
    pTextWriter->Print( "\n" );
    const nn::vfx::EmitterSet* pEmitterSet = pVfxSystem->GetEmitterSetHead( groupID );
    while ( pEmitterSet )
    {
        const nn::vfx::EmitterSetResource* emitterSetResource = pEmitterSet->GetEmitterSetResource();
        if ( emitterSetResource->isVisible )
        {
            _DrawEmitterSetInfo( pCommandBuffer, pTextWriter, pPrimitiveRenderer, pEmitterSet );
            pTextWriter->SetCursorX( x );
            pTextWriter->Print( "\n" );
        }
        pEmitterSet = pEmitterSet->GetNext();
    }
}

void DrawSystemInfo( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::vfx::System* pVfxSystem, float x, float y, size_t dynamicHeapSize ) NN_NOEXCEPT
{
    const float frameWidth = 210.f;
    const float frameHeight = 20.f;
    const float param0Width = 65.f;
    const float param1Width = 65.f;
    const float param2Width = 80.f;

    const float labelPosX  = x;
    const int align = 2;

    pTextWriter->SetCursor( x, y );

    const nn::util::Color4u8 groundColor( 32, 32, 32, 200 );
    const nn::util::Color4u8 captionColor( 79, 129, 189, 200 );

    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "System", labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, 0, captionColor ); pTextWriter->Print( "\n" );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "EmitterSet", pVfxSystem->GetProcessingEmitterSetCount(), pVfxSystem->GetEmitterSetCountMax(), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, align, groundColor ); pTextWriter->Print( "\n" );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Emitter", pVfxSystem->GetProcessingEmitterCount(), pVfxSystem->GetEmitterCountMax(), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, align, groundColor ); pTextWriter->Print( "\n" );

    float posy = pTextWriter->GetCursorY();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "C:", pVfxSystem->GetProcessingEmitterCount( nn::vfx::detail::EmitterCalculationMode_Cpu ), labelPosX, posy, param0Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "G:", pVfxSystem->GetProcessingEmitterCount( nn::vfx::detail::EmitterCalculationMode_Gpu ), labelPosX + param0Width, posy, param1Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "GS:", pVfxSystem->GetProcessingEmitterCount( nn::vfx::detail::EmitterCalculationMode_GpuStreamOut ), labelPosX + param1Width * 2, posy, param2Width, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );

    int particleCount = pVfxSystem->GetProcessingCpuParticleCount() + pVfxSystem->GetProcessingGpuParticleCount() + pVfxSystem->GetProcessingGpuSoParticleCount();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Particle", particleCount, labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, align, groundColor ); pTextWriter->Print( "\n" );
    posy = pTextWriter->GetCursorY();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "C:", pVfxSystem->GetProcessingCpuParticleCount(), labelPosX, posy, param0Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "G:", pVfxSystem->GetProcessingGpuParticleCount(), labelPosX + param0Width, posy, param1Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "GS:", pVfxSystem->GetProcessingGpuSoParticleCount(), labelPosX + param1Width * 2, posy, param2Width, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );

    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Stripe", pVfxSystem->GetProcessingStripeCount(), pVfxSystem->GetStripeCountMax(), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, align, groundColor ); pTextWriter->Print( "\n" );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Stripe2", pVfxSystem->GetProcessingSuperStripeCount(), pVfxSystem->GetSuperStripeCountMax(), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, align, groundColor ); pTextWriter->Print( "\n" );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "C-Stripe", pVfxSystem->GetProcessingConnectionStripeCount(), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, align, groundColor ); pTextWriter->Print( "\n" );

    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "D-Buffer", static_cast< int >( pVfxSystem->GetAllocatedFromDynamicHeapSize() ), static_cast< int >( dynamicHeapSize ), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, align, groundColor ); pTextWriter->Print( "\n" );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "G-Buffer", static_cast< int >( pVfxSystem->GetAllocatedFromGpuHeap() ), static_cast< int >( pVfxSystem->GetGpuHeapSizeMax() ), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, align, groundColor ); pTextWriter->Print( "\n" );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "T-Buffer", static_cast< int >( pVfxSystem->GetUsedTemporaryBufferSize( 0 ) ), static_cast< int >( pVfxSystem->GetTemporaryBufferSize() ), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, align, groundColor ); pTextWriter->Print( "\n" );
}


// vfx ビューア 詳細情報を表示します。
void DrawViewerInfo( nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::util::DebugFontTextWriter* pTextWriter, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::vfx::System* pVfxSystem, float x, float y, size_t dynamicHeapSize, float viewerCounter, float viewerCounterMax ) NN_NOEXCEPT
{
    float posy = 0.0f;

    const float frameWidth = 200.f;
    const float frameHeight = 20.f;

    const float labelWidth = 60.f;
    const float param0Width = 75.f;
    const float param1Width = 65.f;

    const float labelPosX = x;
    const float param0PosX = labelPosX + labelWidth;
    const float param1PosX = param0PosX + param0Width;

    int emitterSetCount             = 0;
    int cpuEmitterCount             = 0;
    int gpuEmitterCount             = 0;
    int gpuSoEmitterCount           = 0;
    int cpuParticleCount            = 0;
    int gpuParticleCount            = 0;
    int gpuSoParticleCount          = 0;
    int stripeCount                 = 0;
    int superStripeCount            = 0;

    size_t dynamicBufferAllocatedSize = 0;
    size_t gpuBufferAllocatedSize     = 0;

    nn::vfx::ProcessingInfo visibleProcessingInfo;
    visibleProcessingInfo.Clear();

    const nn::util::Color4u8 groundColor( 32, 32, 32, 255 );
    const nn::util::Color4u8 captionColor( 79, 129, 189, 255 );

    const nn::vfx::EmitterSet* pEmitterSet = pVfxSystem->GetEmitterSetHead( nn::vfx::SystemParameters_ViewerGroupId );
    while ( pEmitterSet )
    {
        emitterSetCount++;

        if ( pEmitterSet->GetEmitterSetResource()->isVisible )
        {
            const nn::vfx::Emitter* pEmitter = pEmitterSet->GetEmitter( 0 );

            while( pEmitter )
            {
                if ( pEmitter->GetEmitterResource()->m_IsVisible )
                {
                    visibleProcessingInfo.UpdateProcessingInfo( pEmitter );
                }

                for ( int i = 0; i < pEmitter->GetEmitterResource()->m_ChildEmitterResCount; i++ )
                {
                    const nn::vfx::Emitter* pChildEmitter = pEmitter->GetChildEmitterHead( i );
                    while ( pChildEmitter )
                    {
                        if ( pChildEmitter->GetEmitterResource()->m_IsVisible )
                        {
                            visibleProcessingInfo.UpdateProcessingInfo( pChildEmitter );
                        }
                        pChildEmitter = pChildEmitter->GetNextEmitter();
                    }
                }

                pEmitter = pEmitter->GetNextEmitter();
            }
        }

        cpuEmitterCount          += pEmitterSet->GetProcessingCpuEmitterCount();
        gpuEmitterCount          += pEmitterSet->GetProcessingGpuEmitterCount();
        gpuSoEmitterCount        += pEmitterSet->GetProcessingComputeEmitterCount();
        cpuParticleCount         += pEmitterSet->GetProcessingCpuParticleCount();
        gpuParticleCount         += pEmitterSet->GetProcessingGpuParticleCount();
        gpuSoParticleCount       += pEmitterSet->GetProcessingGpuSoParticleCount();
        stripeCount              += pEmitterSet->GetProcessingStripeCount();
        superStripeCount         += pEmitterSet->GetProcessingSuperStripeCount();
        if ( pEmitterSet->GetEmitterSetResource()->isVisible )
        {
            dynamicBufferAllocatedSize += pEmitterSet->GetAllocatedFromDynamicHeapSize();
            gpuBufferAllocatedSize += pEmitterSet->GetAllocatedFromGpuBufferSize();
        }

        pEmitterSet = pEmitterSet->GetNext();
    }

    pTextWriter->SetCursor( x, y );

    posy = pTextWriter->GetCursorY();

    // 暫定: 厳密なGPUカウンタモードを使用する場合は、オプションが通っていることをここでざっくり表示する
    if( pVfxSystem->IsEnabledPreciseGpuCounterMode() )
    {
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Viewer(+)", labelPosX, posy, frameWidth, frameHeight, 0, captionColor );
    }
    else
    {
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Viewer", labelPosX, posy, frameWidth, frameHeight, 0, captionColor );
    }
    pTextWriter->Print( "\n" );

    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "EmitterSet", emitterSetCount, pVfxSystem->GetEmitterSetCountMax(), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor ); pTextWriter->Print( "\n" );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Emitter", ( cpuEmitterCount + gpuEmitterCount + gpuSoEmitterCount ), pVfxSystem->GetEmitterCountMax(), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor ); pTextWriter->Print( "\n" );

    posy = pTextWriter->GetCursorY();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Cpu", labelPosX, posy, labelWidth, frameHeight*2, 0, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "E:", visibleProcessingInfo.cpuEmitterCount, param0PosX, posy, param0Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "", cpuEmitterCount, param1PosX, posy, param1Width, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );
    posy = pTextWriter->GetCursorY();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "P:", visibleProcessingInfo.cpuParticleCount, param0PosX, posy, param0Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "", cpuParticleCount, param1PosX, posy, param1Width, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );

    posy = pTextWriter->GetCursorY();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Gpu", labelPosX, posy, labelWidth, frameHeight * 2, 0, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "E:", visibleProcessingInfo.gpuEmitterCount, param0PosX, posy, param0Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "", gpuEmitterCount, param1PosX, posy, param1Width, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );
    posy = pTextWriter->GetCursorY();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "P:", visibleProcessingInfo.gpuParticleCount, param0PosX, posy, param0Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "", gpuParticleCount, param1PosX, posy, param1Width, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );

    posy = pTextWriter->GetCursorY();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "GpuSo", labelPosX, posy, labelWidth, frameHeight * 2, 0, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "E:", visibleProcessingInfo.computeEmitterCount, param0PosX, posy, param0Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "", gpuSoEmitterCount, param1PosX, posy, param1Width, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );
    posy = pTextWriter->GetCursorY();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "P:", visibleProcessingInfo.gpusoParticleCount, param0PosX, posy, param0Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "", gpuSoParticleCount, param1PosX, posy, param1Width, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );

    posy = pTextWriter->GetCursorY();
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Stripe", labelPosX, posy, labelWidth, frameHeight, 0, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "S:", visibleProcessingInfo.stripeCount, param0PosX, posy, param0Width, frameHeight, 2, groundColor );
    _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "S2:", visibleProcessingInfo.superStripeCount, param1PosX, posy, param1Width, frameHeight, 2, groundColor );
    pTextWriter->Print( "\n" );

    // Viewr Time
    posy = pTextWriter->GetCursorY();
    if ( viewerCounterMax == 0.0f )
    {
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Time", viewerCounter, labelPosX, posy, frameWidth, frameHeight, 2, groundColor );
    }
    else
    {
        _DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "Time", viewerCounter, viewerCounterMax, labelPosX, posy, frameWidth, frameHeight, 2, groundColor );
    }
    pTextWriter->Print( "\n" );

    //posy = pTextWriter->GetCursorY();
    //_DrawTextBox( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "CS:", visibleProcessingInfo.connectionStripeCount, param1PosX, posy, param1Width, frameHeight, 2, groundColor );
    //pTextWriter->Print( "\n" );

    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "D-Buffer", static_cast< int >( dynamicBufferAllocatedSize ), static_cast< int >( dynamicHeapSize ), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor ); pTextWriter->Print( "\n" );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "G-Buffer", static_cast< int >( gpuBufferAllocatedSize ), static_cast< int >( pVfxSystem->GetGpuHeapSizeMax() ), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor ); pTextWriter->Print( "\n" );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "T-Buffer", static_cast< int >( pVfxSystem->GetUsedTemporaryBufferSize( 0 ) ), static_cast< int >( pVfxSystem->GetTemporaryBufferSize() ), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor ); pTextWriter->Print( "\n" );

    int64_t costMax = static_cast<int64_t>( 16.666f * 1000 );

    // CPU Cost
    MeterItem meterItem[ 2 ];
    meterItem[ 0 ].value = static_cast< int >( NN_PERF_GET_ELAPSED_TIME_CPU( nullptr, "ProcVfxCalc", 0 ).GetMicroSeconds() );
    meterItem[ 0 ].color = nn::util::Color4u8( 128, 255, 128, 255 );
    meterItem[ 1 ].value = static_cast< int >( NN_PERF_GET_ELAPSED_TIME_CPU( nullptr, "ProcVfxDraw", 0 ).GetMicroSeconds() );
    meterItem[ 1 ].color = nn::util::Color4u8( 255, 128, 128, 255 );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "CPU", meterItem, 2, static_cast< int >( costMax ), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor, true ); pTextWriter->Print( "\n" );

    // GPU Cost
    meterItem[ 0 ].value = static_cast< int >( NN_PERF_GET_ELAPSED_TIME_GPU( "VfxGpu", 0 ).GetMicroSeconds() );
    meterItem[ 0 ].color = nn::util::Color4u8( 128, 255, 128, 255 );
    meterItem[ 1 ].value = static_cast< int >( NN_PERF_GET_ELAPSED_TIME_GPU( "Compute", 0 ).GetMicroSeconds() );
    meterItem[ 1 ].color = nn::util::Color4u8( 255, 128, 128, 255 );
    _DrawMeter( pCommandBuffer, pTextWriter, pPrimitiveRenderer, "GPU", meterItem, 2, static_cast< int >( costMax ), labelPosX, pTextWriter->GetCursorY(), frameWidth, frameHeight, 2, groundColor, true ); pTextWriter->Print( "\n" );
}
