﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "detail/util_DrawShapes.h"
#include <nn/vfx/vfx_AreaLoop.h>

namespace detail {
//---------------------------------------------------------------------------
//  形状描画のメイン処理です。
//---------------------------------------------------------------------------
void DrawShapes::DrawMain( nn::gfx::CommandBuffer* pCommandBuffer,
                           nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                           const nn::vfx::Emitter* pEmitter,
                           const nn::util::Vector3fType& cameraPos,
                           bool isShapes,
                           bool isField ) NN_NOEXCEPT
{
    if (isShapes)
    {
        switch (pEmitter->GetResEmitter()->volume.volumeType)
        {
        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Point:
                nn::util::Matrix4x3fType matrixSrt;
                GetMatrix( pEmitter, &matrixSrt );
                DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
            break;
        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Circle:
        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_CircleEquallyDivided:
            DrawCircle(pCommandBuffer, pRenderer, pEmitter, cameraPos);
            break;

        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_CircleFill:
            DrawCircleFill(pCommandBuffer, pRenderer, pEmitter, cameraPos);
            break;

        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Sphere:
        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereFill:
        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereEquallyDivided:
        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereEqually64Divided:
            DrawSphere(pCommandBuffer, pRenderer, pEmitter, cameraPos);
            break;

        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Cylinder:
        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_CylinderFill:
            DrawCylinder(pCommandBuffer, pRenderer, pEmitter, cameraPos);
            break;

        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Box:
        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_BoxFill:
            DrawCube(pCommandBuffer, pRenderer, pEmitter, cameraPos);
            break;

        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Line:
        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_LineEquallyDivided:
            DrawLine(pCommandBuffer, pRenderer, pEmitter, cameraPos);
            break;

        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Rectangle:
            DrawRectangle(pCommandBuffer, pRenderer, pEmitter, cameraPos);
            break;

        case nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Primitive:
            DrawPrimitive(pCommandBuffer, pRenderer, pEmitter, cameraPos);
            break;
        }
    }

    // フィールド情報の描画
    if ( pEmitter->GetEmitterResource()->m_IsUseField  && isField )
    {
        if ( pEmitter->GetEmitterResource()->m_pFieldConvergenceData )
        {
            DrawFieldConvergencePosition( pCommandBuffer, pRenderer, pEmitter, cameraPos );
        }

        if ( pEmitter->GetEmitterResource()->m_pFieldMagnetData )
        {
            DrawFieldMagnetPosition(pCommandBuffer, pRenderer, pEmitter, cameraPos);
        }

        if ( pEmitter->GetEmitterResource()->m_pFieldSpinData )
        {
            DrawFieldSpin( pCommandBuffer, pRenderer, pEmitter, cameraPos );
        }
        if( pEmitter->GetEmitterResource()->m_pFieldCollisionData )
        {
            DrawCollision( pCommandBuffer, pRenderer, pEmitter, cameraPos );
        }
    }

    // 範囲内ループ情報の描画
    if ( pEmitter->GetEmitterResource()->m_EmitterPluginIndex == nn::vfx::detail::AreaLoopSystem::PluginId && pEmitter->GetEmitterResource()->m_pEmitterPluginData && isField )
    {
        DrawAreaLoop( pCommandBuffer, pRenderer, pEmitter, cameraPos );
    }
}

//---------------------------------------------------------------------------
//  フィールドの収束点描画
//---------------------------------------------------------------------------
void DrawShapes::DrawFieldConvergencePosition( nn::gfx::CommandBuffer* pCommandBuffer,
                                               nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                               const nn::vfx::Emitter* pEmitter,
                                               const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    nn::util::Vector3fType convergencePosition;
    nn::util::VectorLoad( &convergencePosition, pEmitter->GetEmitterResource()->m_pFieldConvergenceData->fieldConvergencePos );


    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );

    nn::util::Matrix4x3fType rtMatrix;
    nn::util::Matrix4x3fType transMatrix;
    nn::util::MatrixIdentity( &transMatrix );
    nn::util::MatrixSetAxisW( &transMatrix, convergencePosition );
    nn::util::MatrixMultiply( &rtMatrix, transMatrix, matrixSrt );

    pRenderer->SetModelMatrix( &rtMatrix );

    nn::util::Vector3fType pos;
    nn::util::MatrixGetAxisW( &pos, rtMatrix );
    float scale = AjustScreenScaleSize( pos, cameraPos );

    // Pointの描画
    nn::util::Uint8x4 blue  = { { 0, 255, 255, 100 } };
    nn::util::Vector3fType center = { 0.f, 0.f, 0.f };

    pRenderer->SetColor(blue);
    pRenderer->DrawSphere( pCommandBuffer,
                           nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
                           nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
                           center, scale );

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  フィールドの磁力点描画
//---------------------------------------------------------------------------
void DrawShapes::DrawFieldMagnetPosition( nn::gfx::CommandBuffer* pCommandBuffer,
                                          nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                          const nn::vfx::Emitter* pEmitter,
                                          const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT
{
    nn::util::Vector3fType magnetPosition;
    nn::util::VectorLoad(&magnetPosition, pEmitter->GetEmitterResource()->m_pFieldMagnetData->fieldMagnetPos);

    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix(pEmitter, &matrixSrt);

    nn::util::Matrix4x3fType rtMatrix;
    nn::util::Matrix4x3fType transMatrix;
    nn::util::MatrixIdentity(&transMatrix);
    nn::util::MatrixSetAxisW(&transMatrix, magnetPosition);
    nn::util::MatrixMultiply(&rtMatrix, transMatrix, matrixSrt);

    pRenderer->SetModelMatrix(&rtMatrix);

    nn::util::Vector3fType pos;
    nn::util::MatrixGetAxisW(&pos, rtMatrix);
    float scale = AjustScreenScaleSize(pos, cameraPos);

    // Pointの描画
    nn::util::Uint8x4 blue = { { 0, 255, 255, 100 } };
    nn::util::Vector3fType center = { 0.f, 0.f, 0.f };

    pRenderer->SetColor(blue);
    pRenderer->DrawSphere(pCommandBuffer,
        nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
        nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
        center, scale);

    DrawAxis(pCommandBuffer, pRenderer, matrixSrt, cameraPos);
}

//---------------------------------------------------------------------------
//  フィールドのスピン形状描画
//---------------------------------------------------------------------------
void DrawShapes::DrawFieldSpin( nn::gfx::CommandBuffer* pCommandBuffer,
                                nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                const nn::vfx::Emitter* pEmitter,
                                const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    nn::vfx::detail::ResFieldSpinData* pFieldSpinData = pEmitter->GetEmitterResource()->m_pFieldSpinData;
    uint32_t axis = pFieldSpinData->fieldSpinAxis;
    int direction =  ( pFieldSpinData->fieldSpinRotate < 0 ) ? 1 : 0;

    // 矢印頂点
    const nn::util::Float3 TriangleVertexPos[2][ Shapes_TriangleVertexCount ] =
    {{  // 左向き矢印
        NN_UTIL_FLOAT_3_INITIALIZER(  0.0f,  0.2f, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( -0.4f,  0.0f, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER(  0.0f, -0.2f, 0.f )
    },
    {   // 右向き矢印
        NN_UTIL_FLOAT_3_INITIALIZER( 0.4f,  0.0f, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( 0.0f,  0.2f, 0.f ),
        NN_UTIL_FLOAT_3_INITIALIZER( 0.0f, -0.2f, 0.f )
    }};


    nn::util::Matrix4x3fType rtMatrix;
    GetMatrix( pEmitter, &rtMatrix );

    nn::util::Matrix4x3fType scaleRotateMatrix;
    nn::util::MatrixIdentity( &scaleRotateMatrix );

    nn::util::Vector3fType pos;
    nn::util::MatrixGetAxisW( &pos, rtMatrix );
    float screenScale = AjustScreenScaleSize( pos, cameraPos );
    nn::util::Vector3fType size = NN_UTIL_VECTOR_3F_INITIALIZER( screenScale, screenScale, screenScale );

    nn::util::Vector3fType  rotate;
    switch ( axis )
    {
    case 0: // X軸
        nn::util::VectorSet( &rotate, 0.0f, 0.0f, -nn::util::DegreeToRadian(90) );
        nn::util::MatrixSetScaleRotateXyz( &scaleRotateMatrix, size, rotate );
    break;
    case 1: // Y軸
        nn::util::MatrixSetScale( &scaleRotateMatrix, size );
    break;
    case 2: // Z軸
        nn::util::VectorSet( &rotate, nn::util::DegreeToRadian(90), 0.0f, 0.0f );
        nn::util::MatrixSetScaleRotateXyz( &scaleRotateMatrix, size, rotate );
    break;
    };

    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );
    nn::util::MatrixMultiply( &rtMatrix, scaleRotateMatrix, matrixSrt );
    pRenderer->SetModelMatrix( &rtMatrix );

    //　頂点データ作成
    int numVerties = ( detail::Shapes_NumLines - 3 ) + 3 + 2; // 円 + 矢印 + 棒
    int numIndices = ( numVerties - 2 ) * 2 ;
    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBuffer;
    bool ret = meshBuffer.Initialize( pRenderer->GetGpuBuffer(), numVerties, numIndices, nns::gfx::PrimitiveRenderer::VertexFormat_Pos );

    if ( !ret ) return;          // meshBuffer初期化が出来ない場合は処理を終了して戻る。

    nn::util::Float3* vertex = static_cast<nn::util::Float3*>( meshBuffer.GetVertexBufferCpuAddress(  nns::gfx::PrimitiveRenderer::VertexAttribute::VertexAttribute_Pos ) );
    uint32_t* pIndex = meshBuffer.GetIndexBufferCpuAddress();

    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( 1.5f, 9.0f, 1.5f );
    float minLatitude = nn::util::DegreeToRadian( 360.f ) / detail::Shapes_NumLines;  // 最小角度幅

    nn::util::Uint8x4 blue  = { { 0, 255, 255, 100 } };
    pRenderer->SetColor( blue );

    // 回転が逆の時
    if ( direction == 1 )
    {
        minLatitude *= -1.0f;
    }

    int vertexidx = 0;
    int idx = 0;

    // 回転方向を示すので先頭の円は途中まで描画する
    for ( int i = 0; i < detail::Shapes_NumLines - 3; i++ )
    {
        float rot = minLatitude * i;
        CalculatePos( &pos, rot, scale );

        vertex[i].x =  nn::util::VectorGetX( pos );
        vertex[i].y =  nn::util::VectorGetY( pos );
        vertex[i].z =  nn::util::VectorGetZ( pos );
        vertexidx++;

        pIndex[idx] = i;
        pIndex[idx+1] = i+1;
        idx += 2;
    }
    idx -= 2;

    // 三角形
    vertex[vertexidx].x   = TriangleVertexPos[direction][0].x;
    vertex[vertexidx].y   = TriangleVertexPos[direction][0].y + vertex[0].y;
    vertex[vertexidx].z   = TriangleVertexPos[direction][0].z + vertex[0].z;

    vertex[vertexidx+1].x = TriangleVertexPos[direction][1].x;
    vertex[vertexidx+1].y = TriangleVertexPos[direction][1].y + vertex[0].y;
    vertex[vertexidx+1].z = TriangleVertexPos[direction][1].z + vertex[0].z;

    vertex[vertexidx+2].x = TriangleVertexPos[direction][2].x;
    vertex[vertexidx+2].y = TriangleVertexPos[direction][2].y + vertex[0].y;
    vertex[vertexidx+2].z = TriangleVertexPos[direction][2].z + vertex[0].z;

    pIndex[idx]           = vertexidx;
    pIndex[idx+1]         = vertexidx+1;

    pIndex[idx+2]         = vertexidx+1;
    pIndex[idx+3]         = vertexidx+2;

    pIndex[idx+4]         = vertexidx+2;
    pIndex[idx+5]         = vertexidx;

    vertexidx += 3;
    idx += 6;

    // 直線
    vertex[vertexidx].x   = 0.0f;
    vertex[vertexidx].y   = 9.0f;
    vertex[vertexidx].z   = 0.0f;

    vertex[vertexidx+1].x = 0.0f;
    vertex[vertexidx+1].y = -9.0f;
    vertex[vertexidx+1].z = 0.0f;

    pIndex[idx]           = vertexidx;
    pIndex[idx+1]         = vertexidx+1;

    pRenderer->SetLineWidth( 3.f );
    pRenderer->DrawUserMesh( pCommandBuffer,
                             nn::gfx::PrimitiveTopology::PrimitiveTopology_LineList,
                             &meshBuffer, 0, numIndices, 0 );

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  フィールドのコリジョン面描画
//---------------------------------------------------------------------------
void DrawShapes::DrawCollision( nn::gfx::CommandBuffer* pCommandBuffer,
                                nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                const nn::vfx::Emitter* pEmitter,
                                const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT
{
    nn::util::Uint8x4      blue   = { { 0, 255, 255, 100 } };
    nn::util::Vector3fType center = { -50.f, -50.f, 0.f };
    nn::util::Vector3fType size   = { 100.f, 100.f, 50.f };

    nn::util::Vector3fType rotValue = NN_UTIL_VECTOR_3F_INITIALIZER(nn::util::DegreeToRadian(90), 0.f, 0.f);
    nn::util::Matrix4x3f   modelMatrix;
    nn::util::MatrixIdentity(&modelMatrix);
    nn::util::MatrixSetRotateXyz(&modelMatrix, rotValue);

    nn::vfx::detail::ResFieldCollisionData* pFieldCollisionData = pEmitter->GetEmitterResource()->m_pFieldCollisionData;

    // XZ平面のY座標
    float ypos = pFieldCollisionData->fieldCollisionCoord;

    //      ワールド座標系
    if( pFieldCollisionData->fieldCollisionIsWorld )
    {
        nn::util::Vector3fType translate = NN_UTIL_VECTOR_3F_INITIALIZER( 0.f, ypos, 0.f );
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
    }
    else
    {
        nn::util::Vector3fType translate;
        nn::util::MatrixGetAxisW( &translate, pEmitter->GetMatrixSrt() );
        nn::util::VectorSetY( &translate, ypos );
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        nn::util::MatrixMultiply( &modelMatrix, modelMatrix, pEmitter->GetMatrixSrt() );
    }

    pRenderer->SetModelMatrix( &modelMatrix );
    pRenderer->SetColor(blue);
    pRenderer->DrawQuad( pCommandBuffer, center, size );

    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );
    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  範囲内ループ描画
//---------------------------------------------------------------------------
void DrawShapes::DrawAreaLoop( nn::gfx::CommandBuffer* pCommandBuffer,
                               nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                               const nn::vfx::Emitter* pEmitter,
                               const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    nn::vfx::detail::ResEPAreaLoop* areaLoopData = reinterpret_cast< nn::vfx::detail::ResEPAreaLoop* >( pEmitter->GetEmitterResource()->m_pEmitterPluginData );

    if( areaLoopData->isCameraLoop != 0 ) return;       // "箱をカメラ前に固定"の時は描画しない。

    nn::util::Uint8x4      blue   = { { 0, 255, 255, 100 } };
    nn::util::Uint8x4      white  = { { 255, 255, 255, 255 } };
    nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
    nn::util::Vector3fType size   = NN_UTIL_VECTOR_3F_INITIALIZER( areaLoopData->areaSize.v[ 0 ] * 2, areaLoopData->areaSize.v[ 1 ] * 2, areaLoopData->areaSize.v[ 2 ] * 2 );

    // 仮想箱系へのRT行列
    // Rotate
    nn::util::Matrix4x3fType boxMatrix;
    nn::util::MatrixIdentity( &boxMatrix );
    nn::util::Vector3fType rot = NN_UTIL_VECTOR_3F_INITIALIZER( areaLoopData->areaRotate.v[ 0 ], areaLoopData->areaRotate.v[ 1 ], areaLoopData->areaRotate.v[ 2 ] );
    nn::util::MatrixSetRotateXyz( &boxMatrix, rot );

    // Translate
    // 標準の場合、エミッタセットSRTを考慮
    nn::util::Matrix4x3fType m = NN_UTIL_MATRIX_4X3F_INITIALIZER(
        0, 0, 0,
        0, 0, 0,
        0, 0, 0,
        areaLoopData->areaPos.v[ 0 ], areaLoopData->areaPos.v[ 1 ], areaLoopData->areaPos.v[ 2 ] );
    nn::util::MatrixAdd( &boxMatrix, boxMatrix, m );

    // 標準の場合、EmitterSetSRTを適用する
    nn::util::Matrix4x3f   modelMatrix;
    nn::util::MatrixIdentity(&modelMatrix);
    nn::util::MatrixMultiply( &modelMatrix, boxMatrix, pEmitter->GetMatrixSrt() );

    pRenderer->SetLineWidth(1.f);
    pRenderer->SetModelMatrix( &modelMatrix );
    pRenderer->SetColor(blue);
    pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size );
    pRenderer->SetColor(white);
    pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Wired, center, size );

    // クリッピング用の板描画
    if ( areaLoopData->clippingType > 0 )
    {
        nn::util::Vector3fType rotate    = NN_UTIL_VECTOR_3F_INITIALIZER( nn::util::DegreeToRadian(90), 0.0f, 0.0f );
        nn::util::Vector3fType planeSize = NN_UTIL_VECTOR_3F_INITIALIZER( areaLoopData->areaSize.v[ 0 ] * 2, areaLoopData->areaSize.v[ 1 ] * 2, areaLoopData->areaSize.v[ 2 ] * 2 );

        nn::util::Vector3fType translate;
        nn::util::MatrixGetAxisW( &translate, modelMatrix );
        nn::util::VectorSetY( &translate, areaLoopData->clippingWorldHeight );

        nn::util::MatrixIdentity( &boxMatrix );
        nn::util::MatrixSetRotateXyz( &boxMatrix, rotate );
        nn::util::MatrixSetTranslate( &boxMatrix, translate );
        pRenderer->SetModelMatrix( &boxMatrix );

        white.v[3] = 50;
        pRenderer->SetColor(white);

        nn::util::VectorSetX( &center, -areaLoopData->areaSize.v[ 0 ] );
        nn::util::VectorSetY( &center, -areaLoopData->areaSize.v[ 2 ] );
        pRenderer->DrawQuad( pCommandBuffer, center, planeSize );
    }

    DrawAxis( pCommandBuffer, pRenderer, modelMatrix, cameraPos );
}

//---------------------------------------------------------------------------
//  円形状描画処理を行います。
//---------------------------------------------------------------------------
void DrawShapes::DrawCircle( nn::gfx::CommandBuffer* pCommandBuffer,
                            nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                            const nn::vfx::Emitter* pEmitter,
                            const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    const nn::vfx::detail::ResEmitterVolume* pEmitterVolume = &pEmitter->GetResEmitter()->volume;

    float scaleX = pEmitterVolume->volumeRadius.v[ 0 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 0 ];
    float scaleZ = pEmitterVolume->volumeRadius.v[ 2 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 2 ];
    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, 0.0f, scaleZ );

    float minLatitude;  // 最小角度幅
    float sweepStart ;  // 開始角度

    CalculateVolumeSweepParams( pEmitter, pEmitterVolume, &minLatitude, &sweepStart, detail::Shapes_NumLines );

    nn::util::Vector3fType firstStartPos;
    nn::util::Vector3fType beginPos;
    nn::util::Vector3fType endPos;

    CalculatePos( &beginPos, sweepStart, scale );
    firstStartPos = beginPos;

    nn::util::Uint8x4 white = { { 255, 255, 255, 255 } };
    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );

    pRenderer->SetColor( white );
    pRenderer->SetModelMatrix( &matrixSrt );
    pRenderer->SetLineWidth(1.f);

    for ( int i = 0; i < detail::Shapes_NumLines; i++ )
    {
        float rot = minLatitude * (i + 1) + sweepStart;
        CalculatePos( &endPos, rot, scale );

        pRenderer->DrawLine( pCommandBuffer, beginPos, endPos );

        beginPos = endPos;
    }

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  円ポリゴン形状描画処理を行います。
//---------------------------------------------------------------------------
void DrawShapes::DrawCircleFill( nn::gfx::CommandBuffer* pCommandBuffer,
                                 nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                 const nn::vfx::Emitter* pEmitter,
                                 const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    const nn::vfx::detail::ResEmitterVolume* pEmitterVolume = &pEmitter->GetResEmitter()->volume;

    float scaleX = pEmitterVolume->volumeRadius.v[ 0 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 0 ];
    float scaleZ = pEmitterVolume->volumeRadius.v[ 2 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 2 ];
    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, 0.0f, scaleZ );

    float minLatitude;  // 最小角度幅
    float sweepStart ;  // 開始角度

    CalculateVolumeSweepParams( pEmitter, pEmitterVolume, &minLatitude, &sweepStart, detail::Shapes_NumLines );

    nn::util::Float3 beginPos;
    nn::util::Float3 endPos;

    CalculatePos( &beginPos, sweepStart, scale );

    //  MeshBufferの作成
    int numVertices;
    int numIndices;
    CalculateVertexIndexBufferSize( &numVertices, &numIndices, detail::Shapes_NumLines, detail::ShapeBufferType_Triangle );

    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBuffer;
    int ret = meshBuffer.Initialize( pRenderer->GetGpuBuffer(), numVertices, numIndices, nns::gfx::PrimitiveRenderer::VertexFormat_Pos );
    if ( !ret ) return;       // meshBuffer初期化が出来ない場合は処理を終了して戻る。

    nn::util::Float3* pVertex = static_cast<nn::util::Float3*>( meshBuffer.GetVertexBufferCpuAddress(  nns::gfx::PrimitiveRenderer::VertexAttribute::VertexAttribute_Pos ) );
    uint32_t*  pIndex = meshBuffer.GetIndexBufferCpuAddress();

    int verNumTri  = 0;
    for ( int i = 0; i < detail::Shapes_NumLines; i++ )
    {
        float rot = minLatitude * (i + 1) + sweepStart;

        CalculatePos( &endPos, rot, scale );

        // top
        pVertex[verNumTri+0].x = 0;
        pVertex[verNumTri+0].y = beginPos.y;
        pVertex[verNumTri+0].z = 0;

        pVertex[verNumTri+1]   = beginPos;

        pVertex[verNumTri+2]   = endPos;
        pIndex[verNumTri+0]    = 0 + (uint32_t)verNumTri; pIndex[verNumTri+1] = 2 + (uint32_t)verNumTri; pIndex[verNumTri+2] = 1 + (uint32_t)verNumTri;

        beginPos  = endPos;
        verNumTri += 3;
    }

    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );
    pRenderer->SetModelMatrix( &matrixSrt );
    pRenderer->SetColor( m_Color );
    pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBuffer );

    // 円(ボリューム)の空洞率の描画
    if ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_CircleFill )
    {
        nn::util::Matrix4x3fType esetSrt;
        CalculateHoleScale( pEmitter, &esetSrt, true );

        pRenderer->SetModelMatrix( &esetSrt );
        pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBuffer );
    }

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  球形状描画処理を行います。
//---------------------------------------------------------------------------
void DrawShapes::DrawSphere( nn::gfx::CommandBuffer* pCommandBuffer,
                             nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                             const nn::vfx::Emitter* pEmitter,
                             const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    const nn::vfx::detail::ResEmitterVolume* pEmitterVolume = &pEmitter->GetResEmitter()->volume;

    float scaleX = pEmitterVolume->volumeRadius.v[ 0 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 0 ];
    float scaleY = pEmitterVolume->volumeRadius.v[ 1 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 1 ];
    float scaleZ = pEmitterVolume->volumeRadius.v[ 2 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 2 ];
    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, scaleY, scaleZ );

    float minLatitude;  // 最小角度幅
    float sweepStart ;  // 開始角度

    CalculateVolumeSweepParams( pEmitter, pEmitterVolume, &minLatitude, &sweepStart, detail::Shapes_NumShapes );

    //  MeshBufferの作成
    int numVertices;
    int numIndices;
    CalculateVertexIndexBufferSize( &numVertices, &numIndices, detail::Shapes_NumShapes * detail::Shapes_NumShapes, detail::ShapeBufferType_Quad );

    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBuffer;
    bool ret = meshBuffer.Initialize( pRenderer->GetGpuBuffer(), numVertices, numIndices, nns::gfx::PrimitiveRenderer::VertexFormat_Pos );
    if ( !ret ) return;      // meshBuffer初期化が出来ない場合は処理を終了して戻る。

    nn::util::Float3* pVertex = static_cast<nn::util::Float3*>( meshBuffer.GetVertexBufferCpuAddress(  nns::gfx::PrimitiveRenderer::VertexAttribute::VertexAttribute_Pos ) );
    uint32_t*         pIndex  = meshBuffer.GetIndexBufferCpuAddress();

    int indexPos = 0;
    int verNum   = 0;
    nn::util::Float3 beginPos[2];
    nn::util::Float3 endPos[2];
    for (int i = 0; i < detail::Shapes_NumShapes; i++)
    {
        for (int j = 0; j < detail::Shapes_NumShapes; j++)
        {
            PrimitiveSphere( pEmitter, &beginPos[0], minLatitude, sweepStart, i,   j,   scale );
            PrimitiveSphere( pEmitter, &endPos[0],   minLatitude, sweepStart, i,   j+1, scale );
            PrimitiveSphere( pEmitter, &beginPos[1], minLatitude, sweepStart, i+1, j,   scale );
            PrimitiveSphere( pEmitter, &endPos[1],   minLatitude, sweepStart, i+1, j+1, scale );
            {
                pVertex[verNum+0] = beginPos[1];
                pVertex[verNum+1] = beginPos[0];
                pVertex[verNum+2] = endPos[1];
                pVertex[verNum+3] = endPos[0];

                pIndex[indexPos+0] = 0 + (uint32_t)verNum; pIndex[indexPos+1] = 2 + (uint32_t)verNum; pIndex[indexPos+2] = 1 + (uint32_t)verNum;
                pIndex[indexPos+3] = 1 + (uint32_t)verNum; pIndex[indexPos+4] = 3 + (uint32_t)verNum; pIndex[indexPos+5] = 2 + (uint32_t)verNum;

                indexPos += (3 * 2);
                verNum += 4;
            }
        }
    }

    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );

    pRenderer->SetModelMatrix( &matrixSrt );
    pRenderer->SetLineWidth(1.f);
    pRenderer->SetColor( m_Color );
    pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBuffer );  // ポリゴン描画
    pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_LineList, &meshBuffer );      // ワイヤー描画

    // 球(ボリューム)の空洞率の描画
    if ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereFill )
    {
        nn::util::Matrix4x3fType esetSrt;
        CalculateHoleScale( pEmitter, &esetSrt, false );

        pRenderer->SetModelMatrix( &esetSrt );
        pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBuffer );
    }

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  円柱形状描画処理を行います。
//---------------------------------------------------------------------------
void DrawShapes::DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                               nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                               const nn::vfx::Emitter* pEmitter,
                               const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    const nn::vfx::detail::ResEmitterVolume* pEmitterVolume = &pEmitter->GetResEmitter()->volume;

    float scaleX = pEmitterVolume->volumeRadius.v[ 0 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 0 ];
    float scaleY = pEmitterVolume->volumeRadius.v[ 1 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 1 ];
    float scaleZ = pEmitterVolume->volumeRadius.v[ 2 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 2 ];
    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, scaleY, scaleZ );

    float minLatitude;  // 最小角度幅
    float sweepStart ;  // 開始角度

    CalculateVolumeSweepParams( pEmitter, pEmitterVolume, &minLatitude, &sweepStart, detail::Shapes_NumLines );

    nn::util::Float3 beginPos;
    nn::util::Float3 endPos;

    CalculatePos( &beginPos, sweepStart, scale );

    //  MeshBufferの作成
    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBufferTop;
    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBufferBottom;
    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBufferSide;
    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBufferSideWired;

    int numVertices;
    int numIndices;
    CalculateVertexIndexBufferSize( &numVertices, &numIndices, Shapes_NumLines, detail::ShapeBufferType_Triangle );
    bool ret = false;
    ret = meshBufferTop.Initialize( pRenderer->GetGpuBuffer(), numIndices, numIndices, nns::gfx::PrimitiveRenderer::VertexFormat_Pos );
    ret = meshBufferBottom.Initialize( pRenderer->GetGpuBuffer(), numIndices, numIndices, nns::gfx::PrimitiveRenderer::VertexFormat_Pos );

    CalculateVertexIndexBufferSize( &numVertices, &numIndices, Shapes_NumLines, detail::ShapeBufferType_Quad );
    ret = meshBufferSide.Initialize( pRenderer->GetGpuBuffer(), numIndices, numIndices, nns::gfx::PrimitiveRenderer::VertexFormat_Pos );

    CalculateVertexIndexBufferSize( &numVertices, &numIndices, Shapes_NumLines+1, detail::ShapeBufferType_Line );
    ret = meshBufferSideWired.Initialize( pRenderer->GetGpuBuffer(), numIndices, numIndices, nns::gfx::PrimitiveRenderer::VertexFormat_Pos );

    if ( !ret ) return;    // meshBuffer初期化が出来ない場合は処理を終了して戻る。

    nn::util::Float3* pVertexTop       = static_cast<nn::util::Float3*>( meshBufferTop.GetVertexBufferCpuAddress(  nns::gfx::PrimitiveRenderer::VertexAttribute::VertexAttribute_Pos ) );
    uint32_t*         pIndexTop        = meshBufferTop.GetIndexBufferCpuAddress();

    nn::util::Float3* pVertexBottom    = static_cast<nn::util::Float3*>( meshBufferBottom.GetVertexBufferCpuAddress(  nns::gfx::PrimitiveRenderer::VertexAttribute::VertexAttribute_Pos ) );
    uint32_t*         pIndexBottom     = meshBufferBottom.GetIndexBufferCpuAddress();

    nn::util::Float3* pVertexSide      = static_cast<nn::util::Float3*>( meshBufferSide.GetVertexBufferCpuAddress(  nns::gfx::PrimitiveRenderer::VertexAttribute::VertexAttribute_Pos ) );
    uint32_t*         pIndexSide       = meshBufferSide.GetIndexBufferCpuAddress();

    nn::util::Float3* pVertexSideWired = static_cast<nn::util::Float3*>( meshBufferSideWired.GetVertexBufferCpuAddress(  nns::gfx::PrimitiveRenderer::VertexAttribute::VertexAttribute_Pos ) );
    uint32_t*         pIndexSideWired  = meshBufferSideWired.GetIndexBufferCpuAddress();

    int verNumLine  = 0;
    int idxPosLine  = 0;
    int verNumTri   = 0;
    int idxPosTri   = 0;
    int verNumQuad  = 0;
    int idxPosQuad  = 0;

    // 頂点データの作成
    for ( int i = 0; i < detail::Shapes_NumLines; i++ )
    {
        float rot = minLatitude * (i + 1) + sweepStart;

        CalculatePos( &endPos, rot, scale );

        // top
        pVertexTop[verNumTri+0].x = 0;
        pVertexTop[verNumTri+0].y = beginPos.y;
        pVertexTop[verNumTri+0].z = 0;

        pVertexTop[verNumTri+1]   = beginPos;

        pVertexTop[verNumTri+2]   = endPos;
        pIndexTop[idxPosTri+0]    = 2 + (uint32_t)verNumTri; pIndexTop[idxPosTri+1] = 0 + (uint32_t)verNumTri; pIndexTop[idxPosTri+2] = 1 + (uint32_t)verNumTri;

        // bottom
        pVertexBottom[verNumTri+0].x = 0;
        pVertexBottom[verNumTri+0].y = -beginPos.y;
        pVertexBottom[verNumTri+0].z = 0;

        pVertexBottom[verNumTri+1].x = beginPos.x;
        pVertexBottom[verNumTri+1].y = -beginPos.y;
        pVertexBottom[verNumTri+1].z = beginPos.z;

        pVertexBottom[verNumTri+2].x = endPos.x;
        pVertexBottom[verNumTri+2].y = -endPos.y;
        pVertexBottom[verNumTri+2].z = endPos.z;
        pIndexBottom[idxPosTri+0] = 2 + (uint32_t)verNumTri; pIndexBottom[idxPosTri+1] = 0 + (uint32_t)verNumTri; pIndexBottom[idxPosTri+2] = 1 + (uint32_t)verNumTri;

        // 側面
        pVertexSide[verNumQuad+0] = beginPos;

        pVertexSide[verNumQuad+1].x = beginPos.x;
        pVertexSide[verNumQuad+1].y = -beginPos.y;
        pVertexSide[verNumQuad+1].z = beginPos.z;

        pVertexSide[verNumQuad+2] = endPos;

        pVertexSide[verNumQuad+3].x = endPos.x;
        pVertexSide[verNumQuad+3].y = -endPos.y;
        pVertexSide[verNumQuad+3].z = endPos.z;
        pIndexSide[idxPosQuad+0] = 0 + (uint32_t)verNumQuad; pIndexSide[idxPosQuad+1] = 2 + (uint32_t)verNumQuad; pIndexSide[idxPosQuad+2] = 1 + (uint32_t)verNumQuad;
        pIndexSide[idxPosQuad+3] = 1 + (uint32_t)verNumQuad; pIndexSide[idxPosQuad+4] = 3 + (uint32_t)verNumQuad; pIndexSide[idxPosQuad+5] = 2 + (uint32_t)verNumQuad;

        // 側面(ワイヤー）
        pVertexSideWired[verNumLine+0] = beginPos;

        pVertexSideWired[verNumLine+1].x = beginPos.x;
        pVertexSideWired[verNumLine+1].y = -beginPos.y;
        pVertexSideWired[verNumLine+1].z = beginPos.z;
        pIndexSideWired[idxPosLine+0] = 0 + (uint32_t)idxPosLine; pIndexSideWired[idxPosLine+1] = 1 + (uint32_t)idxPosLine;

        beginPos = endPos;

        idxPosLine += 2;
        verNumLine += 2;
        idxPosTri  += 3;
        verNumTri  += 3;
        idxPosQuad += (3 * 2);
        verNumQuad += 4;
    }

    // 側面(ワイヤー）最後のライン
    pVertexSideWired[verNumLine+0] = beginPos;

    pVertexSideWired[verNumLine+1].x = beginPos.x;
    pVertexSideWired[verNumLine+1].y = -beginPos.y;
    pVertexSideWired[verNumLine+1].z = beginPos.z;
    pIndexSideWired[idxPosLine+0] = 0 + (uint32_t)idxPosLine; pIndexSideWired[idxPosLine+1] = 1 + (uint32_t)idxPosLine;

    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );

    pRenderer->SetModelMatrix( &matrixSrt );
    pRenderer->SetLineWidth(1.f);
    pRenderer->SetColor( m_Color );

    // ポリゴンの描画
    if ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_CylinderFill )
    {
        pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBufferTop );
        pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBufferBottom );
    }
    pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBufferSide );

    // ワイヤーの描画
    if ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_CylinderFill )
    {
        pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_LineList, &meshBufferTop );
        pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_LineList, &meshBufferBottom );
    }
    pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_LineList, &meshBufferSideWired );

    // 円柱(ボリューム)の空洞率の描画
    if ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_CylinderFill )
    {
        nn::util::Matrix4x3fType esetSrt;
        CalculateHoleScale( pEmitter, &esetSrt, false );

        pRenderer->SetModelMatrix( &esetSrt );
        pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBufferTop );
        pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBufferBottom );
        pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBufferSide );
    }

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  キューブ形状描画処理を行います。
//---------------------------------------------------------------------------
void DrawShapes::DrawCube( nn::gfx::CommandBuffer* pCommandBuffer,
                           nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                           const nn::vfx::Emitter* pEmitter,
                           const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    const nn::vfx::detail::ResEmitterVolume* pEmitterVolume = &pEmitter->GetResEmitter()->volume;

    float scaleX = pEmitterVolume->volumeRadius.v[ 0 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 0 ] * 2.0f;
    float scaleY = pEmitterVolume->volumeRadius.v[ 1 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 1 ] * 2.0f;
    float scaleZ = pEmitterVolume->volumeRadius.v[ 2 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 2 ] * 2.0f;
    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, scaleY, scaleZ );

    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );

    pRenderer->SetModelMatrix( &matrixSrt );
    pRenderer->SetColor( m_Color );
    pRenderer->SetLineWidth(1.f);

    nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
    pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, scale );  // ポリゴン描画
    pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Wired, center, scale );  // ワイヤー描画

    // 立方体球(ボリューム)の空洞率の描画
    if ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_BoxFill )
    {
        nn::util::Matrix4x3fType esetSrt;
        CalculateHoleScale( pEmitter, &esetSrt, false );

        pRenderer->SetModelMatrix( &esetSrt );
        pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, scale );
    }

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
 }

//---------------------------------------------------------------------------
//  矩形形状描画処理を行います。
//---------------------------------------------------------------------------
void DrawShapes::DrawRectangle( nn::gfx::CommandBuffer* pCommandBuffer,
                                nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                const nn::vfx::Emitter* pEmitter,
                                const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    const nn::vfx::detail::ResEmitterVolume* pEmitterVolume = &pEmitter->GetResEmitter()->volume;

    float scaleX = pEmitterVolume->volumeRadius.v[ 0 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 0 ] * 2;
    float scaleZ = pEmitterVolume->volumeRadius.v[ 2 ] * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 2 ] * 2;

    nn::util::Uint8x4 white = { { 255, 255, 255, 255 } };
    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );

    pRenderer->SetModelMatrix( &matrixSrt );
    pRenderer->SetColor( white );
    pRenderer->SetLineWidth( 1.f );

    nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
    nn::util::Vector3fType scale;
    nn::util::VectorSet( &scale, 0.2f,  0.0f, scaleZ );
    nn::util::VectorSet( &center, scaleX/2,  0.0f, 0.0f );
    pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, scale );  // ポリゴン描画

    nn::util::VectorSet( &scale, 0.2f,  0.0f, scaleZ );
    nn::util::VectorSet( &center, -scaleX/2,  0.0f, 0.0f );
    pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, scale );  // ポリゴン描画

    nn::util::VectorSet( &scale, scaleX,  0.0f, 0.2f );
    nn::util::VectorSet( &center, 0.0f,  0.0f, scaleZ/2 );
    pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, scale );  // ポリゴン描画

    nn::util::VectorSet( &scale, scaleX,  0.0f, 0.2f );
    nn::util::VectorSet( &center, 0.0f,  0.0f, -scaleZ/2 );
    pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, scale );  // ポリゴン描画

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  ライン形状描画処理を行います。
//---------------------------------------------------------------------------
void DrawShapes::DrawLine( nn::gfx::CommandBuffer* pCommandBuffer,
                           nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                           const nn::vfx::Emitter* pEmitter,
                           const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    const nn::vfx::detail::ResEmitterVolume* pEmitterVolume = &pEmitter->GetResEmitter()->volume;
    float scaleZ = pEmitterVolume->lineLength * pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 2 ];

    nn::util::Uint8x4 white = { { 255, 255, 255, 255 } };
    pRenderer->SetColor( white );

    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );
    pRenderer->SetModelMatrix( &matrixSrt );

    nn::util::Vector3fType center = { 0.f, 0.f, 0.f };

    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( 0.2f, 0, scaleZ );
    pRenderer->DrawCube( pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, scale );  // ポリゴン描画

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  プリミティブ形状描画処理を行います。
//---------------------------------------------------------------------------
void DrawShapes::DrawPrimitive( nn::gfx::CommandBuffer* pCommandBuffer,
                                nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                const nn::vfx::Emitter* pEmitter,
                                const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
    nn::vfx::Primitive*    pPrimitive  = pEmitter->GetEmitterResource()->m_pVolumePrimitive;
    if ( pPrimitive == NULL ) return;

    float scaleX = pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 0 ];
    float scaleY = pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 1 ];
    float scaleZ = pEmitter->GetEmitterAnimValue().emitterVolumeScale.v[ 2 ];
    nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( scaleX, scaleY, scaleZ );

    nn::util::Matrix4x3fType emitterMatrixSrt;
    GetMatrix( pEmitter, &emitterMatrixSrt );

    nn::util::Matrix4x3fType scalMatrix;
    nn::util::MatrixIdentity( &scalMatrix );
    nn::util::MatrixSetScale( &scalMatrix, scale );

    nn::util::Matrix4x3fType  matrixSrt;
    nn::util::MatrixMultiply( &matrixSrt, scalMatrix, emitterMatrixSrt );

    int numVertices = pPrimitive->GetVertexBufferArrayCount();
    int numIndices  = pPrimitive->GetIndexCount();

    // プリミティブポリゴン描画
    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBuffer;
    bool ret = meshBuffer.Initialize( pRenderer->GetGpuBuffer(), numVertices, numIndices, nns::gfx::PrimitiveRenderer::VertexFormat_Pos );
    if ( !ret ) return;            // meshBuffer初期化が出来ない場合は処理を終了して戻る。

    nn::util::Float3* pVertex = static_cast<nn::util::Float3*>( meshBuffer.GetVertexBufferCpuAddress(  nns::gfx::PrimitiveRenderer::VertexAttribute::VertexAttribute_Pos ) );
    uint32_t*         pIndex  = meshBuffer.GetIndexBufferCpuAddress();

    // 頂点バッファの設定
    const nn::util::Float4* vertex = pPrimitive->GetVertexBuffer();
    if ( vertex )
    {
        for ( int i = 0; i < numVertices; i++ )
        {
            pVertex[i].x = vertex[i].x;
            pVertex[i].y = vertex[i].y;
            pVertex[i].z = vertex[i].z;
        }
    }

    // インデックスバッファの設定
    const uint32_t* primIndex = static_cast<const uint32_t*>(pPrimitive->GetIndexBuffer());
    if ( primIndex )
    {
        memcpy( pIndex, primIndex , (int)(pPrimitive->GetIndexCount() * sizeof(uint32_t)) );
    }

    pRenderer->SetColor( m_Color );
    pRenderer->SetModelMatrix( &matrixSrt );
    pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_TriangleList, &meshBuffer );

    // プリミティブワイヤー描画
    int indexNum = 0;
    CalculateVertexIndexBufferSize( &numVertices, &indexNum, numIndices, detail::ShapeBufferType_Line );
    ret = meshBuffer.Initialize( pRenderer->GetGpuBuffer(), numVertices, indexNum, nns::gfx::PrimitiveRenderer::VertexFormat_Pos );
    if ( !ret ) return;               // meshBuffer初期化が出来ない場合は処理を終了して戻る。

    pVertex    = static_cast<nn::util::Float3*>( meshBuffer.GetVertexBufferCpuAddress(  nns::gfx::PrimitiveRenderer::VertexAttribute::VertexAttribute_Pos ) );
    pIndex     = meshBuffer.GetIndexBufferCpuAddress();
    int bufIdx = 0;

    for ( int i = 0; i < numIndices; i += 3 )
    {
        uint32_t idx = primIndex[i];
        pVertex[bufIdx].x = vertex[idx].x;
        pVertex[bufIdx].y = vertex[idx].y;
        pVertex[bufIdx].z = vertex[idx].z;
        pIndex[bufIdx]    = bufIdx;

        idx = primIndex[i+1];
        bufIdx++;
        pVertex[bufIdx].x = vertex[idx].x;
        pVertex[bufIdx].y = vertex[idx].y;
        pVertex[bufIdx].z = vertex[idx].z;
        pIndex[bufIdx]    = bufIdx;

        bufIdx++;
        pVertex[bufIdx].x = vertex[idx].x;
        pVertex[bufIdx].y = vertex[idx].y;
        pVertex[bufIdx].z = vertex[idx].z;
        pIndex[bufIdx]    = bufIdx;

        idx = primIndex[i+2];
        bufIdx++;
        pVertex[bufIdx].x = vertex[idx].x;
        pVertex[bufIdx].y = vertex[idx].y;
        pVertex[bufIdx].z = vertex[idx].z;
        pIndex[bufIdx]    = bufIdx;

        bufIdx++;
        pVertex[bufIdx].x = vertex[idx].x;
        pVertex[bufIdx].y = vertex[idx].y;
        pVertex[bufIdx].z = vertex[idx].z;
        pIndex[bufIdx]    = bufIdx;

        idx = primIndex[i];
        bufIdx++;
        pVertex[bufIdx].x = vertex[idx].x;
        pVertex[bufIdx].y = vertex[idx].y;
        pVertex[bufIdx].z = vertex[idx].z;
        pIndex[bufIdx]    = bufIdx;

        bufIdx++;
    }

    pRenderer->SetLineWidth(1.f);
    pRenderer->SetColor( m_Color );
    pRenderer->SetModelMatrix( &matrixSrt );
    pRenderer->DrawUserMesh( pCommandBuffer, nn::gfx::PrimitiveTopology_LineList, &meshBuffer );

    DrawAxis( pCommandBuffer, pRenderer, matrixSrt, cameraPos );
}

//---------------------------------------------------------------------------
//  注視点を描画
//---------------------------------------------------------------------------
void DrawShapes::DrawAxis( nn::gfx::CommandBuffer* pCommandBuffer,
                           nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                           const nn::util::Matrix4x3fType& esetSrt,
                           const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT
{
   // モデルマトリックスを設定(スクリーンに対して一定の大きさで表示)
    nn::util::Vector3fType pos;
    nn::util::MatrixGetAxisW( &pos, esetSrt );

    float scale = AjustScreenScaleSize( pos, cameraPos );
    const nn::util::Vector3fType scale1 = NN_UTIL_VECTOR_3F_INITIALIZER( scale, scale, scale );

    nn::util::Matrix4x3fType rtMatrix;
    nn::util::Matrix4x3fType scalMatrix;
    nn::util::MatrixIdentity( &scalMatrix );
    nn::util::MatrixSetScale( &scalMatrix, scale1 );
    nn::util::MatrixMultiply( &rtMatrix, scalMatrix, esetSrt );

    pRenderer->SetModelMatrix( &rtMatrix );

    // 軸の描画
    nn::util::Uint8x4 red   = { { 255, 0, 0, 255 } };
    nn::util::Uint8x4 blue  = { { 0, 0, 255, 255 } };
    nn::util::Uint8x4 green = { { 0, 255, 0, 255 } };
    nn::util::Vector3fType zeroVector;
    nn::util::VectorSet( &zeroVector, 0.f, 0.f, 0.f );
    nn::util::Vector3fType axisPos;
    nn::util::VectorSet( &axisPos, 1.f, 0.f, 0.0f );

    pRenderer->SetLineWidth( 4.f );
    pRenderer->SetColor( red );
    pRenderer->DrawLine( pCommandBuffer, zeroVector, axisPos );
    pRenderer->SetLineWidth( 4.f );
    nn::util::VectorSet( &axisPos, 0.f, 0.f, 1.f );
    pRenderer->SetColor( blue );
    pRenderer->DrawLine( pCommandBuffer, zeroVector, axisPos );
    pRenderer->SetLineWidth( 4.f );
    nn::util::VectorSet( &axisPos, 0.f, 1.f, 0.f );
    pRenderer->SetColor( green );
    pRenderer->DrawLine( pCommandBuffer, zeroVector, axisPos );
}

//------------------------------------------------------------------------------
//   最小角度幅、開始角度の計算。
//------------------------------------------------------------------------------
void DrawShapes::CalculateVolumeSweepParams( const nn::vfx::Emitter* pEmitter, const nn::vfx::detail::ResEmitterVolume* pEmitterVolume, float* minLatitude, float* sweepStart, int numFace ) NN_NOEXCEPT
{
    if ( ( ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Sphere ) &&
       ( pEmitterVolume->arcType == 1 ) ) ||
       ( ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereFill ) &&
       ( pEmitterVolume->arcType == 1 ) ) ||
       ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereEquallyDivided ) ||
       ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereEqually64Divided ) )
    {
        *minLatitude = pEmitterVolume->sweepLatitude / numFace;     // 最小角度幅
        *sweepStart  = pEmitterVolume->sweepStart - (pEmitterVolume->sweepLatitude / 2);  // 開始角度
    }
    else
    {
        *minLatitude = pEmitterVolume->sweepLongitude / numFace;     // 最小角度幅
        *sweepStart  = pEmitterVolume->sweepStart - (pEmitterVolume->sweepLongitude / 2);  // 開始角度
    }
}

//------------------------------------------------------------------------------
//  位置の計算。
//------------------------------------------------------------------------------
void DrawShapes::CalculatePos( nn::util::Float3* pos, float rot, nn::util::Vector3fType& scale ) NN_NOEXCEPT
{
    float sinV;
    float cosV;

    nn::util::SinCosEst( &sinV, &cosV, rot );
    pos->x = sinV * nn::util::VectorGetX( scale );                // sinV * scaleX;;
    pos->z = cosV * nn::util::VectorGetZ( scale );                // cosV * scaleZ;
    pos->y = nn::util::VectorGetY( scale );                       // scaleY;
}

void DrawShapes::CalculatePos( nn::util::Vector3fType* pos, float rot, nn::util::Vector3fType& scale ) NN_NOEXCEPT
{
    float sinV;
    float cosV;

    nn::util::SinCosEst( &sinV, &cosV, rot );

    nn::util::VectorSet( pos,
                         sinV * nn::util::VectorGetX( scale ),
                         nn::util::VectorGetY( scale ),
                         cosV * nn::util::VectorGetZ( scale ) );
}

//------------------------------------------------------------------------------
//  球を緯度の軸に合うように回転します。
//------------------------------------------------------------------------------
void DrawShapes::RotateDirection( const nn::vfx::detail::ResEmitterVolume* pEmitterVolume, nn::util::Vector3fType* pOutDir ) NN_NOEXCEPT
{
    static const int LatitudeDir_PlusX   = 0;
    static const int LatitudeDir_MinusX  = 1;
    static const int LatitudeDir_PlusY   = 2;
    static const int LatitudeDir_MinusY  = 3;
    static const int LatitudeDir_PlusZ   = 4;
    static const int LatitudeDir_MinusZ  = 5;

    nn::util::Vector3fType latitudeDir = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );
    switch( pEmitterVolume->volumeLatitudeDir )
    {
    case LatitudeDir_PlusX:  nn::util::VectorSetX( &latitudeDir, 1.0f ); break;
    case LatitudeDir_MinusX: nn::util::VectorSetX( &latitudeDir, -1.0f ); break;
    case LatitudeDir_PlusY:  nn::util::VectorSetY( &latitudeDir, 1.0f ); break;
    case LatitudeDir_MinusY: nn::util::VectorSetY( &latitudeDir, -1.0f ); break;
    case LatitudeDir_PlusZ:  nn::util::VectorSetZ( &latitudeDir, 1.0f ); break;
    case LatitudeDir_MinusZ: nn::util::VectorSetZ( &latitudeDir, -1.0f ); break;
    default: break;
    }

    if( nn::util::VectorGetX( latitudeDir ) != 0.0f ||
        nn::util::VectorGetY( latitudeDir ) != 1.0f ||
        nn::util::VectorGetZ( latitudeDir ) != 0.0f )
    {
        // (0, 1, 0)を指定方向に回転させる回転行列を生成し適用する
        nn::util::Vector4fType q;
        nn::util::Vector3fType axis = NN_UTIL_VECTOR_3F_INITIALIZER( 0.0f, 1.0f, 0.0f );
        nn::util::QuaternionMakeVectorRotation( &q, axis, latitudeDir );
        nn::util::Matrix4x3fType qMatrix;
        nn::util::MatrixFromQuaternion( &qMatrix, q );
        nn::util::VectorTransform( pOutDir, *pOutDir, qMatrix );
    }
}

//------------------------------------------------------------------------------
//  空洞率のスケールの計算。
//------------------------------------------------------------------------------
 void DrawShapes::CalculateHoleScale( const nn::vfx::Emitter* pEmitter, nn::util::Matrix4x3fType *esetSrt, bool isYscaleFix ) NN_NOEXCEPT
{
    // 空洞率からスケールを設定
    float inner = 1.0f - pEmitter->GetResEmitter()->volume.caliberRatio;
    nn::util::Vector3fType scale1;
    if ( isYscaleFix == true )
    {
       nn::util::VectorSet( &scale1, inner, 1.0f, inner );
    }
    else
    {
       nn::util::VectorSet( &scale1, inner, inner, inner );
    }

    nn::util::Matrix4x3fType scalMatrix;
    nn::util::MatrixIdentity( &scalMatrix );
    nn::util::MatrixSetScale( &scalMatrix, scale1 );

    nn::util::Matrix4x3fType matrixSrt;
    GetMatrix( pEmitter, &matrixSrt );

    nn::util::MatrixMultiply( esetSrt, scalMatrix, matrixSrt );
}

//------------------------------------------------------------------------------
//  球状の位置の計算。
//------------------------------------------------------------------------------
void DrawShapes::PrimitiveSphere( const nn::vfx::Emitter* pEmitter, nn::util::Float3* pos, float minRot, float volumeSweepStart, int vLine, int hLine, nn::util::Vector3fType& scale ) NN_NOEXCEPT
{
    float rotTheta, rotPhi;
    float sinV[2] , cosV[2];
    nn::util::Float3 dir;
    bool isArcType = false;

    const nn::vfx::detail::ResEmitterVolume* pEmitterVolume = &pEmitter->GetResEmitter()->volume;

    if ( ( pEmitterVolume->volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_Sphere ) && ( pEmitterVolume->arcType == 1 ) )
    {
        isArcType = true;
    }

    if ( ( pEmitterVolume->volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereFill ) && ( pEmitterVolume->arcType == 1 ) )
    {
        isArcType = true;
    }

    if ( ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereEquallyDivided ) ||
       ( pEmitter->GetResEmitter()->volume.volumeType == nn::vfx::detail::EmitterVolumeType::EmitterVolumeType_SphereEqually64Divided ) )
    {
        isArcType = true;
    }

    // 角度計算
    if ( isArcType )
    {
        rotPhi   = (minRot * hLine) - nn::util::FloatPi/2.0f;
        rotTheta = (nn::util::FloatPi*2.0f/detail::Shapes_NumShapes) * vLine + volumeSweepStart;
    }
    else
    {
        rotTheta = minRot * vLine + volumeSweepStart;
        rotPhi   = ((nn::util::FloatPi*2.0f/detail::Shapes_NumShapes) * hLine)/2 - nn::util::FloatPi/2.0f;
    }
    nn::util::SinCosEst( &sinV[0] ,&cosV[0] ,rotTheta );
    nn::util::SinCosEst( &sinV[1] ,&cosV[1] ,rotPhi );

    // 方向
    dir.x = (sinV[0] * cosV[1]);
    dir.z = (cosV[0] * cosV[1]);
    dir.y = -sinV[1];

    if ( isArcType )
    {
        nn::util::Vector3fType dirVector3 = NN_UTIL_VECTOR_3F_INITIALIZER( dir.x, dir.y, dir.z );
        RotateDirection( pEmitterVolume, &dirVector3 );
        dir.x = nn::util::VectorGetX( dirVector3 );
        dir.y = nn::util::VectorGetY( dirVector3 );
        dir.z = nn::util::VectorGetZ( dirVector3 );
    }

    // 位置
    pos->x = dir.x * scale._v[0];
    pos->y = dir.y * scale._v[1];
    pos->z = dir.z * scale._v[2];
}

//---------------------------------------------------------------------------
//  スクリーンのサイズに変換
//---------------------------------------------------------------------------
float DrawShapes::AjustScreenScaleSize( nn::util::Vector3fType& pos, const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT
{
    nn::util::Vector3fType diff;
    nn::util::VectorSubtract( &diff, pos, cameraPos );

    float x = nn::util::VectorGetX( diff );
    float y = nn::util::VectorGetY( diff );
    float z = nn::util::VectorGetZ( diff );
    float length3 = x * x + y * y + z * z;

    return static_cast<float>( ::std::sqrtf( length3 ) * 0.05 );
}

//---------------------------------------------------------------------------
//  頂点、インデックスバッファサイズの計算
//---------------------------------------------------------------------------
void DrawShapes::CalculateVertexIndexBufferSize( int* numVertices, int* numIndices, int numFace, ShapeBufferType bufferType ) NN_NOEXCEPT
{
    int vertex   = 2;
    int triangle = 1;

    switch (bufferType)
    {
        case ShapeBufferType_Line:
            vertex   = 2;
            triangle = 1;
        break;
        case ShapeBufferType_Triangle:
            vertex   = 3;
            triangle = 1;
        break;
        case ShapeBufferType_Quad:
            vertex   = 4;
        break;
    };

    *numVertices = vertex * numFace;
    if ( bufferType == ShapeBufferType_Quad)
    {
        *numIndices  = numFace * ( 3 * 2 );
    }
    else
    {
        *numIndices  = *numVertices * triangle;
    }
}

//------------------------------------------------------------------------------
//  マトリックスを取得する。
//------------------------------------------------------------------------------
void DrawShapes::GetMatrix( const nn::vfx::Emitter* pEmitter,  nn::util::Matrix4x3fType* matrixSrt) NN_NOEXCEPT
{
    // モデルマトリックスを設定
    *matrixSrt = pEmitter->GetMatrixSrt();
}

}  // namespace detail
