﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <nn/nn_Log.h>
#include <nn/util/util_BitUtil.h>
#include <nn/perf/perf_LoadMeter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/util/util_MathTypes.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/font/font_TextWriterBase.h>
#include <nn/font.h>

namespace nn
{
namespace perf
{

//! @brief        負荷メーターを描画するクラスです。
class MeterDrawer
{
public:
    //! @brief        コンストラクタです。
    //! @param[in]    frameMeter フレーム全体時間を計測する負荷メーターです。
    explicit MeterDrawer() NN_NOEXCEPT
      : m_TextWriter( NULL ),
        m_Width( 1260.f ),
        m_Height( 48.f ),
        m_BarHeight( 16.f ),
        m_FontSize( 14.f ),
        m_SectionNum( 0 ),
        m_DrawThreshold( 1.f ),
        m_CurrentSectionNum( 0 ),
        m_AdjustTime( 60 ),
        m_AdjustCounter( m_AdjustTime ),
        m_MinSectionNum( 1 ),
        m_MaxSectionSeparatorNum( 4 ),
        m_Visible( true ),
        m_TextVisible( true ),
        m_BorderColor( nn::util::Color4u8::White() ),
        m_TextColor( nn::util::Color4u8::White() )
    {
        m_Position.x = 10.f;
        m_Position.y = 10.f;
        SetFrameRate( 60.f );
    }

    //! @brief        デストラクタです。
    ~MeterDrawer() NN_NOEXCEPT
    {}

    //! @brief    負荷メーター全体の表示/非表示を設定します。
    //! @param[in]    visible   true を指定した場合、負荷メーターを表示します。
    void SetVisible( bool visible ) NN_NOEXCEPT
    {
        m_Visible = visible;
    }

    //! @brief    負荷メーターの表示/非表示状態を取得します。
    //! @return   表示状態の場合、 true を返します。
    bool IsVisible() const NN_NOEXCEPT
    {
        return m_Visible;
    }

    //! @brief    各負荷メーターの名前や処理率の表示/非表示を設定します。
    //! @param[in]    visible   true を指定した場合、メーター名や処理率を表示します。
    void SetTextVisible( bool visible ) NN_NOEXCEPT
    {
        m_TextVisible = visible;
    }

    //! @brief    各負荷メーターの名前や処理率の表示/非表示状態を取得します。
    //! @return   表示状態の場合、 true を返します。
    bool IsTextVisible() const NN_NOEXCEPT
    {
        return m_TextVisible;
    }

    //! @brief        文字描画に用いる DevTextWriter を設定します。
    //! @param[in]    textWriter 設定する DevTextWriter です。
    void SetTextWriter( nn::font::TextWriter* textWriter ) NN_NOEXCEPT
    {
        m_TextWriter = textWriter;
    }

    //! @brief        設定されている DevTextWriter を取得します。
    //! @return       設定されている DevTextWriter を返します。
    nn::font::TextWriter* GetTextWriter() const NN_NOEXCEPT
    {
        return m_TextWriter;
    }

    //! @brief        負荷メーターの表示のために用いるフレームレートを設定します。
    //! @param[in]    frameRate 設定するフレームレートです。
    void SetFrameRate( float frameRate ) NN_NOEXCEPT
    {
        m_ReferenceFrame = nn::TimeSpan::FromMicroSeconds( static_cast<int64_t>( 1000000.f / frameRate ) );
    }

    //! @brief        負荷メーターの描画位置を設定します。
    //! @param[in]    position       設定する描画位置です。
    void SetPosition( nn::util::Float2 position ) NN_NOEXCEPT
    {
        m_Position = position;
    }

    //! @brief        負荷メーターの描画位置を取得します。
    //! @return       描画位置を返します。
    const nn::util::Float2& GetPosition() const NN_NOEXCEPT
    {
        return m_Position;
    }

    //! @brief        負荷メーターの描画幅を設定します。
    //! @param[in]    width     設定する描画幅です。
    void SetWidth( float width ) NN_NOEXCEPT
    {
        m_Width = width;
    }

    //! @brief        負荷メーターの描画幅を取得します。
    //! @return       描画幅を返します。
    float GetWidth() const NN_NOEXCEPT
    {
        return m_Width;
    }

    //! @brief        負荷メーターの描画高さを設定します。
    //! @param[in]    width     設定する描画高さです。
    //! @details       SetMinimizeBarSize で 0 が設定されている場合に SetHeight で設定した値が使用されます。
    //!               各負荷メーターの高さは ( SetHeight で設定した値 / 負荷メーターの数 ) となります。
    void SetHeight( float height ) NN_NOEXCEPT
    {
        m_Height = height;
    }

    //! @brief        設定されている負荷メーターの描画高さを取得します。
    //! @return       描画高さを返します。
    float GetHeight() const NN_NOEXCEPT
    {
        return m_Height;
    }

    //! @brief        自動調整された負荷メーターの描画高さを取得します。
    //! @return       自動調整された描画高さを返します。
    float GetAdjustedHeight() const NN_NOEXCEPT;

    //! @brief        文字列のフォントサイズを指定します。
    //! @param[in]    size      設定するフォントサイズです。
    void SetFontSize( float size ) NN_NOEXCEPT
    {
        m_FontSize = size;
    }

    //! @brief        文字列のフォントサイズを取得します。
    //! @return       フォントサイズを返します。
    float GetFontSize() const NN_NOEXCEPT
    {
        return m_FontSize;
    }

    //! @brief        画面に表示するフレームの目盛りの数を設定します。
    //! @param[in]    sectionNum 目盛りの数です。
    //! @detai        sectionNum に 0 を指定した場合、すべての負荷メーターが表示できる
    //!               最低限の目盛り数に自動調整します。
    void SetSectionNum( uint32_t sectionNum ) NN_NOEXCEPT
    {
        m_SectionNum = sectionNum;
    }

    //! @brief        設定されているフレームの目盛りの数を取得します。
    //! @return       目盛りの数を返します。
    uint32_t GetSectionNum() const NN_NOEXCEPT
    {
        return m_SectionNum;
    }

    //! @brief        フレームを区切る線の最大本数を設定します。
    //! @param[in]    maxNum    最大本数です。
    //! @details       極端に処理落ちした際に、区切り線が大量に表示されて処理落ちが続く問題を回避します。
    //!               0 を指定すると最大数の制限を行いません。
    void SetMaxSectionSeparatorNum( uint32_t maxNum ) NN_NOEXCEPT
    {
        m_MaxSectionSeparatorNum = maxNum;
    }

    //! @brief        フレームを区切る線の最大本数を取得します。
    //! @return       最大本数を返します。
    uint32_t GetMaxSectionSeparatorNum() const NN_NOEXCEPT
    {
        return m_MaxSectionSeparatorNum;
    }

    //! @brief        目盛りの数が自動調整の場合、自動調整するまでのフレーム数を指定します。
    //! @param[in]    adjustMargin 自動調整するまでのフレーム数です。
    //! @details       自動調整するまでのフレーム数を指定することにより、
    //!               フレームの目盛り表示がチャタリングするのを防ぎます。
    void SetSectionNumAdjustMargin( int adjustMargin ) NN_NOEXCEPT
    {
        m_AdjustTime = adjustMargin;
    }

    //! @brief        目盛りの数を自動調整するまでのフレーム数を取得します。
    //! @return       フレーム数を返します。
    int GetSectionNumAdjustMargin() const NN_NOEXCEPT
    {
        return m_AdjustTime;
    }

    //! @brief        負荷メーター1つあたりの高さを設定します。
    //! @param[in]    barSize   負荷メーター1つあたりの高さです。
    void SetMinimizeBarSize( float barSize ) NN_NOEXCEPT
    {
        m_BarHeight = barSize;
    }

    //! @brief        負荷メーター1つあたりの高さを取得します。
    //! @return       負荷メーター1つあたりの高さを返します。
    float GetMinimizeBarSize() const NN_NOEXCEPT
    {
        return m_BarHeight;
    }

    //! @brief        枠線の色を設定します。
    //! @param[in]    color     枠線の色です。
    void SetBorderColor( const nn::util::Color4u8& color ) NN_NOEXCEPT
    {
        m_BorderColor = color;
    }

    //! @brief        枠線の色を取得します。
    //! @return       枠線の色を返します。
    const nn::util::Color4u8& GetBorderColor() const NN_NOEXCEPT
    {
        return m_BorderColor;
    }

    //! @brief        文字列描画の色を設定します。
    //! @param[in]    color     文字列描画の色です。
    void SetTextColor( const nn::util::Color4u8& color ) NN_NOEXCEPT
    {
        m_TextColor = color;
    }

    //! @brief        文字列描画の色を取得します。
    //! @return       文字列描画の色を返します。
    const nn::util::Color4u8& GetTextColor() const NN_NOEXCEPT
    {
        return m_TextColor;
    }

    //! @brief        負荷メーターの描画を行います。
    //! @param[in]    pCommendBuffer  描画コマンド格納先のコマンドバッファです
    //! @param[in]    renderer    　　PrimitiveRenderer です。
    //! @param[in]    rootMeter       描画する親負荷メーターです（直接の子負荷メータも描画されます）。
    //! @details      PrimitiveRenderer に設定する model, view, projection 行列を MatrixIdentity でデフォルト値に設定します。
    void Draw( nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* renderer, LoadMeterBase* frameMeter) NN_NOEXCEPT;

private:
    typedef nn::util::IntrusiveList<LoadMeterBase, nn::util::IntrusiveListBaseNodeTraits<LoadMeterBase>> LoadMeterList;

    //! @brief ある時刻tがバー上の座標でどこに来るか計算します。
    float CalculateTimeLinePosition( float maxWidth, uint32_t sectionNum, uint64_t time );

    //! @brief ある時間tのバー上での幅を計算します。
    float CalculateTimeLineWidth( float maxWidth, uint32_t sectionNum, uint64_t time );

    //! @brief 表示する必要がある目盛りの数を計算します。
    uint32_t CalculateMaxSectionNum();

    nn::font::TextWriter*           m_TextWriter;           //!< 文字列描画に用いる DevTextWriter です。

    nn::util::Float2   m_Position;                //!< 負荷メーターの位置です。
    float              m_Width;                   //!< 負荷メーターの幅です。
    float              m_Height;                  //!< 負荷メーターの高さです。
    float              m_BarHeight;               //!< 各負荷メーターの高さです。
    float              m_FontSize;                //!< 文字列のフォントサイズです。

    uint32_t           m_SectionNum;              //!< 何区切り分を表示するかです。 0 の場合は、自動で表示範囲を変えます。
    uint32_t           m_CurrentSectionNum;       //!< 自動区切りとき、現在の区切り数です。
    int                m_AdjustTime;              //!< 自動区切りの区切り数をアジャストするまでの時間です。
    int                m_AdjustCounter;           //!< あと何フレームでアジャストするかのカウンターです。
    uint32_t           m_MinSectionNum;           //!< 最低でもこの数だけ目盛りを表示します。
    uint32_t           m_MaxSectionSeparatorNum;  //!< 区切り線の最大本数です。

    nn::TimeSpan       m_ReferenceFrame;          //!< 1 フレームのチックです。
    float              m_DrawThreshold;           //!< メータを描画する最小の横幅です。

    bool               m_Visible;                 //!< 負荷メーター全体を表示するかです。
    bool               m_TextVisible;             //!< テキストを表示するかです。

    nn::util::Color4u8 m_BorderColor;             //!< 枠線の色です。
    nn::util::Color4u8 m_TextColor;               //!< 文字列色です。

    LoadMeterBase*     m_RootMeter;              //!< 親負荷メーターです。
};

} // namespace perf
} // namespace nns
