﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

 //------------------------------------------------------------------------------
 //  ビューア起動時セッティング
 //------------------------------------------------------------------------------
class ViewerLaunchArg
{
public:
    enum LaunchArgIndex
    {
        LaunchArgIndex_DepthMode            = 0,
        LaunchArgIndex_TextureOriginMode    = 1,
        LaunchArgIndex_SwapChainMode        = 2,
        LaunchArgIndex_TemporaryBufferSize  = 3,
        LaunchArgIndex_GpuBufferSize        = 4,
        LaunchArgIndex_DynamicBufferSize    = 5,
        LaunchArgIndex_StripeCount          = 6,
        LaunchArgIndex_SuperStripeCount     = 7,
        LaunchArgIndex_GridSize             = 8,
        LaunchArgIndex_IsEnabledPreciseGpuCounterMode = 9, // Experimental
    };

    enum DepthMode
    {
        DepthMode_Near_Is_Minus_W = 0,          //!< Depth -1 - 1
        DepthMode_Near_Is_Zero = 1,             //!< Depth  0 - 1
    };

    enum TextureOriginMode
    {
        TextureOriginMode_Lower_Left = 0,
        TextureOriginMode_Upper_Left = 1,
    };

    enum SwapChainMode
    {
        SwapChainMode_Linear = 0,
        SwapChainMode_Non_Linear = 1,
    };

    ViewerLaunchArg( int argc, char** argv ) NN_NOEXCEPT
    {
        m_DepathMode            = DepthMode_Near_Is_Minus_W;
        m_TextureOriginMode     = TextureOriginMode_Lower_Left;
        m_SwapCharinMode        = SwapChainMode_Linear;
        m_TemporaryBufferSize   = 1024 * 32;
        m_GpuBufferSize         = 1024 * 1024 * 16;
        m_DynamicBufferSize     = 1024 * 1024 * 16;
        m_StripeCount           = 128;
        m_SuperStripeCount      = 128;
        m_GridSize              = 10.f;
        m_IsEnabledPreciseGpuCounterMode = false;

        const char* viewerArg[] =
        {
            "--depth-mode",                     // LaunchArgIndex_DepthMode
            "--texture-origin",                 // LaunchArgIndex_TextureOriginMode
            "--swap-chain",                     // LaunchArgIndex_SwapChainMode
            "--temporary_buffer_size",          // LaunchArgIndex_TemporaryBufferSize
            "--gpu_buffer_size",                // LaunchArgIndex_GpuBufferSize
            "--dynamic_buffer_size",            // LaunchArgIndex_DynamicBufferSize
            "--stripe_count",                   // LaunchArgIndex_StripeCount
            "--super_stripe_count",             // LaunchArgIndex_SuperStripeCount
            "--grid_size",                      // LaunchArgIndex_GridSize
            "--enable_precise_gpu_counter_mode",// [Experimental] LaunchArgIndex_IsEnabledPreciseGpuCounterMode
        };

        int viewerArgCount = sizeof( viewerArg ) / sizeof( char* );

        for ( int viewerArgIndex = 0; viewerArgIndex < viewerArgCount; viewerArgIndex++ )
        {
            for ( int argIndex = 1; argIndex < argc; argIndex++ )
            {
                if ( strcmp( viewerArg[ viewerArgIndex ], argv[ argIndex ] ) == 0 )
                {
                    switch ( viewerArgIndex )
                    {
                    case LaunchArgIndex_DepthMode:
                    {
                        if ( strcmp( argv[ argIndex + 1 ], "near_is_minus_w" ) == 0 )
                        {
                            m_DepathMode = DepthMode_Near_Is_Minus_W;
                        }
                        else if ( strcmp( argv[ argIndex + 1 ], "near_is_zero" ) == 0 )
                        {
                            m_DepathMode = DepthMode_Near_Is_Zero;
                        }
                    }
                    break;

                    case LaunchArgIndex_TextureOriginMode:
                    {
                        if ( strcmp( argv[ argIndex + 1 ], "lower_left" ) == 0 )
                        {
                            m_TextureOriginMode = TextureOriginMode_Lower_Left;
                        }
                        else if ( strcmp( argv[ argIndex + 1 ], "upper_left" ) == 0 )
                        {
                            m_TextureOriginMode = TextureOriginMode_Upper_Left;
                        }
                    }
                    break;

                    case LaunchArgIndex_SwapChainMode:
                    {
                        if ( strcmp( argv[ argIndex + 1 ], "linear" ) == 0 )
                        {
                            m_SwapCharinMode = SwapChainMode_Linear;
                        }
                        else if ( strcmp( argv[ argIndex + 1 ], "non_linear" ) == 0 )
                        {
                            m_SwapCharinMode = SwapChainMode_Non_Linear;
                        }
                    }
                    break;

                    case LaunchArgIndex_TemporaryBufferSize:
                    {
                        m_TemporaryBufferSize = static_cast<size_t>( atoi( argv[ argIndex + 1 ] ) );

                    }
                    break;

                    case LaunchArgIndex_GpuBufferSize:
                    {
                        m_GpuBufferSize = static_cast< size_t >( atoi( argv[ argIndex + 1 ] ) );

                    }
                    break;

                    case LaunchArgIndex_DynamicBufferSize:
                    {
                        m_DynamicBufferSize = static_cast< size_t >( atoi( argv[ argIndex + 1 ] ) );

                    }
                    break;

                    case LaunchArgIndex_StripeCount:
                    {
                        m_StripeCount = atoi( argv[ argIndex + 1 ] );

                    }
                    break;

                    case LaunchArgIndex_SuperStripeCount:
                    {
                        m_SuperStripeCount = atoi( argv[ argIndex + 1 ] );

                    }
                    break;

                    case LaunchArgIndex_GridSize:
                    {
                        m_GridSize = static_cast< float >( atof( argv[ argIndex + 1 ] ) );
                    }
                    break;

                    case LaunchArgIndex_IsEnabledPreciseGpuCounterMode:
                    {
                        // MEMO: オプションが指定されたら強制 true
                        m_IsEnabledPreciseGpuCounterMode = true;
                    }
                    break;
                    }
                }
            }
        }

        NN_SDK_LOG( "LaunchArgIndex_DepthMode : %d\n", m_DepathMode );
        NN_SDK_LOG( "LaunchArgIndex_TextureOriginMode : %d\n", m_TextureOriginMode );
        NN_SDK_LOG( "LaunchArgIndex_SwapChainMode : %d\n", m_SwapCharinMode );
        NN_SDK_LOG( "LaunchArgIndex_TemporaryBufferSize : %d\n", m_TemporaryBufferSize );
        NN_SDK_LOG( "LaunchArgIndex_GpuBufferSize : %d\n", m_GpuBufferSize );
        NN_SDK_LOG( "LaunchArgIndex_DynamicBufferSize : %d\n", m_DynamicBufferSize );
        NN_SDK_LOG( "LaunchArgIndex_StripeCount : %d\n", m_StripeCount );
        NN_SDK_LOG( "LaunchArgIndex_SuperStripeCount : %d\n", m_SuperStripeCount );
        NN_SDK_LOG( "LaunchArgIndex_GridSize : %f\n", m_GridSize );
    }

    DepthMode GetDepthMode() const NN_NOEXCEPT
    {
        return m_DepathMode;
    }
    TextureOriginMode GetTextureOriginMode() const NN_NOEXCEPT
    {
        return m_TextureOriginMode;
    }
    SwapChainMode GetSwapChainMode() const NN_NOEXCEPT
    {
        return m_SwapCharinMode;
    }
    size_t GetTemporaryBufferSize() const NN_NOEXCEPT
    {
        return m_TemporaryBufferSize;
    }
    size_t GetGpuBufferSize() const NN_NOEXCEPT
    {
        return m_GpuBufferSize;
    }
    size_t GetDynamicBufferSize() const NN_NOEXCEPT
    {
        return m_DynamicBufferSize;
    }
    int GetStripeCount() const NN_NOEXCEPT
    {
        return m_StripeCount;
    }
    int GetSuperStripeCount() const NN_NOEXCEPT
    {
        return m_SuperStripeCount;
    }
    float GetGridSize() const NN_NOEXCEPT
    {
        return m_GridSize;
    }

    // Experimental
    bool IsEnabledPreciseGpuCounterMode() const
    {
        return m_IsEnabledPreciseGpuCounterMode;
    }

private:
    DepthMode           m_DepathMode;
    TextureOriginMode   m_TextureOriginMode;
    SwapChainMode       m_SwapCharinMode;

    size_t              m_TemporaryBufferSize;
    size_t              m_GpuBufferSize;
    size_t              m_DynamicBufferSize;
    int                 m_StripeCount;
    int                 m_SuperStripeCount;
    float               m_GridSize;
    bool                m_IsEnabledPreciseGpuCounterMode;   // Experimental
};

